/*
 * Decompiled with CFR 0.152.
 */
package ffx.potential.commands;

import ffx.crystal.Crystal;
import ffx.numerics.math.DoubleMath;
import ffx.potential.ForceFieldEnergy;
import ffx.potential.bonded.Atom;
import ffx.potential.bonded.Residue;
import ffx.potential.cli.PotentialCommand;
import ffx.potential.parsers.PDBFilter;
import ffx.potential.parsers.SystemFilter;
import ffx.potential.parsers.XYZFilter;
import ffx.utilities.FFXBinding;
import java.util.Arrays;
import java.util.List;
import java.util.stream.Collectors;
import org.apache.commons.math3.util.FastMath;
import picocli.CommandLine;

@CommandLine.Command(description={"Calculates nucleic acid torsions as well as information regarding the sugar pucker."}, name="NucleicAcidAnalysis")
public class NucleicAcidAnalysis
extends PotentialCommand {
    @CommandLine.Option(names={"--at", "--allTorsions"}, paramLabel="false", defaultValue="false", description={"Print all torsions and information."})
    private boolean allTorsions = false;
    @CommandLine.Parameters(arity="1", paramLabel="file", description={"The atomic coordinate file in PDB or XYZ or ARC format."})
    private String filename = null;
    private List<Residue> residues;

    public NucleicAcidAnalysis() {
    }

    public NucleicAcidAnalysis(FFXBinding binding) {
        super(binding);
    }

    public NucleicAcidAnalysis(String[] args) {
        super(args);
    }

    public NucleicAcidAnalysis run() {
        if (!this.init()) {
            return this;
        }
        this.activeAssembly = this.getActiveAssembly(this.filename);
        if (this.activeAssembly == null) {
            logger.info(this.helpString());
            return this;
        }
        ForceFieldEnergy forceFieldEnergy = this.activeAssembly.getPotentialEnergy();
        double[] x = new double[forceFieldEnergy.getNumberOfVariables()];
        forceFieldEnergy.getCoordinates(x);
        forceFieldEnergy.energy(x);
        List<String> nucleicAcidNames = Arrays.asList("DA", "DC", "DG", "DT", "DU", "A", "C", "G", "T", "U", "DAD", "DCY", "DGU", "DTY", "URA");
        this.residues = this.activeAssembly.getResidueList().stream().filter(residue -> nucleicAcidNames.contains(residue.getName().toUpperCase())).collect(Collectors.toList());
        SystemFilter systemFilter = this.potentialFunctions.getFilter();
        if (systemFilter instanceof XYZFilter || systemFilter instanceof PDBFilter) {
            int numModels = systemFilter.countNumModels();
            if (numModels == 1 && this.allTorsions) {
                this.printHeader();
                this.analyzeAndPrintResidues();
            } else {
                int frameNumber = 1;
                logger.info(String.format("\nFrame %d:", frameNumber));
                logger.info("Residue    Name     \u03c7          P         Sugar pucker");
                logger.info("-----------------------------------------------------");
                for (Residue residue2 : this.residues) {
                    Double chiTemp = NucleicAcidAnalysis.getDihedral(residue2, "O4'", "C1'", "N9", "C4");
                    double chi = chiTemp != null ? chiTemp : NucleicAcidAnalysis.getDihedral(residue2, "O4'", "C1'", "N1", "C2");
                    double P = NucleicAcidAnalysis.calculateP(residue2);
                    String sugarPucker = NucleicAcidAnalysis.determineSugarPucker(P);
                    logger.info(String.format("%-10s %-8s %-10s %-10s %-14s", residue2.getResidueNumber(), residue2.getName(), NucleicAcidAnalysis.formatValue(chi), NucleicAcidAnalysis.formatValue(P), sugarPucker));
                }
                ++frameNumber;
                while (systemFilter.readNext()) {
                    Crystal crystal = this.activeAssembly.getCrystal();
                    forceFieldEnergy.setCrystal(crystal);
                    forceFieldEnergy.getCoordinates(x);
                    forceFieldEnergy.energy(x);
                    this.residues = this.activeAssembly.getResidueList().stream().filter(residue -> nucleicAcidNames.contains(residue.getName().toUpperCase())).collect(Collectors.toList());
                    logger.info(String.format("\nFrame %d:", frameNumber));
                    logger.info("Residue    Name     \u03c7          P         Sugar pucker");
                    logger.info("-----------------------------------------------------");
                    for (Residue residue3 : this.residues) {
                        Double chiTemp = NucleicAcidAnalysis.getDihedral(residue3, "O4'", "C1'", "N9", "C4");
                        double chi = chiTemp != null ? chiTemp : NucleicAcidAnalysis.getDihedral(residue3, "O4'", "C1'", "N1", "C2");
                        double P = NucleicAcidAnalysis.calculateP(residue3);
                        String sugarPucker = NucleicAcidAnalysis.determineSugarPucker(P);
                        logger.info(String.format("%-10s %-8s %-10s %-10s %-14s", residue3.getResidueNumber(), residue3.getName(), NucleicAcidAnalysis.formatValue(chi), NucleicAcidAnalysis.formatValue(P), sugarPucker));
                    }
                    ++frameNumber;
                }
            }
        }
        return this;
    }

    private void printHeader() {
        logger.info("Residue    Name      V0         V1         V2         V3         V4         P          \u03bdmax       \u03c7          \u03b3          \u03b1          \u03b2          \u03b4          \u03b5          \u03b6          Sugar pucker                  Stage");
        logger.info("------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------");
    }

    private void analyzeAndPrintResidues() {
        for (Residue residue : this.residues) {
            Double v0 = NucleicAcidAnalysis.getDihedral(residue, "C2'", "C1'", "O4'", "C4'");
            Double v1 = NucleicAcidAnalysis.getDihedral(residue, "O4'", "C1'", "C2'", "C3'");
            Double v2 = NucleicAcidAnalysis.getDihedral(residue, "C1'", "C2'", "C3'", "C4'");
            Double v3 = NucleicAcidAnalysis.getDihedral(residue, "O4'", "C4'", "C3'", "C2'");
            Double v4 = NucleicAcidAnalysis.getDihedral(residue, "C1'", "O4'", "C4'", "C3'");
            Double chiTemp = NucleicAcidAnalysis.getDihedral(residue, "O4'", "C1'", "N9", "C4");
            Double chi = chiTemp != null ? chiTemp : NucleicAcidAnalysis.getDihedral(residue, "O4'", "C1'", "N1", "C2");
            Double gamma = NucleicAcidAnalysis.getDihedral(residue, "O5'", "C5'", "C4'", "C3'");
            Double alpha = NucleicAcidAnalysis.getDihedral(residue, "O3'", "P", "O5'", "C5'");
            Double beta = NucleicAcidAnalysis.getDihedral(residue, "P", "O5'", "C5'", "C4'");
            Double delta = NucleicAcidAnalysis.getDihedral(residue, "C5'", "C4'", "C3'", "O3'");
            Double epsilon = NucleicAcidAnalysis.getDihedral(residue, "C4'", "C3'", "O3'", "P");
            Double zeta = NucleicAcidAnalysis.getDihedral(residue, "C3'", "O3'", "P", "O5'");
            Double P = null;
            if (v0 != null && v1 != null && v3 != null && v4 != null && v2 != null) {
                P = NucleicAcidAnalysis.calculateP(v0, v1, v2, v3, v4);
            }
            Double nuMax = null;
            if (v2 != null && P != null) {
                nuMax = Math.abs(v2 / Math.cos(Math.toRadians(P)));
            }
            String type = NucleicAcidAnalysis.determineType(residue, P);
            String stage = NucleicAcidAnalysis.determineStage(delta, chi, P);
            logger.info(String.format("%-10s %-8s %-10s %-10s %-10s %-10s %-10s %-10s %-10s %-10s %-10s %-10s %-10s %-10s %-10s %-10s %-14s %-18s", residue.getResidueNumber(), residue.getName(), NucleicAcidAnalysis.formatValue(v0), NucleicAcidAnalysis.formatValue(v1), NucleicAcidAnalysis.formatValue(v2), NucleicAcidAnalysis.formatValue(v3), NucleicAcidAnalysis.formatValue(v4), NucleicAcidAnalysis.formatValue(P), NucleicAcidAnalysis.formatValue(nuMax), NucleicAcidAnalysis.formatValue(chi), NucleicAcidAnalysis.formatValue(gamma), NucleicAcidAnalysis.formatValue(alpha), NucleicAcidAnalysis.formatValue(beta), NucleicAcidAnalysis.formatValue(delta), NucleicAcidAnalysis.formatValue(epsilon), NucleicAcidAnalysis.formatValue(zeta), type, stage));
        }
    }

    private static Double getDihedral(Residue residue, String atom1, String atom2, String atom3, String atom4) {
        try {
            Atom a1 = residue.getAtomByName(atom1, true);
            Atom a2 = residue.getAtomByName(atom2, true);
            Atom a3 = residue.getAtomByName(atom3, true);
            Atom a4 = residue.getAtomByName(atom4, true);
            if (a1 != null && a2 != null && a3 != null && a4 != null) {
                return FastMath.toDegrees((double)DoubleMath.dihedralAngle((double[])a1.getXYZ(null), (double[])a2.getXYZ(null), (double[])a3.getXYZ(null), (double[])a4.getXYZ(null)));
            }
        }
        catch (Exception e) {
            logger.warning(String.format("Could not calculate dihedral for %s atoms %s-%s-%s-%s", residue, atom1, atom2, atom3, atom4));
        }
        return null;
    }

    private static String formatValue(Double value) {
        return value != null ? String.format("%.2f", value) : "N/A";
    }

    private static double calculateP(Residue residue) {
        Double v0 = NucleicAcidAnalysis.getDihedral(residue, "C2'", "C1'", "O4'", "C4'");
        Double v1 = NucleicAcidAnalysis.getDihedral(residue, "O4'", "C1'", "C2'", "C3'");
        Double v2 = NucleicAcidAnalysis.getDihedral(residue, "C1'", "C2'", "C3'", "C4'");
        Double v3 = NucleicAcidAnalysis.getDihedral(residue, "O4'", "C4'", "C3'", "C2'");
        Double v4 = NucleicAcidAnalysis.getDihedral(residue, "C1'", "O4'", "C4'", "C3'");
        if (v0 != null && v1 != null && v3 != null && v4 != null && v2 != null) {
            return NucleicAcidAnalysis.calculateP(v0, v1, v2, v3, v4);
        }
        return Double.NaN;
    }

    private static double calculateP(double v0, double v1, double v2, double v3, double v4) {
        try {
            double sin36 = Math.sin(Math.toRadians(36.0));
            double sin72 = Math.sin(Math.toRadians(72.0));
            double denominator = 2.0 * v2 * (sin36 + sin72);
            if (Math.abs(denominator) < 1.0E-10) {
                return Double.NaN;
            }
            double p = (v4 - v0 - (v3 - v1)) / denominator;
            double P = v2 < 0.0 ? Math.toDegrees(Math.atan(p)) + 180.0 : (p < 0.0 ? Math.toDegrees(Math.atan(p)) + 360.0 : Math.toDegrees(Math.atan(p)));
            return P;
        }
        catch (Exception e) {
            return Double.NaN;
        }
    }

    private static String determineType(Residue residue, Double P) {
        if (P == null) {
            return "Unknown";
        }
        String base = switch (residue.getName()) {
            case "DAD" -> "Ade";
            case "DGU" -> "Gua";
            case "DCY" -> "Cyt";
            case "DTY" -> "Thy";
            case "URA" -> "Ura";
            default -> "Unknown";
        };
        String sugarPucker = NucleicAcidAnalysis.determineSugarPucker(P);
        return base + ", " + sugarPucker;
    }

    private static String determineSugarPucker(Double P) {
        if (P == null) {
            return "Unknown";
        }
        if (P >= 0.0 && P < 36.0) {
            return "C3'-endo";
        }
        if (P >= 36.0 && P < 72.0) {
            return "C4'-endo";
        }
        if (P >= 72.0 && P < 108.0) {
            return "O4'-endo";
        }
        if (P >= 108.0 && P < 144.0) {
            return "C1'-exo";
        }
        if (P >= 144.0 && P < 180.0) {
            return "C2'-endo";
        }
        if (P >= 180.0 && P < 216.0) {
            return "C3'-exo";
        }
        if (P >= 216.0 && P < 252.0) {
            return "C4'-exo";
        }
        if (P >= 252.0 && P < 288.0) {
            return "O4'-exo";
        }
        if (P >= 288.0 && P < 324.0) {
            return "C1'-endo";
        }
        if (P >= 324.0 || P < 0.0) {
            return "C2'-endo";
        }
        return "Unknown";
    }

    private static String determineStage(Double delta, Double chi, Double P) {
        if (delta == null || chi == null || P == null) {
            return "Unknown";
        }
        if (delta > 120.0 && chi > -160.0) {
            return "Stage 1: Slide and Roll Change";
        }
        if (delta < 120.0 && P >= 0.0 && P < 180.0) {
            return "Stage 2: Backbone and Sugar Pucker Change";
        }
        if (delta < 120.0 && P >= 180.0 && P < 360.0) {
            return "Stage 3: Roll and Inclination to Finish Transition";
        }
        return "Unknown";
    }
}

