/*
 * Decompiled with CFR 0.152.
 */
package ffx.potential.bonded;

import ffx.numerics.math.ScalarMath;
import ffx.potential.MolecularAssembly;
import ffx.potential.bonded.AminoAcidUtils;
import ffx.potential.bonded.Angle;
import ffx.potential.bonded.Atom;
import ffx.potential.bonded.Bond;
import ffx.potential.bonded.BondedUtils;
import ffx.potential.bonded.MultiResidue;
import ffx.potential.bonded.NucleicAcidUtils;
import ffx.potential.bonded.Polymer;
import ffx.potential.bonded.Residue;
import ffx.potential.bonded.ResidueState;
import ffx.potential.bonded.Rotamer;
import ffx.potential.bonded.Torsion;
import ffx.potential.parameters.AngleType;
import ffx.potential.parameters.TitrationUtils;
import java.io.BufferedReader;
import java.io.File;
import java.io.FileReader;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.logging.Level;
import java.util.logging.Logger;
import javax.annotation.Nullable;
import org.apache.commons.math3.util.FastMath;

public class RotamerLibrary {
    private static final Logger logger = Logger.getLogger(RotamerLibrary.class.getName());
    private static final int numberOfAminoAcids = AminoAcidUtils.AminoAcid3.values().length;
    private static final int numberOfNucleicAcids = NucleicAcidUtils.NucleicAcid3.values().length;
    private static final ProteinLibrary DEFAULT_PROTEIN_LIB = ProteinLibrary.Richardson;
    private static final NucleicAcidLibrary DEFAULT_NA_LIB = NucleicAcidLibrary.RICHARDSON;
    private static final Map<String, NonstandardRotLibrary> nonstdRotCache = new HashMap<String, NonstandardRotLibrary>();
    private static final RotamerLibrary defaultRotamerLibrary = new RotamerLibrary(ProteinLibrary.PonderAndRichards, false);
    private static final boolean useIdealRingGeometries = Boolean.parseBoolean(System.getProperty("useIdealRingGeo", "true"));
    private static final Map<AminoAcidUtils.AminoAcid3, Map<String, Double>> idealAngleGeometries = Map.of(AminoAcidUtils.AminoAcid3.PHE, Map.of("CD1-CG-CB", 120.3, "CD2-CG-CB", 120.3, "CE1-CD1-CG", 120.3, "CE2-CD2-CG", 120.3, "CZ-CE1-CD1", 120.0, "CZ-CE2-CD2", 120.0), AminoAcidUtils.AminoAcid3.TYR, Map.of("CD1-CG-CB", 120.3, "CD2-CG-CB", 120.3, "CE1-CD1-CG", 120.3, "CE2-CD2-CG", 120.3, "CZ-CE1-CD1", 120.0, "CZ-CE2-CD2", 120.0), AminoAcidUtils.AminoAcid3.TYD, Map.of("CD1-CG-CB", 120.5, "CD2-CG-CB", 120.5, "CE1-CD1-CG", 120.4, "CE2-CD2-CG", 120.4, "CZ-CE1-CD1", 120.8, "CZ-CE2-CD2", 120.8), AminoAcidUtils.AminoAcid3.HIS, Map.of("ND1-CG-CB", 122.1, "CD2-CG-CB", 131.0, "CD2-CG-ND1", 106.8, "CE1-ND1-CG", 109.5, "NE2-CD2-CG", 107.1), AminoAcidUtils.AminoAcid3.HID, Map.of("ND1-CG-CB", 123.5, "CD2-CG-CB", 132.3, "CD2-CG-ND1", 104.2, "CE1-ND1-CG", 108.8, "NE2-CD2-CG", 111.2), AminoAcidUtils.AminoAcid3.HIE, Map.of("ND1-CG-CB", 120.2, "CD2-CG-CB", 129.1, "CD2-CG-ND1", 110.7, "CE1-ND1-CG", 105.1, "NE2-CD2-CG", 104.6), AminoAcidUtils.AminoAcid3.TRP, Map.of("CD1-CG-CB", 126.4, "CD2-CG-CB", 126.5, "CD2-CG-CD1", 107.1, "NE1-CD1-CG", 106.9, "CE2-NE1-CD1", 109.4, "CE3-CD2-CE2", 121.6, "CZ2-CE2-CD2", 123.5, "CZ3-CE3-CD2", 116.7, "CH2-CZ2-CE2", 116.2));
    private final Rotamer[][] aminoAcidRotamerCache = new Rotamer[numberOfAminoAcids][];
    private final Rotamer[][] nucleicAcidRotamerCache = new Rotamer[numberOfNucleicAcids][];
    private final ProteinLibrary proteinLibrary;
    private final NucleicAcidLibrary nucleicAcidLibrary;
    private boolean useOrigCoordsRotamer;

    public RotamerLibrary(boolean origCoords) {
        this(DEFAULT_PROTEIN_LIB, DEFAULT_NA_LIB, origCoords);
    }

    public RotamerLibrary(ProteinLibrary protLibrary, boolean origCoords) {
        this(protLibrary, DEFAULT_NA_LIB, origCoords);
    }

    public RotamerLibrary(NucleicAcidLibrary naLibrary, boolean origCoords) {
        this(DEFAULT_PROTEIN_LIB, naLibrary, origCoords);
    }

    public RotamerLibrary(ProteinLibrary protLibrary, NucleicAcidLibrary naLibrary, boolean origCoords) {
        this.proteinLibrary = protLibrary;
        this.nucleicAcidLibrary = naLibrary;
        this.useOrigCoordsRotamer = origCoords;
    }

    public static boolean addRotPatch(String rotFileName) {
        File rotPatchFile = new File(rotFileName);
        if (rotPatchFile.exists() && rotPatchFile.canRead()) {
            return RotamerLibrary.addRotPatch(new File(rotFileName));
        }
        return false;
    }

    public static void applyRotamer(Residue residue, Rotamer rotamer) {
        RotamerLibrary.applyRotamer(residue, rotamer, false);
    }

    public static void applyRotamer(Residue residue, Rotamer rotamer, boolean independent) {
        residue.setRotamer(rotamer);
        if (rotamer.isTitrating) {
            rotamer.updateParameters(residue);
        }
        if (rotamer.isState) {
            RotamerLibrary.applyState(residue, rotamer);
        } else {
            switch (residue.getResidueType()) {
                case AA: {
                    RotamerLibrary.applyAARotamer(residue, rotamer);
                    break;
                }
                case NA: {
                    RotamerLibrary.applyNARotamer(residue, rotamer, independent);
                    break;
                }
            }
        }
    }

    public static double[] applySugarPucker(Residue residue, NucleicSugarPucker pucker, boolean isDeoxy, boolean place) {
        double C2_SP_SOUTH_RNA = 24.2;
        double C3_SP_SOUTH_RNA = 357.7;
        double O3_SP_SOUTH_RNA = 268.1;
        double C2_SP_NORTH_RNA = 324.7;
        double C3_SP_NORTH_RNA = 20.4;
        double O3_SP_NORTH_RNA = 201.8;
        double C2_SP_SOUTH_DNA = 22.6;
        double C3_SP_SOUTH_DNA = 357.7;
        double O3_SP_SOUTH_DNA = 265.8;
        double C2_SP_NORTH_DNA = 327.7;
        double C3_SP_NORTH_DNA = 16.4;
        double O3_SP_NORTH_DNA = 205.4;
        double[] ret = new double[3];
        Atom C1s = (Atom)residue.getAtomNode("C1'");
        Atom C3s = (Atom)residue.getAtomNode("C3'");
        Atom O4s = (Atom)residue.getAtomNode("O4'");
        Atom C4s = (Atom)residue.getAtomNode("C4'");
        Atom O3s = (Atom)residue.getAtomNode("O3'");
        Bond C3s_C4s = C4s.getBond(C3s);
        double dC3s_C4s = C3s_C4s.bondType.distance;
        Angle C3s_C4s_O4s = O4s.getAngle(C4s, C3s);
        double dC3s_C4s_O4s = C3s_C4s_O4s.angleType.angle[C3s_C4s_O4s.nh];
        Bond C3s_O3s = C3s.getBond(O3s);
        double dC3s_O3s = C3s_O3s.bondType.distance;
        Angle C4s_C3s_O3s = C4s.getAngle(C3s, O3s);
        double dC4s_C3s_O3s = C4s_C3s_O3s.angleType.angle[C4s_C3s_O3s.nh];
        if (place) {
            Atom C2s = (Atom)residue.getAtomNode("C2'");
            Bond C3s_C2s = C3s.getBond(C2s);
            double dC3s_C2s = C3s_C2s.bondType.distance;
            Angle C4s_C3s_C2s = C4s.getAngle(C3s, C2s);
            double dC4s_C3s_C2s = C4s_C3s_C2s.angleType.angle[C4s_C3s_C2s.nh];
            if (isDeoxy) {
                if (pucker == NucleicSugarPucker.C3_ENDO) {
                    BondedUtils.intxyz(C3s, C4s, dC3s_C4s, O4s, dC3s_C4s_O4s, C1s, 16.4, 0);
                    BondedUtils.intxyz(C2s, C3s, dC3s_C2s, C4s, dC4s_C3s_C2s, O4s, 327.7, 0);
                    BondedUtils.intxyz(O3s, C3s, dC3s_O3s, C4s, dC4s_C3s_O3s, O4s, 205.4, 0);
                } else {
                    BondedUtils.intxyz(C3s, C4s, dC3s_C4s, O4s, dC3s_C4s_O4s, C1s, 357.7, 0);
                    BondedUtils.intxyz(C2s, C3s, dC3s_C2s, C4s, dC4s_C3s_C2s, O4s, 22.6, 0);
                    BondedUtils.intxyz(O3s, C3s, dC3s_O3s, C4s, dC4s_C3s_O3s, O4s, 265.8, 0);
                }
            } else if (pucker == NucleicSugarPucker.C3_ENDO) {
                BondedUtils.intxyz(C3s, C4s, dC3s_C4s, O4s, dC3s_C4s_O4s, C1s, 20.4, 0);
                BondedUtils.intxyz(C2s, C3s, dC3s_C2s, C4s, dC4s_C3s_C2s, O4s, 324.7, 0);
                BondedUtils.intxyz(O3s, C3s, dC3s_O3s, C4s, dC4s_C3s_O3s, O4s, 201.8, 0);
            } else {
                BondedUtils.intxyz(C3s, C4s, dC3s_C4s, O4s, dC3s_C4s_O4s, C1s, 357.7, 0);
                BondedUtils.intxyz(C2s, C3s, dC3s_C2s, C4s, dC4s_C3s_C2s, O4s, 24.2, 0);
                BondedUtils.intxyz(O3s, C3s, dC3s_O3s, C4s, dC4s_C3s_O3s, O4s, 268.1, 0);
            }
        } else {
            double[] C1sXYZ = new double[3];
            double[] O4sXYZ = new double[3];
            double[] C4sXYZ = new double[3];
            C1s.getXYZ(C1sXYZ);
            O4s.getXYZ(O4sXYZ);
            C4s.getXYZ(C4sXYZ);
            if (isDeoxy) {
                if (pucker == NucleicSugarPucker.C3_ENDO) {
                    double[] C3sXYZ = BondedUtils.determineIntxyz(C4sXYZ, dC3s_C4s, O4sXYZ, dC3s_C4s_O4s, C1sXYZ, 16.4, 0);
                    ret = BondedUtils.determineIntxyz(C3sXYZ, dC3s_O3s, C4sXYZ, dC4s_C3s_O3s, O4sXYZ, 205.4, 0);
                } else {
                    double[] C3sXYZ = BondedUtils.determineIntxyz(C4sXYZ, dC3s_C4s, O4sXYZ, dC3s_C4s_O4s, C1sXYZ, 357.7, 0);
                    ret = BondedUtils.determineIntxyz(C3sXYZ, dC3s_O3s, C4sXYZ, dC4s_C3s_O3s, O4sXYZ, 265.8, 0);
                }
            } else if (pucker == NucleicSugarPucker.C3_ENDO) {
                double[] C3sXYZ = BondedUtils.determineIntxyz(C4sXYZ, dC3s_C4s, O4sXYZ, dC3s_C4s_O4s, C1sXYZ, 20.4, 0);
                ret = BondedUtils.determineIntxyz(C3sXYZ, dC3s_O3s, C4sXYZ, dC4s_C3s_O3s, O4sXYZ, 201.8, 0);
            } else {
                double[] C3sXYZ = BondedUtils.determineIntxyz(C4sXYZ, dC3s_C4s, O4sXYZ, dC3s_C4s_O4s, C1sXYZ, 357.7, 0);
                ret = BondedUtils.determineIntxyz(C3sXYZ, dC3s_O3s, C4sXYZ, dC4s_C3s_O3s, O4sXYZ, 268.1, 0);
            }
        }
        return ret;
    }

    public static RotamerLibrary getDefaultLibrary() {
        return defaultRotamerLibrary;
    }

    public static void initializeDefaultAtomicCoordinates(Polymer[] polymers) {
        for (Polymer polymer : polymers) {
            List<Residue> current = polymer.getResidues();
            for (Residue residuej : current) {
                if (residuej.getResidueType() != Residue.ResidueType.NA) continue;
                residuej.initializeDefaultAtomicCoordinates();
            }
        }
    }

    public static int measureAARotamer(Residue residue, double[] chi, boolean print) {
        if (residue instanceof MultiResidue) {
            residue = ((MultiResidue)residue).getActive();
        }
        AminoAcidUtils.AminoAcid3 name = residue.getAminoAcid3();
        switch (name) {
            case VAL: {
                Atom N = (Atom)residue.getAtomNode("N");
                Atom CA = (Atom)residue.getAtomNode("CA");
                Atom CB = (Atom)residue.getAtomNode("CB");
                Atom CG1 = (Atom)residue.getAtomNode("CG1");
                for (Torsion torsion : residue.getTorsionList()) {
                    if (!torsion.compare(N, CA, CB, CG1)) continue;
                    chi[0] = torsion.measure();
                    if (!print) break;
                    logger.info(torsion.toString());
                    break;
                }
                return 1;
            }
            case LEU: {
                Atom N = (Atom)residue.getAtomNode("N");
                Atom CA = (Atom)residue.getAtomNode("CA");
                Atom CB = (Atom)residue.getAtomNode("CB");
                Atom CG = (Atom)residue.getAtomNode("CG");
                Atom CD1 = (Atom)residue.getAtomNode("CD1");
                for (Torsion torsion : residue.getTorsionList()) {
                    if (torsion.compare(N, CA, CB, CG)) {
                        chi[0] = torsion.measure();
                        if (print) {
                            logger.info(torsion.toString());
                        }
                    }
                    if (!torsion.compare(CA, CB, CG, CD1)) continue;
                    chi[1] = torsion.measure();
                    if (!print) continue;
                    logger.info(torsion.toString());
                }
                return 2;
            }
            case ILE: {
                Atom N = (Atom)residue.getAtomNode("N");
                Atom CA = (Atom)residue.getAtomNode("CA");
                Atom CB = (Atom)residue.getAtomNode("CB");
                Atom CD1 = (Atom)residue.getAtomNode("CD1");
                Atom CG1 = (Atom)residue.getAtomNode("CG1");
                for (Torsion torsion : residue.getTorsionList()) {
                    if (torsion.compare(N, CA, CB, CG1)) {
                        chi[0] = torsion.measure();
                        if (print) {
                            logger.info(torsion.toString());
                        }
                    }
                    if (!torsion.compare(CA, CB, CG1, CD1)) continue;
                    chi[1] = torsion.measure();
                    if (!print) continue;
                    logger.info(torsion.toString());
                }
                return 2;
            }
            case SER: {
                Atom N = (Atom)residue.getAtomNode("N");
                Atom CA = (Atom)residue.getAtomNode("CA");
                Atom CB = (Atom)residue.getAtomNode("CB");
                Atom OG = (Atom)residue.getAtomNode("OG");
                Atom HG = (Atom)residue.getAtomNode("HG");
                for (Torsion torsion : residue.getTorsionList()) {
                    if (torsion.compare(N, CA, CB, OG)) {
                        chi[0] = torsion.measure();
                        if (print) {
                            logger.info(torsion.toString());
                        }
                    }
                    if (!torsion.compare(CA, CB, OG, HG)) continue;
                    chi[1] = torsion.measure();
                    if (Double.isNaN(chi[1])) {
                        chi[1] = 180.0;
                    }
                    if (!print) continue;
                    logger.info(torsion.toString());
                }
                return 2;
            }
            case THR: {
                Atom N = (Atom)residue.getAtomNode("N");
                Atom CA = (Atom)residue.getAtomNode("CA");
                Atom CB = (Atom)residue.getAtomNode("CB");
                Atom OG1 = (Atom)residue.getAtomNode("OG1");
                Atom HG1 = (Atom)residue.getAtomNode("HG1");
                for (Torsion torsion : residue.getTorsionList()) {
                    if (torsion.compare(N, CA, CB, OG1)) {
                        chi[0] = torsion.measure();
                        if (print) {
                            logger.info(torsion.toString());
                        }
                    }
                    if (!torsion.compare(CA, CB, OG1, HG1)) continue;
                    chi[1] = torsion.measure();
                    if (Double.isNaN(chi[1])) {
                        chi[1] = 180.0;
                    }
                    if (!print) continue;
                    logger.info(torsion.toString());
                }
                return 2;
            }
            case CYS: 
            case CYX: 
            case CYD: {
                Atom N = (Atom)residue.getAtomNode("N");
                Atom CA = (Atom)residue.getAtomNode("CA");
                Atom CB = (Atom)residue.getAtomNode("CB");
                Atom SG = (Atom)residue.getAtomNode("SG");
                for (Torsion torsion : residue.getTorsionList()) {
                    if (!torsion.compare(N, CA, CB, SG)) continue;
                    chi[0] = torsion.measure();
                    if (!print) break;
                    logger.info(torsion.toString());
                    break;
                }
                return 1;
            }
            case PHE: {
                Atom N = (Atom)residue.getAtomNode("N");
                Atom CA = (Atom)residue.getAtomNode("CA");
                Atom CB = (Atom)residue.getAtomNode("CB");
                Atom CG = (Atom)residue.getAtomNode("CG");
                for (Torsion torsion : residue.getTorsionList()) {
                    if (!torsion.compare(N, CA, CB, CG)) continue;
                    chi[0] = torsion.measure();
                    if (!print) break;
                    logger.info(torsion.toString());
                    break;
                }
                return 1;
            }
            case PRO: {
                Atom N = (Atom)residue.getAtomNode("N");
                Atom CA = (Atom)residue.getAtomNode("CA");
                Atom CB = (Atom)residue.getAtomNode("CB");
                Atom CD = (Atom)residue.getAtomNode("CD");
                Atom CG = (Atom)residue.getAtomNode("CG");
                for (Torsion torsion : residue.getTorsionList()) {
                    if (torsion.compare(N, CA, CB, CG)) {
                        chi[0] = torsion.measure();
                        if (print) {
                            logger.info(torsion.toString());
                        }
                    }
                    if (!torsion.compare(CA, CB, CG, CD)) continue;
                    chi[1] = torsion.measure();
                    if (!print) continue;
                    logger.info(torsion.toString());
                }
                return 2;
            }
            case TYR: {
                Atom N = (Atom)residue.getAtomNode("N");
                Atom CA = (Atom)residue.getAtomNode("CA");
                Atom CB = (Atom)residue.getAtomNode("CB");
                Atom CD1 = (Atom)residue.getAtomNode("CD1");
                Atom CE2 = (Atom)residue.getAtomNode("CE2");
                Atom CG = (Atom)residue.getAtomNode("CG");
                Atom CZ = (Atom)residue.getAtomNode("CZ");
                Atom OH = (Atom)residue.getAtomNode("OH");
                Atom HH = (Atom)residue.getAtomNode("HH");
                for (Torsion torsion : residue.getTorsionList()) {
                    if (torsion.compare(N, CA, CB, CG)) {
                        chi[0] = torsion.measure();
                        if (print) {
                            logger.info(torsion.toString());
                        }
                    }
                    if (torsion.compare(CA, CB, CG, CD1)) {
                        chi[1] = torsion.measure();
                        if (print) {
                            logger.info(torsion.toString());
                        }
                    }
                    if (!torsion.compare(CE2, CZ, OH, HH)) continue;
                    chi[2] = torsion.measure();
                    if (Double.isNaN(chi[2])) {
                        chi[2] = 180.0;
                    }
                    if (!print) continue;
                    logger.info(torsion.toString());
                }
                return 3;
            }
            case TYD: 
            case TRP: {
                Atom N = (Atom)residue.getAtomNode("N");
                Atom CA = (Atom)residue.getAtomNode("CA");
                Atom CB = (Atom)residue.getAtomNode("CB");
                Atom CD1 = (Atom)residue.getAtomNode("CD1");
                Atom CG = (Atom)residue.getAtomNode("CG");
                for (Torsion torsion : residue.getTorsionList()) {
                    if (torsion.compare(N, CA, CB, CG)) {
                        chi[0] = torsion.measure();
                        if (print) {
                            logger.info(torsion.toString());
                        }
                    }
                    if (!torsion.compare(CA, CB, CG, CD1)) continue;
                    chi[1] = torsion.measure();
                    if (!print) continue;
                    logger.info(torsion.toString());
                }
                return 2;
            }
            case HIS: 
            case HIE: 
            case HID: {
                Atom N = (Atom)residue.getAtomNode("N");
                Atom CA = (Atom)residue.getAtomNode("CA");
                Atom CB = (Atom)residue.getAtomNode("CB");
                Atom CG = (Atom)residue.getAtomNode("CG");
                Atom ND1 = (Atom)residue.getAtomNode("ND1");
                for (Torsion torsion : residue.getTorsionList()) {
                    if (torsion.compare(N, CA, CB, CG)) {
                        chi[0] = torsion.measure();
                        if (print) {
                            logger.info(torsion.toString());
                        }
                    }
                    if (!torsion.compare(CA, CB, CG, ND1)) continue;
                    chi[1] = torsion.measure();
                    if (!print) continue;
                    logger.info(torsion.toString());
                }
                return 2;
            }
            case ASP: {
                Atom N = (Atom)residue.getAtomNode("N");
                Atom CA = (Atom)residue.getAtomNode("CA");
                Atom CB = (Atom)residue.getAtomNode("CB");
                Atom CG = (Atom)residue.getAtomNode("CG");
                for (Torsion torsion : residue.getTorsionList()) {
                    if (!torsion.compare(N, CA, CB, CG)) continue;
                    chi[0] = torsion.measure();
                    if (!print) break;
                    logger.info(torsion.toString());
                    break;
                }
                return 1;
            }
            case ASH: 
            case ASN: {
                Atom N = (Atom)residue.getAtomNode("N");
                Atom CA = (Atom)residue.getAtomNode("CA");
                Atom CB = (Atom)residue.getAtomNode("CB");
                Atom CG = (Atom)residue.getAtomNode("CG");
                Atom OD1 = (Atom)residue.getAtomNode("OD1");
                for (Torsion torsion : residue.getTorsionList()) {
                    if (torsion.compare(N, CA, CB, CG)) {
                        chi[0] = torsion.measure();
                        if (print) {
                            logger.info(torsion.toString());
                        }
                    }
                    if (!torsion.compare(CA, CB, CG, OD1)) continue;
                    chi[1] = torsion.measure();
                    if (!print) continue;
                    logger.info(torsion.toString());
                }
                return 2;
            }
            case GLU: 
            case GLN: 
            case GLH: {
                Atom N = (Atom)residue.getAtomNode("N");
                Atom CA = (Atom)residue.getAtomNode("CA");
                Atom CB = (Atom)residue.getAtomNode("CB");
                Atom CG = (Atom)residue.getAtomNode("CG");
                Atom CD = (Atom)residue.getAtomNode("CD");
                Atom OE1 = (Atom)residue.getAtomNode("OE1");
                for (Torsion torsion : residue.getTorsionList()) {
                    if (torsion.compare(N, CA, CB, CG)) {
                        chi[0] = torsion.measure();
                        if (print) {
                            logger.info(torsion.toString());
                        }
                    }
                    if (torsion.compare(CA, CB, CG, CD)) {
                        chi[1] = torsion.measure();
                        if (print) {
                            logger.info(torsion.toString());
                        }
                    }
                    if (!torsion.compare(CB, CG, CD, OE1)) continue;
                    chi[2] = torsion.measure();
                    if (!print) continue;
                    logger.info(torsion.toString());
                }
                return 3;
            }
            case MET: {
                Atom N = (Atom)residue.getAtomNode("N");
                Atom CA = (Atom)residue.getAtomNode("CA");
                Atom CB = (Atom)residue.getAtomNode("CB");
                Atom CG = (Atom)residue.getAtomNode("CG");
                Atom CE = (Atom)residue.getAtomNode("CE");
                Atom SD = (Atom)residue.getAtomNode("SD");
                for (Torsion torsion : residue.getTorsionList()) {
                    if (torsion.compare(N, CA, CB, CG)) {
                        chi[0] = torsion.measure();
                        if (print) {
                            logger.info(torsion.toString());
                        }
                    }
                    if (torsion.compare(CA, CB, CG, SD)) {
                        chi[1] = torsion.measure();
                        if (print) {
                            logger.info(torsion.toString());
                        }
                    }
                    if (!torsion.compare(CB, CG, SD, CE)) continue;
                    chi[2] = torsion.measure();
                    if (!print) continue;
                    logger.info(torsion.toString());
                }
                return 3;
            }
            case LYS: 
            case LYD: {
                Atom N = (Atom)residue.getAtomNode("N");
                Atom CA = (Atom)residue.getAtomNode("CA");
                Atom CB = (Atom)residue.getAtomNode("CB");
                Atom CD = (Atom)residue.getAtomNode("CD");
                Atom CE = (Atom)residue.getAtomNode("CE");
                Atom CG = (Atom)residue.getAtomNode("CG");
                Atom NZ = (Atom)residue.getAtomNode("NZ");
                for (Torsion torsion : residue.getTorsionList()) {
                    if (torsion.compare(N, CA, CB, CG)) {
                        chi[0] = torsion.measure();
                        if (print) {
                            logger.info(torsion.toString());
                        }
                    }
                    if (torsion.compare(CA, CB, CG, CD)) {
                        chi[1] = torsion.measure();
                        if (print) {
                            logger.info(torsion.toString());
                        }
                    }
                    if (torsion.compare(CB, CG, CD, CE)) {
                        chi[2] = torsion.measure();
                        if (print) {
                            logger.info(torsion.toString());
                        }
                    }
                    if (!torsion.compare(CG, CD, CE, NZ)) continue;
                    chi[3] = torsion.measure();
                    if (!print) continue;
                    logger.info(torsion.toString());
                }
                return 4;
            }
            case ARG: {
                Atom N = (Atom)residue.getAtomNode("N");
                Atom CA = (Atom)residue.getAtomNode("CA");
                Atom CB = (Atom)residue.getAtomNode("CB");
                Atom CD = (Atom)residue.getAtomNode("CD");
                Atom CG = (Atom)residue.getAtomNode("CG");
                Atom CZ = (Atom)residue.getAtomNode("CZ");
                Atom NE = (Atom)residue.getAtomNode("NE");
                for (Torsion torsion : residue.getTorsionList()) {
                    if (torsion.compare(N, CA, CB, CG)) {
                        chi[0] = torsion.measure();
                        if (print) {
                            logger.info(torsion.toString());
                        }
                    }
                    if (torsion.compare(CA, CB, CG, CD)) {
                        chi[1] = torsion.measure();
                        if (print) {
                            logger.info(torsion.toString());
                        }
                    }
                    if (torsion.compare(CB, CG, CD, NE)) {
                        chi[2] = torsion.measure();
                        if (print) {
                            logger.info(torsion.toString());
                        }
                    }
                    if (!torsion.compare(CG, CD, NE, CZ)) continue;
                    chi[3] = torsion.measure();
                    if (!print) continue;
                    logger.info(torsion.toString());
                }
                return 4;
            }
            case UNK: {
                chi = new double[7];
                String resName = residue.getName().toUpperCase();
                if (nonstdRotCache.containsKey(resName)) {
                    return nonstdRotCache.get(resName).measureNonstdRot(residue, chi, print);
                }
                throw new IllegalArgumentException(String.format("(IAE) valueOf(%s)", residue.getName()));
            }
        }
        return 0;
    }

    public static double measureDelta(Residue residue) {
        Atom C5s = (Atom)residue.getAtomNode("C5'");
        Atom C4s = (Atom)residue.getAtomNode("C4'");
        Atom C3s = (Atom)residue.getAtomNode("C3'");
        Atom O3s = (Atom)residue.getAtomNode("O3'");
        Torsion torsion = O3s.getTorsion(C3s, C4s, C5s);
        return torsion.measure();
    }

    public static double[] measureRotamer(@Nullable Residue residue, boolean print) {
        if (residue != null) {
            switch (residue.getResidueType()) {
                case AA: {
                    double[] chi = new double[4];
                    try {
                        RotamerLibrary.measureRotamer(residue, chi, print);
                    }
                    catch (ArrayIndexOutOfBoundsException e) {
                        String message = " Array passed to measureRotamer was not of sufficient size.";
                        logger.log(Level.WARNING, message, e);
                    }
                    return chi;
                }
                case NA: {
                    double[] chi = new double[7];
                    try {
                        RotamerLibrary.measureRotamer(residue, chi, print);
                    }
                    catch (ArrayIndexOutOfBoundsException e) {
                        String message = " Array passed to measureRotamer was not of sufficient size.";
                        logger.log(Level.WARNING, message, e);
                    }
                    return chi;
                }
            }
            return null;
        }
        return null;
    }

    public static int measureRotamer(Residue residue, double[] chi, boolean print) {
        if (residue == null) {
            return -1;
        }
        int nRot = -1;
        switch (residue.getResidueType()) {
            case AA: {
                try {
                    nRot = RotamerLibrary.measureAARotamer(residue, chi, print);
                }
                catch (ArrayIndexOutOfBoundsException e) {
                    String message = " Array passed to measureRotamer was not of sufficient size.";
                    logger.log(Level.WARNING, message, e);
                }
                break;
            }
            case NA: {
                try {
                    nRot = RotamerLibrary.measureNARotamer(residue, chi, print);
                }
                catch (ArrayIndexOutOfBoundsException e) {
                    String message = "Array passed to measureRotamer was not of sufficient size.";
                    logger.log(Level.WARNING, message, e);
                }
                break;
            }
            default: {
                try {
                    RotamerLibrary.measureUNKRotamer(residue, chi, print);
                    break;
                }
                catch (ArrayIndexOutOfBoundsException e) {
                    String message = "Array passed to measureRotamer was not of sufficient size.";
                    logger.log(Level.WARNING, message, e);
                }
            }
        }
        return nRot;
    }

    public static void measureRotamers(List<Residue> residueList, boolean print) {
        double[] chi = new double[7];
        if (residueList.isEmpty()) {
            return;
        }
        logger.info("\n Residue    Current Torsional Angles");
        block4: for (Residue residue : residueList) {
            Arrays.fill(chi, 0.0);
            RotamerLibrary.measureRotamer(residue, chi, print);
            switch (residue.getResidueType()) {
                case AA: {
                    logger.info(String.format(" %c %8s %8.3f %8.3f %8.3f %8.3f", residue.getChainID(), residue, chi[0], chi[1], chi[2], chi[3]));
                    continue block4;
                }
                case NA: {
                    logger.info(String.format(" %c %8s %8.3f %8.3f %8.3f %8.3f %8.3f %8.3f %8.3f", residue.getChainID(), residue, chi[0], chi[1], chi[2], chi[3], chi[4], chi[5], chi[6]));
                    continue block4;
                }
            }
            logger.info(" Not recognized as a nucleic or amino acid residue");
        }
    }

    public static void readRotFile(File rotamerFile, MolecularAssembly assembly) throws IOException {
        RotamerLibrary.readRotFile(rotamerFile, assembly, 1);
    }

    /*
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    private static boolean addRotPatch(File rpatchFile) {
        try {
            Object object;
            ArrayList<Rotamer> rotamers;
            Residue.ResidueType rType;
            ArrayList<String> applyLines;
            String resName;
            block34: {
                BufferedReader br = new BufferedReader(new FileReader(rpatchFile));
                resName = null;
                applyLines = new ArrayList<String>();
                ArrayList<String> rotLines = new ArrayList<String>();
                rType = Residue.ResidueType.AA;
                String line = br.readLine();
                while (line != null) {
                    if ((line = line.trim()).startsWith("PLACE")) {
                        applyLines.add(line);
                    } else if (line.startsWith("RESNAME")) {
                        toks = line.split("\\s+");
                        resName = toks[1];
                    } else if (line.startsWith("RESTYPE")) {
                        toks = line.split("\\s+");
                        rType = switch (toks[1]) {
                            case "AA" -> Residue.ResidueType.AA;
                            case "NA" -> Residue.ResidueType.NA;
                            default -> Residue.ResidueType.UNK;
                        };
                    } else if (line.startsWith("ROTAMER")) {
                        rotLines.add(line);
                    }
                    line = br.readLine();
                }
                if (resName != null) {
                    rotamers = new ArrayList<Rotamer>();
                    object = rotLines.iterator();
                    break block34;
                } else {
                    boolean bl = false;
                    return bl;
                }
                finally {
                    br.close();
                }
            }
            block21: while (object.hasNext()) {
                String string = (String)object.next();
                String[] toks = string.split("\\s+");
                int nVals = toks.length - 1;
                double[] values = new double[nVals];
                for (int i = 0; i < nVals; ++i) {
                    values[i] = Double.parseDouble(toks[i + 1]);
                }
                switch (rType) {
                    case AA: {
                        rotamers.add(new Rotamer(AminoAcidUtils.AminoAcid3.UNK, values));
                        continue block21;
                    }
                    case NA: {
                        rotamers.add(new Rotamer(NucleicAcidUtils.NucleicAcid3.UNK, values));
                        continue block21;
                    }
                }
                rotamers.add(new Rotamer(values));
            }
            if (nonstdRotCache.containsKey(resName)) {
                logger.warning(String.format(" Rotamer library already contains rotamer definition for residue %s!", resName));
            } else {
                NonstandardRotLibrary nrlib = new NonstandardRotLibrary(resName, applyLines.toArray(new String[0]), rotamers.toArray(new Rotamer[0]));
                nonstdRotCache.put(resName, nrlib);
            }
            boolean bl = true;
            return bl;
        }
        catch (IOException ex) {
            logger.warning(String.format(" Exception in parsing rotamer patch file %s: %s", rpatchFile.getName(), ex));
            return false;
        }
    }

    private static double angleFromForceField(Atom a1, Atom a2, Atom a3) {
        Angle a = a1.getAngle(a2, a3);
        AngleType at = a.getAngleType();
        return at.angle[a.nh];
    }

    private static void applyAARotamer(Residue residue, Rotamer rotamer) {
        AminoAcidUtils.AminoAcid3 name;
        if (residue == null || rotamer == null) {
            return;
        }
        if (residue instanceof MultiResidue) {
            name = rotamer.aminoAcid3;
            if (!((MultiResidue)residue).setActiveResidue(name)) {
                logger.warning(String.format(" Could not set residue %s for multi-residue %s", new Object[]{name, residue}));
            }
        } else {
            name = rotamer.aminoAcid3;
        }
        switch (name) {
            case VAL: {
                Atom CA = (Atom)residue.getAtomNode("CA");
                Atom CB = (Atom)residue.getAtomNode("CB");
                Atom N = (Atom)residue.getAtomNode("N");
                Atom CG1 = (Atom)residue.getAtomNode("CG1");
                Atom CG2 = (Atom)residue.getAtomNode("CG2");
                Atom HB = (Atom)residue.getAtomNode("HB");
                Atom HG11 = (Atom)residue.getAtomNode("HG11");
                Atom HG12 = (Atom)residue.getAtomNode("HG12");
                Atom HG13 = (Atom)residue.getAtomNode("HG13");
                Atom HG21 = (Atom)residue.getAtomNode("HG21");
                Atom HG22 = (Atom)residue.getAtomNode("HG22");
                Atom HG23 = (Atom)residue.getAtomNode("HG23");
                Bond CG_CB = CB.getBond(CG1);
                Bond HB_CB = CB.getBond(HB);
                Bond HG_CG = HG11.getBond(CG1);
                double dCG_CB = CG_CB.bondType.distance;
                double dHB_CB = HB_CB.bondType.distance;
                double dHG_CG = HG_CG.bondType.distance;
                Angle CG_CB_CA = CG1.getAngle(CB, CA);
                Angle HB_CB_CA = HB.getAngle(CB, CA);
                Angle HG_CG_CB = HG11.getAngle(CG1, CB);
                double dCG_CB_CA = CG_CB_CA.angleType.angle[CG_CB_CA.nh];
                double dHB_CB_CA = HB_CB_CA.angleType.angle[HB_CB_CA.nh];
                double dHG_CG_CB = HG_CG_CB.angleType.angle[HG_CG_CB.nh];
                BondedUtils.intxyz(CG1, CB, dCG_CB, CA, dCG_CB_CA, N, rotamer.chi1, 0);
                BondedUtils.intxyz(CG2, CB, dCG_CB, CA, dCG_CB_CA, CG1, 109.5, -1);
                BondedUtils.intxyz(HB, CB, dHB_CB, CA, dHB_CB_CA, CG1, 109.4, 1);
                BondedUtils.intxyz(HG11, CG1, dHG_CG, CB, dHG_CG_CB, CA, 180.0, 0);
                BondedUtils.intxyz(HG12, CG1, dHG_CG, CB, dHG_CG_CB, HG11, 109.4, 1);
                BondedUtils.intxyz(HG13, CG1, dHG_CG, CB, dHG_CG_CB, HG11, 109.4, -1);
                BondedUtils.intxyz(HG21, CG2, dHG_CG, CB, dHG_CG_CB, CA, 180.0, 0);
                BondedUtils.intxyz(HG22, CG2, dHG_CG, CB, dHG_CG_CB, HG21, 109.4, 1);
                BondedUtils.intxyz(HG23, CG2, dHG_CG, CB, dHG_CG_CB, HG21, 109.4, -1);
                break;
            }
            case LEU: {
                Atom CA = (Atom)residue.getAtomNode("CA");
                Atom CB = (Atom)residue.getAtomNode("CB");
                Atom N = (Atom)residue.getAtomNode("N");
                Atom CG = (Atom)residue.getAtomNode("CG");
                Atom CD1 = (Atom)residue.getAtomNode("CD1");
                Atom CD2 = (Atom)residue.getAtomNode("CD2");
                Atom HB2 = (Atom)residue.getAtomNode("HB2");
                Atom HB3 = (Atom)residue.getAtomNode("HB3");
                Atom HG = (Atom)residue.getAtomNode("HG");
                Atom HD11 = (Atom)residue.getAtomNode("HD11");
                Atom HD12 = (Atom)residue.getAtomNode("HD12");
                Atom HD13 = (Atom)residue.getAtomNode("HD13");
                Atom HD21 = (Atom)residue.getAtomNode("HD21");
                Atom HD22 = (Atom)residue.getAtomNode("HD22");
                Atom HD23 = (Atom)residue.getAtomNode("HD23");
                Bond CG_CB = CG.getBond(CB);
                Bond CD_CG = CD1.getBond(CG);
                Bond HB_CB = HB2.getBond(CB);
                Bond HG_CG = HG.getBond(CG);
                Bond HD_CD = HD11.getBond(CD1);
                double dCG_CB = CG_CB.bondType.distance;
                double dCD_CG = CD_CG.bondType.distance;
                double dHB_CB = HB_CB.bondType.distance;
                double dHG_CG = HG_CG.bondType.distance;
                double dHD_CD = HD_CD.bondType.distance;
                Angle CG_CB_CA = CG.getAngle(CB, CA);
                Angle CD_CG_CB = CD1.getAngle(CG, CB);
                Angle HB_CB_CA = HB2.getAngle(CB, CA);
                Angle HG_CG_CB = HG.getAngle(CG, CB);
                Angle HD_CD_CG = HD11.getAngle(CD1, CG);
                double dCG_CB_CA = CG_CB_CA.angleType.angle[CG_CB_CA.nh];
                double dCD_CG_CB = CD_CG_CB.angleType.angle[CD_CG_CB.nh];
                double dHB_CB_CA = HB_CB_CA.angleType.angle[HB_CB_CA.nh];
                double dHG_CG_CB = HG_CG_CB.angleType.angle[HG_CG_CB.nh];
                double dHD_CD_CG = HD_CD_CG.angleType.angle[HD_CD_CG.nh];
                BondedUtils.intxyz(CG, CB, dCG_CB, CA, dCG_CB_CA, N, rotamer.chi1, 0);
                BondedUtils.intxyz(CD1, CG, dCD_CG, CB, dCD_CG_CB, CA, rotamer.chi2, 0);
                BondedUtils.intxyz(CD2, CG, dCD_CG, CB, dCD_CG_CB, CD1, 109.5, -1);
                BondedUtils.intxyz(HB2, CB, dHB_CB, CA, dHB_CB_CA, CG, 109.4, 1);
                BondedUtils.intxyz(HB3, CB, dHB_CB, CA, dHB_CB_CA, CG, 109.4, -1);
                BondedUtils.intxyz(HG, CG, dHG_CG, CB, dHG_CG_CB, CD1, 109.4, 1);
                BondedUtils.intxyz(HD11, CD1, dHD_CD, CG, dHD_CD_CG, CB, 180.0, 0);
                BondedUtils.intxyz(HD12, CD1, dHD_CD, CG, dHD_CD_CG, HD11, 109.4, 1);
                BondedUtils.intxyz(HD13, CD1, dHD_CD, CG, dHD_CD_CG, HD11, 109.4, -1);
                BondedUtils.intxyz(HD21, CD2, dHD_CD, CG, dHD_CD_CG, CB, 180.0, 0);
                BondedUtils.intxyz(HD22, CD2, dHD_CD, CG, dHD_CD_CG, HD21, 109.4, 1);
                BondedUtils.intxyz(HD23, CD2, dHD_CD, CG, dHD_CD_CG, HD21, 109.4, -1);
                break;
            }
            case ILE: {
                Atom CA = (Atom)residue.getAtomNode("CA");
                Atom CB = (Atom)residue.getAtomNode("CB");
                Atom N = (Atom)residue.getAtomNode("N");
                Atom CG1 = (Atom)residue.getAtomNode("CG1");
                Atom CG2 = (Atom)residue.getAtomNode("CG2");
                Atom CD1 = (Atom)residue.getAtomNode("CD1");
                Atom HB = (Atom)residue.getAtomNode("HB");
                Atom HG12 = (Atom)residue.getAtomNode("HG12");
                Atom HG13 = (Atom)residue.getAtomNode("HG13");
                Atom HG21 = (Atom)residue.getAtomNode("HG21");
                Atom HG22 = (Atom)residue.getAtomNode("HG22");
                Atom HG23 = (Atom)residue.getAtomNode("HG23");
                Atom HD11 = (Atom)residue.getAtomNode("HD11");
                Atom HD12 = (Atom)residue.getAtomNode("HD12");
                Atom HD13 = (Atom)residue.getAtomNode("HD13");
                Bond CG1_CB = CG1.getBond(CB);
                Bond CG2_CB = CG2.getBond(CB);
                Bond CD1_CG1 = CD1.getBond(CG1);
                Bond HB_CB = HB.getBond(CB);
                Bond HG1_CG = HG12.getBond(CG1);
                Bond HG2_CG = HG22.getBond(CG2);
                Bond HD_CD = HD12.getBond(CD1);
                double dCG1_CB = CG1_CB.bondType.distance;
                double dCG2_CB = CG2_CB.bondType.distance;
                double dCD1_CG1 = CD1_CG1.bondType.distance;
                double dHB_CB = HB_CB.bondType.distance;
                double dHG1_CG = HG1_CG.bondType.distance;
                double dHG2_CG = HG2_CG.bondType.distance;
                double dHD_CD = HD_CD.bondType.distance;
                Angle CG1_CB_CA = CG1.getAngle(CB, CA);
                Angle CG2_CB_CA = CG2.getAngle(CB, CA);
                Angle CD1_CG1_CB = CD1.getAngle(CG1, CB);
                Angle HB_CB_CA = HB.getAngle(CB, CA);
                Angle HG1_CG_CB = HG12.getAngle(CG1, CB);
                Angle HG2_CG_CB = HG21.getAngle(CG2, CB);
                Angle HD_CD1_CG1 = HD11.getAngle(CD1, CG1);
                double dCG1_CB_CA = CG1_CB_CA.angleType.angle[CG1_CB_CA.nh];
                double dCG2_CB_CA = CG2_CB_CA.angleType.angle[CG2_CB_CA.nh];
                double dCD1_CG1_CB = CD1_CG1_CB.angleType.angle[CD1_CG1_CB.nh];
                double dHB_CB_CA = HB_CB_CA.angleType.angle[HB_CB_CA.nh];
                double dHG1_CG_CB = HG1_CG_CB.angleType.angle[HG1_CG_CB.nh];
                double dHG2_CG_CB = HG2_CG_CB.angleType.angle[HG2_CG_CB.nh];
                double dHD_CD1_CG1 = HD_CD1_CG1.angleType.angle[HD_CD1_CG1.nh];
                BondedUtils.intxyz(CG1, CB, dCG1_CB, CA, dCG1_CB_CA, N, rotamer.chi1, 0);
                BondedUtils.intxyz(CG2, CB, dCG2_CB, CA, dCG2_CB_CA, CG1, 109.5, 1);
                BondedUtils.intxyz(CD1, CG1, dCD1_CG1, CB, dCD1_CG1_CB, CA, rotamer.chi2, 0);
                BondedUtils.intxyz(HB, CB, dHB_CB, CA, dHB_CB_CA, CG2, 109.4, 1);
                BondedUtils.intxyz(HG12, CG1, dHG1_CG, CB, dHG1_CG_CB, CD1, 109.4, 1);
                BondedUtils.intxyz(HG13, CG1, dHG1_CG, CB, dHG1_CG_CB, CD1, 109.4, -1);
                BondedUtils.intxyz(HG21, CG2, dHG2_CG, CB, dHG2_CG_CB, CG1, 180.0, 0);
                BondedUtils.intxyz(HG22, CG2, dHG2_CG, CB, dHG2_CG_CB, HG21, 109.0, 1);
                BondedUtils.intxyz(HG23, CG2, dHG2_CG, CB, dHG2_CG_CB, HG21, 109.0, -1);
                BondedUtils.intxyz(HD11, CD1, dHD_CD, CG1, dHD_CD1_CG1, CB, 180.0, 0);
                BondedUtils.intxyz(HD12, CD1, dHD_CD, CG1, dHD_CD1_CG1, HD11, 109.0, 1);
                BondedUtils.intxyz(HD13, CD1, dHD_CD, CG1, dHD_CD1_CG1, HD11, 109.0, -1);
                break;
            }
            case SER: {
                Atom CA = (Atom)residue.getAtomNode("CA");
                Atom CB = (Atom)residue.getAtomNode("CB");
                Atom N = (Atom)residue.getAtomNode("N");
                Atom OG = (Atom)residue.getAtomNode("OG");
                Atom HB2 = (Atom)residue.getAtomNode("HB2");
                Atom HB3 = (Atom)residue.getAtomNode("HB3");
                Atom HG = (Atom)residue.getAtomNode("HG");
                Bond OG_CB = OG.getBond(CB);
                Bond HB_CB = HB2.getBond(CB);
                Bond HG_OG = HG.getBond(OG);
                double dOG_CB = OG_CB.bondType.distance;
                double dHB_CB = HB_CB.bondType.distance;
                double dHG_OG = HG_OG.bondType.distance;
                Angle OG_CB_CA = OG.getAngle(CB, CA);
                Angle HB_CB_CA = HB2.getAngle(CB, CA);
                Angle HG_OG_CB = HG.getAngle(OG, CB);
                double dOG_CB_CA = OG_CB_CA.angleType.angle[OG_CB_CA.nh];
                double dHB_CB_CA = HB_CB_CA.angleType.angle[HB_CB_CA.nh];
                double dHG_OG_CB = HG_OG_CB.angleType.angle[HG_OG_CB.nh];
                BondedUtils.intxyz(OG, CB, dOG_CB, CA, dOG_CB_CA, N, rotamer.chi1, 0);
                BondedUtils.intxyz(HB2, CB, dHB_CB, CA, dHB_CB_CA, OG, 106.7, 1);
                BondedUtils.intxyz(HB3, CB, dHB_CB, CA, dHB_CB_CA, OG, 106.7, -1);
                BondedUtils.intxyz(HG, OG, dHG_OG, CB, dHG_OG_CB, CA, 180.0, 0);
                if (rotamer.length == 2) {
                    BondedUtils.intxyz(HG, OG, dHG_OG, CB, dHG_OG_CB, CA, rotamer.chi2, 0);
                    break;
                }
                BondedUtils.intxyz(HG, OG, dHG_OG, CB, dHG_OG_CB, CA, 180.0, 0);
                break;
            }
            case THR: {
                Atom CA = (Atom)residue.getAtomNode("CA");
                Atom CB = (Atom)residue.getAtomNode("CB");
                Atom N = (Atom)residue.getAtomNode("N");
                Atom OG1 = (Atom)residue.getAtomNode("OG1");
                Atom CG2 = (Atom)residue.getAtomNode("CG2");
                Atom HB = (Atom)residue.getAtomNode("HB");
                Atom HG1 = (Atom)residue.getAtomNode("HG1");
                Atom HG21 = (Atom)residue.getAtomNode("HG21");
                Atom HG22 = (Atom)residue.getAtomNode("HG22");
                Atom HG23 = (Atom)residue.getAtomNode("HG23");
                Bond OG1_CB = OG1.getBond(CB);
                Bond CG2_CB = CG2.getBond(CB);
                Bond HB_CB = HB.getBond(CB);
                Bond HG1_OG1 = HG1.getBond(OG1);
                Bond HG2_CG2 = HG21.getBond(CG2);
                double dOG1_CB = OG1_CB.bondType.distance;
                double dCG2_CB = CG2_CB.bondType.distance;
                double dHB_CB = HB_CB.bondType.distance;
                double dHG1_OG1 = HG1_OG1.bondType.distance;
                double dHG2_CG2 = HG2_CG2.bondType.distance;
                Angle OG1_CB_CA = OG1.getAngle(CB, CA);
                Angle CG2_CB_CA = CG2.getAngle(CB, CA);
                Angle HB_CB_CA = HB.getAngle(CB, CA);
                Angle HG1_OG1_CB = HG1.getAngle(OG1, CB);
                Angle HG2_CG2_CB = HG21.getAngle(CG2, CB);
                double dOG1_CB_CA = OG1_CB_CA.angleType.angle[OG1_CB_CA.nh];
                double dCG2_CB_CA = CG2_CB_CA.angleType.angle[CG2_CB_CA.nh];
                double dHB_CB_CA = HB_CB_CA.angleType.angle[HB_CB_CA.nh];
                double dHG1_OG1_CB = HG1_OG1_CB.angleType.angle[HG1_OG1_CB.nh];
                double dHG2_CG2_CB = HG2_CG2_CB.angleType.angle[HG2_CG2_CB.nh];
                BondedUtils.intxyz(OG1, CB, dOG1_CB, CA, dOG1_CB_CA, N, rotamer.chi1, 0);
                BondedUtils.intxyz(CG2, CB, dCG2_CB, CA, dCG2_CB_CA, OG1, 107.7, 1);
                BondedUtils.intxyz(HB, CB, dHB_CB, CA, dHB_CB_CA, OG1, 106.7, -1);
                BondedUtils.intxyz(HG1, OG1, dHG1_OG1, CB, dHG1_OG1_CB, CA, 180.0, 0);
                if (rotamer.length == 2) {
                    BondedUtils.intxyz(HG1, OG1, dHG1_OG1, CB, dHG1_OG1_CB, CA, rotamer.chi2, 0);
                } else {
                    BondedUtils.intxyz(HG1, OG1, dHG1_OG1, CB, dHG1_OG1_CB, CA, 180.0, 0);
                }
                BondedUtils.intxyz(HG21, CG2, dHG2_CG2, CB, dHG2_CG2_CB, CA, 180.0, 0);
                BondedUtils.intxyz(HG22, CG2, dHG2_CG2, CB, dHG2_CG2_CB, HG21, 109.0, 1);
                BondedUtils.intxyz(HG23, CG2, dHG2_CG2, CB, dHG2_CG2_CB, HG21, 109.0, -1);
                break;
            }
            case CYS: 
            case CYX: {
                Atom CA = (Atom)residue.getAtomNode("CA");
                Atom CB = (Atom)residue.getAtomNode("CB");
                Atom N = (Atom)residue.getAtomNode("N");
                Atom SG = (Atom)residue.getAtomNode("SG");
                Atom HB2 = (Atom)residue.getAtomNode("HB2");
                Atom HB3 = (Atom)residue.getAtomNode("HB3");
                Atom HG = (Atom)residue.getAtomNode("HG");
                if (CA == null || CB == null || N == null || SG == null || HB2 == null || HB3 == null || HG == null) break;
                Bond SG_CB = SG.getBond(CB);
                Bond HB_CB = HB2.getBond(CB);
                Bond HG_SG = HG.getBond(SG);
                double dSG_CB = SG_CB.bondType.distance;
                double dHB_CB = HB_CB.bondType.distance;
                double dHG_SG = HG_SG.bondType.distance;
                Angle SG_CB_CA = SG.getAngle(CB, CA);
                Angle HB_CB_CA = HB2.getAngle(CB, CA);
                Angle HG_SG_CB = HG.getAngle(SG, CB);
                double dSG_CB_CA = SG_CB_CA.angleType.angle[SG_CB_CA.nh];
                double dHB_CB_CA = HB_CB_CA.angleType.angle[HB_CB_CA.nh];
                double dHG_SG_CB = HG_SG_CB.angleType.angle[HG_SG_CB.nh];
                BondedUtils.intxyz(SG, CB, dSG_CB, CA, dSG_CB_CA, N, rotamer.chi1, 0);
                BondedUtils.intxyz(HB2, CB, dHB_CB, CA, dHB_CB_CA, SG, 112.0, 1);
                BondedUtils.intxyz(HB3, CB, dHB_CB, CA, dHB_CB_CA, SG, 112.0, -1);
                BondedUtils.intxyz(HG, SG, dHG_SG, CB, dHG_SG_CB, CA, 180.0, 0);
                break;
            }
            case CYD: {
                Atom CA = (Atom)residue.getAtomNode("CA");
                Atom CB = (Atom)residue.getAtomNode("CB");
                Atom N = (Atom)residue.getAtomNode("N");
                Atom SG = (Atom)residue.getAtomNode("SG");
                Atom HB2 = (Atom)residue.getAtomNode("HB2");
                Atom HB3 = (Atom)residue.getAtomNode("HB3");
                Bond SG_CB = SG.getBond(CB);
                Bond HB_CB = HB2.getBond(CB);
                double dSG_CB = SG_CB.bondType.distance;
                double dHB_CB = HB_CB.bondType.distance;
                Angle SG_CB_CA = SG.getAngle(CB, CA);
                Angle HB_CB_CA = HB2.getAngle(CB, CA);
                double dSG_CB_CA = SG_CB_CA.angleType.angle[SG_CB_CA.nh];
                double dHB_CB_CA = HB_CB_CA.angleType.angle[HB_CB_CA.nh];
                BondedUtils.intxyz(SG, CB, dSG_CB, CA, dSG_CB_CA, N, rotamer.chi1, 0);
                BondedUtils.intxyz(HB2, CB, dHB_CB, CA, dHB_CB_CA, SG, 112.0, 1);
                BondedUtils.intxyz(HB3, CB, dHB_CB, CA, dHB_CB_CA, SG, 112.0, -1);
                break;
            }
            case PHE: {
                Atom CA = (Atom)residue.getAtomNode("CA");
                Atom CB = (Atom)residue.getAtomNode("CB");
                Atom N = (Atom)residue.getAtomNode("N");
                Atom CG = (Atom)residue.getAtomNode("CG");
                Atom CD1 = (Atom)residue.getAtomNode("CD1");
                Atom CD2 = (Atom)residue.getAtomNode("CD2");
                Atom CE1 = (Atom)residue.getAtomNode("CE1");
                Atom CE2 = (Atom)residue.getAtomNode("CE2");
                Atom CZ = (Atom)residue.getAtomNode("CZ");
                Atom HB2 = (Atom)residue.getAtomNode("HB2");
                Atom HB3 = (Atom)residue.getAtomNode("HB3");
                Atom HD1 = (Atom)residue.getAtomNode("HD1");
                Atom HD2 = (Atom)residue.getAtomNode("HD2");
                Atom HE1 = (Atom)residue.getAtomNode("HE1");
                Atom HE2 = (Atom)residue.getAtomNode("HE2");
                Atom HZ = (Atom)residue.getAtomNode("HZ");
                Bond CG_CB = CG.getBond(CB);
                Bond CD_CG = CD1.getBond(CG);
                Bond CE_CD = CE1.getBond(CD1);
                Bond HB_CB = HB2.getBond(CB);
                double dCG_CB = CG_CB.bondType.distance;
                double dCD_CG = CD_CG.bondType.distance;
                double dCE_CD = CE_CD.bondType.distance;
                double dHB_CB = HB_CB.bondType.distance;
                double dHD_CD = HD1.getBond((Atom)CD1).bondType.distance;
                double dHE_CE = HE1.getBond((Atom)CE1).bondType.distance;
                double dHZ_CZ = HZ.getBond((Atom)CZ).bondType.distance;
                double dCG_CB_CA = RotamerLibrary.getAngle(name, CG, CB, CA);
                double dCD_CG_CB = RotamerLibrary.getAngle(name, CD1, CG, CB);
                double dCE_CD_CG = RotamerLibrary.getAngle(name, CE1, CD1, CG);
                double dHB_CB_CA = RotamerLibrary.getAngle(name, HB2, CB, CA);
                double dHD_CD_CG = RotamerLibrary.getAngle(name, HD1, CD1, CG);
                double dHD_CD_CE = RotamerLibrary.getAngle(name, HD1, CD1, CE1);
                double dHE_CE_CD = RotamerLibrary.getAngle(name, HE1, CE1, CD1);
                double dHE_CE_CZ = RotamerLibrary.getAngle(name, HE1, CE1, CZ);
                double dHZ_CZ_CE = RotamerLibrary.getAngle(name, HZ, CZ, CE1);
                BondedUtils.intxyz(CG, CB, dCG_CB, CA, dCG_CB_CA, N, rotamer.chi1, 0);
                BondedUtils.intxyz(CD1, CG, dCD_CG, CB, dCD_CG_CB, CA, rotamer.chi2, 0);
                BondedUtils.intxyz(CD2, CG, dCD_CG, CB, dCD_CG_CB, CA, rotamer.chi2 + 180.0, 0);
                BondedUtils.intxyz(CE1, CD1, dCE_CD, CG, dCE_CD_CG, CB, 180.0, 0);
                BondedUtils.intxyz(CE2, CD2, dCE_CD, CG, dCE_CD_CG, CB, 180.0, 0);
                RotamerLibrary.applyCZ(name, CZ, CG, CE1, CD1, CE2, CD2);
                BondedUtils.intxyz(HB2, CB, dHB_CB, CA, dHB_CB_CA, CG, 109.4, 1);
                BondedUtils.intxyz(HB3, CB, dHB_CB, CA, dHB_CB_CA, CG, 109.4, -1);
                BondedUtils.intxyz(HD1, CD1, dHD_CD, CG, dHD_CD_CG, CE1, dHD_CD_CE, 3);
                BondedUtils.intxyz(HD2, CD2, dHD_CD, CG, dHD_CD_CG, CE2, dHD_CD_CE, 3);
                BondedUtils.intxyz(HE1, CE1, dHE_CE, CD1, dHE_CE_CD, CZ, dHE_CE_CZ, 3);
                BondedUtils.intxyz(HE2, CE2, dHE_CE, CD2, dHE_CE_CD, CZ, dHE_CE_CZ, 3);
                BondedUtils.intxyz(HZ, CZ, dHZ_CZ, CE1, dHZ_CZ_CE, CE2, dHZ_CZ_CE, 3);
                break;
            }
            case PRO: {
                Atom CA = (Atom)residue.getAtomNode("CA");
                Atom CB = (Atom)residue.getAtomNode("CB");
                Atom N = (Atom)residue.getAtomNode("N");
                Atom CG = (Atom)residue.getAtomNode("CG");
                Atom CD = (Atom)residue.getAtomNode("CD");
                Atom HB2 = (Atom)residue.getAtomNode("HB2");
                Atom HB3 = (Atom)residue.getAtomNode("HB3");
                Atom HG2 = (Atom)residue.getAtomNode("HG2");
                Atom HG3 = (Atom)residue.getAtomNode("HG3");
                Atom HD2 = (Atom)residue.getAtomNode("HD2");
                Atom HD3 = (Atom)residue.getAtomNode("HD3");
                Bond CG_CB = CG.getBond(CB);
                Bond CD_CG = CD.getBond(CG);
                Bond HB_CB = HB2.getBond(CB);
                Bond HG_CG = HG2.getBond(CG);
                Bond HD_CD = HD2.getBond(CD);
                double dCG_CB = CG_CB.bondType.distance;
                double dCD_CG = CD_CG.bondType.distance;
                double dHB_CB = HB_CB.bondType.distance;
                double dHG_CG = HG_CG.bondType.distance;
                double dHD_CD = HD_CD.bondType.distance;
                Angle CG_CB_CA = CG.getAngle(CB, CA);
                Angle CD_CG_CB = CD.getAngle(CG, CB);
                Angle HB_CB_CA = HB2.getAngle(CB, CA);
                Angle HG_CG_CB = HG2.getAngle(CG, CB);
                Angle HD_CD_CG = HD2.getAngle(CD, CG);
                double dCG_CB_CA = CG_CB_CA.angleType.angle[CG_CB_CA.nh];
                double dCD_CG_CB = CD_CG_CB.angleType.angle[CD_CG_CB.nh];
                double dHB_CB_CA = HB_CB_CA.angleType.angle[HB_CB_CA.nh];
                double dHG_CG_CB = HG_CG_CB.angleType.angle[HG_CG_CB.nh];
                double dHD_CD_CG = HD_CD_CG.angleType.angle[HD_CD_CG.nh];
                BondedUtils.intxyz(CG, CB, dCG_CB, CA, dCG_CB_CA, N, rotamer.chi1, 0);
                BondedUtils.intxyz(CD, CG, dCD_CG, CB, dCD_CG_CB, CA, rotamer.chi2, 0);
                BondedUtils.intxyz(HB2, CB, dHB_CB, CA, dHB_CB_CA, CG, 109.4, 1);
                BondedUtils.intxyz(HB3, CB, dHB_CB, CA, dHB_CB_CA, CG, 109.4, -1);
                BondedUtils.intxyz(HG2, CG, dHG_CG, CB, dHG_CG_CB, CD, 109.4, 1);
                BondedUtils.intxyz(HG3, CG, dHG_CG, CB, dHG_CG_CB, CD, 109.4, -1);
                BondedUtils.intxyz(HD2, CD, dHD_CD, CG, dHD_CD_CG, N, 109.4, 1);
                BondedUtils.intxyz(HD3, CD, dHD_CD, CG, dHD_CD_CG, N, 109.4, -1);
                break;
            }
            case TYR: {
                Atom CA = (Atom)residue.getAtomNode("CA");
                Atom CB = (Atom)residue.getAtomNode("CB");
                Atom N = (Atom)residue.getAtomNode("N");
                Atom CG = (Atom)residue.getAtomNode("CG");
                Atom CD1 = (Atom)residue.getAtomNode("CD1");
                Atom CD2 = (Atom)residue.getAtomNode("CD2");
                Atom CE1 = (Atom)residue.getAtomNode("CE1");
                Atom CE2 = (Atom)residue.getAtomNode("CE2");
                Atom CZ = (Atom)residue.getAtomNode("CZ");
                Atom HB2 = (Atom)residue.getAtomNode("HB2");
                Atom HB3 = (Atom)residue.getAtomNode("HB3");
                Atom HD1 = (Atom)residue.getAtomNode("HD1");
                Atom HD2 = (Atom)residue.getAtomNode("HD2");
                Atom HE1 = (Atom)residue.getAtomNode("HE1");
                Atom HE2 = (Atom)residue.getAtomNode("HE2");
                Atom OH = (Atom)residue.getAtomNode("OH");
                Atom HH = (Atom)residue.getAtomNode("HH");
                Bond CG_CB = CG.getBond(CB);
                Bond CD_CG = CD1.getBond(CG);
                Bond CE_CD = CE1.getBond(CD1);
                Bond HB_CB = HB2.getBond(CB);
                double dCG_CB = CG_CB.bondType.distance;
                double dCD_CG = CD_CG.bondType.distance;
                double dCE_CD = CE_CD.bondType.distance;
                double dHB_CB = HB_CB.bondType.distance;
                double dHD_CD = HD1.getBond((Atom)CD1).bondType.distance;
                double dHE_CE = HE1.getBond((Atom)CE1).bondType.distance;
                double dOH_CZ = OH.getBond((Atom)CZ).bondType.distance;
                double dHH_OH = HH.getBond((Atom)OH).bondType.distance;
                double dCG_CB_CA = RotamerLibrary.getAngle(name, CG, CB, CA);
                double dCD_CG_CB = RotamerLibrary.getAngle(name, CD1, CG, CB);
                double dCE_CD_CG = RotamerLibrary.getAngle(name, CE1, CD1, CG);
                double dHB_CB_CA = RotamerLibrary.getAngle(name, HB2, CB, CA);
                double dHD_CD_CG = RotamerLibrary.getAngle(name, HD1, CD1, CG);
                double dHD_CD_CE = RotamerLibrary.getAngle(name, HD1, CD1, CE1);
                double dHE_CE_CD = RotamerLibrary.getAngle(name, HE1, CE1, CD1);
                double dHE_CE_CZ = RotamerLibrary.getAngle(name, HE1, CE1, CZ);
                double dOH_CZ_CE = RotamerLibrary.getAngle(name, OH, CZ, CE1);
                double dHH_OH_CZ = RotamerLibrary.getAngle(name, HH, OH, CZ);
                BondedUtils.intxyz(CG, CB, dCG_CB, CA, dCG_CB_CA, N, rotamer.chi1, 0);
                BondedUtils.intxyz(CD1, CG, dCD_CG, CB, dCD_CG_CB, CA, rotamer.chi2, 0);
                BondedUtils.intxyz(CD2, CG, dCD_CG, CB, dCD_CG_CB, CA, rotamer.chi2 + 180.0, 0);
                BondedUtils.intxyz(CE1, CD1, dCE_CD, CG, dCE_CD_CG, CB, 180.0, 0);
                BondedUtils.intxyz(CE2, CD2, dCE_CD, CG, dCE_CD_CG, CB, 180.0, 0);
                RotamerLibrary.applyCZ(name, CZ, CG, CE1, CD1, CE2, CD2);
                BondedUtils.intxyz(HB2, CB, dHB_CB, CA, dHB_CB_CA, CG, 109.4, 1);
                BondedUtils.intxyz(HB3, CB, dHB_CB, CA, dHB_CB_CA, CG, 109.4, -1);
                BondedUtils.intxyz(HD1, CD1, dHD_CD, CG, dHD_CD_CG, CE1, dHD_CD_CE, 3);
                BondedUtils.intxyz(HD2, CD2, dHD_CD, CG, dHD_CD_CG, CE2, dHD_CD_CE, 3);
                BondedUtils.intxyz(HE1, CE1, dHE_CE, CD1, dHE_CE_CD, CZ, dHE_CE_CZ, 3);
                BondedUtils.intxyz(HE2, CE2, dHE_CE, CD2, dHE_CE_CD, CZ, dHE_CE_CZ, 3);
                BondedUtils.intxyz(OH, CZ, dOH_CZ, CE1, dOH_CZ_CE, CE2, dOH_CZ_CE, 3);
                if (rotamer.length == 3) {
                    BondedUtils.intxyz(HH, OH, dHH_OH, CZ, dHH_OH_CZ, CE2, rotamer.chi3, 0);
                    break;
                }
                BondedUtils.intxyz(HH, OH, dHH_OH, CZ, dHH_OH_CZ, CE2, 0.0, 0);
                break;
            }
            case TYD: {
                Atom CA = (Atom)residue.getAtomNode("CA");
                Atom CB = (Atom)residue.getAtomNode("CB");
                Atom N = (Atom)residue.getAtomNode("N");
                Atom CG = (Atom)residue.getAtomNode("CG");
                Atom CD1 = (Atom)residue.getAtomNode("CD1");
                Atom CD2 = (Atom)residue.getAtomNode("CD2");
                Atom CE1 = (Atom)residue.getAtomNode("CE1");
                Atom CE2 = (Atom)residue.getAtomNode("CE2");
                Atom CZ = (Atom)residue.getAtomNode("CZ");
                Atom HB2 = (Atom)residue.getAtomNode("HB2");
                Atom HB3 = (Atom)residue.getAtomNode("HB3");
                Atom HD1 = (Atom)residue.getAtomNode("HD1");
                Atom HD2 = (Atom)residue.getAtomNode("HD2");
                Atom HE1 = (Atom)residue.getAtomNode("HE1");
                Atom HE2 = (Atom)residue.getAtomNode("HE2");
                Atom OH = (Atom)residue.getAtomNode("OH");
                Bond CG_CB = CG.getBond(CB);
                Bond CD_CG = CD1.getBond(CG);
                Bond CE_CD = CE1.getBond(CD1);
                Bond HB_CB = HB2.getBond(CB);
                double dCG_CB = CG_CB.bondType.distance;
                double dCD_CG = CD_CG.bondType.distance;
                double dCE_CD = CE_CD.bondType.distance;
                double dHB_CB = HB_CB.bondType.distance;
                double dHD_CD = HD1.getBond((Atom)CD1).bondType.distance;
                double dHE_CE = HE1.getBond((Atom)CE1).bondType.distance;
                double dOH_CZ = OH.getBond((Atom)CZ).bondType.distance;
                double dCG_CB_CA = RotamerLibrary.getAngle(name, CG, CB, CA);
                double dCD_CG_CB = RotamerLibrary.getAngle(name, CD1, CG, CB);
                double dCE_CD_CG = RotamerLibrary.getAngle(name, CE1, CD1, CG);
                double dHB_CB_CA = RotamerLibrary.getAngle(name, HB2, CB, CA);
                double dHD_CD_CG = RotamerLibrary.getAngle(name, HD1, CD1, CG);
                double dHD_CD_CE = RotamerLibrary.getAngle(name, HD1, CD1, CE1);
                double dHE_CE_CD = RotamerLibrary.getAngle(name, HE1, CE1, CD1);
                double dHE_CE_CZ = RotamerLibrary.getAngle(name, HE1, CE1, CZ);
                double dOH_CZ_CE = RotamerLibrary.getAngle(name, OH, CZ, CE1);
                BondedUtils.intxyz(CG, CB, dCG_CB, CA, dCG_CB_CA, N, rotamer.chi1, 0);
                BondedUtils.intxyz(CD1, CG, dCD_CG, CB, dCD_CG_CB, CA, rotamer.chi2, 0);
                BondedUtils.intxyz(CD2, CG, dCD_CG, CB, dCD_CG_CB, CA, rotamer.chi2 + 180.0, 0);
                BondedUtils.intxyz(CE1, CD1, dCE_CD, CG, dCE_CD_CG, CB, 180.0, 0);
                BondedUtils.intxyz(CE2, CD2, dCE_CD, CG, dCE_CD_CG, CB, 180.0, 0);
                RotamerLibrary.applyCZ(name, CZ, CG, CE1, CD1, CE2, CD2);
                BondedUtils.intxyz(HB2, CB, dHB_CB, CA, dHB_CB_CA, CG, 109.4, 1);
                BondedUtils.intxyz(HB3, CB, dHB_CB, CA, dHB_CB_CA, CG, 109.4, -1);
                BondedUtils.intxyz(HD1, CD1, dHD_CD, CG, dHD_CD_CG, CE1, dHD_CD_CE, 3);
                BondedUtils.intxyz(HD2, CD2, dHD_CD, CG, dHD_CD_CG, CE2, dHD_CD_CE, 3);
                BondedUtils.intxyz(HE1, CE1, dHE_CE, CD1, dHE_CE_CD, CZ, dHE_CE_CZ, 3);
                BondedUtils.intxyz(HE2, CE2, dHE_CE, CD2, dHE_CE_CD, CZ, dHE_CE_CZ, 3);
                BondedUtils.intxyz(OH, CZ, dOH_CZ, CE1, dOH_CZ_CE, CE2, dOH_CZ_CE, 3);
                break;
            }
            case TRP: {
                Atom CA = (Atom)residue.getAtomNode("CA");
                Atom CB = (Atom)residue.getAtomNode("CB");
                Atom N = (Atom)residue.getAtomNode("N");
                Atom CG = (Atom)residue.getAtomNode("CG");
                Atom CD1 = (Atom)residue.getAtomNode("CD1");
                Atom CD2 = (Atom)residue.getAtomNode("CD2");
                Atom NE1 = (Atom)residue.getAtomNode("NE1");
                Atom CE2 = (Atom)residue.getAtomNode("CE2");
                Atom CE3 = (Atom)residue.getAtomNode("CE3");
                Atom CZ2 = (Atom)residue.getAtomNode("CZ2");
                Atom CZ3 = (Atom)residue.getAtomNode("CZ3");
                Atom CH2 = (Atom)residue.getAtomNode("CH2");
                Atom HB2 = (Atom)residue.getAtomNode("HB2");
                Atom HB3 = (Atom)residue.getAtomNode("HB3");
                Atom HD1 = (Atom)residue.getAtomNode("HD1");
                Atom HE1 = (Atom)residue.getAtomNode("HE1");
                Atom HE3 = (Atom)residue.getAtomNode("HE3");
                Atom HZ2 = (Atom)residue.getAtomNode("HZ2");
                Atom HZ3 = (Atom)residue.getAtomNode("HZ3");
                Atom HH2 = (Atom)residue.getAtomNode("HH2");
                Bond CG_CB = CG.getBond(CB);
                Bond CD1_CG = CD1.getBond(CG);
                Bond CD2_CG = CD2.getBond(CG);
                Bond NE1_CD1 = NE1.getBond(CD1);
                Bond CE2_NE1 = CE2.getBond(NE1);
                Bond CE3_CD2 = CE3.getBond(CD2);
                Bond CZ2_CE2 = CZ2.getBond(CE2);
                Bond CZ3_CE3 = CZ3.getBond(CE3);
                Bond CH2_CZ2 = CH2.getBond(CZ2);
                Bond HB_CB = HB2.getBond(CB);
                Bond HD1_CD1 = HD1.getBond(CD1);
                Bond HE1_NE1 = HE1.getBond(NE1);
                Bond HE3_CE3 = HE3.getBond(CE3);
                Bond HZ2_CZ2 = HZ2.getBond(CZ2);
                Bond HZ3_CZ3 = HZ3.getBond(CZ3);
                Bond HH2_CH2 = HH2.getBond(CH2);
                double dCG_CB = CG_CB.bondType.distance;
                double dCD1_CG = CD1_CG.bondType.distance;
                double dCD2_CG = CD2_CG.bondType.distance;
                double dNE1_CD1 = NE1_CD1.bondType.distance;
                double dCE2_NE1 = CE2_NE1.bondType.distance;
                double dCE3_CD2 = CE3_CD2.bondType.distance;
                double dCZ2_CE2 = CZ2_CE2.bondType.distance;
                double dCZ3_CE3 = CZ3_CE3.bondType.distance;
                double dCH2_CZ2 = CH2_CZ2.bondType.distance;
                double dHB_CB = HB_CB.bondType.distance;
                double dHD1_CD1 = HD1_CD1.bondType.distance;
                double dHE1_NE1 = HE1_NE1.bondType.distance;
                double dHE3_CE3 = HE3_CE3.bondType.distance;
                double dHZ2_CZ2 = HZ2_CZ2.bondType.distance;
                double dHZ3_CZ3 = HZ3_CZ3.bondType.distance;
                double dHH2_CH2 = HH2_CH2.bondType.distance;
                double dCG_CB_CA = RotamerLibrary.getAngle(name, CG, CB, CA);
                double dCD1_CG_CB = RotamerLibrary.getAngle(name, CD1, CG, CB);
                double dCD2_CG_CB = RotamerLibrary.getAngle(name, CD2, CG, CB);
                double dNE1_CD1_CG = RotamerLibrary.getAngle(name, NE1, CD1, CG);
                double dCE2_NE1_CD1 = RotamerLibrary.getAngle(name, CE2, NE1, CD1);
                double dCE3_CD2_CE2 = RotamerLibrary.getAngle(name, CE3, CD2, CE2);
                double dCZ2_CE2_CD2 = RotamerLibrary.getAngle(name, CZ2, CE2, CD2);
                double dCZ3_CE3_CD2 = RotamerLibrary.getAngle(name, CZ3, CE3, CD2);
                double dCH2_CZ2_CE2 = RotamerLibrary.getAngle(name, CH2, CZ2, CE2);
                double dHB_CB_CA = RotamerLibrary.getAngle(name, HB2, CB, CA);
                double dHD1_CD1_CG = RotamerLibrary.getAngle(name, HD1, CD1, CG);
                double dHD1_CD1_NE1 = RotamerLibrary.getAngle(name, HD1, CD1, NE1);
                double dHE1_NE1_CD1 = RotamerLibrary.getAngle(name, HE1, NE1, CD1);
                double dHE1_NE1_CE2 = RotamerLibrary.getAngle(name, HE1, NE1, CE2);
                double dHE3_CE3_CD2 = RotamerLibrary.getAngle(name, HE3, CE3, CD2);
                double dHE3_CE3_CZ3 = RotamerLibrary.getAngle(name, HE3, CE3, CZ3);
                double dHZ2_CZ2_CE2 = RotamerLibrary.getAngle(name, HZ2, CZ2, CE2);
                double dHZ2_CZ2_CH2 = RotamerLibrary.getAngle(name, HZ2, CZ2, CH2);
                double dHZ3_CZ3_CE3 = RotamerLibrary.getAngle(name, HZ3, CZ3, CE3);
                double dHZ3_CZ3_CH2 = RotamerLibrary.getAngle(name, HZ3, CZ3, CH2);
                double dHH2_CH2_CZ2 = RotamerLibrary.getAngle(name, HH2, CH2, CZ2);
                double dHH2_CH2_CZ3 = RotamerLibrary.getAngle(name, HH2, CH2, CZ3);
                BondedUtils.intxyz(CG, CB, dCG_CB, CA, dCG_CB_CA, N, rotamer.chi1, 0);
                BondedUtils.intxyz(CD1, CG, dCD1_CG, CB, dCD1_CG_CB, CA, rotamer.chi2, 0);
                BondedUtils.intxyz(CD2, CG, dCD2_CG, CB, dCD2_CG_CB, CA, rotamer.chi2 + 180.0, 0);
                BondedUtils.intxyz(NE1, CD1, dNE1_CD1, CG, dNE1_CD1_CG, CD2, 0.0, 0);
                BondedUtils.intxyz(CE2, NE1, dCE2_NE1, CD1, dCE2_NE1_CD1, CG, 0.0, 0);
                BondedUtils.intxyz(CE3, CD2, dCE3_CD2, CE2, dCE3_CD2_CE2, NE1, 180.0, 0);
                BondedUtils.intxyz(CZ2, CE2, dCZ2_CE2, CD2, dCZ2_CE2_CD2, CE3, 0.0, 0);
                BondedUtils.intxyz(CZ3, CE3, dCZ3_CE3, CD2, dCZ3_CE3_CD2, CE2, 0.0, 0);
                BondedUtils.intxyz(CH2, CZ2, dCH2_CZ2, CE2, dCH2_CZ2_CE2, CD2, 0.0, 0);
                BondedUtils.intxyz(HB2, CB, dHB_CB, CA, dHB_CB_CA, CG, 109.4, 1);
                BondedUtils.intxyz(HB3, CB, dHB_CB, CA, dHB_CB_CA, CG, 109.4, -1);
                BondedUtils.intxyz(HD1, CD1, dHD1_CD1, CG, dHD1_CD1_CG, NE1, dHD1_CD1_NE1, 3);
                BondedUtils.intxyz(HE1, NE1, dHE1_NE1, CD1, dHE1_NE1_CD1, CE2, dHE1_NE1_CE2, 3);
                BondedUtils.intxyz(HE3, CE3, dHE3_CE3, CD2, dHE3_CE3_CD2, CZ3, dHE3_CE3_CZ3, 3);
                BondedUtils.intxyz(HZ2, CZ2, dHZ2_CZ2, CE2, dHZ2_CZ2_CE2, CH2, dHZ2_CZ2_CH2, 3);
                BondedUtils.intxyz(HZ3, CZ3, dHZ3_CZ3, CE3, dHZ3_CZ3_CE3, CH2, dHZ3_CZ3_CH2, 3);
                BondedUtils.intxyz(HH2, CH2, dHH2_CH2, CZ2, dHH2_CH2_CZ2, CZ3, dHH2_CH2_CZ3, 3);
                break;
            }
            case HIS: {
                Atom CA = (Atom)residue.getAtomNode("CA");
                Atom CB = (Atom)residue.getAtomNode("CB");
                Atom N = (Atom)residue.getAtomNode("N");
                Atom CG = (Atom)residue.getAtomNode("CG");
                Atom ND1 = (Atom)residue.getAtomNode("ND1");
                Atom CD2 = (Atom)residue.getAtomNode("CD2");
                Atom CE1 = (Atom)residue.getAtomNode("CE1");
                Atom NE2 = (Atom)residue.getAtomNode("NE2");
                Atom HB2 = (Atom)residue.getAtomNode("HB2");
                Atom HB3 = (Atom)residue.getAtomNode("HB3");
                Atom HD1 = (Atom)residue.getAtomNode("HD1");
                Atom HD2 = (Atom)residue.getAtomNode("HD2");
                Atom HE1 = (Atom)residue.getAtomNode("HE1");
                Atom HE2 = (Atom)residue.getAtomNode("HE2");
                Bond CG_CB = CG.getBond(CB);
                Bond ND1_CG = ND1.getBond(CG);
                Bond CD2_CG = CD2.getBond(CG);
                Bond CE1_ND1 = CE1.getBond(ND1);
                Bond NE2_CD2 = NE2.getBond(CD2);
                Bond HB_CB = HB2.getBond(CB);
                Bond HD1_ND1 = HD1.getBond(ND1);
                Bond HD2_CD2 = HD2.getBond(CD2);
                Bond HE1_CE1 = HE1.getBond(CE1);
                Bond HE2_NE2 = HE2.getBond(NE2);
                double dCG_CB = CG_CB.bondType.distance;
                double dND1_CG = ND1_CG.bondType.distance;
                double dCD2_CG = CD2_CG.bondType.distance;
                double dCE1_ND1 = CE1_ND1.bondType.distance;
                double dNE2_CD2 = NE2_CD2.bondType.distance;
                double dHB_CB = HB_CB.bondType.distance;
                double dHD1_ND1 = HD1_ND1.bondType.distance;
                double dHD2_CD2 = HD2_CD2.bondType.distance;
                double dHE1_CE1 = HE1_CE1.bondType.distance;
                double dHE2_NE2 = HE2_NE2.bondType.distance;
                double dCG_CB_CA = RotamerLibrary.getAngle(name, CG, CB, CA);
                double dND1_CG_CB = RotamerLibrary.getAngle(name, ND1, CG, CB);
                double dCD2_CG_CB = RotamerLibrary.getAngle(name, CD2, CG, CB);
                double dCE1_ND1_CG = RotamerLibrary.getAngle(name, CE1, ND1, CG);
                double dNE2_CD2_CG = RotamerLibrary.getAngle(name, NE2, CD2, CG);
                double dHB_CB_CA = RotamerLibrary.getAngle(name, HB2, CB, CA);
                double dHD1_ND1_CG = RotamerLibrary.getAngle(name, HD1, ND1, CG);
                double dHD1_ND1_CE1 = RotamerLibrary.getAngle(name, HD1, ND1, CE1);
                double dHD2_CD2_CG = RotamerLibrary.getAngle(name, HD2, CD2, CG);
                double dHD2_CD2_NE2 = RotamerLibrary.getAngle(name, HD2, CD2, NE2);
                double dHE1_CE1_ND1 = RotamerLibrary.getAngle(name, HE1, CE1, ND1);
                double dHE1_CE1_NE2 = RotamerLibrary.getAngle(name, HE1, CE1, NE2);
                double dHE2_NE2_CD2 = RotamerLibrary.getAngle(name, HE2, NE2, CD2);
                double dHE2_NE2_CE1 = RotamerLibrary.getAngle(name, HE2, NE2, CE1);
                BondedUtils.intxyz(CG, CB, dCG_CB, CA, dCG_CB_CA, N, rotamer.chi1, 0);
                BondedUtils.intxyz(ND1, CG, dND1_CG, CB, dND1_CG_CB, CA, rotamer.chi2, 0);
                BondedUtils.intxyz(CD2, CG, dCD2_CG, CB, dCD2_CG_CB, CA, rotamer.chi2 + 180.0, 0);
                BondedUtils.intxyz(CE1, ND1, dCE1_ND1, CG, dCE1_ND1_CG, CD2, 0.0, 0);
                BondedUtils.intxyz(NE2, CD2, dNE2_CD2, CG, dNE2_CD2_CG, ND1, 0.0, 0);
                BondedUtils.intxyz(HB2, CB, dHB_CB, CA, dHB_CB_CA, CG, 109.4, 1);
                BondedUtils.intxyz(HB3, CB, dHB_CB, CA, dHB_CB_CA, CG, 109.4, -1);
                BondedUtils.intxyz(HD1, ND1, dHD1_ND1, CG, dHD1_ND1_CG, CE1, dHD1_ND1_CE1, 3);
                BondedUtils.intxyz(HD2, CD2, dHD2_CD2, CG, dHD2_CD2_CG, NE2, dHD2_CD2_NE2, 3);
                BondedUtils.intxyz(HE1, CE1, dHE1_CE1, ND1, dHE1_CE1_ND1, NE2, dHE1_CE1_NE2, 3);
                BondedUtils.intxyz(HE2, NE2, dHE2_NE2, CD2, dHE2_NE2_CD2, CE1, dHE2_NE2_CE1, 3);
                break;
            }
            case HID: {
                Atom CA = (Atom)residue.getAtomNode("CA");
                Atom CB = (Atom)residue.getAtomNode("CB");
                Atom N = (Atom)residue.getAtomNode("N");
                Atom CG = (Atom)residue.getAtomNode("CG");
                Atom ND1 = (Atom)residue.getAtomNode("ND1");
                Atom CD2 = (Atom)residue.getAtomNode("CD2");
                Atom CE1 = (Atom)residue.getAtomNode("CE1");
                Atom NE2 = (Atom)residue.getAtomNode("NE2");
                Atom HB2 = (Atom)residue.getAtomNode("HB2");
                Atom HB3 = (Atom)residue.getAtomNode("HB3");
                Atom HD1 = (Atom)residue.getAtomNode("HD1");
                Atom HD2 = (Atom)residue.getAtomNode("HD2");
                Atom HE1 = (Atom)residue.getAtomNode("HE1");
                Bond CG_CB = CG.getBond(CB);
                Bond ND1_CG = ND1.getBond(CG);
                Bond CD2_CG = CD2.getBond(CG);
                Bond CE1_ND1 = CE1.getBond(ND1);
                Bond NE2_CD2 = NE2.getBond(CD2);
                Bond HB_CB = HB2.getBond(CB);
                Bond HD1_ND1 = HD1.getBond(ND1);
                Bond HD2_CD2 = HD2.getBond(CD2);
                Bond HE1_CE1 = HE1.getBond(CE1);
                double dCG_CB = CG_CB.bondType.distance;
                double dND1_CG = ND1_CG.bondType.distance;
                double dCD2_CG = CD2_CG.bondType.distance;
                double dCE1_ND1 = CE1_ND1.bondType.distance;
                double dNE2_CD2 = NE2_CD2.bondType.distance;
                double dHB_CB = HB_CB.bondType.distance;
                double dHD1_ND1 = HD1_ND1.bondType.distance;
                double dHD2_CD2 = HD2_CD2.bondType.distance;
                double dHE1_CE1 = HE1_CE1.bondType.distance;
                double dCG_CB_CA = RotamerLibrary.getAngle(name, CG, CB, CA);
                double dND1_CG_CB = RotamerLibrary.getAngle(name, ND1, CG, CB);
                double dCD2_CG_CB = RotamerLibrary.getAngle(name, CD2, CG, CB);
                double dCE1_ND1_CG = RotamerLibrary.getAngle(name, CE1, ND1, CG);
                double dNE2_CD2_CG = RotamerLibrary.getAngle(name, NE2, CD2, CG);
                double dHB_CB_CA = RotamerLibrary.getAngle(name, HB2, CB, CA);
                double dHD1_ND1_CG = RotamerLibrary.getAngle(name, HD1, ND1, CG);
                double dHD1_ND1_CE1 = RotamerLibrary.getAngle(name, HD1, ND1, CE1);
                double dHD2_CD2_CG = RotamerLibrary.getAngle(name, HD2, CD2, CG);
                double dHD2_CD2_NE2 = RotamerLibrary.getAngle(name, HD2, CD2, NE2);
                double dHE1_CE1_ND1 = RotamerLibrary.getAngle(name, HE1, CE1, ND1);
                double dHE1_CE1_NE2 = RotamerLibrary.getAngle(name, HE1, CE1, NE2);
                BondedUtils.intxyz(CG, CB, dCG_CB, CA, dCG_CB_CA, N, rotamer.chi1, 0);
                BondedUtils.intxyz(ND1, CG, dND1_CG, CB, dND1_CG_CB, CA, rotamer.chi2, 0);
                BondedUtils.intxyz(CD2, CG, dCD2_CG, CB, dCD2_CG_CB, CA, rotamer.chi2 + 180.0, 0);
                BondedUtils.intxyz(CE1, ND1, dCE1_ND1, CG, dCE1_ND1_CG, CD2, 0.0, 0);
                BondedUtils.intxyz(NE2, CD2, dNE2_CD2, CG, dNE2_CD2_CG, ND1, 0.0, 0);
                BondedUtils.intxyz(HB2, CB, dHB_CB, CA, dHB_CB_CA, CG, 109.4, 1);
                BondedUtils.intxyz(HB3, CB, dHB_CB, CA, dHB_CB_CA, CG, 109.4, -1);
                BondedUtils.intxyz(HD1, ND1, dHD1_ND1, CG, dHD1_ND1_CG, CE1, dHD1_ND1_CE1, 3);
                BondedUtils.intxyz(HD2, CD2, dHD2_CD2, CG, dHD2_CD2_CG, NE2, dHD2_CD2_NE2, 3);
                BondedUtils.intxyz(HE1, CE1, dHE1_CE1, ND1, dHE1_CE1_ND1, NE2, dHE1_CE1_NE2, 3);
                if (!rotamer.isTitrating) break;
                Atom HE2 = (Atom)residue.getAtomNode("HE2");
                Bond HE2_NE2 = HE2.getBond(NE2);
                double dHE2_NE2 = HE2_NE2.bondType.distance;
                double dHE2_NE2_CD2 = RotamerLibrary.getAngle(name, HE2, NE2, CD2);
                double dHE2_NE2_CE1 = RotamerLibrary.getAngle(name, HE2, NE2, CE1);
                BondedUtils.intxyz(HE2, NE2, dHE2_NE2, CD2, dHE2_NE2_CD2, CE1, dHE2_NE2_CE1, 3);
                break;
            }
            case HIE: {
                Atom CA = (Atom)residue.getAtomNode("CA");
                Atom CB = (Atom)residue.getAtomNode("CB");
                Atom N = (Atom)residue.getAtomNode("N");
                Atom CG = (Atom)residue.getAtomNode("CG");
                Atom ND1 = (Atom)residue.getAtomNode("ND1");
                Atom CD2 = (Atom)residue.getAtomNode("CD2");
                Atom CE1 = (Atom)residue.getAtomNode("CE1");
                Atom NE2 = (Atom)residue.getAtomNode("NE2");
                Atom HB2 = (Atom)residue.getAtomNode("HB2");
                Atom HB3 = (Atom)residue.getAtomNode("HB3");
                Atom HD2 = (Atom)residue.getAtomNode("HD2");
                Atom HE1 = (Atom)residue.getAtomNode("HE1");
                Atom HE2 = (Atom)residue.getAtomNode("HE2");
                Bond CG_CB = CG.getBond(CB);
                Bond ND1_CG = ND1.getBond(CG);
                Bond CD2_CG = CD2.getBond(CG);
                Bond CE1_ND1 = CE1.getBond(ND1);
                Bond NE2_CD2 = NE2.getBond(CD2);
                Bond HB_CB = HB2.getBond(CB);
                Bond HD2_CD2 = HD2.getBond(CD2);
                Bond HE1_CE1 = HE1.getBond(CE1);
                Bond HE2_NE2 = HE2.getBond(NE2);
                double dCG_CB = CG_CB.bondType.distance;
                double dND1_CG = ND1_CG.bondType.distance;
                double dCD2_CG = CD2_CG.bondType.distance;
                double dCE1_ND1 = CE1_ND1.bondType.distance;
                double dNE2_CD2 = NE2_CD2.bondType.distance;
                double dHB_CB = HB_CB.bondType.distance;
                double dHD2_CD2 = HD2_CD2.bondType.distance;
                double dHE1_CE1 = HE1_CE1.bondType.distance;
                double dHE2_NE2 = HE2_NE2.bondType.distance;
                double dCG_CB_CA = RotamerLibrary.getAngle(name, CG, CB, CA);
                double dND1_CG_CB = RotamerLibrary.getAngle(name, ND1, CG, CB);
                double dCD2_CG_CB = RotamerLibrary.getAngle(name, CD2, CG, CB);
                double dCE1_ND1_CG = RotamerLibrary.getAngle(name, CE1, ND1, CG);
                double dNE2_CD2_CG = RotamerLibrary.getAngle(name, NE2, CD2, CG);
                double dHB_CB_CA = RotamerLibrary.getAngle(name, HB2, CB, CA);
                double dHD2_CD2_CG = RotamerLibrary.getAngle(name, HD2, CD2, CG);
                double dHD2_CD2_NE2 = RotamerLibrary.getAngle(name, HD2, CD2, NE2);
                double dHE1_CE1_ND1 = RotamerLibrary.getAngle(name, HE1, CE1, ND1);
                double dHE1_CE1_NE2 = RotamerLibrary.getAngle(name, HE1, CE1, NE2);
                double dHE2_NE2_CD2 = RotamerLibrary.getAngle(name, HE2, NE2, CD2);
                double dHE2_NE2_CE1 = RotamerLibrary.getAngle(name, HE2, NE2, CE1);
                BondedUtils.intxyz(CG, CB, dCG_CB, CA, dCG_CB_CA, N, rotamer.chi1, 0);
                BondedUtils.intxyz(ND1, CG, dND1_CG, CB, dND1_CG_CB, CA, rotamer.chi2, 0);
                BondedUtils.intxyz(CD2, CG, dCD2_CG, CB, dCD2_CG_CB, CA, rotamer.chi2 + 180.0, 0);
                BondedUtils.intxyz(CE1, ND1, dCE1_ND1, CG, dCE1_ND1_CG, CD2, 0.0, 0);
                BondedUtils.intxyz(NE2, CD2, dNE2_CD2, CG, dNE2_CD2_CG, ND1, 0.0, 0);
                BondedUtils.intxyz(HB2, CB, dHB_CB, CA, dHB_CB_CA, CG, 109.4, 1);
                BondedUtils.intxyz(HB3, CB, dHB_CB, CA, dHB_CB_CA, CG, 109.4, -1);
                BondedUtils.intxyz(HD2, CD2, dHD2_CD2, CG, dHD2_CD2_CG, NE2, dHD2_CD2_NE2, 3);
                BondedUtils.intxyz(HE1, CE1, dHE1_CE1, ND1, dHE1_CE1_ND1, NE2, dHE1_CE1_NE2, 3);
                BondedUtils.intxyz(HE2, NE2, dHE2_NE2, CD2, dHE2_NE2_CD2, CE1, dHE2_NE2_CE1, 3);
                if (!rotamer.isTitrating) break;
                Atom HD1 = (Atom)residue.getAtomNode("HD1");
                Bond HD1_ND1 = HD1.getBond(ND1);
                double dHD1_ND1 = HD1_ND1.bondType.distance;
                double dHD1_ND1_CG = RotamerLibrary.getAngle(name, HD1, ND1, CG);
                double dHD1_ND1_CE1 = RotamerLibrary.getAngle(name, HD1, ND1, CE1);
                BondedUtils.intxyz(HD1, ND1, dHD1_ND1, CG, dHD1_ND1_CG, CE1, dHD1_ND1_CE1, 3);
                break;
            }
            case ASP: {
                Atom CA = (Atom)residue.getAtomNode("CA");
                Atom CB = (Atom)residue.getAtomNode("CB");
                Atom N = (Atom)residue.getAtomNode("N");
                Atom CG = (Atom)residue.getAtomNode("CG");
                Atom OD1 = (Atom)residue.getAtomNode("OD1");
                Atom OD2 = (Atom)residue.getAtomNode("OD2");
                Atom HB2 = (Atom)residue.getAtomNode("HB2");
                Atom HB3 = (Atom)residue.getAtomNode("HB3");
                Bond CG_CB = CG.getBond(CB);
                Bond OD1_CG = OD1.getBond(CG);
                Bond OD2_CG = OD2.getBond(CG);
                Bond HB_CB = HB2.getBond(CB);
                double dCG_CB = CG_CB.bondType.distance;
                double dOD1_CG = OD1_CG.bondType.distance;
                double dOD2_CG = OD2_CG.bondType.distance;
                double dHB_CB = HB_CB.bondType.distance;
                Angle CG_CB_CA = CG.getAngle(CB, CA);
                Angle OD1_CG_CB = OD1.getAngle(CG, CB);
                Angle OD2_CG_CB = OD2.getAngle(CG, CB);
                Angle HB_CB_CA = HB2.getAngle(CB, CA);
                double dCG_CB_CA = CG_CB_CA.angleType.angle[CG_CB_CA.nh];
                double dOD1_CG_CB = OD1_CG_CB.angleType.angle[OD1_CG_CB.nh];
                double dOD2_CG_CB = OD2_CG_CB.angleType.angle[OD2_CG_CB.nh];
                double dHB_CB_CA = HB_CB_CA.angleType.angle[HB_CB_CA.nh];
                BondedUtils.intxyz(CG, CB, dCG_CB, CA, dCG_CB_CA, N, rotamer.chi1, 0);
                BondedUtils.intxyz(OD1, CG, dOD1_CG, CB, dOD1_CG_CB, CA, rotamer.chi2, 0);
                BondedUtils.intxyz(OD2, CG, dOD2_CG, CB, dOD2_CG_CB, OD1, 126.0, 1);
                BondedUtils.intxyz(HB2, CB, dHB_CB, CA, dHB_CB_CA, CG, 107.9, 1);
                BondedUtils.intxyz(HB3, CB, dHB_CB, CA, dHB_CB_CA, CG, 107.9, -1);
                if (!rotamer.isTitrating) break;
                Atom HD2 = (Atom)residue.getAtomNode("HD2");
                Bond HD2_OD2 = HD2.getBond(OD2);
                double dHD2_OD2 = HD2_OD2.bondType.distance;
                Angle HD2_OD2_CG = HD2.getAngle(OD2, CG);
                double dHD2_OD2_CG = HD2_OD2_CG.angleType.angle[HD2_OD2_CG.nh];
                BondedUtils.intxyz(HD2, OD2, dHD2_OD2, CG, dHD2_OD2_CG, OD1, 0.0, 0);
                break;
            }
            case ASH: {
                Atom CA = (Atom)residue.getAtomNode("CA");
                Atom CB = (Atom)residue.getAtomNode("CB");
                Atom N = (Atom)residue.getAtomNode("N");
                Atom CG = (Atom)residue.getAtomNode("CG");
                Atom OD1 = (Atom)residue.getAtomNode("OD1");
                Atom OD2 = (Atom)residue.getAtomNode("OD2");
                Atom HB2 = (Atom)residue.getAtomNode("HB2");
                Atom HB3 = (Atom)residue.getAtomNode("HB3");
                Atom HD2 = (Atom)residue.getAtomNode("HD2");
                Bond CG_CB = CG.getBond(CB);
                Bond OD1_CG = OD1.getBond(CG);
                Bond OD2_CG = OD2.getBond(CG);
                Bond HB_CB = HB2.getBond(CB);
                Bond HD2_OD2 = HD2.getBond(OD2);
                double dCG_CB = CG_CB.bondType.distance;
                double dOD1_CG = OD1_CG.bondType.distance;
                double dOD2_CG = OD2_CG.bondType.distance;
                double dHB_CB = HB_CB.bondType.distance;
                double dHD2_OD2 = HD2_OD2.bondType.distance;
                Angle CG_CB_CA = CG.getAngle(CB, CA);
                Angle OD1_CG_CB = OD1.getAngle(CG, CB);
                Angle OD2_CG_CB = OD2.getAngle(CG, CB);
                Angle HB_CB_CA = HB2.getAngle(CB, CA);
                Angle HD2_OD2_CG = HD2.getAngle(OD2, CG);
                double dCG_CB_CA = CG_CB_CA.angleType.angle[CG_CB_CA.nh];
                double dOD1_CG_CB = OD1_CG_CB.angleType.angle[OD1_CG_CB.nh];
                double dOD2_CG_CB = OD2_CG_CB.angleType.angle[OD2_CG_CB.nh];
                double dHB_CB_CA = HB_CB_CA.angleType.angle[HB_CB_CA.nh];
                double dHD2_OD2_CG = HD2_OD2_CG.angleType.angle[HD2_OD2_CG.nh];
                BondedUtils.intxyz(CG, CB, dCG_CB, CA, dCG_CB_CA, N, rotamer.chi1, 0);
                BondedUtils.intxyz(OD1, CG, dOD1_CG, CB, dOD1_CG_CB, CA, rotamer.chi2, 0);
                BondedUtils.intxyz(OD2, CG, dOD2_CG, CB, dOD2_CG_CB, OD1, 126.0, 1);
                BondedUtils.intxyz(HB2, CB, dHB_CB, CA, dHB_CB_CA, CG, 107.9, 1);
                BondedUtils.intxyz(HB3, CB, dHB_CB, CA, dHB_CB_CA, CG, 107.9, -1);
                BondedUtils.intxyz(HD2, OD2, dHD2_OD2, CG, dHD2_OD2_CG, OD1, rotamer.chi3, 0);
                break;
            }
            case ASN: {
                Atom CA = (Atom)residue.getAtomNode("CA");
                Atom CB = (Atom)residue.getAtomNode("CB");
                Atom N = (Atom)residue.getAtomNode("N");
                Atom CG = (Atom)residue.getAtomNode("CG");
                Atom OD1 = (Atom)residue.getAtomNode("OD1");
                Atom ND2 = (Atom)residue.getAtomNode("ND2");
                Atom HB2 = (Atom)residue.getAtomNode("HB2");
                Atom HB3 = (Atom)residue.getAtomNode("HB3");
                Atom HD21 = (Atom)residue.getAtomNode("HD21");
                Atom HD22 = (Atom)residue.getAtomNode("HD22");
                Bond CG_CB = CG.getBond(CB);
                Bond OD1_CG = OD1.getBond(CG);
                Bond ND2_CG = ND2.getBond(CG);
                Bond HB_CB = HB2.getBond(CB);
                Bond HD2_ND2 = HD21.getBond(ND2);
                double dCG_CB = CG_CB.bondType.distance;
                double dOD1_CG = OD1_CG.bondType.distance;
                double dND2_CG = ND2_CG.bondType.distance;
                double dHB_CB = HB_CB.bondType.distance;
                double dHD2_ND2 = HD2_ND2.bondType.distance;
                Angle CG_CB_CA = CG.getAngle(CB, CA);
                Angle OD1_CG_CB = OD1.getAngle(CG, CB);
                Angle ND2_CG_CB = ND2.getAngle(CG, CB);
                Angle HB_CB_CA = HB2.getAngle(CB, CA);
                Angle HD2_ND2_CG = HD21.getAngle(ND2, CG);
                double dCG_CB_CA = CG_CB_CA.angleType.angle[CG_CB_CA.nh];
                double dOD1_CG_CB = OD1_CG_CB.angleType.angle[OD1_CG_CB.nh];
                double dND2_CG_CB = ND2_CG_CB.angleType.angle[ND2_CG_CB.nh];
                double dHB_CB_CA = HB_CB_CA.angleType.angle[HB_CB_CA.nh];
                double dHD2_ND2_CG = HD2_ND2_CG.angleType.angle[HD2_ND2_CG.nh];
                BondedUtils.intxyz(CG, CB, dCG_CB, CA, dCG_CB_CA, N, rotamer.chi1, 0);
                BondedUtils.intxyz(OD1, CG, dOD1_CG, CB, dOD1_CG_CB, CA, rotamer.chi2, 0);
                BondedUtils.intxyz(ND2, CG, dND2_CG, CB, dND2_CG_CB, OD1, 124.0, 1);
                BondedUtils.intxyz(HB2, CB, dHB_CB, CA, dHB_CB_CA, CG, 107.9, 1);
                BondedUtils.intxyz(HB3, CB, dHB_CB, CA, dHB_CB_CA, CG, 107.9, -1);
                BondedUtils.intxyz(HD21, ND2, dHD2_ND2, CG, dHD2_ND2_CG, CB, 0.0, 0);
                BondedUtils.intxyz(HD22, ND2, dHD2_ND2, CG, dHD2_ND2_CG, HD21, 120.0, 1);
                break;
            }
            case GLU: {
                Atom CA = (Atom)residue.getAtomNode("CA");
                Atom CB = (Atom)residue.getAtomNode("CB");
                Atom N = (Atom)residue.getAtomNode("N");
                Atom CG = (Atom)residue.getAtomNode("CG");
                Atom CD = (Atom)residue.getAtomNode("CD");
                Atom OE1 = (Atom)residue.getAtomNode("OE1");
                Atom OE2 = (Atom)residue.getAtomNode("OE2");
                Atom HB2 = (Atom)residue.getAtomNode("HB2");
                Atom HB3 = (Atom)residue.getAtomNode("HB3");
                Atom HG2 = (Atom)residue.getAtomNode("HG2");
                Atom HG3 = (Atom)residue.getAtomNode("HG3");
                Bond CG_CB = CG.getBond(CB);
                Bond CD_CG = CD.getBond(CG);
                Bond OE1_CD = OE1.getBond(CD);
                Bond OE2_CD = OE2.getBond(CD);
                Bond HB_CB = HB2.getBond(CB);
                Bond HG_CG = HG2.getBond(CG);
                double dCG_CB = CG_CB.bondType.distance;
                double dCD_CG = CD_CG.bondType.distance;
                double dOE1_CD = OE1_CD.bondType.distance;
                double dOE2_CD = OE2_CD.bondType.distance;
                double dHB_CB = HB_CB.bondType.distance;
                double dHG_CG = HG_CG.bondType.distance;
                Angle CG_CB_CA = CG.getAngle(CB, CA);
                Angle CD_CG_CB = CD.getAngle(CG, CB);
                Angle OE1_CD_CG = OE1.getAngle(CD, CG);
                Angle OE2_CD_CG = OE2.getAngle(CD, CG);
                Angle HB_CB_CA = HB2.getAngle(CB, CA);
                Angle HG_CG_CB = HG2.getAngle(CG, CB);
                double dCG_CB_CA = CG_CB_CA.angleType.angle[CG_CB_CA.nh];
                double dCD_CG_CB = CD_CG_CB.angleType.angle[CD_CG_CB.nh];
                double dOE1_CD_CG = OE1_CD_CG.angleType.angle[OE1_CD_CG.nh];
                double dOE2_CD_CG = OE2_CD_CG.angleType.angle[OE2_CD_CG.nh];
                double dHB_CB_CA = HB_CB_CA.angleType.angle[HB_CB_CA.nh];
                double dHG_CG_CB = HG_CG_CB.angleType.angle[HG_CG_CB.nh];
                BondedUtils.intxyz(CG, CB, dCG_CB, CA, dCG_CB_CA, N, rotamer.chi1, 0);
                BondedUtils.intxyz(CD, CG, dCD_CG, CB, dCD_CG_CB, CA, rotamer.chi2, 0);
                BondedUtils.intxyz(OE1, CD, dOE1_CD, CG, dOE1_CD_CG, CB, rotamer.chi3, 0);
                BondedUtils.intxyz(OE2, CD, dOE2_CD, CG, dOE2_CD_CG, OE1, 126.0, 1);
                BondedUtils.intxyz(HB2, CB, dHB_CB, CA, dHB_CB_CA, CG, 109.4, 1);
                BondedUtils.intxyz(HB3, CB, dHB_CB, CA, dHB_CB_CA, CG, 109.4, -1);
                BondedUtils.intxyz(HG2, CG, dHG_CG, CB, dHG_CG_CB, CD, 107.9, 1);
                BondedUtils.intxyz(HG3, CG, dHG_CG, CB, dHG_CG_CB, CD, 107.9, -1);
                if (!rotamer.isTitrating) break;
                Atom HE2 = (Atom)residue.getAtomNode("HE2");
                Bond HE2_OE2 = HE2.getBond(OE2);
                double dHE2_OE2 = HE2_OE2.bondType.distance;
                Angle HE2_OE2_CD = HE2.getAngle(OE2, CD);
                double dHE2_OE2_CD = HE2_OE2_CD.angleType.angle[HE2_OE2_CD.nh];
                BondedUtils.intxyz(HE2, OE2, dHE2_OE2, CD, dHE2_OE2_CD, OE1, 0.0, 0);
                break;
            }
            case GLH: {
                Atom CA = (Atom)residue.getAtomNode("CA");
                Atom CB = (Atom)residue.getAtomNode("CB");
                Atom N = (Atom)residue.getAtomNode("N");
                Atom CG = (Atom)residue.getAtomNode("CG");
                Atom CD = (Atom)residue.getAtomNode("CD");
                Atom OE1 = (Atom)residue.getAtomNode("OE1");
                Atom OE2 = (Atom)residue.getAtomNode("OE2");
                Atom HB2 = (Atom)residue.getAtomNode("HB2");
                Atom HB3 = (Atom)residue.getAtomNode("HB3");
                Atom HG2 = (Atom)residue.getAtomNode("HG2");
                Atom HG3 = (Atom)residue.getAtomNode("HG3");
                Atom HE2 = (Atom)residue.getAtomNode("HE2");
                Bond CG_CB = CG.getBond(CB);
                Bond CD_CG = CD.getBond(CG);
                Bond OE1_CD = OE1.getBond(CD);
                Bond OE2_CD = OE2.getBond(CD);
                Bond HB_CB = HB2.getBond(CB);
                Bond HG_CG = HG2.getBond(CG);
                Bond HE2_OE2 = HE2.getBond(OE2);
                double dCG_CB = CG_CB.bondType.distance;
                double dCD_CG = CD_CG.bondType.distance;
                double dOE1_CD = OE1_CD.bondType.distance;
                double dOE2_CD = OE2_CD.bondType.distance;
                double dHB_CB = HB_CB.bondType.distance;
                double dHG_CG = HG_CG.bondType.distance;
                double dHE2_OE2 = HE2_OE2.bondType.distance;
                Angle CG_CB_CA = CG.getAngle(CB, CA);
                Angle CD_CG_CB = CD.getAngle(CG, CB);
                Angle OE1_CD_CG = OE1.getAngle(CD, CG);
                Angle OE2_CD_CG = OE2.getAngle(CD, CG);
                Angle HB_CB_CA = HB2.getAngle(CB, CA);
                Angle HG_CG_CB = HG2.getAngle(CG, CB);
                Angle HE2_OE2_CD = HE2.getAngle(OE2, CD);
                double dCG_CB_CA = CG_CB_CA.angleType.angle[CG_CB_CA.nh];
                double dCD_CG_CB = CD_CG_CB.angleType.angle[CD_CG_CB.nh];
                double dOE1_CD_CG = OE1_CD_CG.angleType.angle[OE1_CD_CG.nh];
                double dOE2_CD_CG = OE2_CD_CG.angleType.angle[OE2_CD_CG.nh];
                double dHB_CB_CA = HB_CB_CA.angleType.angle[HB_CB_CA.nh];
                double dHG_CG_CB = HG_CG_CB.angleType.angle[HG_CG_CB.nh];
                double dHE2_OE2_CD = HE2_OE2_CD.angleType.angle[HE2_OE2_CD.nh];
                BondedUtils.intxyz(CG, CB, dCG_CB, CA, dCG_CB_CA, N, rotamer.chi1, 0);
                BondedUtils.intxyz(CD, CG, dCD_CG, CB, dCD_CG_CB, CA, rotamer.chi2, 0);
                BondedUtils.intxyz(OE1, CD, dOE1_CD, CG, dOE1_CD_CG, CB, rotamer.chi3, 0);
                BondedUtils.intxyz(OE2, CD, dOE2_CD, CG, dOE2_CD_CG, OE1, 126.0, 1);
                BondedUtils.intxyz(HB2, CB, dHB_CB, CA, dHB_CB_CA, CG, 109.4, 1);
                BondedUtils.intxyz(HB3, CB, dHB_CB, CA, dHB_CB_CA, CG, 109.4, -1);
                BondedUtils.intxyz(HG2, CG, dHG_CG, CB, dHG_CG_CB, CD, 107.9, 1);
                BondedUtils.intxyz(HG3, CG, dHG_CG, CB, dHG_CG_CB, CD, 107.9, -1);
                BondedUtils.intxyz(HE2, OE2, dHE2_OE2, CD, dHE2_OE2_CD, OE1, rotamer.chi4, 0);
                break;
            }
            case GLN: {
                Atom CA = (Atom)residue.getAtomNode("CA");
                Atom CB = (Atom)residue.getAtomNode("CB");
                Atom N = (Atom)residue.getAtomNode("N");
                Atom CG = (Atom)residue.getAtomNode("CG");
                Atom CD = (Atom)residue.getAtomNode("CD");
                Atom OE1 = (Atom)residue.getAtomNode("OE1");
                Atom NE2 = (Atom)residue.getAtomNode("NE2");
                Atom HB2 = (Atom)residue.getAtomNode("HB2");
                Atom HB3 = (Atom)residue.getAtomNode("HB3");
                Atom HG2 = (Atom)residue.getAtomNode("HG2");
                Atom HG3 = (Atom)residue.getAtomNode("HG3");
                Atom HE21 = (Atom)residue.getAtomNode("HE21");
                Atom HE22 = (Atom)residue.getAtomNode("HE22");
                Bond CG_CB = CG.getBond(CB);
                Bond CD_CG = CD.getBond(CG);
                Bond OE1_CD = OE1.getBond(CD);
                Bond NE2_CD = NE2.getBond(CD);
                Bond HB_CB = HB2.getBond(CB);
                Bond HG_CG = HG2.getBond(CG);
                Bond HE2_NE2 = HE21.getBond(NE2);
                double dCG_CB = CG_CB.bondType.distance;
                double dCD_CG = CD_CG.bondType.distance;
                double dOE1_CD = OE1_CD.bondType.distance;
                double dNE2_CD = NE2_CD.bondType.distance;
                double dHB_CB = HB_CB.bondType.distance;
                double dHG_CG = HG_CG.bondType.distance;
                double dHE2_NE2 = HE2_NE2.bondType.distance;
                Angle CG_CB_CA = CG.getAngle(CB, CA);
                Angle CD_CG_CB = CD.getAngle(CG, CB);
                Angle OE1_CD_CG = OE1.getAngle(CD, CG);
                Angle NE2_CD_CG = NE2.getAngle(CD, CG);
                Angle HB_CB_CA = HB2.getAngle(CB, CA);
                Angle HG_CG_CB = HG2.getAngle(CG, CB);
                Angle HE2_NE2_CD = HE21.getAngle(NE2, CD);
                double dCG_CB_CA = CG_CB_CA.angleType.angle[CG_CB_CA.nh];
                double dCD_CG_CB = CD_CG_CB.angleType.angle[CD_CG_CB.nh];
                double dOE1_CD_CG = OE1_CD_CG.angleType.angle[OE1_CD_CG.nh];
                double dNE2_CD_CG = NE2_CD_CG.angleType.angle[NE2_CD_CG.nh];
                double dHB_CB_CA = HB_CB_CA.angleType.angle[HB_CB_CA.nh];
                double dHG_CG_CB = HG_CG_CB.angleType.angle[HG_CG_CB.nh];
                double dHE2_NE2_CD = HE2_NE2_CD.angleType.angle[HE2_NE2_CD.nh];
                BondedUtils.intxyz(CG, CB, dCG_CB, CA, dCG_CB_CA, N, rotamer.chi1, 0);
                BondedUtils.intxyz(CD, CG, dCD_CG, CB, dCD_CG_CB, CA, rotamer.chi2, 0);
                BondedUtils.intxyz(OE1, CD, dOE1_CD, CG, dOE1_CD_CG, CB, rotamer.chi3, 0);
                BondedUtils.intxyz(NE2, CD, dNE2_CD, CG, dNE2_CD_CG, OE1, 124.0, 1);
                BondedUtils.intxyz(HB2, CB, dHB_CB, CA, dHB_CB_CA, CG, 109.4, 1);
                BondedUtils.intxyz(HB3, CB, dHB_CB, CA, dHB_CB_CA, CG, 109.4, -1);
                BondedUtils.intxyz(HG2, CG, dHG_CG, CB, dHG_CG_CB, CD, 107.9, 1);
                BondedUtils.intxyz(HG3, CG, dHG_CG, CB, dHG_CG_CB, CD, 107.9, -1);
                BondedUtils.intxyz(HE21, NE2, dHE2_NE2, CD, dHE2_NE2_CD, CG, 0.0, 0);
                BondedUtils.intxyz(HE22, NE2, dHE2_NE2, CD, dHE2_NE2_CD, HE21, 120.0, 1);
                break;
            }
            case MET: {
                Atom CA = (Atom)residue.getAtomNode("CA");
                Atom CB = (Atom)residue.getAtomNode("CB");
                Atom N = (Atom)residue.getAtomNode("N");
                Atom CG = (Atom)residue.getAtomNode("CG");
                Atom SD = (Atom)residue.getAtomNode("SD");
                Atom CE = (Atom)residue.getAtomNode("CE");
                Atom HB2 = (Atom)residue.getAtomNode("HB2");
                Atom HB3 = (Atom)residue.getAtomNode("HB3");
                Atom HG2 = (Atom)residue.getAtomNode("HG2");
                Atom HG3 = (Atom)residue.getAtomNode("HG3");
                Atom HE1 = (Atom)residue.getAtomNode("HE1");
                Atom HE2 = (Atom)residue.getAtomNode("HE2");
                Atom HE3 = (Atom)residue.getAtomNode("HE3");
                Bond CG_CB = CG.getBond(CB);
                Bond SD_CG = SD.getBond(CG);
                Bond CE_SD = CE.getBond(SD);
                Bond HB_CB = HB2.getBond(CB);
                Bond HG_CG = HG2.getBond(CG);
                Bond HE_CE = HE1.getBond(CE);
                double dCG_CB = CG_CB.bondType.distance;
                double dSD_CG = SD_CG.bondType.distance;
                double dCE_SD = CE_SD.bondType.distance;
                double dHB_CB = HB_CB.bondType.distance;
                double dHG_CG = HG_CG.bondType.distance;
                double dHE_CE = HE_CE.bondType.distance;
                Angle CG_CB_CA = CG.getAngle(CB, CA);
                Angle SD_CG_CB = SD.getAngle(CG, CB);
                Angle CE_SD_CG = CE.getAngle(SD, CG);
                Angle HB_CB_CA = HB2.getAngle(CB, CA);
                Angle HG_CG_CB = HG2.getAngle(CG, CB);
                Angle HE_CE_SD = HE1.getAngle(CE, SD);
                double dCG_CB_CA = CG_CB_CA.angleType.angle[CG_CB_CA.nh];
                double dSD_CG_CB = SD_CG_CB.angleType.angle[SD_CG_CB.nh];
                double dCE_SD_CG = CE_SD_CG.angleType.angle[CE_SD_CG.nh];
                double dHB_CB_CA = HB_CB_CA.angleType.angle[HB_CB_CA.nh];
                double dHG_CG_CB = HG_CG_CB.angleType.angle[HG_CG_CB.nh];
                double dHE_CE_SD = HE_CE_SD.angleType.angle[HE_CE_SD.nh];
                BondedUtils.intxyz(CG, CB, dCG_CB, CA, dCG_CB_CA, N, rotamer.chi1, 0);
                BondedUtils.intxyz(SD, CG, dSD_CG, CB, dSD_CG_CB, CA, rotamer.chi2, 0);
                BondedUtils.intxyz(CE, SD, dCE_SD, CG, dCE_SD_CG, CB, rotamer.chi3, 0);
                BondedUtils.intxyz(HB2, CB, dHB_CB, CA, dHB_CB_CA, CG, 109.4, 1);
                BondedUtils.intxyz(HB3, CB, dHB_CB, CA, dHB_CB_CA, CG, 109.4, -1);
                BondedUtils.intxyz(HG2, CG, dHG_CG, CB, dHG_CG_CB, SD, 112.0, 1);
                BondedUtils.intxyz(HG3, CG, dHG_CG, CB, dHG_CG_CB, SD, 112.0, -1);
                BondedUtils.intxyz(HE1, CE, dHE_CE, SD, dHE_CE_SD, CG, 180.0, 0);
                BondedUtils.intxyz(HE2, CE, dHE_CE, SD, dHE_CE_SD, HE1, 109.4, 1);
                BondedUtils.intxyz(HE3, CE, dHE_CE, SD, dHE_CE_SD, HE1, 109.4, -1);
                break;
            }
            case LYS: {
                Atom CA = (Atom)residue.getAtomNode("CA");
                Atom CB = (Atom)residue.getAtomNode("CB");
                Atom N = (Atom)residue.getAtomNode("N");
                Atom CG = (Atom)residue.getAtomNode("CG");
                Atom CD = (Atom)residue.getAtomNode("CD");
                Atom CE = (Atom)residue.getAtomNode("CE");
                Atom NZ = (Atom)residue.getAtomNode("NZ");
                Atom HB2 = (Atom)residue.getAtomNode("HB2");
                Atom HB3 = (Atom)residue.getAtomNode("HB3");
                Atom HG2 = (Atom)residue.getAtomNode("HG2");
                Atom HG3 = (Atom)residue.getAtomNode("HG3");
                Atom HD2 = (Atom)residue.getAtomNode("HD2");
                Atom HD3 = (Atom)residue.getAtomNode("HD3");
                Atom HE2 = (Atom)residue.getAtomNode("HE2");
                Atom HE3 = (Atom)residue.getAtomNode("HE3");
                Atom HZ1 = (Atom)residue.getAtomNode("HZ1");
                Atom HZ2 = (Atom)residue.getAtomNode("HZ2");
                Atom HZ3 = (Atom)residue.getAtomNode("HZ3");
                Bond CG_CB = CG.getBond(CB);
                Bond CD_CG = CD.getBond(CG);
                Bond CE_CD = CE.getBond(CD);
                Bond NZ_CE = NZ.getBond(CE);
                Bond HB_CB = HB2.getBond(CB);
                Bond HG_CG = HG2.getBond(CG);
                Bond HD_CD = HD2.getBond(CD);
                Bond HE_CE = HE2.getBond(CE);
                Bond HZ_NZ = HZ1.getBond(NZ);
                double dCG_CB = CG_CB.bondType.distance;
                double dCD_CG = CD_CG.bondType.distance;
                double dCE_CD = CE_CD.bondType.distance;
                double dNZ_CE = NZ_CE.bondType.distance;
                double dHB_CB = HB_CB.bondType.distance;
                double dHG_CG = HG_CG.bondType.distance;
                double dHD_CD = HD_CD.bondType.distance;
                double dHE_CE = HE_CE.bondType.distance;
                double dHZ_NZ = HZ_NZ.bondType.distance;
                Angle CG_CB_CA = CG.getAngle(CB, CA);
                Angle CD_CG_CB = CD.getAngle(CG, CB);
                Angle CE_CD_CG = CE.getAngle(CD, CG);
                Angle NZ_CE_CD = NZ.getAngle(CE, CD);
                Angle HB_CB_CA = HB2.getAngle(CB, CA);
                Angle HG_CG_CB = HG2.getAngle(CG, CB);
                Angle HD_CD_CG = HD2.getAngle(CD, CG);
                Angle HE_CE_CD = HE2.getAngle(CE, CD);
                Angle HZ_NZ_CE = HZ1.getAngle(NZ, CE);
                double dCG_CB_CA = CG_CB_CA.angleType.angle[CG_CB_CA.nh];
                double dCD_CG_CB = CD_CG_CB.angleType.angle[CD_CG_CB.nh];
                double dCE_CD_CG = CE_CD_CG.angleType.angle[CE_CD_CG.nh];
                double dNZ_CE_CD = NZ_CE_CD.angleType.angle[NZ_CE_CD.nh];
                double dHB_CB_CA = HB_CB_CA.angleType.angle[HB_CB_CA.nh];
                double dHG_CG_CB = HG_CG_CB.angleType.angle[HG_CG_CB.nh];
                double dHD_CD_CG = HD_CD_CG.angleType.angle[HD_CD_CG.nh];
                double dHE_CE_CD = HE_CE_CD.angleType.angle[HE_CE_CD.nh];
                double dHZ_NZ_CE = HZ_NZ_CE.angleType.angle[HZ_NZ_CE.nh];
                BondedUtils.intxyz(CG, CB, dCG_CB, CA, dCG_CB_CA, N, rotamer.chi1, 0);
                BondedUtils.intxyz(CD, CG, dCD_CG, CB, dCD_CG_CB, CA, rotamer.chi2, 0);
                BondedUtils.intxyz(CE, CD, dCE_CD, CG, dCE_CD_CG, CB, rotamer.chi3, 0);
                BondedUtils.intxyz(NZ, CE, dNZ_CE, CD, dNZ_CE_CD, CG, rotamer.chi4, 0);
                BondedUtils.intxyz(HB2, CB, dHB_CB, CA, dHB_CB_CA, CG, 109.4, 1);
                BondedUtils.intxyz(HB3, CB, dHB_CB, CA, dHB_CB_CA, CG, 109.4, -1);
                BondedUtils.intxyz(HG2, CG, dHG_CG, CB, dHG_CG_CB, CD, 109.4, 1);
                BondedUtils.intxyz(HG3, CG, dHG_CG, CB, dHG_CG_CB, CD, 109.4, -1);
                BondedUtils.intxyz(HD2, CD, dHD_CD, CG, dHD_CD_CG, CE, 109.4, 1);
                BondedUtils.intxyz(HD3, CD, dHD_CD, CG, dHD_CD_CG, CE, 109.4, -1);
                BondedUtils.intxyz(HE2, CE, dHE_CE, CD, dHE_CE_CD, NZ, 108.8, 1);
                BondedUtils.intxyz(HE3, CE, dHE_CE, CD, dHE_CE_CD, NZ, 108.8, -1);
                BondedUtils.intxyz(HZ1, NZ, dHZ_NZ, CE, dHZ_NZ_CE, CD, 180.0, 0);
                BondedUtils.intxyz(HZ2, NZ, dHZ_NZ, CE, dHZ_NZ_CE, HZ1, 109.5, 1);
                BondedUtils.intxyz(HZ3, NZ, dHZ_NZ, CE, dHZ_NZ_CE, HZ1, 109.5, -1);
                break;
            }
            case LYD: {
                Atom CA = (Atom)residue.getAtomNode("CA");
                Atom CB = (Atom)residue.getAtomNode("CB");
                Atom N = (Atom)residue.getAtomNode("N");
                Atom CG = (Atom)residue.getAtomNode("CG");
                Atom CD = (Atom)residue.getAtomNode("CD");
                Atom CE = (Atom)residue.getAtomNode("CE");
                Atom NZ = (Atom)residue.getAtomNode("NZ");
                Atom HB2 = (Atom)residue.getAtomNode("HB2");
                Atom HB3 = (Atom)residue.getAtomNode("HB3");
                Atom HG2 = (Atom)residue.getAtomNode("HG2");
                Atom HG3 = (Atom)residue.getAtomNode("HG3");
                Atom HD2 = (Atom)residue.getAtomNode("HD2");
                Atom HD3 = (Atom)residue.getAtomNode("HD3");
                Atom HE2 = (Atom)residue.getAtomNode("HE2");
                Atom HE3 = (Atom)residue.getAtomNode("HE3");
                Atom HZ1 = (Atom)residue.getAtomNode("HZ1");
                Atom HZ2 = (Atom)residue.getAtomNode("HZ2");
                Bond CG_CB = CG.getBond(CB);
                Bond CD_CG = CD.getBond(CG);
                Bond CE_CD = CE.getBond(CD);
                Bond NZ_CE = NZ.getBond(CE);
                Bond HB_CB = HB2.getBond(CB);
                Bond HG_CG = HG2.getBond(CG);
                Bond HD_CD = HD2.getBond(CD);
                Bond HE_CE = HE2.getBond(CE);
                Bond HZ_NZ = HZ1.getBond(NZ);
                double dCG_CB = CG_CB.bondType.distance;
                double dCD_CG = CD_CG.bondType.distance;
                double dCE_CD = CE_CD.bondType.distance;
                double dNZ_CE = NZ_CE.bondType.distance;
                double dHB_CB = HB_CB.bondType.distance;
                double dHG_CG = HG_CG.bondType.distance;
                double dHD_CD = HD_CD.bondType.distance;
                double dHE_CE = HE_CE.bondType.distance;
                double dHZ_NZ = HZ_NZ.bondType.distance;
                Angle CG_CB_CA = CG.getAngle(CB, CA);
                Angle CD_CG_CB = CD.getAngle(CG, CB);
                Angle CE_CD_CG = CE.getAngle(CD, CG);
                Angle NZ_CE_CD = NZ.getAngle(CE, CD);
                Angle HB_CB_CA = HB2.getAngle(CB, CA);
                Angle HG_CG_CB = HG2.getAngle(CG, CB);
                Angle HD_CD_CG = HD2.getAngle(CD, CG);
                Angle HE_CE_CD = HE2.getAngle(CE, CD);
                Angle HZ_NZ_CE = HZ1.getAngle(NZ, CE);
                double dCG_CB_CA = CG_CB_CA.angleType.angle[CG_CB_CA.nh];
                double dCD_CG_CB = CD_CG_CB.angleType.angle[CD_CG_CB.nh];
                double dCE_CD_CG = CE_CD_CG.angleType.angle[CE_CD_CG.nh];
                double dNZ_CE_CD = NZ_CE_CD.angleType.angle[NZ_CE_CD.nh];
                double dHB_CB_CA = HB_CB_CA.angleType.angle[HB_CB_CA.nh];
                double dHG_CG_CB = HG_CG_CB.angleType.angle[HG_CG_CB.nh];
                double dHD_CD_CG = HD_CD_CG.angleType.angle[HD_CD_CG.nh];
                double dHE_CE_CD = HE_CE_CD.angleType.angle[HE_CE_CD.nh];
                double dHZ_NZ_CE = HZ_NZ_CE.angleType.angle[HZ_NZ_CE.nh];
                BondedUtils.intxyz(CG, CB, dCG_CB, CA, dCG_CB_CA, N, rotamer.chi1, 0);
                BondedUtils.intxyz(CD, CG, dCD_CG, CB, dCD_CG_CB, CA, rotamer.chi2, 0);
                BondedUtils.intxyz(CE, CD, dCE_CD, CG, dCE_CD_CG, CB, rotamer.chi3, 0);
                BondedUtils.intxyz(NZ, CE, dNZ_CE, CD, dNZ_CE_CD, CG, rotamer.chi4, 0);
                BondedUtils.intxyz(HB2, CB, dHB_CB, CA, dHB_CB_CA, CG, 109.4, 1);
                BondedUtils.intxyz(HB3, CB, dHB_CB, CA, dHB_CB_CA, CG, 109.4, -1);
                BondedUtils.intxyz(HG2, CG, dHG_CG, CB, dHG_CG_CB, CD, 109.4, 1);
                BondedUtils.intxyz(HG3, CG, dHG_CG, CB, dHG_CG_CB, CD, 109.4, -1);
                BondedUtils.intxyz(HD2, CD, dHD_CD, CG, dHD_CD_CG, CE, 109.4, 1);
                BondedUtils.intxyz(HD3, CD, dHD_CD, CG, dHD_CD_CG, CE, 109.4, -1);
                BondedUtils.intxyz(HE2, CE, dHE_CE, CD, dHE_CE_CD, NZ, 108.8, 1);
                BondedUtils.intxyz(HE3, CE, dHE_CE, CD, dHE_CE_CD, NZ, 108.8, -1);
                BondedUtils.intxyz(HZ1, NZ, dHZ_NZ, CE, dHZ_NZ_CE, CD, 180.0, 0);
                BondedUtils.intxyz(HZ2, NZ, dHZ_NZ, CE, dHZ_NZ_CE, HZ1, 109.5, 1);
                if (!rotamer.isTitrating) break;
                Atom HZ3 = (Atom)residue.getAtomNode("HZ3");
                BondedUtils.intxyz(HZ3, NZ, dHZ_NZ, CE, dHZ_NZ_CE, HZ1, 109.5, -1);
                break;
            }
            case ARG: {
                Atom CA = (Atom)residue.getAtomNode("CA");
                Atom CB = (Atom)residue.getAtomNode("CB");
                Atom N = (Atom)residue.getAtomNode("N");
                Atom CG = (Atom)residue.getAtomNode("CG");
                Atom CD = (Atom)residue.getAtomNode("CD");
                Atom NE = (Atom)residue.getAtomNode("NE");
                Atom CZ = (Atom)residue.getAtomNode("CZ");
                Atom NH1 = (Atom)residue.getAtomNode("NH1");
                Atom NH2 = (Atom)residue.getAtomNode("NH2");
                Atom HB2 = (Atom)residue.getAtomNode("HB2");
                Atom HB3 = (Atom)residue.getAtomNode("HB3");
                Atom HG2 = (Atom)residue.getAtomNode("HG2");
                Atom HG3 = (Atom)residue.getAtomNode("HG3");
                Atom HD2 = (Atom)residue.getAtomNode("HD2");
                Atom HD3 = (Atom)residue.getAtomNode("HD3");
                Atom HE = (Atom)residue.getAtomNode("HE");
                Atom HH11 = (Atom)residue.getAtomNode("HH11");
                Atom HH12 = (Atom)residue.getAtomNode("HH12");
                Atom HH21 = (Atom)residue.getAtomNode("HH21");
                Atom HH22 = (Atom)residue.getAtomNode("HH22");
                Bond CG_CB = CG.getBond(CB);
                Bond CD_CG = CD.getBond(CG);
                Bond NE_CD = NE.getBond(CD);
                Bond CZ_NE = CZ.getBond(NE);
                Bond NH_CZ = NH1.getBond(CZ);
                Bond HB_CB = HB2.getBond(CB);
                Bond HG_CG = HG2.getBond(CG);
                Bond HD_CD = HD2.getBond(CD);
                Bond HE_NE = HE.getBond(NE);
                Bond HH_NH = HH11.getBond(NH1);
                double dCG_CB = CG_CB.bondType.distance;
                double dCD_CG = CD_CG.bondType.distance;
                double dNE_CD = NE_CD.bondType.distance;
                double dCZ_NE = CZ_NE.bondType.distance;
                double dNH_CZ = NH_CZ.bondType.distance;
                double dHB_CB = HB_CB.bondType.distance;
                double dHG_CG = HG_CG.bondType.distance;
                double dHD_CD = HD_CD.bondType.distance;
                double dHE_NE = HE_NE.bondType.distance;
                double dHH_NH = HH_NH.bondType.distance;
                Angle CG_CB_CA = CG.getAngle(CB, CA);
                Angle CD_CG_CB = CD.getAngle(CG, CB);
                Angle NE_CD_CG = NE.getAngle(CD, CG);
                Angle CZ_NE_CD = CZ.getAngle(NE, CD);
                Angle NH_CZ_NE = NH1.getAngle(CZ, NE);
                Angle HB_CB_CA = HB2.getAngle(CB, CA);
                Angle HG_CG_CB = HG2.getAngle(CG, CB);
                Angle HD_CD_CG = HD2.getAngle(CD, CG);
                Angle HE_NE_CD = HE.getAngle(NE, CD);
                Angle HH_NH_CZ = HH11.getAngle(NH1, CZ);
                double dCG_CB_CA = CG_CB_CA.angleType.angle[CG_CB_CA.nh];
                double dCD_CG_CB = CD_CG_CB.angleType.angle[CD_CG_CB.nh];
                double dNE_CD_CG = NE_CD_CG.angleType.angle[NE_CD_CG.nh];
                double dCZ_NE_CD = CZ_NE_CD.angleType.angle[CZ_NE_CD.nh];
                double dNH_CZ_NE = NH_CZ_NE.angleType.angle[NH_CZ_NE.nh];
                double dHB_CB_CA = HB_CB_CA.angleType.angle[HB_CB_CA.nh];
                double dHG_CG_CB = HG_CG_CB.angleType.angle[HG_CG_CB.nh];
                double dHD_CD_CG = HD_CD_CG.angleType.angle[HD_CD_CG.nh];
                double dHE_NE_CD = HE_NE_CD.angleType.angle[HE_NE_CD.nh];
                double dHH_NH_CZ = HH_NH_CZ.angleType.angle[HH_NH_CZ.nh];
                BondedUtils.intxyz(CG, CB, dCG_CB, CA, dCG_CB_CA, N, rotamer.chi1, 0);
                BondedUtils.intxyz(CD, CG, dCD_CG, CB, dCD_CG_CB, CA, rotamer.chi2, 0);
                BondedUtils.intxyz(NE, CD, dNE_CD, CG, dNE_CD_CG, CB, rotamer.chi3, 0);
                BondedUtils.intxyz(CZ, NE, dCZ_NE, CD, dCZ_NE_CD, CG, rotamer.chi4, 0);
                BondedUtils.intxyz(NH1, CZ, dNH_CZ, NE, dNH_CZ_NE, CD, 180.0, 0);
                BondedUtils.intxyz(NH2, CZ, dNH_CZ, NE, dNH_CZ_NE, NH1, 120.0, 1);
                BondedUtils.intxyz(HB2, CB, dHB_CB, CA, dHB_CB_CA, CG, 109.4, 1);
                BondedUtils.intxyz(HB3, CB, dHB_CB, CA, dHB_CB_CA, CG, 109.4, -1);
                BondedUtils.intxyz(HG2, CG, dHG_CG, CB, dHG_CG_CB, CD, 109.4, 1);
                BondedUtils.intxyz(HG3, CG, dHG_CG, CB, dHG_CG_CB, CD, 109.4, -1);
                BondedUtils.intxyz(HD2, CD, dHD_CD, CG, dHD_CD_CG, NE, 109.4, 1);
                BondedUtils.intxyz(HD3, CD, dHD_CD, CG, dHD_CD_CG, NE, 109.4, -1);
                BondedUtils.intxyz(HE, NE, dHE_NE, CD, dHE_NE_CD, CZ, 120.0, 1);
                BondedUtils.intxyz(HH11, NH1, dHH_NH, CZ, dHH_NH_CZ, NE, 180.0, 0);
                BondedUtils.intxyz(HH12, NH1, dHH_NH, CZ, dHH_NH_CZ, HH11, 120.0, 1);
                BondedUtils.intxyz(HH21, NH2, dHH_NH, CZ, dHH_NH_CZ, NE, 180.0, 0);
                BondedUtils.intxyz(HH22, NH2, dHH_NH, CZ, dHH_NH_CZ, HH21, 120.0, 1);
                break;
            }
            case UNK: {
                String resName = residue.getName().toUpperCase();
                if (!nonstdRotCache.containsKey(resName)) break;
                nonstdRotCache.get(resName).applyNonstdRotamer(residue, rotamer);
                break;
            }
        }
    }

    private static void applyCZ(AminoAcidUtils.AminoAcid3 resName, Atom CZ, Atom CG, Atom CE1, Atom CD1, Atom CE2, Atom CD2) {
        CZ.moveTo(RotamerLibrary.drawCZ(resName, CZ, CG, CE1, CD1, CE2, CD2));
    }

    private static void applyNABackbone(Residue residue, Rotamer rotamer, Residue prevResidue) {
        Atom O3s = (Atom)residue.getAtomNode("O3'");
        Atom C3s = (Atom)residue.getAtomNode("C3'");
        Atom C4s = (Atom)residue.getAtomNode("C4'");
        Atom C5s = (Atom)residue.getAtomNode("C5'");
        Atom O5s = (Atom)residue.getAtomNode("O5'");
        Atom P = (Atom)residue.getAtomNode("P");
        Atom OP3 = (Atom)residue.getAtomNode("OP3");
        Atom HO5s = (Atom)residue.getAtomNode("HO5'");
        Bond C4s_C5s = C4s.getBond(C5s);
        double dC4s_C5s = C4s_C5s.bondType.distance;
        Angle C3s_C4s_C5s = C3s.getAngle(C4s, C5s);
        double dC3s_C4s_C5s = C3s_C4s_C5s.angleType.angle[C3s_C4s_C5s.nh];
        BondedUtils.intxyz(C5s, C4s, dC4s_C5s, C3s, dC3s_C4s_C5s, O3s, rotamer.chi7, 0);
        Bond C5s_O5s = C5s.getBond(O5s);
        double dC5s_O5s = C5s_O5s.bondType.distance;
        Angle C4s_C5s_O5s = C4s.getAngle(C5s, O5s);
        double dC4s_C5s_O5s = C4s_C5s_O5s.angleType.angle[C4s_C5s_O5s.nh];
        BondedUtils.intxyz(O5s, C5s, dC5s_O5s, C4s, dC4s_C5s_O5s, C3s, rotamer.chi6, 0);
        if (prevResidue == null) {
            if (HO5s != null) {
                Bond O5s_HO5s = O5s.getBond(HO5s);
                double dO5s_HO5s = O5s_HO5s.bondType.distance;
                Angle C5s_O5s_HO5s = C5s.getAngle(O5s, HO5s);
                double dC5s_O5s_HO5s = C5s_O5s_HO5s.angleType.angle[C5s_O5s_HO5s.nh];
                BondedUtils.intxyz(HO5s, O5s, dO5s_HO5s, C5s, dC5s_O5s_HO5s, C4s, rotamer.chi5, 0);
            } else {
                Bond O5s_P = O5s.getBond(P);
                double dO5s_P = O5s_P.bondType.distance;
                Angle C5s_O5s_P = C5s.getAngle(O5s, P);
                double dC5s_O5s_P = C5s_O5s_P.angleType.angle[C5s_O5s_P.nh];
                BondedUtils.intxyz(P, O5s, dO5s_P, C5s, dC5s_O5s_P, C4s, rotamer.chi5, 0);
                Bond P_OP3 = P.getBond(OP3);
                double dP_OP3 = P_OP3.bondType.distance;
                Angle O5s_P_OP3 = C5s.getAngle(O5s, P);
                double dO5s_P_OP3 = O5s_P_OP3.angleType.angle[O5s_P_OP3.nh];
                BondedUtils.intxyz(OP3, P, dP_OP3, O5s, dO5s_P_OP3, C5s, rotamer.chi4, 0);
            }
        } else {
            Bond O5s_P = O5s.getBond(P);
            double dO5s_P = O5s_P.bondType.distance;
            Angle C5s_O5s_P = C5s.getAngle(O5s, P);
            double dC5s_O5s_P = C5s_O5s_P.angleType.angle[C5s_O5s_P.nh];
            BondedUtils.intxyz(P, O5s, dO5s_P, C5s, dC5s_O5s_P, C4s, rotamer.chi5, 0);
        }
    }

    private static double applyNACorrections(Residue residue, Residue prevResidue, Rotamer rotamer, NucleicSugarPucker prevSugarPucker, boolean isDeoxy, boolean is3sTerminal) {
        Atom C3s = (Atom)residue.getAtomNode("C3'");
        Atom O4s = (Atom)residue.getAtomNode("O4'");
        Atom C4s = (Atom)residue.getAtomNode("C4'");
        Atom C5s = (Atom)residue.getAtomNode("C5'");
        Atom O5s = (Atom)residue.getAtomNode("O5'");
        Atom P = (Atom)residue.getAtomNode("P");
        Atom C1s = (Atom)residue.getAtomNode("C1'");
        Atom C2s = (Atom)residue.getAtomNode("C2'");
        Atom O3sPrev = (Atom)prevResidue.getAtomNode("O3'");
        double[] O3sPriorCoords = prevSugarPucker == NucleicSugarPucker.C3_ENDO ? prevResidue.getO3sNorth() : prevResidue.getO3sSouth();
        Bond P_O3sPrev = P.getBond(O3sPrev);
        double dP_O3sPrev = P_O3sPrev.bondType.distance;
        Angle O5s_P_O3sPrev = O5s.getAngle(P, O3sPrev);
        double dO5s_P_O3sPrev = O5s_P_O3sPrev.angleType.angle[O5s_P_O3sPrev.nh];
        double[] O3sHypCoords = BondedUtils.determineIntxyz(P.getXYZ(null), dP_O3sPrev, O5s.getXYZ(null), dO5s_P_O3sPrev, C5s.getXYZ(null), rotamer.chi4, 0);
        double[][] corrections = new double[7][3];
        for (int i = 0; i < 3; ++i) {
            corrections[5][i] = O3sPriorCoords[i] - O3sHypCoords[i];
            corrections[0][i] = 0.16666666666666666 * corrections[5][i];
            corrections[1][i] = 0.3333333333333333 * corrections[5][i];
            corrections[2][i] = 0.5 * corrections[5][i];
            corrections[3][i] = 0.6666666666666666 * corrections[5][i];
            corrections[4][i] = 0.8333333333333334 * corrections[5][i];
            corrections[6][i] = 0.08333333333333333 * corrections[5][i];
        }
        O4s.move(corrections[0]);
        C3s.move(corrections[0]);
        C4s.move(corrections[1]);
        C5s.move(corrections[2]);
        O5s.move(corrections[3]);
        P.move(corrections[4]);
        C1s.move(corrections[6]);
        C2s.move(corrections[6]);
        return corrections[5][0] * corrections[5][0] + corrections[5][1] * corrections[5][1] + corrections[5][2] * corrections[5][2];
    }

    private static double applyNARotamer(Residue residue, Rotamer rotamer, boolean independent) {
        if (rotamer.isState) {
            RotamerLibrary.applyState(residue, rotamer);
            return 0.0;
        }
        Residue prevResidue = residue.getPreviousResidue();
        boolean is3sTerminal = residue.getNextResidue() == null;
        boolean isDeoxy = residue.getAtomNode("O2'") == null;
        NucleicSugarPucker sugarPucker = NucleicSugarPucker.checkPucker(rotamer.chi7, isDeoxy);
        NucleicSugarPucker prevSugarPucker = NucleicSugarPucker.checkPucker(rotamer.chi1, isDeoxy);
        Atom C1s = (Atom)residue.getAtomNode("C1'");
        C1s.moveTo(residue.getC1sCoords());
        Atom O4s = (Atom)residue.getAtomNode("O4'");
        O4s.moveTo(residue.getO4sCoords());
        Atom C4s = (Atom)residue.getAtomNode("C4'");
        C4s.moveTo(residue.getC4sCoords());
        RotamerLibrary.applySugarPucker(residue, sugarPucker, isDeoxy, true);
        RotamerLibrary.applyNABackbone(residue, rotamer, prevResidue);
        double naCorrection = 0.0;
        if (prevResidue != null && !independent) {
            naCorrection = RotamerLibrary.applyNACorrections(residue, prevResidue, rotamer, prevSugarPucker, isDeoxy, is3sTerminal);
        }
        RotamerLibrary.applyNASideAtoms(residue, rotamer, prevResidue, isDeoxy, is3sTerminal, prevSugarPucker);
        return naCorrection;
    }

    private static void applyNASideAtoms(Residue residue, Rotamer rotamer, Residue prevResidue, boolean isDeoxy, boolean is3sTerminal, NucleicSugarPucker prevSugarPucker) {
        Atom C1s = (Atom)residue.getAtomNode("C1'");
        Atom C2s = (Atom)residue.getAtomNode("C2'");
        Atom C3s = (Atom)residue.getAtomNode("C3'");
        Atom C4s = (Atom)residue.getAtomNode("C4'");
        Atom O4s = (Atom)residue.getAtomNode("O4'");
        Atom O3s = (Atom)residue.getAtomNode("O3'");
        Atom O2s = (Atom)residue.getAtomNode("O2'");
        Atom H2ss = (Atom)residue.getAtomNode("H2''");
        Atom H2s = (Atom)residue.getAtomNode("H2'");
        Atom HO2s = (Atom)residue.getAtomNode("HO2'");
        Atom H3s = (Atom)residue.getAtomNode("H3'");
        Atom H4s = (Atom)residue.getAtomNode("H4'");
        Atom H1s = (Atom)residue.getAtomNode("H1'");
        Atom H5s = (Atom)residue.getAtomNode("H5'");
        Atom H5ss = (Atom)residue.getAtomNode("H5''");
        Atom C5s = (Atom)residue.getAtomNode("C5'");
        Atom O5s = (Atom)residue.getAtomNode("O5'");
        Atom P = (Atom)residue.getAtomNode("P");
        Atom OP1 = (Atom)residue.getAtomNode("OP1");
        Atom OP2 = (Atom)residue.getAtomNode("OP2");
        Bond C2s_H2s = C2s.getBond(H2s);
        double dC2s_H2s = C2s_H2s.bondType.distance;
        Angle C3s_C2s_H2s = C3s.getAngle(C2s, H2s);
        double dC3s_C2s_H2s = C3s_C2s_H2s.angleType.angle[C3s_C2s_H2s.nh];
        if (isDeoxy) {
            BondedUtils.intxyz(H2s, C2s, dC2s_H2s, C3s, dC3s_C2s_H2s, C1s, 109.4, 1);
            Bond C2s_H2ss = C2s.getBond(H2ss);
            double dC2s_H2ss = C2s_H2ss.bondType.distance;
            Angle C3s_C2s_H2ss = C3s.getAngle(C2s, H2ss);
            double dC3s_C2s_H2ss = C3s_C2s_H2ss.angleType.angle[C3s_C2s_H2ss.nh];
            BondedUtils.intxyz(H2ss, C2s, dC2s_H2ss, C3s, dC3s_C2s_H2ss, C1s, 109.4, -1);
        } else {
            BondedUtils.intxyz(H2s, C2s, dC2s_H2s, C3s, dC3s_C2s_H2s, C1s, 109.4, -1);
            Bond C2s_O2s = C2s.getBond(O2s);
            double dC2s_O2s = C2s_O2s.bondType.distance;
            Angle C3s_C2s_O2s = C3s.getAngle(C2s, O2s);
            double dC3s_C2s_O2s = C3s_C2s_O2s.angleType.angle[C3s_C2s_O2s.nh];
            BondedUtils.intxyz(O2s, C2s, dC2s_O2s, C3s, dC3s_C2s_O2s, C1s, 109.4, 1);
            Bond O2s_HO2s = O2s.getBond(HO2s);
            double dO2s_HO2s = O2s_HO2s.bondType.distance;
            Angle C2s_O2s_HO2s = C2s.getAngle(O2s, HO2s);
            double dC2s_O2s_HO2s = C2s_O2s_HO2s.angleType.angle[C2s_O2s_HO2s.nh];
            BondedUtils.intxyz(HO2s, O2s, dO2s_HO2s, C2s, dC2s_O2s_HO2s, C1s, 70.0, 0);
        }
        Bond C1s_H1s = C1s.getBond(H1s);
        double dC1s_H1s = C1s_H1s.bondType.distance;
        Angle C2s_C1s_H1s = C2s.getAngle(C1s, H1s);
        double dC2s_C1s_H1s = C2s_C1s_H1s.angleType.angle[C2s_C1s_H1s.nh];
        BondedUtils.intxyz(H1s, C1s, dC1s_H1s, C2s, dC2s_C1s_H1s, O4s, 109.4, 1);
        Bond C3s_H3s = C3s.getBond(H3s);
        double dC3s_H3s = C3s_H3s.bondType.distance;
        Angle C2s_C3s_H3s = C2s.getAngle(C3s, H3s);
        double dC2s_C3s_H3s = C2s_C3s_H3s.angleType.angle[C2s_C3s_H3s.nh];
        BondedUtils.intxyz(H3s, C3s, dC3s_H3s, C2s, dC2s_C3s_H3s, C4s, 109.4, 1);
        Bond C4s_H4s = C4s.getBond(H4s);
        double dC4s_H4s = C4s_H4s.bondType.distance;
        Angle C3s_C4s_H4s = C3s.getAngle(C4s, H4s);
        double dC3s_C4s_H4s = C3s_C4s_H4s.angleType.angle[C3s_C4s_H4s.nh];
        BondedUtils.intxyz(H4s, C4s, dC4s_H4s, C3s, dC3s_C4s_H4s, O4s, 109.4, -1);
        Bond C5s_H5s = C5s.getBond(H5s);
        double dC5s_H5s = C5s_H5s.bondType.distance;
        Angle C4s_C5s_H5s = C4s.getAngle(C5s, H5s);
        double dC4s_C5s_H5s = C4s_C5s_H5s.angleType.angle[C4s_C5s_H5s.nh];
        BondedUtils.intxyz(H5s, C5s, dC5s_H5s, C4s, dC4s_C5s_H5s, O5s, 109.4, -1);
        Bond C5s_H5ss = C5s.getBond(H5ss);
        double dC5s_H5ss = C5s_H5ss.bondType.distance;
        Angle C4s_C5s_H5ss = C4s.getAngle(C5s, H5ss);
        double dC4s_C5s_H5ss = C4s_C5s_H5ss.angleType.angle[C4s_C5s_H5ss.nh];
        BondedUtils.intxyz(H5ss, C5s, dC5s_H5ss, C4s, dC4s_C5s_H5ss, O5s, 109.4, 1);
        if (is3sTerminal) {
            Atom HO3s = (Atom)residue.getAtomNode("HO3'");
            Bond O3s_HO3s = O3s.getBond(HO3s);
            double dO3s_HO3s = O3s_HO3s.bondType.distance;
            Angle C3s_O3s_HO3s = C3s.getAngle(O3s, HO3s);
            double dC3s_O3s_HO3s = C3s_O3s_HO3s.angleType.angle[C3s_O3s_HO3s.nh];
            BondedUtils.intxyz(HO3s, O3s, dO3s_HO3s, C3s, dC3s_O3s_HO3s, C4s, 180.0, 0);
        }
        if (P != null) {
            double[] PXYZ = new double[3];
            P.getXYZ(PXYZ);
            double[] O5sXYZ = new double[3];
            O5s.getXYZ(O5sXYZ);
            double[] C5sXYZ = new double[3];
            C5s.getXYZ(C5sXYZ);
            Bond P_OP1 = P.getBond(OP1);
            double dP_OP1 = P_OP1.bondType.distance;
            Angle O5s_P_OP1 = C5s.getAngle(O5s, P);
            double dO5s_P_OP1 = O5s_P_OP1.angleType.angle[O5s_P_OP1.nh];
            Bond P_OP2 = P.getBond(OP2);
            double dP_OP2 = P_OP2.bondType.distance;
            Angle O5s_P_OP2 = C5s.getAngle(O5s, P);
            double dO5s_P_OP2 = O5s_P_OP2.angleType.angle[O5s_P_OP2.nh];
            if (prevResidue != null) {
                double[] O3sPriorCoords = prevSugarPucker == NucleicSugarPucker.C3_ENDO ? prevResidue.getO3sNorth() : prevResidue.getO3sSouth();
                double[] OP1XYZ = BondedUtils.determineIntxyz(PXYZ, dP_OP1, O5sXYZ, dO5s_P_OP1, O3sPriorCoords, 109.4, 1);
                double[] OP2XYZ = BondedUtils.determineIntxyz(PXYZ, dP_OP2, O5sXYZ, dO5s_P_OP2, O3sPriorCoords, 109.4, -1);
                OP1.moveTo(OP1XYZ);
                OP2.moveTo(OP2XYZ);
            } else {
                Atom OP3 = (Atom)residue.getAtomNode("OP3");
                double[] OP3XYZ = new double[3];
                OP3.getXYZ(OP3XYZ);
                double[] OP1XYZ = BondedUtils.determineIntxyz(PXYZ, dP_OP1, O5sXYZ, dO5s_P_OP1, OP3XYZ, 109.4, 1);
                double[] OP2XYZ = BondedUtils.determineIntxyz(PXYZ, dP_OP2, O5sXYZ, dO5s_P_OP2, OP3XYZ, 109.4, -1);
                OP1.moveTo(OP1XYZ);
                OP2.moveTo(OP2XYZ);
            }
        } else {
            Atom HO5s = (Atom)residue.getAtomNode("HO5'");
            Bond O5s_HO5s = O5s.getBond(HO5s);
            double dO5s_HO5s = O5s_HO5s.bondType.distance;
            Angle C5s_O5s_HO5s = C5s.getAngle(O5s, HO5s);
            double dC5s_O5s_HO5s = C5s_O5s_HO5s.angleType.angle[C5s_O5s_HO5s.nh];
            BondedUtils.intxyz(HO5s, O5s, dO5s_HO5s, C5s, dC5s_O5s_HO5s, C4s, rotamer.chi5, 0);
        }
    }

    private static void applyState(Residue residue, Rotamer rotamer) {
        if (rotamer.isState) {
            residue.revertState(rotamer.originalState);
        } else {
            logger.warning(String.format(" Attempting to apply a ResidueState for a torsion-based rotamer %s for residue %s", rotamer, residue));
        }
    }

    private static double[] drawCZ(AminoAcidUtils.AminoAcid3 resName, Atom CZ, Atom CG, Atom CE1, Atom CD1, Atom CE2, Atom CD2) {
        double bondLen = CZ.getBond((Atom)CE1).bondType.distance;
        double ang = RotamerLibrary.getAngle(resName, CZ, CE1, CD1);
        double[] xCG = new double[3];
        xCG = CG.getXYZ(xCG);
        double[] xCE = new double[3];
        xCE = CE1.getXYZ(xCE);
        double[] xCD = new double[3];
        xCD = CD1.getXYZ(xCD);
        double[] xyz1 = BondedUtils.determineIntxyz(xCE, bondLen, xCD, ang, xCG, 0.0, 0);
        xCE = CE2.getXYZ(xCE);
        xCD = CD2.getXYZ(xCD);
        double[] xyz2 = BondedUtils.determineIntxyz(xCE, bondLen, xCD, ang, xCG, 0.0, 0);
        int i = 0;
        while (i < 3) {
            int n = i;
            xyz1[n] = xyz1[n] + xyz2[i];
            int n2 = i++;
            xyz1[n2] = xyz1[n2] * 0.5;
        }
        return xyz1;
    }

    private static double getAngle(AminoAcidUtils.AminoAcid3 resName, Atom a1, Atom a2, Atom a3) {
        if (useIdealRingGeometries) {
            return RotamerLibrary.idealGeometryAngle(resName, a1, a2, a3);
        }
        return RotamerLibrary.angleFromForceField(a1, a2, a3);
    }

    private static double idealGeometryAngle(AminoAcidUtils.AminoAcid3 resName, Atom a1, Atom a2, Atom a3) {
        StringBuilder sb = new StringBuilder(a1.getName());
        sb.append("-").append(a2.getName()).append("-").append(a3.getName());
        Map<String, Double> resMap = idealAngleGeometries.get((Object)resName);
        String atomString = sb.toString();
        if (resMap.containsKey(atomString)) {
            return resMap.get(atomString);
        }
        sb = new StringBuilder(a3.getName());
        sb.append("-").append(a2.getName()).append("-").append(a1.getName());
        atomString = sb.toString();
        if (resMap.containsKey(atomString)) {
            return resMap.get(atomString);
        }
        logger.finest(String.format(" Could not find an ideal-geometry angle for %s %s-%s-%s", new Object[]{resName, a1, a2, a3}));
        return RotamerLibrary.angleFromForceField(a1, a2, a3);
    }

    private static boolean isModRes(Residue residue) {
        List<Atom> resAtoms = residue.getAtomList();
        return resAtoms != null && !resAtoms.isEmpty() && resAtoms.get(0).isModRes();
    }

    private static int measureNARotamer(Residue residue, double[] chi, boolean print) {
        Torsion torsion;
        int nRot;
        Atom O5s;
        Atom O3s;
        Atom C3s;
        Atom C4s;
        Atom C5s;
        block18: {
            Atom P;
            Residue prevResidue;
            NucleicAcidUtils.NucleicAcid3 name;
            block17: {
                name = NucleicAcidUtils.NucleicAcid3.valueOf(residue.getName());
                prevResidue = residue.getPreviousResidue();
                C5s = (Atom)residue.getAtomNode("C5'");
                C4s = (Atom)residue.getAtomNode("C4'");
                C3s = (Atom)residue.getAtomNode("C3'");
                O3s = (Atom)residue.getAtomNode("O3'");
                O5s = (Atom)residue.getAtomNode("O5'");
                P = (Atom)residue.getAtomNode("P");
                nRot = 7;
                if (prevResidue != null) break block17;
                switch (name) {
                    case GUA: 
                    case ADE: 
                    case DGU: 
                    case DAD: 
                    case CYT: 
                    case URI: 
                    case THY: 
                    case DCY: 
                    case DTY: {
                        Atom HO5s = (Atom)residue.getAtomNode("HO5'");
                        if (HO5s != null) {
                            torsion = HO5s.getTorsion(O5s, C5s, C4s);
                            chi[4] = torsion.measure();
                            if (print) {
                                logger.info(torsion.toString());
                            }
                            nRot = 3;
                            break;
                        }
                        Atom OP3 = (Atom)residue.getAtomNode("OP3");
                        nRot = 3;
                        if (OP3 != null) {
                            torsion = OP3.getTorsion(P, O5s, C5s);
                            chi[3] = torsion.measure();
                            if (print) {
                                logger.info(torsion.toString());
                            }
                            nRot = 4;
                        }
                        torsion = P.getTorsion(O5s, C5s, C4s);
                        chi[4] = torsion.measure();
                        if (print) {
                            logger.info(torsion.toString());
                            break;
                        } else {
                            break;
                        }
                    }
                }
                break block18;
            }
            switch (name) {
                case GUA: 
                case ADE: 
                case DGU: 
                case DAD: 
                case CYT: 
                case URI: 
                case THY: 
                case DCY: 
                case DTY: {
                    Atom O3sPrev = (Atom)prevResidue.getAtomNode("O3'");
                    Atom C3sPrev = (Atom)prevResidue.getAtomNode("C3'");
                    Atom C4sPrev = (Atom)prevResidue.getAtomNode("C4'");
                    Atom C5sPrev = (Atom)prevResidue.getAtomNode("C5'");
                    torsion = C5sPrev.getTorsion(C4sPrev, C3sPrev, O3sPrev);
                    chi[0] = torsion.measure();
                    if (print) {
                        logger.info(torsion.toString());
                    }
                    torsion = C4sPrev.getTorsion(C3sPrev, O3sPrev, P);
                    chi[1] = torsion.measure();
                    if (print) {
                        logger.info(torsion.toString());
                    }
                    torsion = C3sPrev.getTorsion(O3sPrev, P, O5s);
                    chi[2] = torsion.measure();
                    if (print) {
                        logger.info(torsion.toString());
                    }
                    torsion = O3sPrev.getTorsion(P, O5s, C5s);
                    chi[3] = torsion.measure();
                    if (print) {
                        logger.info(torsion.toString());
                    }
                    torsion = P.getTorsion(O5s, C5s, C4s);
                    chi[4] = torsion.measure();
                    if (!print) break;
                    logger.info(torsion.toString());
                    break;
                }
            }
        }
        torsion = O5s.getTorsion(C5s, C4s, C3s);
        chi[5] = torsion.measure();
        if (print) {
            logger.info(torsion.toString());
        }
        torsion = C5s.getTorsion(C4s, C3s, O3s);
        chi[6] = torsion.measure();
        if (print) {
            logger.info(torsion.toString());
        }
        return nRot;
    }

    private static void measureUNKRotamer(Residue residue, double[] chi, boolean print) {
        String resName = residue.getName().toUpperCase();
        if (nonstdRotCache.containsKey(resName)) {
            nonstdRotCache.get(resName).measureNonstdRot(residue, chi, print);
        } else {
            logger.warning(String.format(" Could not measure chi angles for residue %s", residue));
        }
    }

    private static void readRotFile(File rotamerFile, MolecularAssembly assembly, int boxWindowIndex) throws IOException {
        try (BufferedReader br = new BufferedReader(new FileReader(rotamerFile));){
            Polymer[] polys = assembly.getChains();
            Residue currentRes = null;
            ResidueState origState = null;
            String line = br.readLine();
            boolean doRead = false;
            while (line != null) {
                String[] toks = line.trim().split(":");
                if (toks[0].equals("ALGORITHM")) {
                    doRead = Integer.parseInt(toks[2]) == boxWindowIndex;
                    logger.info(String.format(" Readabilifications %b with %s", doRead, line));
                } else if (doRead && !toks[0].startsWith("#")) {
                    block8 : switch (toks[0]) {
                        case "RES": {
                            String segID = toks[2];
                            int resnum = Integer.parseInt(toks[4]);
                            for (Polymer poly : polys) {
                                if (!poly.getName().equals(segID)) continue;
                                currentRes = poly.getResidue(resnum);
                                break block8;
                            }
                            break;
                        }
                        case "ENDROT": {
                            currentRes.addRotamers(Rotamer.defaultRotamerFactory(currentRes));
                            currentRes.revertState(origState);
                            logger.info(String.format(" Adding a rotamer to %s", currentRes));
                            break;
                        }
                        case "ROT": {
                            origState = currentRes.storeState();
                            break;
                        }
                        case "ATOM": {
                            String name = toks[1];
                            Atom atom = (Atom)currentRes.getAtomNode(name);
                            double[] xyz = new double[3];
                            for (int i = 0; i < 3; ++i) {
                                xyz[i] = Double.parseDouble(toks[i + 2]);
                            }
                            atom.setXYZ(xyz);
                            break;
                        }
                        default: {
                            logger.warning(" Unrecognized line! " + line);
                        }
                    }
                }
                line = br.readLine();
            }
        }
    }

    public ProteinLibrary getLibrary() {
        return this.proteinLibrary;
    }

    public Rotamer[] getRotamers(AminoAcidUtils.AminoAcid3 name) {
        return this.getRotamers(name, null);
    }

    public Rotamer[] getRotamers(AminoAcidUtils.AminoAcid3 name, TitrationUtils titrationUtils) {
        return switch (this.proteinLibrary.ordinal()) {
            default -> throw new MatchException(null, null);
            case 0 -> this.getPonderAndRichardsRotamers(name, titrationUtils);
            case 1 -> this.getRichardsonRotamers(name, titrationUtils);
            case 2 -> null;
        };
    }

    public Rotamer[] getRotamers(NucleicAcidUtils.NucleicAcid3 name) {
        return this.getRichardsonRNARotamers(name);
    }

    public boolean getUsingOrigCoordsRotamer() {
        return this.useOrigCoordsRotamer;
    }

    public RotamerGuess guessRotamer(Residue residue) {
        assert (!this.useOrigCoordsRotamer);
        if (residue == null) {
            throw new IllegalArgumentException(" Residue cannot be null!");
        }
        Rotamer[] rotamers = residue.getRotamers();
        if (rotamers == null) {
            rotamers = residue.setRotamers(this);
        }
        if (rotamers == null || rotamers.length == 0) {
            throw new IllegalArgumentException(String.format(" Residue %s does not have rotamers!", residue));
        }
        int nRot = rotamers.length;
        double[] currChi = residue.getResidueType().equals((Object)Residue.ResidueType.AA) ? new double[4] : new double[7];
        int numChis = RotamerLibrary.measureRotamer(residue, currChi, false);
        double[] chiRMSD = new double[nRot];
        double lowestRMSD = Double.MAX_VALUE;
        int indexLowest = -1;
        for (int i = 0; i < nRot; ++i) {
            double rmsd = 0.0;
            Rotamer roti = rotamers[i];
            double[] rotChi = roti.angles;
            for (int j = 0; j < numChis; ++j) {
                double dChi = Math.abs(currChi[j] - rotChi[j]);
                if (dChi > 180.0) {
                    dChi = 360.0 - dChi;
                }
                dChi *= dChi;
                rmsd += dChi;
            }
            rmsd /= (double)numChis;
            chiRMSD[i] = rmsd = FastMath.sqrt((double)rmsd);
            if (!(rmsd < lowestRMSD)) continue;
            lowestRMSD = rmsd;
            indexLowest = i;
        }
        if (indexLowest < 0) {
            logger.warning(String.format(" Residue %s could not be IDd!", residue));
            logger.info(Arrays.toString(currChi));
            logger.info(String.format("%12.5g", lowestRMSD));
            logger.info(String.format("%d", nRot));
        }
        return new RotamerGuess(residue, rotamers[indexLowest], indexLowest, lowestRMSD);
    }

    public void setUseOrigCoordsRotamer(boolean set) {
        this.useOrigCoordsRotamer = set;
    }

    Rotamer[] getRotamers(Residue residue) {
        return this.getRotamers(residue, null);
    }

    Rotamer[] getRotamers(Residue residue, TitrationUtils titrationUtils) {
        if (residue == null) {
            return null;
        }
        if (RotamerLibrary.isModRes(residue)) {
            if (nonstdRotCache.containsKey(residue.getName().toUpperCase())) {
                return nonstdRotCache.get(residue.getName().toUpperCase()).getRotamers();
            }
            return null;
        }
        switch (residue.getResidueType()) {
            case AA: {
                AminoAcidUtils.AminoAcid3 aa = AminoAcidUtils.AminoAcid3.valueOf(residue.getName());
                switch (aa) {
                    case CYS: 
                    case CYX: 
                    case CYD: {
                        List<Atom> cysAtoms = residue.getAtomList();
                        Optional<Atom> s = cysAtoms.stream().filter(a -> a.getAtomType().atomicNumber == 16).findAny();
                        if (s.isEmpty()) {
                            s = cysAtoms.stream().filter(a -> a.getName().startsWith("S")).findAny();
                        }
                        if (s.isPresent()) {
                            Atom theS = s.get();
                            boolean attachedS = theS.getBonds().stream().map(b -> b.get1_2(theS)).anyMatch(a -> a.getAtomType().atomicNumber == 16 || a.getName().startsWith("S"));
                            if (!attachedS) break;
                            return null;
                        }
                        logger.warning(String.format(" No sulfur atom found attached to %s residue %s!", new Object[]{aa, residue}));
                    }
                }
                return this.getRotamers(aa, titrationUtils);
            }
            case NA: {
                NucleicAcidUtils.NucleicAcid3 na = NucleicAcidUtils.NucleicAcid3.valueOf(residue.getName());
                return this.getRotamers(na);
            }
        }
        if (nonstdRotCache.containsKey(residue.getName().toUpperCase())) {
            return nonstdRotCache.get(residue.getName().toUpperCase()).getRotamers();
        }
        return null;
    }

    private Rotamer[] getPonderAndRichardsRotamers(AminoAcidUtils.AminoAcid3 name, TitrationUtils titrationUtils) {
        int n = name.ordinal();
        if (this.aminoAcidRotamerCache[n] != null) {
            return this.aminoAcidRotamerCache[n];
        }
        switch (name) {
            case VAL: {
                this.aminoAcidRotamerCache[n] = new Rotamer[3];
                this.aminoAcidRotamerCache[n][0] = new Rotamer(name, 173.5, 9.0);
                this.aminoAcidRotamerCache[n][1] = new Rotamer(name, -63.4, 8.1);
                this.aminoAcidRotamerCache[n][2] = new Rotamer(name, 69.3, 9.6);
                break;
            }
            case LEU: {
                this.aminoAcidRotamerCache[n] = new Rotamer[4];
                this.aminoAcidRotamerCache[n][0] = new Rotamer(name, -64.9, 8.2, 176.0, 9.9);
                this.aminoAcidRotamerCache[n][1] = new Rotamer(name, -176.4, 10.2, 63.1, 8.2);
                this.aminoAcidRotamerCache[n][2] = new Rotamer(name, -165.3, 10.0, 168.2, 34.2);
                this.aminoAcidRotamerCache[n][3] = new Rotamer(name, 44.3, 20.0, 60.4, 18.8);
                break;
            }
            case ILE: {
                this.aminoAcidRotamerCache[n] = new Rotamer[5];
                this.aminoAcidRotamerCache[n][0] = new Rotamer(name, -60.9, 7.5, 168.7, 11.6);
                this.aminoAcidRotamerCache[n][1] = new Rotamer(name, -59.6, 9.6, -64.1, 14.3);
                this.aminoAcidRotamerCache[n][2] = new Rotamer(name, 61.7, 5.0, 163.8, 16.4);
                this.aminoAcidRotamerCache[n][3] = new Rotamer(name, -166.6, 10.1, 166.0, 8.9);
                this.aminoAcidRotamerCache[n][4] = new Rotamer(name, -174.8, 24.9, 72.1, 10.5);
                break;
            }
            case SER: {
                this.aminoAcidRotamerCache[n] = new Rotamer[3];
                this.aminoAcidRotamerCache[n][0] = new Rotamer(name, 64.7, 16.1);
                this.aminoAcidRotamerCache[n][1] = new Rotamer(name, -69.7, 14.6);
                this.aminoAcidRotamerCache[n][2] = new Rotamer(name, -176.1, 20.2);
                break;
            }
            case THR: {
                this.aminoAcidRotamerCache[n] = new Rotamer[3];
                this.aminoAcidRotamerCache[n][0] = new Rotamer(name, 62.7, 8.5);
                this.aminoAcidRotamerCache[n][1] = new Rotamer(name, -59.7, 9.4);
                this.aminoAcidRotamerCache[n][2] = new Rotamer(name, -169.5, 6.6);
                break;
            }
            case CYS: 
            case CYD: {
                if (titrationUtils == null) {
                    this.aminoAcidRotamerCache[n] = new Rotamer[3];
                    this.aminoAcidRotamerCache[n][0] = new Rotamer(name, -65.2, 10.1);
                    this.aminoAcidRotamerCache[n][1] = new Rotamer(name, -179.6, 9.5);
                    this.aminoAcidRotamerCache[n][2] = new Rotamer(name, 63.5, 9.6);
                    break;
                }
                this.aminoAcidRotamerCache[n] = new Rotamer[6];
                this.aminoAcidRotamerCache[n][0] = new Rotamer(AminoAcidUtils.AminoAcid3.CYS, titrationUtils, -65.2, 10.1);
                this.aminoAcidRotamerCache[n][1] = new Rotamer(AminoAcidUtils.AminoAcid3.CYS, titrationUtils, -179.6, 9.5);
                this.aminoAcidRotamerCache[n][2] = new Rotamer(AminoAcidUtils.AminoAcid3.CYS, titrationUtils, 63.5, 9.6);
                this.aminoAcidRotamerCache[n][3] = new Rotamer(AminoAcidUtils.AminoAcid3.CYD, titrationUtils, -65.2, 10.1);
                this.aminoAcidRotamerCache[n][4] = new Rotamer(AminoAcidUtils.AminoAcid3.CYD, titrationUtils, -179.6, 9.5);
                this.aminoAcidRotamerCache[n][5] = new Rotamer(AminoAcidUtils.AminoAcid3.CYD, titrationUtils, 63.5, 9.6);
                break;
            }
            case PRO: {
                this.aminoAcidRotamerCache[n] = new Rotamer[3];
                this.aminoAcidRotamerCache[n][0] = new Rotamer(name, 24.0, 8.0);
                this.aminoAcidRotamerCache[n][1] = new Rotamer(name, 0.0, 8.0);
                this.aminoAcidRotamerCache[n][2] = new Rotamer(name, -24.0, 8.0);
                break;
            }
            case PHE: {
                this.aminoAcidRotamerCache[n] = new Rotamer[4];
                this.aminoAcidRotamerCache[n][0] = new Rotamer(name, -66.3, 10.2, 94.3, 19.5);
                this.aminoAcidRotamerCache[n][1] = new Rotamer(name, -179.2, 9.3, 78.9, 8.9);
                this.aminoAcidRotamerCache[n][2] = new Rotamer(name, 66.0, 12.0, 90.7, 9.4);
                this.aminoAcidRotamerCache[n][3] = new Rotamer(name, -71.9, 16.3, -0.4, 26.1);
                break;
            }
            case TYR: {
                this.aminoAcidRotamerCache[n] = new Rotamer[8];
                this.aminoAcidRotamerCache[n][0] = new Rotamer(name, -66.5, 11.4, 96.6, 21.8, 0.0, 0.0);
                this.aminoAcidRotamerCache[n][1] = new Rotamer(name, -179.7, 12.6, 71.9, 13.4, 0.0, 0.0);
                this.aminoAcidRotamerCache[n][2] = new Rotamer(name, 63.3, 9.4, 89.1, 13.0, 0.0, 0.0);
                this.aminoAcidRotamerCache[n][3] = new Rotamer(name, -67.2, 13.2, -1.0, 20.1, 0.0, 0.0);
                this.aminoAcidRotamerCache[n][4] = new Rotamer(name, -66.5, 11.4, 96.6, 21.8, 180.0, 0.0);
                this.aminoAcidRotamerCache[n][5] = new Rotamer(name, -179.7, 12.6, 71.9, 13.4, 180.0, 0.0);
                this.aminoAcidRotamerCache[n][6] = new Rotamer(name, 63.3, 9.4, 89.1, 13.0, 180.0, 0.0);
                this.aminoAcidRotamerCache[n][7] = new Rotamer(name, -67.2, 13.2, -1.0, 20.1, 180.0, 0.0);
                break;
            }
            case TYD: {
                this.aminoAcidRotamerCache[n] = new Rotamer[4];
                this.aminoAcidRotamerCache[n][0] = new Rotamer(name, -66.5, 11.4, 96.6, 21.8);
                this.aminoAcidRotamerCache[n][1] = new Rotamer(name, -179.7, 12.6, 71.9, 13.4);
                this.aminoAcidRotamerCache[n][2] = new Rotamer(name, 63.3, 9.4, 89.1, 13.0);
                this.aminoAcidRotamerCache[n][3] = new Rotamer(name, -67.2, 13.2, -1.0, 20.1);
                break;
            }
            case TRP: {
                this.aminoAcidRotamerCache[n] = new Rotamer[6];
                this.aminoAcidRotamerCache[n][0] = new Rotamer(name, -70.4, 7.0, 100.5, 18.2);
                this.aminoAcidRotamerCache[n][1] = new Rotamer(name, 64.8, 13.0, -88.9, 5.3);
                this.aminoAcidRotamerCache[n][2] = new Rotamer(name, -177.3, 7.9, -95.1, 7.6);
                this.aminoAcidRotamerCache[n][3] = new Rotamer(name, -179.5, 3.4, 87.5, 3.8);
                this.aminoAcidRotamerCache[n][4] = new Rotamer(name, -73.3, 6.5, -87.7, 8.1);
                this.aminoAcidRotamerCache[n][5] = new Rotamer(name, 62.2, 10.0, 112.5, 15.0);
                break;
            }
            case HIS: 
            case HIE: 
            case HID: {
                if (titrationUtils == null) {
                    this.aminoAcidRotamerCache[n] = new Rotamer[6];
                    this.aminoAcidRotamerCache[n][0] = new Rotamer(name, -62.8, 10.0, -74.3, 17.2);
                    this.aminoAcidRotamerCache[n][1] = new Rotamer(name, -175.2, 15.4, -88.7, 43.5);
                    this.aminoAcidRotamerCache[n][2] = new Rotamer(name, -69.8, 5.9, 96.1, 32.2);
                    this.aminoAcidRotamerCache[n][3] = new Rotamer(name, 67.9, 17.4, -80.5, 40.7);
                    this.aminoAcidRotamerCache[n][4] = new Rotamer(name, -177.3, 6.3, 100.5, 14.0);
                    this.aminoAcidRotamerCache[n][5] = new Rotamer(name, 48.8, 10.0, 89.5, 30.0);
                    break;
                }
                this.aminoAcidRotamerCache[n] = new Rotamer[18];
                this.aminoAcidRotamerCache[n][0] = new Rotamer(AminoAcidUtils.AminoAcid3.HIS, titrationUtils, -62.8, 10.0, -74.3, 17.2);
                this.aminoAcidRotamerCache[n][1] = new Rotamer(AminoAcidUtils.AminoAcid3.HIS, titrationUtils, -175.2, 15.4, -88.7, 43.5);
                this.aminoAcidRotamerCache[n][2] = new Rotamer(AminoAcidUtils.AminoAcid3.HIS, titrationUtils, -69.8, 5.9, 96.1, 32.2);
                this.aminoAcidRotamerCache[n][3] = new Rotamer(AminoAcidUtils.AminoAcid3.HIS, titrationUtils, 67.9, 17.4, -80.5, 40.7);
                this.aminoAcidRotamerCache[n][4] = new Rotamer(AminoAcidUtils.AminoAcid3.HIS, titrationUtils, -177.3, 6.3, 100.5, 14.0);
                this.aminoAcidRotamerCache[n][5] = new Rotamer(AminoAcidUtils.AminoAcid3.HIS, titrationUtils, 48.8, 10.0, 89.5, 30.0);
                this.aminoAcidRotamerCache[n][6] = new Rotamer(AminoAcidUtils.AminoAcid3.HIE, titrationUtils, -62.8, 10.0, -74.3, 17.2);
                this.aminoAcidRotamerCache[n][7] = new Rotamer(AminoAcidUtils.AminoAcid3.HIE, titrationUtils, -175.2, 15.4, -88.7, 43.5);
                this.aminoAcidRotamerCache[n][8] = new Rotamer(AminoAcidUtils.AminoAcid3.HIE, titrationUtils, -69.8, 5.9, 96.1, 32.2);
                this.aminoAcidRotamerCache[n][9] = new Rotamer(AminoAcidUtils.AminoAcid3.HIE, titrationUtils, 67.9, 17.4, -80.5, 40.7);
                this.aminoAcidRotamerCache[n][10] = new Rotamer(AminoAcidUtils.AminoAcid3.HIE, titrationUtils, -177.3, 6.3, 100.5, 14.0);
                this.aminoAcidRotamerCache[n][11] = new Rotamer(AminoAcidUtils.AminoAcid3.HIE, titrationUtils, 48.8, 10.0, 89.5, 30.0);
                this.aminoAcidRotamerCache[n][12] = new Rotamer(AminoAcidUtils.AminoAcid3.HID, titrationUtils, -62.8, 10.0, -74.3, 17.2);
                this.aminoAcidRotamerCache[n][13] = new Rotamer(AminoAcidUtils.AminoAcid3.HID, titrationUtils, -175.2, 15.4, -88.7, 43.5);
                this.aminoAcidRotamerCache[n][14] = new Rotamer(AminoAcidUtils.AminoAcid3.HID, titrationUtils, -69.8, 5.9, 96.1, 32.2);
                this.aminoAcidRotamerCache[n][15] = new Rotamer(AminoAcidUtils.AminoAcid3.HID, titrationUtils, 67.9, 17.4, -80.5, 40.7);
                this.aminoAcidRotamerCache[n][16] = new Rotamer(AminoAcidUtils.AminoAcid3.HID, titrationUtils, -177.3, 6.3, 100.5, 14.0);
                this.aminoAcidRotamerCache[n][17] = new Rotamer(AminoAcidUtils.AminoAcid3.HID, titrationUtils, 48.8, 10.0, 89.5, 30.0);
                break;
            }
            case ASH: {
                if (titrationUtils == null) {
                    this.aminoAcidRotamerCache[n] = new Rotamer[12];
                    this.aminoAcidRotamerCache[n][0] = new Rotamer(name, -68.3, 9.2, -25.7, 31.1, 0.0, 0.0);
                    this.aminoAcidRotamerCache[n][1] = new Rotamer(name, -169.1, 9.5, 3.9, 38.9, 0.0, 0.0);
                    this.aminoAcidRotamerCache[n][2] = new Rotamer(name, 63.7, 9.9, 2.4, 29.4, 0.0, 0.0);
                    this.aminoAcidRotamerCache[n][3] = new Rotamer(name, -68.3, 9.2, 154.3, 31.1, 0.0, 0.0);
                    this.aminoAcidRotamerCache[n][4] = new Rotamer(name, -169.1, 9.5, -176.1, 38.9, 0.0, 0.0);
                    this.aminoAcidRotamerCache[n][5] = new Rotamer(name, 63.7, 9.9, -177.6, 29.4, 0.0, 0.0);
                    this.aminoAcidRotamerCache[n][6] = new Rotamer(name, -68.3, 9.2, -25.7, 31.1, 180.0, 0.0);
                    this.aminoAcidRotamerCache[n][7] = new Rotamer(name, -169.1, 9.5, 3.9, 38.9, 180.0, 0.0);
                    this.aminoAcidRotamerCache[n][8] = new Rotamer(name, 63.7, 9.9, 2.4, 29.4, 180.0, 0.0);
                    this.aminoAcidRotamerCache[n][9] = new Rotamer(name, -68.3, 9.2, 154.3, 31.1, 180.0, 0.0);
                    this.aminoAcidRotamerCache[n][10] = new Rotamer(name, -169.1, 9.5, -176.1, 38.9, 180.0, 0.0);
                    this.aminoAcidRotamerCache[n][11] = new Rotamer(name, 63.7, 9.9, -177.6, 29.4, 180.0, 0.0);
                    break;
                }
                this.aminoAcidRotamerCache[n] = new Rotamer[15];
                this.aminoAcidRotamerCache[n][0] = new Rotamer(AminoAcidUtils.AminoAcid3.ASH, titrationUtils, -68.3, 9.2, -25.7, 31.1, 0.0, 0.0);
                this.aminoAcidRotamerCache[n][1] = new Rotamer(AminoAcidUtils.AminoAcid3.ASH, titrationUtils, -169.1, 9.5, 3.9, 38.9, 0.0, 0.0);
                this.aminoAcidRotamerCache[n][2] = new Rotamer(AminoAcidUtils.AminoAcid3.ASH, titrationUtils, 63.7, 9.9, 2.4, 29.4, 0.0, 0.0);
                this.aminoAcidRotamerCache[n][3] = new Rotamer(AminoAcidUtils.AminoAcid3.ASH, titrationUtils, -68.3, 9.2, 154.3, 31.1, 0.0, 0.0);
                this.aminoAcidRotamerCache[n][4] = new Rotamer(AminoAcidUtils.AminoAcid3.ASH, titrationUtils, -169.1, 9.5, -176.1, 38.9, 0.0, 0.0);
                this.aminoAcidRotamerCache[n][5] = new Rotamer(AminoAcidUtils.AminoAcid3.ASH, titrationUtils, 63.7, 9.9, -177.6, 29.4, 0.0, 0.0);
                this.aminoAcidRotamerCache[n][6] = new Rotamer(AminoAcidUtils.AminoAcid3.ASH, titrationUtils, -68.3, 9.2, -25.7, 31.1, 180.0, 0.0);
                this.aminoAcidRotamerCache[n][7] = new Rotamer(AminoAcidUtils.AminoAcid3.ASH, titrationUtils, -169.1, 9.5, 3.9, 38.9, 180.0, 0.0);
                this.aminoAcidRotamerCache[n][8] = new Rotamer(AminoAcidUtils.AminoAcid3.ASH, titrationUtils, 63.7, 9.9, 2.4, 29.4, 180.0, 0.0);
                this.aminoAcidRotamerCache[n][9] = new Rotamer(AminoAcidUtils.AminoAcid3.ASH, titrationUtils, -68.3, 9.2, 154.3, 31.1, 180.0, 0.0);
                this.aminoAcidRotamerCache[n][10] = new Rotamer(AminoAcidUtils.AminoAcid3.ASH, titrationUtils, -169.1, 9.5, -176.1, 38.9, 180.0, 0.0);
                this.aminoAcidRotamerCache[n][11] = new Rotamer(AminoAcidUtils.AminoAcid3.ASH, titrationUtils, 63.7, 9.9, -177.6, 29.4, 180.0, 0.0);
                this.aminoAcidRotamerCache[n][12] = new Rotamer(AminoAcidUtils.AminoAcid3.ASP, titrationUtils, -68.3, 9.2, -25.7, 31.1);
                this.aminoAcidRotamerCache[n][13] = new Rotamer(AminoAcidUtils.AminoAcid3.ASP, titrationUtils, -169.1, 9.5, 3.9, 38.9);
                this.aminoAcidRotamerCache[n][14] = new Rotamer(AminoAcidUtils.AminoAcid3.ASP, titrationUtils, 63.7, 9.9, 2.4, 29.4);
                break;
            }
            case ASP: {
                this.aminoAcidRotamerCache[n] = new Rotamer[3];
                this.aminoAcidRotamerCache[n][0] = new Rotamer(name, -68.3, 9.2, -25.7, 31.1);
                this.aminoAcidRotamerCache[n][1] = new Rotamer(name, -169.1, 9.5, 3.9, 38.9);
                this.aminoAcidRotamerCache[n][2] = new Rotamer(name, 63.7, 9.9, 2.4, 29.4);
                break;
            }
            case ASN: {
                this.aminoAcidRotamerCache[n] = new Rotamer[6];
                this.aminoAcidRotamerCache[n][0] = new Rotamer(name, -68.3, 12.3, -36.8, 25.2);
                this.aminoAcidRotamerCache[n][1] = new Rotamer(name, -177.1, 8.8, 1.3, 34.1);
                this.aminoAcidRotamerCache[n][2] = new Rotamer(name, -67.2, 10.8, 128.8, 24.2);
                this.aminoAcidRotamerCache[n][3] = new Rotamer(name, 63.9, 3.7, -6.8, 13.5);
                this.aminoAcidRotamerCache[n][4] = new Rotamer(name, -174.9, 17.9, -156.8, 58.9);
                this.aminoAcidRotamerCache[n][5] = new Rotamer(name, 63.6, 6.6, 53.8, 17.1);
                break;
            }
            case GLU: {
                this.aminoAcidRotamerCache[n] = new Rotamer[7];
                this.aminoAcidRotamerCache[n][0] = new Rotamer(name, -69.6, 19.2, -177.2, 21.7, -11.4, 44.8);
                this.aminoAcidRotamerCache[n][1] = new Rotamer(name, -176.2, 14.9, 175.4, 10.6, -6.7, 39.0);
                this.aminoAcidRotamerCache[n][2] = new Rotamer(name, -64.6, 13.5, -69.1, 17.3, -33.4, 27.4);
                this.aminoAcidRotamerCache[n][3] = new Rotamer(name, -55.6, 10.6, 77.0, 6.8, 25.3, 32.6);
                this.aminoAcidRotamerCache[n][4] = new Rotamer(name, 69.8, 10.6, -179.0, 23.7, 6.6, 64.2);
                this.aminoAcidRotamerCache[n][5] = new Rotamer(name, -173.6, 14.6, 70.6, 8.7, 14.0, 37.1);
                this.aminoAcidRotamerCache[n][6] = new Rotamer(name, 63.0, 4.3, -80.4, 13.9, 16.3, 20.8);
                break;
            }
            case GLH: {
                if (titrationUtils == null) {
                    this.aminoAcidRotamerCache[n] = new Rotamer[28];
                    this.aminoAcidRotamerCache[n][0] = new Rotamer(name, -69.6, 19.2, -177.2, 21.7, -11.4, 44.8, 0.0, 0.0);
                    this.aminoAcidRotamerCache[n][1] = new Rotamer(name, -176.2, 14.9, 175.4, 10.6, -6.7, 39.0, 0.0, 0.0);
                    this.aminoAcidRotamerCache[n][2] = new Rotamer(name, -64.6, 13.5, -69.1, 17.3, -33.4, 27.4, 0.0, 0.0);
                    this.aminoAcidRotamerCache[n][3] = new Rotamer(name, -55.6, 10.6, 77.0, 6.8, 25.3, 32.6, 0.0, 0.0);
                    this.aminoAcidRotamerCache[n][4] = new Rotamer(name, 69.8, 10.6, -179.0, 23.7, 6.6, 64.2, 0.0, 0.0);
                    this.aminoAcidRotamerCache[n][5] = new Rotamer(name, -173.6, 14.6, 70.6, 8.7, 14.0, 37.1, 0.0, 0.0);
                    this.aminoAcidRotamerCache[n][6] = new Rotamer(name, 63.0, 4.3, -80.4, 13.9, 16.3, 20.8, 0.0, 0.0);
                    this.aminoAcidRotamerCache[n][7] = new Rotamer(name, -69.6, 19.2, -177.2, 21.7, 168.6, 44.8, 0.0, 0.0);
                    this.aminoAcidRotamerCache[n][8] = new Rotamer(name, -176.2, 14.9, 175.4, 10.6, 175.3, 39.0, 0.0, 0.0);
                    this.aminoAcidRotamerCache[n][9] = new Rotamer(name, -64.6, 13.5, -69.1, 17.3, 146.6, 27.4, 0.0, 0.0);
                    this.aminoAcidRotamerCache[n][10] = new Rotamer(name, -55.6, 10.6, 77.0, 6.8, -154.7, 32.6, 0.0, 0.0);
                    this.aminoAcidRotamerCache[n][11] = new Rotamer(name, 69.8, 10.6, -179.0, 23.7, -173.4, 64.2, 0.0, 0.0);
                    this.aminoAcidRotamerCache[n][12] = new Rotamer(name, -173.6, 14.6, 70.6, 8.7, -166.0, 37.1, 0.0, 0.0);
                    this.aminoAcidRotamerCache[n][13] = new Rotamer(name, 63.0, 4.3, -80.4, 13.9, -163.7, 20.8, 0.0, 0.0);
                    this.aminoAcidRotamerCache[n][14] = new Rotamer(name, -69.6, 19.2, -177.2, 21.7, -11.4, 44.8, 180.0, 0.0);
                    this.aminoAcidRotamerCache[n][15] = new Rotamer(name, -176.2, 14.9, 175.4, 10.6, -6.7, 39.0, 180.0, 0.0);
                    this.aminoAcidRotamerCache[n][16] = new Rotamer(name, -64.6, 13.5, -69.1, 17.3, -33.4, 27.4, 180.0, 0.0);
                    this.aminoAcidRotamerCache[n][17] = new Rotamer(name, -55.6, 10.6, 77.0, 6.8, 25.3, 32.6, 180.0, 0.0);
                    this.aminoAcidRotamerCache[n][18] = new Rotamer(name, 69.8, 10.6, -179.0, 23.7, 6.6, 64.2, 180.0, 0.0);
                    this.aminoAcidRotamerCache[n][19] = new Rotamer(name, -173.6, 14.6, 70.6, 8.7, 14.0, 37.1, 180.0, 0.0);
                    this.aminoAcidRotamerCache[n][20] = new Rotamer(name, 63.0, 4.3, -80.4, 13.9, 16.3, 20.8, 180.0, 0.0);
                    this.aminoAcidRotamerCache[n][21] = new Rotamer(name, -69.6, 19.2, -177.2, 21.7, 168.6, 44.8, 180.0, 0.0);
                    this.aminoAcidRotamerCache[n][22] = new Rotamer(name, -176.2, 14.9, 175.4, 10.6, 175.3, 39.0, 180.0, 0.0);
                    this.aminoAcidRotamerCache[n][23] = new Rotamer(name, -64.6, 13.5, -69.1, 17.3, 146.6, 27.4, 180.0, 0.0);
                    this.aminoAcidRotamerCache[n][24] = new Rotamer(name, -55.6, 10.6, 77.0, 6.8, -154.7, 32.6, 180.0, 0.0);
                    this.aminoAcidRotamerCache[n][25] = new Rotamer(name, 69.8, 10.6, -179.0, 23.7, -173.4, 64.2, 180.0, 0.0);
                    this.aminoAcidRotamerCache[n][26] = new Rotamer(name, -173.6, 14.6, 70.6, 8.7, -166.0, 37.1, 180.0, 0.0);
                    this.aminoAcidRotamerCache[n][27] = new Rotamer(name, 63.0, 4.3, -80.4, 13.9, -163.7, 20.8, 180.0, 0.0);
                    break;
                }
                this.aminoAcidRotamerCache[n] = new Rotamer[35];
                this.aminoAcidRotamerCache[n][0] = new Rotamer(AminoAcidUtils.AminoAcid3.GLH, titrationUtils, -69.6, 19.2, -177.2, 21.7, -11.4, 44.8, 0.0, 0.0);
                this.aminoAcidRotamerCache[n][1] = new Rotamer(AminoAcidUtils.AminoAcid3.GLH, titrationUtils, -176.2, 14.9, 175.4, 10.6, -6.7, 39.0, 0.0, 0.0);
                this.aminoAcidRotamerCache[n][2] = new Rotamer(AminoAcidUtils.AminoAcid3.GLH, titrationUtils, -64.6, 13.5, -69.1, 17.3, -33.4, 27.4, 0.0, 0.0);
                this.aminoAcidRotamerCache[n][3] = new Rotamer(AminoAcidUtils.AminoAcid3.GLH, titrationUtils, -55.6, 10.6, 77.0, 6.8, 25.3, 32.6, 0.0, 0.0);
                this.aminoAcidRotamerCache[n][4] = new Rotamer(AminoAcidUtils.AminoAcid3.GLH, titrationUtils, 69.8, 10.6, -179.0, 23.7, 6.6, 64.2, 0.0, 0.0);
                this.aminoAcidRotamerCache[n][5] = new Rotamer(AminoAcidUtils.AminoAcid3.GLH, titrationUtils, -173.6, 14.6, 70.6, 8.7, 14.0, 37.1, 0.0, 0.0);
                this.aminoAcidRotamerCache[n][6] = new Rotamer(AminoAcidUtils.AminoAcid3.GLH, titrationUtils, 63.0, 4.3, -80.4, 13.9, 16.3, 20.8, 0.0, 0.0);
                this.aminoAcidRotamerCache[n][7] = new Rotamer(AminoAcidUtils.AminoAcid3.GLH, titrationUtils, -69.6, 19.2, -177.2, 21.7, 168.6, 44.8, 0.0, 0.0);
                this.aminoAcidRotamerCache[n][8] = new Rotamer(AminoAcidUtils.AminoAcid3.GLH, titrationUtils, -176.2, 14.9, 175.4, 10.6, 175.3, 39.0, 0.0, 0.0);
                this.aminoAcidRotamerCache[n][9] = new Rotamer(AminoAcidUtils.AminoAcid3.GLH, titrationUtils, -64.6, 13.5, -69.1, 17.3, 146.6, 27.4, 0.0, 0.0);
                this.aminoAcidRotamerCache[n][10] = new Rotamer(AminoAcidUtils.AminoAcid3.GLH, titrationUtils, -54.6, 10.6, 77.0, 6.8, -154.7, 32.6, 0.0, 0.0);
                this.aminoAcidRotamerCache[n][11] = new Rotamer(AminoAcidUtils.AminoAcid3.GLH, titrationUtils, 69.8, 10.6, -179.0, 23.7, -173.4, 64.2, 0.0, 0.0);
                this.aminoAcidRotamerCache[n][12] = new Rotamer(AminoAcidUtils.AminoAcid3.GLH, titrationUtils, -173.6, 14.6, 70.6, 8.7, -166.0, 37.1, 0.0, 0.0);
                this.aminoAcidRotamerCache[n][13] = new Rotamer(AminoAcidUtils.AminoAcid3.GLH, titrationUtils, 63.0, 4.3, -80.4, 13.9, -163.7, 20.8, 0.0, 0.0);
                this.aminoAcidRotamerCache[n][14] = new Rotamer(AminoAcidUtils.AminoAcid3.GLH, titrationUtils, -69.6, 19.2, -177.2, 21.7, -11.4, 44.8, 180.0, 0.0);
                this.aminoAcidRotamerCache[n][15] = new Rotamer(AminoAcidUtils.AminoAcid3.GLH, titrationUtils, -176.2, 14.9, 175.4, 10.6, -6.7, 39.0, 180.0, 0.0);
                this.aminoAcidRotamerCache[n][16] = new Rotamer(AminoAcidUtils.AminoAcid3.GLH, titrationUtils, -64.6, 13.5, -69.1, 17.3, -33.4, 27.4, 180.0, 0.0);
                this.aminoAcidRotamerCache[n][17] = new Rotamer(AminoAcidUtils.AminoAcid3.GLH, titrationUtils, -55.6, 10.6, 77.0, 6.8, 25.3, 32.6, 180.0, 0.0);
                this.aminoAcidRotamerCache[n][18] = new Rotamer(AminoAcidUtils.AminoAcid3.GLH, titrationUtils, 69.8, 10.6, -179.0, 23.7, 6.6, 64.2, 180.0, 0.0);
                this.aminoAcidRotamerCache[n][19] = new Rotamer(AminoAcidUtils.AminoAcid3.GLH, titrationUtils, -173.6, 14.6, 70.6, 8.7, 14.0, 37.1, 180.0, 0.0);
                this.aminoAcidRotamerCache[n][20] = new Rotamer(AminoAcidUtils.AminoAcid3.GLH, titrationUtils, 63.0, 4.3, -80.4, 13.9, 16.3, 20.8, 180.0, 0.0);
                this.aminoAcidRotamerCache[n][21] = new Rotamer(AminoAcidUtils.AminoAcid3.GLH, titrationUtils, -69.6, 19.2, -177.2, 21.7, 168.6, 44.8, 180.0, 0.0);
                this.aminoAcidRotamerCache[n][22] = new Rotamer(AminoAcidUtils.AminoAcid3.GLH, titrationUtils, -176.2, 14.9, 175.4, 10.6, 175.3, 39.0, 180.0, 0.0);
                this.aminoAcidRotamerCache[n][23] = new Rotamer(AminoAcidUtils.AminoAcid3.GLH, titrationUtils, -64.6, 13.5, -69.1, 17.3, 146.6, 27.4, 180.0, 0.0);
                this.aminoAcidRotamerCache[n][24] = new Rotamer(AminoAcidUtils.AminoAcid3.GLH, titrationUtils, -55.6, 10.6, 77.0, 6.8, -154.7, 32.6, 180.0, 0.0);
                this.aminoAcidRotamerCache[n][25] = new Rotamer(AminoAcidUtils.AminoAcid3.GLH, titrationUtils, 69.8, 10.6, -179.0, 23.7, -173.4, 64.2, 180.0, 0.0);
                this.aminoAcidRotamerCache[n][26] = new Rotamer(AminoAcidUtils.AminoAcid3.GLH, titrationUtils, -173.6, 14.6, 70.6, 8.7, -166.0, 37.1, 180.0, 0.0);
                this.aminoAcidRotamerCache[n][27] = new Rotamer(AminoAcidUtils.AminoAcid3.GLH, titrationUtils, 63.0, 4.3, -80.4, 13.9, -163.7, 20.8, 180.0, 0.0);
                this.aminoAcidRotamerCache[n][28] = new Rotamer(AminoAcidUtils.AminoAcid3.GLU, titrationUtils, -69.6, 19.2, -177.2, 21.7, -11.4, 44.8);
                this.aminoAcidRotamerCache[n][29] = new Rotamer(AminoAcidUtils.AminoAcid3.GLU, titrationUtils, -176.2, 14.9, 175.4, 10.6, -6.7, 39.0);
                this.aminoAcidRotamerCache[n][30] = new Rotamer(AminoAcidUtils.AminoAcid3.GLU, titrationUtils, -64.6, 13.5, -69.1, 17.3, -33.4, 27.4);
                this.aminoAcidRotamerCache[n][31] = new Rotamer(AminoAcidUtils.AminoAcid3.GLU, titrationUtils, -55.6, 10.6, 77.0, 6.8, 25.3, 32.6);
                this.aminoAcidRotamerCache[n][32] = new Rotamer(AminoAcidUtils.AminoAcid3.GLU, titrationUtils, 69.8, 10.6, -179.0, 23.7, 6.6, 64.2);
                this.aminoAcidRotamerCache[n][33] = new Rotamer(AminoAcidUtils.AminoAcid3.GLU, titrationUtils, -173.6, 14.6, 70.6, 8.7, 14.0, 37.1);
                this.aminoAcidRotamerCache[n][34] = new Rotamer(AminoAcidUtils.AminoAcid3.GLU, titrationUtils, 63.0, 4.3, -80.4, 13.9, 16.3, 20.8);
                break;
            }
            case GLN: {
                this.aminoAcidRotamerCache[n] = new Rotamer[10];
                this.aminoAcidRotamerCache[n][0] = new Rotamer(name, -66.7, 14.1, -178.5, 14.9, -24.0, 38.0);
                this.aminoAcidRotamerCache[n][1] = new Rotamer(name, -66.7, 14.1, -178.5, 14.9, 156.0, 38.0);
                this.aminoAcidRotamerCache[n][2] = new Rotamer(name, -174.6, 11.5, -177.7, 17.2, -24.0, 38.0);
                this.aminoAcidRotamerCache[n][3] = new Rotamer(name, -174.6, 11.5, -177.7, 17.2, 156.0, 38.0);
                this.aminoAcidRotamerCache[n][4] = new Rotamer(name, -58.7, 11.2, -63.8, 16.1, -46.3, 27.7);
                this.aminoAcidRotamerCache[n][5] = new Rotamer(name, -51.3, 7.3, -90.4, 22.8, 165.0, 38.2);
                this.aminoAcidRotamerCache[n][6] = new Rotamer(name, -179.4, 21.5, 67.3, 7.9, 26.8, 38.4);
                this.aminoAcidRotamerCache[n][7] = new Rotamer(name, 167.5, 14.8, 70.9, 3.7, 174.2, 7.1);
                this.aminoAcidRotamerCache[n][8] = new Rotamer(name, 70.8, 13.0, -165.6, 9.5, -24.0, 38.0);
                this.aminoAcidRotamerCache[n][9] = new Rotamer(name, 70.8, 13.0, -165.6, 9.5, 156.0, 38.0);
                break;
            }
            case MET: {
                this.aminoAcidRotamerCache[n] = new Rotamer[13];
                this.aminoAcidRotamerCache[n][0] = new Rotamer(name, -64.5, 12.7, -68.5, 6.0, -75.6, 14.1);
                this.aminoAcidRotamerCache[n][1] = new Rotamer(name, -78.3, 5.4, -174.7, 15.7, 65.0, 20.0);
                this.aminoAcidRotamerCache[n][2] = new Rotamer(name, -78.3, 5.4, -174.7, 15.7, 180.0, 20.0);
                this.aminoAcidRotamerCache[n][3] = new Rotamer(name, -78.3, 5.4, -174.7, 15.7, -65.0, 20.0);
                this.aminoAcidRotamerCache[n][4] = new Rotamer(name, 178.9, 8.7, 179.0, 13.4, 65.0, 20.0);
                this.aminoAcidRotamerCache[n][5] = new Rotamer(name, 178.9, 8.7, 179.0, 13.4, 65.0, 20.0);
                this.aminoAcidRotamerCache[n][6] = new Rotamer(name, 178.9, 8.7, 179.0, 13.4, -65.0, 20.0);
                this.aminoAcidRotamerCache[n][7] = new Rotamer(name, -70.0, 21.0, -65.0, 20.0, 65.0, 20.0);
                this.aminoAcidRotamerCache[n][8] = new Rotamer(name, -170.0, 24.0, 65.0, 20.0, 180.0, 20.0);
                this.aminoAcidRotamerCache[n][9] = new Rotamer(name, -170.0, 24.0, -65.0, 20.0, 180.0, 20.0);
                this.aminoAcidRotamerCache[n][10] = new Rotamer(name, -70.0, 21.0, 65.0, 20.0, 180.0, 20.0);
                this.aminoAcidRotamerCache[n][11] = new Rotamer(name, -70.0, 21.0, -65.0, 20.0, 180.0, 20.0);
                this.aminoAcidRotamerCache[n][12] = new Rotamer(name, 61.0, 21.0, 65.0, 20.0, 180.0, 20.0);
                break;
            }
            case LYS: 
            case LYD: {
                if (titrationUtils == null) {
                    this.aminoAcidRotamerCache[n] = new Rotamer[12];
                    this.aminoAcidRotamerCache[n][0] = new Rotamer(name, -170.0, 24.0, 180.0, 20.0, 65.0, 20.0, 180.0, 20.0);
                    this.aminoAcidRotamerCache[n][1] = new Rotamer(name, -170.0, 24.0, 180.0, 20.0, 180.0, 20.0, 65.0, 20.0);
                    this.aminoAcidRotamerCache[n][2] = new Rotamer(name, -170.0, 24.0, 180.0, 20.0, 180.0, 20.0, 180.0, 20.0);
                    this.aminoAcidRotamerCache[n][3] = new Rotamer(name, -170.0, 24.0, 180.0, 20.0, -65.0, 20.0, 180.0, 20.0);
                    this.aminoAcidRotamerCache[n][4] = new Rotamer(name, -170.0, 24.0, -65.0, 20.0, 180.0, 20.0, 180.0, 20.0);
                    this.aminoAcidRotamerCache[n][5] = new Rotamer(name, -70.0, 21.0, 65.0, 20.0, 180.0, 20.0, 180.0, 20.0);
                    this.aminoAcidRotamerCache[n][6] = new Rotamer(name, -70.0, 21.0, 180.0, 20.0, 65.0, 20.0, 180.0, 20.0);
                    this.aminoAcidRotamerCache[n][7] = new Rotamer(name, -70.0, 21.0, 180.0, 20.0, 180.0, 20.0, 180.0, 20.0);
                    this.aminoAcidRotamerCache[n][8] = new Rotamer(name, -70.0, 21.0, 180.0, 20.0, 180.0, 20.0, -65.0, 20.0);
                    this.aminoAcidRotamerCache[n][9] = new Rotamer(name, -70.0, 21.0, 180.0, 20.0, -65.0, 20.0, 180.0, 20.0);
                    this.aminoAcidRotamerCache[n][10] = new Rotamer(name, -70.0, 21.0, -65.0, 20.0, 180.0, 20.0, 180.0, 20.0);
                    this.aminoAcidRotamerCache[n][11] = new Rotamer(name, -70.0, 21.0, -65.0, 20.0, 180.0, 20.0, -65.0, 20.0);
                    break;
                }
                this.aminoAcidRotamerCache[n] = new Rotamer[24];
                this.aminoAcidRotamerCache[n][0] = new Rotamer(AminoAcidUtils.AminoAcid3.LYS, titrationUtils, -170.0, 24.0, 180.0, 20.0, 65.0, 20.0, 180.0, 20.0);
                this.aminoAcidRotamerCache[n][1] = new Rotamer(AminoAcidUtils.AminoAcid3.LYS, titrationUtils, -170.0, 24.0, 180.0, 20.0, 180.0, 20.0, 65.0, 20.0);
                this.aminoAcidRotamerCache[n][2] = new Rotamer(AminoAcidUtils.AminoAcid3.LYS, titrationUtils, -170.0, 24.0, 180.0, 20.0, 180.0, 20.0, 180.0, 20.0);
                this.aminoAcidRotamerCache[n][3] = new Rotamer(AminoAcidUtils.AminoAcid3.LYS, titrationUtils, -170.0, 24.0, 180.0, 20.0, -65.0, 20.0, 180.0, 20.0);
                this.aminoAcidRotamerCache[n][4] = new Rotamer(AminoAcidUtils.AminoAcid3.LYS, titrationUtils, -170.0, 24.0, -65.0, 20.0, 180.0, 20.0, 180.0, 20.0);
                this.aminoAcidRotamerCache[n][5] = new Rotamer(AminoAcidUtils.AminoAcid3.LYS, titrationUtils, -70.0, 21.0, 65.0, 20.0, 180.0, 20.0, 180.0, 20.0);
                this.aminoAcidRotamerCache[n][6] = new Rotamer(AminoAcidUtils.AminoAcid3.LYS, titrationUtils, -70.0, 21.0, 180.0, 20.0, 65.0, 20.0, 180.0, 20.0);
                this.aminoAcidRotamerCache[n][7] = new Rotamer(AminoAcidUtils.AminoAcid3.LYS, titrationUtils, -70.0, 21.0, 180.0, 20.0, 180.0, 20.0, 180.0, 20.0);
                this.aminoAcidRotamerCache[n][8] = new Rotamer(AminoAcidUtils.AminoAcid3.LYS, titrationUtils, -70.0, 21.0, 180.0, 20.0, 180.0, 20.0, -65.0, 20.0);
                this.aminoAcidRotamerCache[n][9] = new Rotamer(AminoAcidUtils.AminoAcid3.LYS, titrationUtils, -70.0, 21.0, 180.0, 20.0, -65.0, 20.0, 180.0, 20.0);
                this.aminoAcidRotamerCache[n][10] = new Rotamer(AminoAcidUtils.AminoAcid3.LYS, titrationUtils, -70.0, 21.0, -65.0, 20.0, 180.0, 20.0, 180.0, 20.0);
                this.aminoAcidRotamerCache[n][11] = new Rotamer(AminoAcidUtils.AminoAcid3.LYS, titrationUtils, -70.0, 21.0, -65.0, 20.0, 180.0, 20.0, -65.0, 20.0);
                this.aminoAcidRotamerCache[n][12] = new Rotamer(AminoAcidUtils.AminoAcid3.LYD, titrationUtils, -170.0, 24.0, 180.0, 20.0, 65.0, 20.0, 180.0, 20.0);
                this.aminoAcidRotamerCache[n][13] = new Rotamer(AminoAcidUtils.AminoAcid3.LYD, titrationUtils, -170.0, 24.0, 180.0, 20.0, 180.0, 20.0, 65.0, 20.0);
                this.aminoAcidRotamerCache[n][14] = new Rotamer(AminoAcidUtils.AminoAcid3.LYD, titrationUtils, -170.0, 24.0, 180.0, 20.0, 180.0, 20.0, 180.0, 20.0);
                this.aminoAcidRotamerCache[n][15] = new Rotamer(AminoAcidUtils.AminoAcid3.LYD, titrationUtils, -170.0, 24.0, 180.0, 20.0, -65.0, 20.0, 180.0, 20.0);
                this.aminoAcidRotamerCache[n][16] = new Rotamer(AminoAcidUtils.AminoAcid3.LYD, titrationUtils, -170.0, 24.0, -65.0, 20.0, 180.0, 20.0, 180.0, 20.0);
                this.aminoAcidRotamerCache[n][17] = new Rotamer(AminoAcidUtils.AminoAcid3.LYD, titrationUtils, -70.0, 21.0, 65.0, 20.0, 180.0, 20.0, 180.0, 20.0);
                this.aminoAcidRotamerCache[n][18] = new Rotamer(AminoAcidUtils.AminoAcid3.LYD, titrationUtils, -70.0, 21.0, 180.0, 20.0, 65.0, 20.0, 180.0, 20.0);
                this.aminoAcidRotamerCache[n][19] = new Rotamer(AminoAcidUtils.AminoAcid3.LYD, titrationUtils, -70.0, 21.0, 180.0, 20.0, 180.0, 20.0, 180.0, 20.0);
                this.aminoAcidRotamerCache[n][20] = new Rotamer(AminoAcidUtils.AminoAcid3.LYD, titrationUtils, -70.0, 21.0, 180.0, 20.0, 180.0, 20.0, -65.0, 20.0);
                this.aminoAcidRotamerCache[n][21] = new Rotamer(AminoAcidUtils.AminoAcid3.LYD, titrationUtils, -70.0, 21.0, 180.0, 20.0, -65.0, 20.0, 180.0, 20.0);
                this.aminoAcidRotamerCache[n][22] = new Rotamer(AminoAcidUtils.AminoAcid3.LYD, titrationUtils, -70.0, 21.0, -65.0, 20.0, 180.0, 20.0, 180.0, 20.0);
                this.aminoAcidRotamerCache[n][23] = new Rotamer(AminoAcidUtils.AminoAcid3.LYD, titrationUtils, -70.0, 21.0, -65.0, 20.0, 180.0, 20.0, -65.0, 20.0);
                break;
            }
            case ARG: {
                this.aminoAcidRotamerCache[n] = new Rotamer[14];
                this.aminoAcidRotamerCache[n][0] = new Rotamer(name, 61.0, 25.0, 180.0, 20.0, 65.0, 20.0, 90.0, 20.0);
                this.aminoAcidRotamerCache[n][1] = new Rotamer(name, 61.0, 25.0, 180.0, 20.0, 180.0, 20.0, 180.0, 20.0);
                this.aminoAcidRotamerCache[n][2] = new Rotamer(name, -170.0, 24.0, 180.0, 20.0, 65.0, 20.0, 90.0, 20.0);
                this.aminoAcidRotamerCache[n][3] = new Rotamer(name, -170.0, 24.0, 180.0, 20.0, 180.0, 20.0, 90.0, 20.0);
                this.aminoAcidRotamerCache[n][4] = new Rotamer(name, -170.0, 24.0, 180.0, 20.0, 180.0, 20.0, 180.0, 20.0);
                this.aminoAcidRotamerCache[n][5] = new Rotamer(name, -170.0, 24.0, 180.0, 20.0, 180.0, 20.0, -90.0, 20.0);
                this.aminoAcidRotamerCache[n][6] = new Rotamer(name, -170.0, 24.0, 180.0, 20.0, -65.0, 20.0, 180.0, 20.0);
                this.aminoAcidRotamerCache[n][7] = new Rotamer(name, -70.0, 21.0, 180.0, 20.0, 65.0, 20.0, 90.0, 20.0);
                this.aminoAcidRotamerCache[n][8] = new Rotamer(name, -70.0, 21.0, 180.0, 20.0, 65.0, 20.0, 180.0, 20.0);
                this.aminoAcidRotamerCache[n][9] = new Rotamer(name, -70.0, 21.0, 180.0, 20.0, 180.0, 20.0, 180.0, 20.0);
                this.aminoAcidRotamerCache[n][10] = new Rotamer(name, -70.0, 21.0, 180.0, 20.0, 180.0, 20.0, -90.0, 20.0);
                this.aminoAcidRotamerCache[n][11] = new Rotamer(name, -70.0, 21.0, 180.0, 20.0, -65.0, 20.0, 180.0, 20.0);
                this.aminoAcidRotamerCache[n][12] = new Rotamer(name, -170.0, 21.0, 65.0, 20.0, 65.0, 20.0, 180.0, 20.0);
                this.aminoAcidRotamerCache[n][13] = new Rotamer(name, -70.0, 21.0, -65.0, 20.0, -65.0, 20.0, 180.0, 20.0);
                break;
            }
        }
        return this.aminoAcidRotamerCache[n];
    }

    /*
     * Opcode count of 14638 triggered aggressive code reduction.  Override with --aggressivesizethreshold.
     */
    private Rotamer[] getRichardsonRotamers(AminoAcidUtils.AminoAcid3 name, TitrationUtils titrationUtils) {
        int n = name.ordinal();
        if (this.aminoAcidRotamerCache[n] != null) {
            return this.aminoAcidRotamerCache[n];
        }
        switch (name) {
            case VAL: {
                this.aminoAcidRotamerCache[n] = new Rotamer[3];
                this.aminoAcidRotamerCache[n][0] = new Rotamer(name, 64.0, 0.0);
                this.aminoAcidRotamerCache[n][1] = new Rotamer(name, 175.0, 0.0);
                this.aminoAcidRotamerCache[n][2] = new Rotamer(name, -60.0, 0.0);
                break;
            }
            case LEU: {
                this.aminoAcidRotamerCache[n] = new Rotamer[5];
                this.aminoAcidRotamerCache[n][0] = new Rotamer(name, 62.0, 0.0, 80.0, 0.0);
                this.aminoAcidRotamerCache[n][1] = new Rotamer(name, -177.0, 0.0, 65.0, 0.0);
                this.aminoAcidRotamerCache[n][2] = new Rotamer(name, -172.0, 0.0, 145.0, 0.0);
                this.aminoAcidRotamerCache[n][3] = new Rotamer(name, -85.0, 0.0, 65.0, 0.0);
                this.aminoAcidRotamerCache[n][4] = new Rotamer(name, -65.0, 0.0, 175.0, 0.0);
                break;
            }
            case ILE: {
                this.aminoAcidRotamerCache[n] = new Rotamer[7];
                this.aminoAcidRotamerCache[n][0] = new Rotamer(name, 62.0, 0.0, 100.0, 0.0);
                this.aminoAcidRotamerCache[n][1] = new Rotamer(name, 62.0, 0.0, 170.0, 0.0);
                this.aminoAcidRotamerCache[n][2] = new Rotamer(name, -177.0, 0.0, 66.0, 0.0);
                this.aminoAcidRotamerCache[n][3] = new Rotamer(name, -177.0, 0.0, 165.0, 0.0);
                this.aminoAcidRotamerCache[n][4] = new Rotamer(name, -65.0, 0.0, 100.0, 0.0);
                this.aminoAcidRotamerCache[n][5] = new Rotamer(name, -65.0, 0.0, 170.0, 0.0);
                this.aminoAcidRotamerCache[n][6] = new Rotamer(name, -57.0, 0.0, -60.0, 0.0);
                break;
            }
            case SER: {
                this.aminoAcidRotamerCache[n] = new Rotamer[18];
                this.aminoAcidRotamerCache[n][0] = new Rotamer(name, 62.0, 0.0, 0.0, 0.0);
                this.aminoAcidRotamerCache[n][1] = new Rotamer(name, 62.0, 0.0, 60.0, 0.0);
                this.aminoAcidRotamerCache[n][2] = new Rotamer(name, 62.0, 0.0, 120.0, 0.0);
                this.aminoAcidRotamerCache[n][3] = new Rotamer(name, 62.0, 0.0, 180.0, 0.0);
                this.aminoAcidRotamerCache[n][4] = new Rotamer(name, 62.0, 0.0, -60.0, 0.0);
                this.aminoAcidRotamerCache[n][5] = new Rotamer(name, 62.0, 0.0, -120.0, 0.0);
                this.aminoAcidRotamerCache[n][6] = new Rotamer(name, -177.0, 0.0, 0.0, 0.0);
                this.aminoAcidRotamerCache[n][7] = new Rotamer(name, -177.0, 0.0, 60.0, 0.0);
                this.aminoAcidRotamerCache[n][8] = new Rotamer(name, -177.0, 0.0, 120.0, 0.0);
                this.aminoAcidRotamerCache[n][9] = new Rotamer(name, -177.0, 0.0, 180.0, 0.0);
                this.aminoAcidRotamerCache[n][10] = new Rotamer(name, -177.0, 0.0, -60.0, 0.0);
                this.aminoAcidRotamerCache[n][11] = new Rotamer(name, -177.0, 0.0, -120.0, 0.0);
                this.aminoAcidRotamerCache[n][12] = new Rotamer(name, -65.0, 0.0, 0.0, 0.0);
                this.aminoAcidRotamerCache[n][13] = new Rotamer(name, -65.0, 0.0, 60.0, 0.0);
                this.aminoAcidRotamerCache[n][14] = new Rotamer(name, -65.0, 0.0, 120.0, 0.0);
                this.aminoAcidRotamerCache[n][15] = new Rotamer(name, -65.0, 0.0, 180.0, 0.0);
                this.aminoAcidRotamerCache[n][16] = new Rotamer(name, -65.0, 0.0, -60.0, 0.0);
                this.aminoAcidRotamerCache[n][17] = new Rotamer(name, -65.0, 0.0, -120.0, 0.0);
                break;
            }
            case THR: {
                this.aminoAcidRotamerCache[n] = new Rotamer[18];
                this.aminoAcidRotamerCache[n][0] = new Rotamer(name, 62.0, 0.0, 0.0, 0.0);
                this.aminoAcidRotamerCache[n][1] = new Rotamer(name, 62.0, 0.0, 60.0, 0.0);
                this.aminoAcidRotamerCache[n][2] = new Rotamer(name, 62.0, 0.0, 120.0, 0.0);
                this.aminoAcidRotamerCache[n][3] = new Rotamer(name, 62.0, 0.0, 180.0, 0.0);
                this.aminoAcidRotamerCache[n][4] = new Rotamer(name, 62.0, 0.0, -60.0, 0.0);
                this.aminoAcidRotamerCache[n][5] = new Rotamer(name, 62.0, 0.0, -120.0, 0.0);
                this.aminoAcidRotamerCache[n][6] = new Rotamer(name, -175.0, 0.0, 0.0, 0.0);
                this.aminoAcidRotamerCache[n][7] = new Rotamer(name, -175.0, 0.0, 60.0, 0.0);
                this.aminoAcidRotamerCache[n][8] = new Rotamer(name, -175.0, 0.0, 120.0, 0.0);
                this.aminoAcidRotamerCache[n][9] = new Rotamer(name, -175.0, 0.0, 180.0, 0.0);
                this.aminoAcidRotamerCache[n][10] = new Rotamer(name, -175.0, 0.0, -60.0, 0.0);
                this.aminoAcidRotamerCache[n][11] = new Rotamer(name, -175.0, 0.0, -120.0, 0.0);
                this.aminoAcidRotamerCache[n][12] = new Rotamer(name, -65.0, 0.0, 0.0, 0.0);
                this.aminoAcidRotamerCache[n][13] = new Rotamer(name, -65.0, 0.0, 60.0, 0.0);
                this.aminoAcidRotamerCache[n][14] = new Rotamer(name, -65.0, 0.0, 120.0, 0.0);
                this.aminoAcidRotamerCache[n][15] = new Rotamer(name, -65.0, 0.0, 180.0, 0.0);
                this.aminoAcidRotamerCache[n][16] = new Rotamer(name, -65.0, 0.0, -60.0, 0.0);
                this.aminoAcidRotamerCache[n][17] = new Rotamer(name, -65.0, 0.0, -120.0, 0.0);
                break;
            }
            case CYS: 
            case CYD: {
                if (titrationUtils == null) {
                    this.aminoAcidRotamerCache[n] = new Rotamer[3];
                    this.aminoAcidRotamerCache[n][0] = new Rotamer(name, 62.0, 0.0);
                    this.aminoAcidRotamerCache[n][1] = new Rotamer(name, -177.0, 0.0);
                    this.aminoAcidRotamerCache[n][2] = new Rotamer(name, -65.0, 0.0);
                    break;
                }
                this.aminoAcidRotamerCache[n] = new Rotamer[9];
                this.aminoAcidRotamerCache[n][0] = new Rotamer(AminoAcidUtils.AminoAcid3.CYS, titrationUtils, 62.0, 0.0);
                this.aminoAcidRotamerCache[n][1] = new Rotamer(AminoAcidUtils.AminoAcid3.CYS, titrationUtils, -177.0, 0.0);
                this.aminoAcidRotamerCache[n][2] = new Rotamer(AminoAcidUtils.AminoAcid3.CYS, titrationUtils, -65.0, 0.0);
                this.aminoAcidRotamerCache[n][3] = new Rotamer(AminoAcidUtils.AminoAcid3.CYS, titrationUtils, 62.0, 180.0);
                this.aminoAcidRotamerCache[n][4] = new Rotamer(AminoAcidUtils.AminoAcid3.CYS, titrationUtils, -177.0, 180.0);
                this.aminoAcidRotamerCache[n][5] = new Rotamer(AminoAcidUtils.AminoAcid3.CYS, titrationUtils, -65.0, 18.0);
                this.aminoAcidRotamerCache[n][6] = new Rotamer(AminoAcidUtils.AminoAcid3.CYD, titrationUtils, 62.0, 0.0);
                this.aminoAcidRotamerCache[n][7] = new Rotamer(AminoAcidUtils.AminoAcid3.CYD, titrationUtils, -177.0, 0.0);
                this.aminoAcidRotamerCache[n][8] = new Rotamer(AminoAcidUtils.AminoAcid3.CYD, titrationUtils, -65.0, 0.0);
                break;
            }
            case PHE: 
            case TYD: {
                this.aminoAcidRotamerCache[n] = new Rotamer[4];
                this.aminoAcidRotamerCache[n][0] = new Rotamer(name, 62.0, 0.0, 90.0, 0.0);
                this.aminoAcidRotamerCache[n][1] = new Rotamer(name, -177.0, 0.0, 80.0, 0.0);
                this.aminoAcidRotamerCache[n][2] = new Rotamer(name, -65.0, 0.0, -85.0, 0.0);
                this.aminoAcidRotamerCache[n][3] = new Rotamer(name, -65.0, 0.0, -30.0, 0.0);
                break;
            }
            case TYR: {
                this.aminoAcidRotamerCache[n] = new Rotamer[8];
                this.aminoAcidRotamerCache[n][0] = new Rotamer(name, 62.0, 0.0, 90.0, 0.0, 0.0, 0.0);
                this.aminoAcidRotamerCache[n][1] = new Rotamer(name, 62.0, 0.0, 90.0, 0.0, 180.0, 0.0);
                this.aminoAcidRotamerCache[n][2] = new Rotamer(name, -177.0, 0.0, 80.0, 0.0, 0.0, 0.0);
                this.aminoAcidRotamerCache[n][3] = new Rotamer(name, -177.0, 0.0, 80.0, 0.0, 180.0, 0.0);
                this.aminoAcidRotamerCache[n][4] = new Rotamer(name, -65.0, 0.0, -85.0, 0.0, 0.0, 0.0);
                this.aminoAcidRotamerCache[n][5] = new Rotamer(name, -65.0, 0.0, -85.0, 0.0, 180.0, 0.0);
                this.aminoAcidRotamerCache[n][6] = new Rotamer(name, -65.0, 0.0, -30.0, 0.0, 0.0, 0.0);
                this.aminoAcidRotamerCache[n][7] = new Rotamer(name, -65.0, 0.0, -30.0, 0.0, 180.0, 0.0);
                break;
            }
            case TRP: {
                this.aminoAcidRotamerCache[n] = new Rotamer[7];
                this.aminoAcidRotamerCache[n][0] = new Rotamer(name, 62.0, 0.0, -90.0, 0.0);
                this.aminoAcidRotamerCache[n][1] = new Rotamer(name, 62.0, 0.0, 90.0, 0.0);
                this.aminoAcidRotamerCache[n][2] = new Rotamer(name, -177.0, 0.0, -105.0, 0.0);
                this.aminoAcidRotamerCache[n][3] = new Rotamer(name, -177.0, 0.0, 90.0, 0.0);
                this.aminoAcidRotamerCache[n][4] = new Rotamer(name, -65.0, 0.0, -90.0, 0.0);
                this.aminoAcidRotamerCache[n][5] = new Rotamer(name, -65.0, 0.0, -5.0, 0.0);
                this.aminoAcidRotamerCache[n][6] = new Rotamer(name, -65.0, 0.0, 95.0, 0.0);
                break;
            }
            case HIS: 
            case HIE: 
            case HID: {
                if (titrationUtils == null) {
                    this.aminoAcidRotamerCache[n] = new Rotamer[8];
                    this.aminoAcidRotamerCache[n][0] = new Rotamer(name, 62.0, 0.0, -75.0, 0.0);
                    this.aminoAcidRotamerCache[n][1] = new Rotamer(name, 62.0, 0.0, 80.0, 0.0);
                    this.aminoAcidRotamerCache[n][2] = new Rotamer(name, -177.0, 0.0, -165.0, 0.0);
                    this.aminoAcidRotamerCache[n][3] = new Rotamer(name, -177.0, 0.0, -80.0, 0.0);
                    this.aminoAcidRotamerCache[n][4] = new Rotamer(name, -177.0, 0.0, 60.0, 0.0);
                    this.aminoAcidRotamerCache[n][5] = new Rotamer(name, -65.0, 0.0, -70.0, 0.0);
                    this.aminoAcidRotamerCache[n][6] = new Rotamer(name, -65.0, 0.0, 165.0, 0.0);
                    this.aminoAcidRotamerCache[n][7] = new Rotamer(name, -65.0, 0.0, 80.0, 0.0);
                    break;
                }
                this.aminoAcidRotamerCache[n] = new Rotamer[24];
                this.aminoAcidRotamerCache[n][0] = new Rotamer(AminoAcidUtils.AminoAcid3.HIS, titrationUtils, 62.0, 0.0, -75.0, 0.0);
                this.aminoAcidRotamerCache[n][1] = new Rotamer(AminoAcidUtils.AminoAcid3.HIS, titrationUtils, 62.0, 0.0, 80.0, 0.0);
                this.aminoAcidRotamerCache[n][2] = new Rotamer(AminoAcidUtils.AminoAcid3.HIS, titrationUtils, -177.0, 0.0, -165.0, 0.0);
                this.aminoAcidRotamerCache[n][3] = new Rotamer(AminoAcidUtils.AminoAcid3.HIS, titrationUtils, -177.0, 0.0, -80.0, 0.0);
                this.aminoAcidRotamerCache[n][4] = new Rotamer(AminoAcidUtils.AminoAcid3.HIS, titrationUtils, -177.0, 0.0, 60.0, 0.0);
                this.aminoAcidRotamerCache[n][5] = new Rotamer(AminoAcidUtils.AminoAcid3.HIS, titrationUtils, -65.0, 0.0, -70.0, 0.0);
                this.aminoAcidRotamerCache[n][6] = new Rotamer(AminoAcidUtils.AminoAcid3.HIS, titrationUtils, -65.0, 0.0, 165.0, 0.0);
                this.aminoAcidRotamerCache[n][7] = new Rotamer(AminoAcidUtils.AminoAcid3.HIS, titrationUtils, -65.0, 0.0, 80.0, 0.0);
                this.aminoAcidRotamerCache[n][8] = new Rotamer(AminoAcidUtils.AminoAcid3.HID, titrationUtils, 62.0, 0.0, -75.0, 0.0);
                this.aminoAcidRotamerCache[n][9] = new Rotamer(AminoAcidUtils.AminoAcid3.HID, titrationUtils, 62.0, 0.0, 80.0, 0.0);
                this.aminoAcidRotamerCache[n][10] = new Rotamer(AminoAcidUtils.AminoAcid3.HID, titrationUtils, -177.0, 0.0, -165.0, 0.0);
                this.aminoAcidRotamerCache[n][11] = new Rotamer(AminoAcidUtils.AminoAcid3.HID, titrationUtils, -177.0, 0.0, -80.0, 0.0);
                this.aminoAcidRotamerCache[n][12] = new Rotamer(AminoAcidUtils.AminoAcid3.HID, titrationUtils, -177.0, 0.0, 60.0, 0.0);
                this.aminoAcidRotamerCache[n][13] = new Rotamer(AminoAcidUtils.AminoAcid3.HID, titrationUtils, -65.0, 0.0, -70.0, 0.0);
                this.aminoAcidRotamerCache[n][14] = new Rotamer(AminoAcidUtils.AminoAcid3.HID, titrationUtils, -65.0, 0.0, 165.0, 0.0);
                this.aminoAcidRotamerCache[n][15] = new Rotamer(AminoAcidUtils.AminoAcid3.HID, titrationUtils, -65.0, 0.0, 80.0, 0.0);
                this.aminoAcidRotamerCache[n][16] = new Rotamer(AminoAcidUtils.AminoAcid3.HIE, titrationUtils, 62.0, 0.0, -75.0, 0.0);
                this.aminoAcidRotamerCache[n][17] = new Rotamer(AminoAcidUtils.AminoAcid3.HIE, titrationUtils, 62.0, 0.0, 80.0, 0.0);
                this.aminoAcidRotamerCache[n][18] = new Rotamer(AminoAcidUtils.AminoAcid3.HIE, titrationUtils, -177.0, 0.0, -165.0, 0.0);
                this.aminoAcidRotamerCache[n][19] = new Rotamer(AminoAcidUtils.AminoAcid3.HIE, titrationUtils, -177.0, 0.0, -80.0, 0.0);
                this.aminoAcidRotamerCache[n][20] = new Rotamer(AminoAcidUtils.AminoAcid3.HIE, titrationUtils, -177.0, 0.0, 60.0, 0.0);
                this.aminoAcidRotamerCache[n][21] = new Rotamer(AminoAcidUtils.AminoAcid3.HIE, titrationUtils, -65.0, 0.0, -70.0, 0.0);
                this.aminoAcidRotamerCache[n][22] = new Rotamer(AminoAcidUtils.AminoAcid3.HIE, titrationUtils, -65.0, 0.0, 165.0, 0.0);
                this.aminoAcidRotamerCache[n][23] = new Rotamer(AminoAcidUtils.AminoAcid3.HIE, titrationUtils, -65.0, 0.0, 80.0, 0.0);
                break;
            }
            case ASP: {
                this.aminoAcidRotamerCache[n] = new Rotamer[5];
                this.aminoAcidRotamerCache[n][0] = new Rotamer(name, 62.0, 0.0, 10.0, 0.0);
                this.aminoAcidRotamerCache[n][1] = new Rotamer(name, 62.0, 0.0, 30.0, 0.0);
                this.aminoAcidRotamerCache[n][2] = new Rotamer(name, -177.0, 0.0, 0.0, 0.0);
                this.aminoAcidRotamerCache[n][3] = new Rotamer(name, -177.0, 0.0, 65.0, 0.0);
                this.aminoAcidRotamerCache[n][4] = new Rotamer(name, -70.0, 0.0, -15.0, 0.0);
                break;
            }
            case ASH: {
                if (titrationUtils == null) {
                    this.aminoAcidRotamerCache[n] = new Rotamer[20];
                    this.aminoAcidRotamerCache[n][0] = new Rotamer(name, 62.0, 0.0, 10.0, 0.0, 0.0, 0.0);
                    this.aminoAcidRotamerCache[n][1] = new Rotamer(name, 62.0, 0.0, 30.0, 0.0, 0.0, 0.0);
                    this.aminoAcidRotamerCache[n][2] = new Rotamer(name, -177.0, 0.0, 0.0, 0.0, 0.0, 0.0);
                    this.aminoAcidRotamerCache[n][3] = new Rotamer(name, -177.0, 0.0, 65.0, 0.0, 0.0, 0.0);
                    this.aminoAcidRotamerCache[n][4] = new Rotamer(name, -70.0, 0.0, -15.0, 0.0, 0.0, 0.0);
                    this.aminoAcidRotamerCache[n][5] = new Rotamer(name, 62.0, 0.0, -170.0, 0.0, 0.0, 0.0);
                    this.aminoAcidRotamerCache[n][6] = new Rotamer(name, 62.0, 0.0, -150.0, 0.0, 0.0, 0.0);
                    this.aminoAcidRotamerCache[n][7] = new Rotamer(name, -177.0, 0.0, -180.0, 0.0, 0.0, 0.0);
                    this.aminoAcidRotamerCache[n][8] = new Rotamer(name, -177.0, 0.0, -115.0, 0.0, 0.0, 0.0);
                    this.aminoAcidRotamerCache[n][9] = new Rotamer(name, -70.0, 0.0, 165.0, 0.0, 0.0, 0.0);
                    this.aminoAcidRotamerCache[n][10] = new Rotamer(name, 62.0, 0.0, 10.0, 0.0, 180.0, 0.0);
                    this.aminoAcidRotamerCache[n][11] = new Rotamer(name, 62.0, 0.0, 30.0, 0.0, 180.0, 0.0);
                    this.aminoAcidRotamerCache[n][12] = new Rotamer(name, -177.0, 0.0, 0.0, 0.0, 180.0, 0.0);
                    this.aminoAcidRotamerCache[n][13] = new Rotamer(name, -177.0, 0.0, 65.0, 0.0, 180.0, 0.0);
                    this.aminoAcidRotamerCache[n][14] = new Rotamer(name, -70.0, 0.0, -15.0, 0.0, 180.0, 0.0);
                    this.aminoAcidRotamerCache[n][15] = new Rotamer(name, 62.0, 0.0, -170.0, 0.0, 180.0, 0.0);
                    this.aminoAcidRotamerCache[n][16] = new Rotamer(name, 62.0, 0.0, -150.0, 0.0, 180.0, 0.0);
                    this.aminoAcidRotamerCache[n][17] = new Rotamer(name, -177.0, 0.0, -180.0, 0.0, 180.0, 0.0);
                    this.aminoAcidRotamerCache[n][18] = new Rotamer(name, -177.0, 0.0, -115.0, 0.0, 180.0, 0.0);
                    this.aminoAcidRotamerCache[n][19] = new Rotamer(name, -70.0, 0.0, 165.0, 0.0, 180.0, 0.0);
                    break;
                }
                this.aminoAcidRotamerCache[n] = new Rotamer[25];
                this.aminoAcidRotamerCache[n][0] = new Rotamer(AminoAcidUtils.AminoAcid3.ASH, titrationUtils, 62.0, 0.0, 10.0, 0.0, 0.0, 0.0);
                this.aminoAcidRotamerCache[n][1] = new Rotamer(AminoAcidUtils.AminoAcid3.ASH, titrationUtils, 62.0, 0.0, 30.0, 0.0, 0.0, 0.0);
                this.aminoAcidRotamerCache[n][2] = new Rotamer(AminoAcidUtils.AminoAcid3.ASH, titrationUtils, -177.0, 0.0, 0.0, 0.0, 0.0, 0.0);
                this.aminoAcidRotamerCache[n][3] = new Rotamer(AminoAcidUtils.AminoAcid3.ASH, titrationUtils, -177.0, 0.0, 65.0, 0.0, 0.0, 0.0);
                this.aminoAcidRotamerCache[n][4] = new Rotamer(AminoAcidUtils.AminoAcid3.ASH, titrationUtils, -70.0, 0.0, -15.0, 0.0, 0.0, 0.0);
                this.aminoAcidRotamerCache[n][5] = new Rotamer(AminoAcidUtils.AminoAcid3.ASH, titrationUtils, 62.0, 0.0, -170.0, 0.0, 0.0, 0.0);
                this.aminoAcidRotamerCache[n][6] = new Rotamer(AminoAcidUtils.AminoAcid3.ASH, titrationUtils, 62.0, 0.0, -150.0, 0.0, 0.0, 0.0);
                this.aminoAcidRotamerCache[n][7] = new Rotamer(AminoAcidUtils.AminoAcid3.ASH, titrationUtils, -177.0, 0.0, -180.0, 0.0, 0.0, 0.0);
                this.aminoAcidRotamerCache[n][8] = new Rotamer(AminoAcidUtils.AminoAcid3.ASH, titrationUtils, -177.0, 0.0, -115.0, 0.0, 0.0, 0.0);
                this.aminoAcidRotamerCache[n][9] = new Rotamer(AminoAcidUtils.AminoAcid3.ASH, titrationUtils, -70.0, 0.0, 165.0, 0.0, 0.0, 0.0);
                this.aminoAcidRotamerCache[n][10] = new Rotamer(AminoAcidUtils.AminoAcid3.ASH, titrationUtils, 62.0, 0.0, 10.0, 0.0, 180.0, 0.0);
                this.aminoAcidRotamerCache[n][11] = new Rotamer(AminoAcidUtils.AminoAcid3.ASH, titrationUtils, 62.0, 0.0, 30.0, 0.0, 180.0, 0.0);
                this.aminoAcidRotamerCache[n][12] = new Rotamer(AminoAcidUtils.AminoAcid3.ASH, titrationUtils, -177.0, 0.0, 0.0, 0.0, 180.0, 0.0);
                this.aminoAcidRotamerCache[n][13] = new Rotamer(AminoAcidUtils.AminoAcid3.ASH, titrationUtils, -177.0, 0.0, 65.0, 0.0, 180.0, 0.0);
                this.aminoAcidRotamerCache[n][14] = new Rotamer(AminoAcidUtils.AminoAcid3.ASH, titrationUtils, -70.0, 0.0, -15.0, 0.0, 180.0, 0.0);
                this.aminoAcidRotamerCache[n][15] = new Rotamer(AminoAcidUtils.AminoAcid3.ASH, titrationUtils, 62.0, 0.0, -170.0, 0.0, 180.0, 0.0);
                this.aminoAcidRotamerCache[n][16] = new Rotamer(AminoAcidUtils.AminoAcid3.ASH, titrationUtils, 62.0, 0.0, -150.0, 0.0, 180.0, 0.0);
                this.aminoAcidRotamerCache[n][17] = new Rotamer(AminoAcidUtils.AminoAcid3.ASH, titrationUtils, -177.0, 0.0, -180.0, 0.0, 180.0, 0.0);
                this.aminoAcidRotamerCache[n][18] = new Rotamer(AminoAcidUtils.AminoAcid3.ASH, titrationUtils, -177.0, 0.0, -115.0, 0.0, 180.0, 0.0);
                this.aminoAcidRotamerCache[n][19] = new Rotamer(AminoAcidUtils.AminoAcid3.ASH, titrationUtils, -70.0, 0.0, 165.0, 0.0, 180.0, 0.0);
                this.aminoAcidRotamerCache[n][20] = new Rotamer(AminoAcidUtils.AminoAcid3.ASP, titrationUtils, 62.0, 0.0, 10.0, 0.0);
                this.aminoAcidRotamerCache[n][21] = new Rotamer(AminoAcidUtils.AminoAcid3.ASP, titrationUtils, 62.0, 0.0, 30.0, 0.0);
                this.aminoAcidRotamerCache[n][22] = new Rotamer(AminoAcidUtils.AminoAcid3.ASP, titrationUtils, -177.0, 0.0, 0.0, 0.0);
                this.aminoAcidRotamerCache[n][23] = new Rotamer(AminoAcidUtils.AminoAcid3.ASP, titrationUtils, -177.0, 0.0, 65.0, 0.0);
                this.aminoAcidRotamerCache[n][24] = new Rotamer(AminoAcidUtils.AminoAcid3.ASP, titrationUtils, -70.0, 0.0, -15.0, 0.0);
                break;
            }
            case ASN: {
                this.aminoAcidRotamerCache[n] = new Rotamer[7];
                this.aminoAcidRotamerCache[n][0] = new Rotamer(name, 62.0, 0.0, -10.0, 0.0);
                this.aminoAcidRotamerCache[n][1] = new Rotamer(name, 62.0, 0.0, 30.0, 0.0);
                this.aminoAcidRotamerCache[n][2] = new Rotamer(name, -174.0, 0.0, -20.0, 0.0);
                this.aminoAcidRotamerCache[n][3] = new Rotamer(name, -177.0, 0.0, 30.0, 0.0);
                this.aminoAcidRotamerCache[n][4] = new Rotamer(name, -65.0, 0.0, -20.0, 0.0);
                this.aminoAcidRotamerCache[n][5] = new Rotamer(name, -65.0, 0.0, -75.0, 0.0);
                this.aminoAcidRotamerCache[n][6] = new Rotamer(name, -65.0, 0.0, 120.0, 0.0);
                break;
            }
            case GLU: {
                this.aminoAcidRotamerCache[n] = new Rotamer[8];
                this.aminoAcidRotamerCache[n][0] = new Rotamer(name, 62.0, 0.0, 180.0, 0.0, -20.0, 0.0);
                this.aminoAcidRotamerCache[n][1] = new Rotamer(name, 70.0, 0.0, -80.0, 0.0, 0.0, 0.0);
                this.aminoAcidRotamerCache[n][2] = new Rotamer(name, -177.0, 0.0, 65.0, 0.0, 10.0, 0.0);
                this.aminoAcidRotamerCache[n][3] = new Rotamer(name, -177.0, 0.0, 180.0, 0.0, 0.0, 0.0);
                this.aminoAcidRotamerCache[n][4] = new Rotamer(name, -177.0, 0.0, -80.0, 0.0, -25.0, 0.0);
                this.aminoAcidRotamerCache[n][5] = new Rotamer(name, -65.0, 0.0, 85.0, 0.0, 0.0, 0.0);
                this.aminoAcidRotamerCache[n][6] = new Rotamer(name, -67.0, 0.0, -180.0, 0.0, -10.0, 0.0);
                this.aminoAcidRotamerCache[n][7] = new Rotamer(name, -65.0, 0.0, -65.0, 0.0, -40.0, 0.0);
                break;
            }
            case GLH: {
                if (titrationUtils == null) {
                    this.aminoAcidRotamerCache[n] = new Rotamer[32];
                    this.aminoAcidRotamerCache[n][0] = new Rotamer(name, 62.0, 0.0, 180.0, 0.0, -20.0, 0.0, 0.0, 0.0);
                    this.aminoAcidRotamerCache[n][1] = new Rotamer(name, 70.0, 0.0, -80.0, 0.0, 0.0, 0.0, 0.0, 0.0);
                    this.aminoAcidRotamerCache[n][2] = new Rotamer(name, -177.0, 0.0, 65.0, 0.0, 10.0, 0.0, 0.0, 0.0);
                    this.aminoAcidRotamerCache[n][3] = new Rotamer(name, -177.0, 0.0, 180.0, 0.0, 0.0, 0.0, 0.0, 0.0);
                    this.aminoAcidRotamerCache[n][4] = new Rotamer(name, -177.0, 0.0, -80.0, 0.0, -25.0, 0.0, 0.0, 0.0);
                    this.aminoAcidRotamerCache[n][5] = new Rotamer(name, -65.0, 0.0, 85.0, 0.0, 0.0, 0.0, 0.0, 0.0);
                    this.aminoAcidRotamerCache[n][6] = new Rotamer(name, -67.0, 0.0, -180.0, 0.0, -10.0, 0.0, 0.0, 0.0);
                    this.aminoAcidRotamerCache[n][7] = new Rotamer(name, -65.0, 0.0, -65.0, 0.0, -40.0, 0.0, 0.0, 0.0);
                    this.aminoAcidRotamerCache[n][8] = new Rotamer(name, 62.0, 0.0, 180.0, 0.0, 160.0, 0.0, 0.0, 0.0);
                    this.aminoAcidRotamerCache[n][9] = new Rotamer(name, 70.0, 0.0, -80.0, 0.0, -180.0, 0.0, 0.0, 0.0);
                    this.aminoAcidRotamerCache[n][10] = new Rotamer(name, -177.0, 0.0, 65.0, 0.0, -170.0, 0.0, 0.0, 0.0);
                    this.aminoAcidRotamerCache[n][11] = new Rotamer(name, -177.0, 0.0, 180.0, 0.0, -180.0, 0.0, 0.0, 0.0);
                    this.aminoAcidRotamerCache[n][12] = new Rotamer(name, -177.0, 0.0, -80.0, 0.0, 155.0, 0.0, 0.0, 0.0);
                    this.aminoAcidRotamerCache[n][13] = new Rotamer(name, -65.0, 0.0, 85.0, 0.0, -180.0, 0.0, 0.0, 0.0);
                    this.aminoAcidRotamerCache[n][14] = new Rotamer(name, -67.0, 0.0, -180.0, 0.0, 170.0, 0.0, 0.0, 0.0);
                    this.aminoAcidRotamerCache[n][15] = new Rotamer(name, -65.0, 0.0, -65.0, 0.0, 140.0, 0.0, 0.0, 0.0);
                    this.aminoAcidRotamerCache[n][16] = new Rotamer(name, 62.0, 0.0, 180.0, 0.0, -20.0, 0.0, 180.0, 0.0);
                    this.aminoAcidRotamerCache[n][17] = new Rotamer(name, 70.0, 0.0, -80.0, 0.0, 0.0, 0.0, 180.0, 0.0);
                    this.aminoAcidRotamerCache[n][18] = new Rotamer(name, -177.0, 0.0, 65.0, 0.0, 10.0, 0.0, 180.0, 0.0);
                    this.aminoAcidRotamerCache[n][19] = new Rotamer(name, -177.0, 0.0, 180.0, 0.0, 0.0, 0.0, 180.0, 0.0);
                    this.aminoAcidRotamerCache[n][20] = new Rotamer(name, -177.0, 0.0, -80.0, 0.0, -25.0, 0.0, 180.0, 0.0);
                    this.aminoAcidRotamerCache[n][21] = new Rotamer(name, -65.0, 0.0, 85.0, 0.0, 0.0, 0.0, 180.0, 0.0);
                    this.aminoAcidRotamerCache[n][22] = new Rotamer(name, -67.0, 0.0, -180.0, 0.0, -10.0, 0.0, 180.0, 0.0);
                    this.aminoAcidRotamerCache[n][23] = new Rotamer(name, -65.0, 0.0, -65.0, 0.0, -40.0, 0.0, 180.0, 0.0);
                    this.aminoAcidRotamerCache[n][24] = new Rotamer(name, 62.0, 0.0, 180.0, 0.0, 160.0, 0.0, 180.0, 0.0);
                    this.aminoAcidRotamerCache[n][25] = new Rotamer(name, 70.0, 0.0, -80.0, 0.0, -180.0, 0.0, 180.0, 0.0);
                    this.aminoAcidRotamerCache[n][26] = new Rotamer(name, -177.0, 0.0, 65.0, 0.0, -170.0, 0.0, 180.0, 0.0);
                    this.aminoAcidRotamerCache[n][27] = new Rotamer(name, -177.0, 0.0, 180.0, 0.0, -180.0, 0.0, 180.0, 0.0);
                    this.aminoAcidRotamerCache[n][28] = new Rotamer(name, -177.0, 0.0, -80.0, 0.0, 155.0, 0.0, 180.0, 0.0);
                    this.aminoAcidRotamerCache[n][29] = new Rotamer(name, -65.0, 0.0, 85.0, 0.0, -180.0, 0.0, 180.0, 0.0);
                    this.aminoAcidRotamerCache[n][30] = new Rotamer(name, -67.0, 0.0, -180.0, 0.0, 170.0, 0.0, 180.0, 0.0);
                    this.aminoAcidRotamerCache[n][31] = new Rotamer(name, -65.0, 0.0, -65.0, 0.0, 140.0, 0.0, 180.0, 0.0);
                    break;
                }
                this.aminoAcidRotamerCache[n] = new Rotamer[40];
                this.aminoAcidRotamerCache[n][0] = new Rotamer(AminoAcidUtils.AminoAcid3.GLH, titrationUtils, 62.0, 0.0, 180.0, 0.0, -20.0, 0.0, 0.0, 0.0);
                this.aminoAcidRotamerCache[n][1] = new Rotamer(AminoAcidUtils.AminoAcid3.GLH, titrationUtils, 70.0, 0.0, -80.0, 0.0, 0.0, 0.0, 0.0, 0.0);
                this.aminoAcidRotamerCache[n][2] = new Rotamer(AminoAcidUtils.AminoAcid3.GLH, titrationUtils, -177.0, 0.0, 65.0, 0.0, 10.0, 0.0, 0.0, 0.0);
                this.aminoAcidRotamerCache[n][3] = new Rotamer(AminoAcidUtils.AminoAcid3.GLH, titrationUtils, -177.0, 0.0, 180.0, 0.0, 0.0, 0.0, 0.0, 0.0);
                this.aminoAcidRotamerCache[n][4] = new Rotamer(AminoAcidUtils.AminoAcid3.GLH, titrationUtils, -177.0, 0.0, -80.0, 0.0, -25.0, 0.0, 0.0, 0.0);
                this.aminoAcidRotamerCache[n][5] = new Rotamer(AminoAcidUtils.AminoAcid3.GLH, titrationUtils, -65.0, 0.0, 85.0, 0.0, 0.0, 0.0, 0.0, 0.0);
                this.aminoAcidRotamerCache[n][6] = new Rotamer(AminoAcidUtils.AminoAcid3.GLH, titrationUtils, -67.0, 0.0, -180.0, 0.0, -10.0, 0.0, 0.0, 0.0);
                this.aminoAcidRotamerCache[n][7] = new Rotamer(AminoAcidUtils.AminoAcid3.GLH, titrationUtils, -65.0, 0.0, -65.0, 0.0, -40.0, 0.0, 0.0, 0.0);
                this.aminoAcidRotamerCache[n][8] = new Rotamer(AminoAcidUtils.AminoAcid3.GLH, titrationUtils, 62.0, 0.0, 180.0, 0.0, 160.0, 0.0, 0.0, 0.0);
                this.aminoAcidRotamerCache[n][9] = new Rotamer(AminoAcidUtils.AminoAcid3.GLH, titrationUtils, 70.0, 0.0, -80.0, 0.0, -180.0, 0.0, 0.0, 0.0);
                this.aminoAcidRotamerCache[n][10] = new Rotamer(AminoAcidUtils.AminoAcid3.GLH, titrationUtils, -177.0, 0.0, 65.0, 0.0, -170.0, 0.0, 0.0, 0.0);
                this.aminoAcidRotamerCache[n][11] = new Rotamer(AminoAcidUtils.AminoAcid3.GLH, titrationUtils, -177.0, 0.0, 180.0, 0.0, -180.0, 0.0, 0.0, 0.0);
                this.aminoAcidRotamerCache[n][12] = new Rotamer(AminoAcidUtils.AminoAcid3.GLH, titrationUtils, -177.0, 0.0, -80.0, 0.0, 155.0, 0.0, 0.0, 0.0);
                this.aminoAcidRotamerCache[n][13] = new Rotamer(AminoAcidUtils.AminoAcid3.GLH, titrationUtils, -65.0, 0.0, 85.0, 0.0, -180.0, 0.0, 0.0, 0.0);
                this.aminoAcidRotamerCache[n][14] = new Rotamer(AminoAcidUtils.AminoAcid3.GLH, titrationUtils, -67.0, 0.0, -180.0, 0.0, 170.0, 0.0, 0.0, 0.0);
                this.aminoAcidRotamerCache[n][15] = new Rotamer(AminoAcidUtils.AminoAcid3.GLH, titrationUtils, -65.0, 0.0, -65.0, 0.0, 140.0, 0.0, 0.0, 0.0);
                this.aminoAcidRotamerCache[n][16] = new Rotamer(AminoAcidUtils.AminoAcid3.GLH, titrationUtils, 62.0, 0.0, 180.0, 0.0, -20.0, 0.0, 180.0, 0.0);
                this.aminoAcidRotamerCache[n][17] = new Rotamer(AminoAcidUtils.AminoAcid3.GLH, titrationUtils, 70.0, 0.0, -80.0, 0.0, 0.0, 0.0, 180.0, 0.0);
                this.aminoAcidRotamerCache[n][18] = new Rotamer(AminoAcidUtils.AminoAcid3.GLH, titrationUtils, -177.0, 0.0, 65.0, 0.0, 10.0, 0.0, 180.0, 0.0);
                this.aminoAcidRotamerCache[n][19] = new Rotamer(AminoAcidUtils.AminoAcid3.GLH, titrationUtils, -177.0, 0.0, 180.0, 0.0, 0.0, 0.0, 180.0, 0.0);
                this.aminoAcidRotamerCache[n][20] = new Rotamer(AminoAcidUtils.AminoAcid3.GLH, titrationUtils, -177.0, 0.0, -80.0, 0.0, -25.0, 0.0, 180.0, 0.0);
                this.aminoAcidRotamerCache[n][21] = new Rotamer(AminoAcidUtils.AminoAcid3.GLH, titrationUtils, -65.0, 0.0, 85.0, 0.0, 0.0, 0.0, 180.0, 0.0);
                this.aminoAcidRotamerCache[n][22] = new Rotamer(AminoAcidUtils.AminoAcid3.GLH, titrationUtils, -67.0, 0.0, -180.0, 0.0, -10.0, 0.0, 180.0, 0.0);
                this.aminoAcidRotamerCache[n][23] = new Rotamer(AminoAcidUtils.AminoAcid3.GLH, titrationUtils, -65.0, 0.0, -65.0, 0.0, -40.0, 0.0, 180.0, 0.0);
                this.aminoAcidRotamerCache[n][24] = new Rotamer(AminoAcidUtils.AminoAcid3.GLH, titrationUtils, 62.0, 0.0, 180.0, 0.0, 160.0, 0.0, 180.0, 0.0);
                this.aminoAcidRotamerCache[n][25] = new Rotamer(AminoAcidUtils.AminoAcid3.GLH, titrationUtils, 70.0, 0.0, -80.0, 0.0, -180.0, 0.0, 180.0, 0.0);
                this.aminoAcidRotamerCache[n][26] = new Rotamer(AminoAcidUtils.AminoAcid3.GLH, titrationUtils, -177.0, 0.0, 65.0, 0.0, -170.0, 0.0, 180.0, 0.0);
                this.aminoAcidRotamerCache[n][27] = new Rotamer(AminoAcidUtils.AminoAcid3.GLH, titrationUtils, -177.0, 0.0, 180.0, 0.0, -180.0, 0.0, 180.0, 0.0);
                this.aminoAcidRotamerCache[n][28] = new Rotamer(AminoAcidUtils.AminoAcid3.GLH, titrationUtils, -177.0, 0.0, -80.0, 0.0, 155.0, 0.0, 180.0, 0.0);
                this.aminoAcidRotamerCache[n][29] = new Rotamer(AminoAcidUtils.AminoAcid3.GLH, titrationUtils, -65.0, 0.0, 85.0, 0.0, -180.0, 0.0, 180.0, 0.0);
                this.aminoAcidRotamerCache[n][30] = new Rotamer(AminoAcidUtils.AminoAcid3.GLH, titrationUtils, -67.0, 0.0, -180.0, 0.0, 170.0, 0.0, 180.0, 0.0);
                this.aminoAcidRotamerCache[n][31] = new Rotamer(AminoAcidUtils.AminoAcid3.GLH, titrationUtils, -65.0, 0.0, -65.0, 0.0, 140.0, 0.0, 180.0, 0.0);
                this.aminoAcidRotamerCache[n][32] = new Rotamer(AminoAcidUtils.AminoAcid3.GLU, titrationUtils, 62.0, 0.0, 180.0, 0.0, -20.0, 0.0);
                this.aminoAcidRotamerCache[n][33] = new Rotamer(AminoAcidUtils.AminoAcid3.GLU, titrationUtils, 70.0, 0.0, -80.0, 0.0, 0.0, 0.0);
                this.aminoAcidRotamerCache[n][34] = new Rotamer(AminoAcidUtils.AminoAcid3.GLU, titrationUtils, -177.0, 0.0, 65.0, 0.0, 10.0, 0.0);
                this.aminoAcidRotamerCache[n][35] = new Rotamer(AminoAcidUtils.AminoAcid3.GLU, titrationUtils, -177.0, 0.0, 180.0, 0.0, 0.0, 0.0);
                this.aminoAcidRotamerCache[n][36] = new Rotamer(AminoAcidUtils.AminoAcid3.GLU, titrationUtils, -177.0, 0.0, -80.0, 0.0, -25.0, 0.0);
                this.aminoAcidRotamerCache[n][37] = new Rotamer(AminoAcidUtils.AminoAcid3.GLU, titrationUtils, -65.0, 0.0, 85.0, 0.0, 0.0, 0.0);
                this.aminoAcidRotamerCache[n][38] = new Rotamer(AminoAcidUtils.AminoAcid3.GLU, titrationUtils, -67.0, 0.0, -180.0, 0.0, -10.0, 0.0);
                this.aminoAcidRotamerCache[n][39] = new Rotamer(AminoAcidUtils.AminoAcid3.GLU, titrationUtils, -65.0, 0.0, -65.0, 0.0, -40.0, 0.0);
                break;
            }
            case GLN: {
                this.aminoAcidRotamerCache[n] = new Rotamer[9];
                this.aminoAcidRotamerCache[n][0] = new Rotamer(name, 62.0, 0.0, 180.0, 0.0, 20.0, 0.0);
                this.aminoAcidRotamerCache[n][1] = new Rotamer(name, 70.0, 0.0, -75.0, 0.0, 0.0, 0.0);
                this.aminoAcidRotamerCache[n][2] = new Rotamer(name, -177.0, 0.0, 65.0, 0.0, -100.0, 0.0);
                this.aminoAcidRotamerCache[n][3] = new Rotamer(name, -177.0, 0.0, 65.0, 0.0, 60.0, 0.0);
                this.aminoAcidRotamerCache[n][4] = new Rotamer(name, -177.0, 0.0, 180.0, 0.0, 0.0, 0.0);
                this.aminoAcidRotamerCache[n][5] = new Rotamer(name, -65.0, 0.0, 85.0, 0.0, 0.0, 0.0);
                this.aminoAcidRotamerCache[n][6] = new Rotamer(name, -67.0, 0.0, 180.0, 0.0, -25.0, 0.0);
                this.aminoAcidRotamerCache[n][7] = new Rotamer(name, -65.0, 0.0, -65.0, 0.0, -40.0, 0.0);
                this.aminoAcidRotamerCache[n][8] = new Rotamer(name, -65.0, 0.0, -65.0, 0.0, 100.0, 0.0);
                break;
            }
            case MET: {
                this.aminoAcidRotamerCache[n] = new Rotamer[13];
                this.aminoAcidRotamerCache[n][0] = new Rotamer(name, 62.0, 0.0, 180.0, 0.0, 75.0, 0.0);
                this.aminoAcidRotamerCache[n][1] = new Rotamer(name, 62.0, 0.0, 180.0, 0.0, -75.0, 0.0);
                this.aminoAcidRotamerCache[n][2] = new Rotamer(name, -177.0, 0.0, 65.0, 0.0, 75.0, 0.0);
                this.aminoAcidRotamerCache[n][3] = new Rotamer(name, -177.0, 0.0, 65.0, 0.0, 180.0, 0.0);
                this.aminoAcidRotamerCache[n][4] = new Rotamer(name, -177.0, 0.0, 180.0, 0.0, 75.0, 0.0);
                this.aminoAcidRotamerCache[n][5] = new Rotamer(name, -177.0, 0.0, 180.0, 0.0, 180.0, 0.0);
                this.aminoAcidRotamerCache[n][6] = new Rotamer(name, -177.0, 0.0, 180.0, 0.0, -75.0, 0.0);
                this.aminoAcidRotamerCache[n][7] = new Rotamer(name, -67.0, 0.0, 180.0, 0.0, 75.0, 0.0);
                this.aminoAcidRotamerCache[n][8] = new Rotamer(name, -67.0, 0.0, 180.0, 0.0, 180.0, 0.0);
                this.aminoAcidRotamerCache[n][9] = new Rotamer(name, -67.0, 0.0, 180.0, 0.0, -75.0, 0.0);
                this.aminoAcidRotamerCache[n][10] = new Rotamer(name, -65.0, 0.0, -65.0, 0.0, 103.0, 0.0);
                this.aminoAcidRotamerCache[n][11] = new Rotamer(name, -65.0, 0.0, -65.0, 0.0, 180.0, 0.0);
                this.aminoAcidRotamerCache[n][12] = new Rotamer(name, -65.0, 0.0, -65.0, 0.0, -70.0, 0.0);
                break;
            }
            case LYS: 
            case LYD: {
                if (titrationUtils == null) {
                    this.aminoAcidRotamerCache[n] = new Rotamer[27];
                    this.aminoAcidRotamerCache[n][0] = new Rotamer(name, 62.0, 0.0, 180.0, 0.0, 68.0, 0.0, 180.0, 0.0);
                    this.aminoAcidRotamerCache[n][1] = new Rotamer(name, 62.0, 0.0, 180.0, 0.0, 180.0, 0.0, 65.0, 0.0);
                    this.aminoAcidRotamerCache[n][2] = new Rotamer(name, 62.0, 0.0, 180.0, 0.0, 180.0, 0.0, 180.0, 0.0);
                    this.aminoAcidRotamerCache[n][3] = new Rotamer(name, 62.0, 0.0, 180.0, 0.0, 180.0, 0.0, -65.0, 0.0);
                    this.aminoAcidRotamerCache[n][4] = new Rotamer(name, 62.0, 0.0, 180.0, 0.0, -68.0, 0.0, 180.0, 0.0);
                    this.aminoAcidRotamerCache[n][5] = new Rotamer(name, -177.0, 0.0, 68.0, 0.0, 180.0, 0.0, 65.0, 0.0);
                    this.aminoAcidRotamerCache[n][6] = new Rotamer(name, -177.0, 0.0, 68.0, 0.0, 180.0, 0.0, 180.0, 0.0);
                    this.aminoAcidRotamerCache[n][7] = new Rotamer(name, -177.0, 0.0, 68.0, 0.0, 180.0, 0.0, -65.0, 0.0);
                    this.aminoAcidRotamerCache[n][8] = new Rotamer(name, -177.0, 0.0, 180.0, 0.0, 68.0, 0.0, 65.0, 0.0);
                    this.aminoAcidRotamerCache[n][9] = new Rotamer(name, -177.0, 0.0, 180.0, 0.0, 68.0, 0.0, 180.0, 0.0);
                    this.aminoAcidRotamerCache[n][10] = new Rotamer(name, -177.0, 0.0, 180.0, 0.0, 180.0, 0.0, 65.0, 0.0);
                    this.aminoAcidRotamerCache[n][11] = new Rotamer(name, -177.0, 0.0, 180.0, 0.0, 180.0, 0.0, 180.0, 0.0);
                    this.aminoAcidRotamerCache[n][12] = new Rotamer(name, -177.0, 0.0, 180.0, 0.0, 180.0, 0.0, -65.0, 0.0);
                    this.aminoAcidRotamerCache[n][13] = new Rotamer(name, -177.0, 0.0, 180.0, 0.0, -68.0, 0.0, 180.0, 0.0);
                    this.aminoAcidRotamerCache[n][14] = new Rotamer(name, -177.0, 0.0, 180.0, 0.0, -68.0, 0.0, -65.0, 0.0);
                    this.aminoAcidRotamerCache[n][15] = new Rotamer(name, -90.0, 0.0, 68.0, 0.0, 180.0, 0.0, 180.0);
                    this.aminoAcidRotamerCache[n][16] = new Rotamer(name, -67.0, 0.0, 180.0, 0.0, 68.0, 0.0, -65.0, 0.0);
                    this.aminoAcidRotamerCache[n][17] = new Rotamer(name, -67.0, 0.0, 180.0, 0.0, 68.0, 0.0, 180.0, 0.0);
                    this.aminoAcidRotamerCache[n][18] = new Rotamer(name, -67.0, 0.0, 180.0, 0.0, 180.0, 0.0, 65.0, 0.0);
                    this.aminoAcidRotamerCache[n][19] = new Rotamer(name, -67.0, 0.0, 180.0, 0.0, 180.0, 0.0, 180.0, 0.0);
                    this.aminoAcidRotamerCache[n][20] = new Rotamer(name, -67.0, 0.0, 180.0, 0.0, 180.0, 0.0, -65.0, 0.0);
                    this.aminoAcidRotamerCache[n][21] = new Rotamer(name, -67.0, 0.0, 180.0, 0.0, -68.0, 0.0, 180.0, 0.0);
                    this.aminoAcidRotamerCache[n][22] = new Rotamer(name, -67.0, 0.0, 180.0, 0.0, -68.0, 0.0, -65.0, 0.0);
                    this.aminoAcidRotamerCache[n][23] = new Rotamer(name, -62.0, 0.0, -68.0, 0.0, 180.0, 0.0, 65.0, 0.0);
                    this.aminoAcidRotamerCache[n][24] = new Rotamer(name, -62.0, 0.0, -68.0, 0.0, 180.0, 0.0, 180.0, 0.0);
                    this.aminoAcidRotamerCache[n][25] = new Rotamer(name, -62.0, 0.0, -68.0, 0.0, 180.0, 0.0, -65.0, 0.0);
                    this.aminoAcidRotamerCache[n][26] = new Rotamer(name, -62.0, 0.0, -68.0, 0.0, -68.0, 0.0, 180.0, 0.0);
                    break;
                }
                this.aminoAcidRotamerCache[n] = new Rotamer[54];
                this.aminoAcidRotamerCache[n][0] = new Rotamer(AminoAcidUtils.AminoAcid3.LYD, titrationUtils, 62.0, 0.0, 180.0, 0.0, 68.0, 0.0, 180.0, 0.0);
                this.aminoAcidRotamerCache[n][1] = new Rotamer(AminoAcidUtils.AminoAcid3.LYD, titrationUtils, 62.0, 0.0, 180.0, 0.0, 180.0, 0.0, 65.0, 0.0);
                this.aminoAcidRotamerCache[n][2] = new Rotamer(AminoAcidUtils.AminoAcid3.LYD, titrationUtils, 62.0, 0.0, 180.0, 0.0, 180.0, 0.0, 180.0, 0.0);
                this.aminoAcidRotamerCache[n][3] = new Rotamer(AminoAcidUtils.AminoAcid3.LYD, titrationUtils, 62.0, 0.0, 180.0, 0.0, 180.0, 0.0, -65.0, 0.0);
                this.aminoAcidRotamerCache[n][4] = new Rotamer(AminoAcidUtils.AminoAcid3.LYD, titrationUtils, 62.0, 0.0, 180.0, 0.0, -68.0, 0.0, 180.0, 0.0);
                this.aminoAcidRotamerCache[n][5] = new Rotamer(AminoAcidUtils.AminoAcid3.LYD, titrationUtils, -177.0, 0.0, 68.0, 0.0, 180.0, 0.0, 65.0, 0.0);
                this.aminoAcidRotamerCache[n][6] = new Rotamer(AminoAcidUtils.AminoAcid3.LYD, titrationUtils, -177.0, 0.0, 68.0, 0.0, 180.0, 0.0, 180.0, 0.0);
                this.aminoAcidRotamerCache[n][7] = new Rotamer(AminoAcidUtils.AminoAcid3.LYD, titrationUtils, -177.0, 0.0, 68.0, 0.0, 180.0, 0.0, -65.0, 0.0);
                this.aminoAcidRotamerCache[n][8] = new Rotamer(AminoAcidUtils.AminoAcid3.LYD, titrationUtils, -177.0, 0.0, 180.0, 0.0, 68.0, 0.0, 65.0, 0.0);
                this.aminoAcidRotamerCache[n][9] = new Rotamer(AminoAcidUtils.AminoAcid3.LYD, titrationUtils, -177.0, 0.0, 180.0, 0.0, 68.0, 0.0, 180.0, 0.0);
                this.aminoAcidRotamerCache[n][10] = new Rotamer(AminoAcidUtils.AminoAcid3.LYD, titrationUtils, -177.0, 0.0, 180.0, 0.0, 180.0, 0.0, 65.0, 0.0);
                this.aminoAcidRotamerCache[n][11] = new Rotamer(AminoAcidUtils.AminoAcid3.LYD, titrationUtils, -177.0, 0.0, 180.0, 0.0, 180.0, 0.0, 180.0, 0.0);
                this.aminoAcidRotamerCache[n][12] = new Rotamer(AminoAcidUtils.AminoAcid3.LYD, titrationUtils, -177.0, 0.0, 180.0, 0.0, 180.0, 0.0, -65.0, 0.0);
                this.aminoAcidRotamerCache[n][13] = new Rotamer(AminoAcidUtils.AminoAcid3.LYD, titrationUtils, -177.0, 0.0, 180.0, 0.0, -68.0, 0.0, 180.0, 0.0);
                this.aminoAcidRotamerCache[n][14] = new Rotamer(AminoAcidUtils.AminoAcid3.LYD, titrationUtils, -177.0, 0.0, 180.0, 0.0, -68.0, 0.0, -65.0, 0.0);
                this.aminoAcidRotamerCache[n][15] = new Rotamer(AminoAcidUtils.AminoAcid3.LYD, titrationUtils, -90.0, 0.0, 68.0, 0.0, 180.0, 0.0, 180.0);
                this.aminoAcidRotamerCache[n][16] = new Rotamer(AminoAcidUtils.AminoAcid3.LYD, titrationUtils, -67.0, 0.0, 180.0, 0.0, 68.0, 0.0, -65.0, 0.0);
                this.aminoAcidRotamerCache[n][17] = new Rotamer(AminoAcidUtils.AminoAcid3.LYD, titrationUtils, -67.0, 0.0, 180.0, 0.0, 68.0, 0.0, 180.0, 0.0);
                this.aminoAcidRotamerCache[n][18] = new Rotamer(AminoAcidUtils.AminoAcid3.LYD, titrationUtils, -67.0, 0.0, 180.0, 0.0, 180.0, 0.0, 65.0, 0.0);
                this.aminoAcidRotamerCache[n][19] = new Rotamer(AminoAcidUtils.AminoAcid3.LYD, titrationUtils, -67.0, 0.0, 180.0, 0.0, 180.0, 0.0, 180.0, 0.0);
                this.aminoAcidRotamerCache[n][20] = new Rotamer(AminoAcidUtils.AminoAcid3.LYD, titrationUtils, -67.0, 0.0, 180.0, 0.0, 180.0, 0.0, -65.0, 0.0);
                this.aminoAcidRotamerCache[n][21] = new Rotamer(AminoAcidUtils.AminoAcid3.LYD, titrationUtils, -67.0, 0.0, 180.0, 0.0, -68.0, 0.0, 180.0, 0.0);
                this.aminoAcidRotamerCache[n][22] = new Rotamer(AminoAcidUtils.AminoAcid3.LYD, titrationUtils, -67.0, 0.0, 180.0, 0.0, -68.0, 0.0, -65.0, 0.0);
                this.aminoAcidRotamerCache[n][23] = new Rotamer(AminoAcidUtils.AminoAcid3.LYD, titrationUtils, -62.0, 0.0, -68.0, 0.0, 180.0, 0.0, 65.0, 0.0);
                this.aminoAcidRotamerCache[n][24] = new Rotamer(AminoAcidUtils.AminoAcid3.LYD, titrationUtils, -62.0, 0.0, -68.0, 0.0, 180.0, 0.0, 180.0, 0.0);
                this.aminoAcidRotamerCache[n][25] = new Rotamer(AminoAcidUtils.AminoAcid3.LYD, titrationUtils, -62.0, 0.0, -68.0, 0.0, 180.0, 0.0, -65.0, 0.0);
                this.aminoAcidRotamerCache[n][26] = new Rotamer(AminoAcidUtils.AminoAcid3.LYD, titrationUtils, -62.0, 0.0, -68.0, 0.0, -68.0, 0.0, 180.0, 0.0);
                this.aminoAcidRotamerCache[n][27] = new Rotamer(AminoAcidUtils.AminoAcid3.LYS, titrationUtils, 62.0, 0.0, 180.0, 0.0, 68.0, 0.0, 180.0, 0.0);
                this.aminoAcidRotamerCache[n][28] = new Rotamer(AminoAcidUtils.AminoAcid3.LYS, titrationUtils, 262.0, 0.0, 180.0, 0.0, 180.0, 0.0, 65.0, 0.0);
                this.aminoAcidRotamerCache[n][29] = new Rotamer(AminoAcidUtils.AminoAcid3.LYS, titrationUtils, 62.0, 0.0, 180.0, 0.0, 180.0, 0.0, 180.0, 0.0);
                this.aminoAcidRotamerCache[n][30] = new Rotamer(AminoAcidUtils.AminoAcid3.LYS, titrationUtils, 62.0, 0.0, 180.0, 0.0, 180.0, 0.0, -65.0, 0.0);
                this.aminoAcidRotamerCache[n][31] = new Rotamer(AminoAcidUtils.AminoAcid3.LYS, titrationUtils, 62.0, 0.0, 180.0, 0.0, -68.0, 0.0, 180.0, 0.0);
                this.aminoAcidRotamerCache[n][32] = new Rotamer(AminoAcidUtils.AminoAcid3.LYS, titrationUtils, -177.0, 0.0, 68.0, 0.0, 180.0, 0.0, 65.0, 0.0);
                this.aminoAcidRotamerCache[n][33] = new Rotamer(AminoAcidUtils.AminoAcid3.LYS, titrationUtils, -177.0, 0.0, 68.0, 0.0, 180.0, 0.0, 180.0, 0.0);
                this.aminoAcidRotamerCache[n][34] = new Rotamer(AminoAcidUtils.AminoAcid3.LYS, titrationUtils, -177.0, 0.0, 68.0, 0.0, 180.0, 0.0, -65.0, 0.0);
                this.aminoAcidRotamerCache[n][35] = new Rotamer(AminoAcidUtils.AminoAcid3.LYS, titrationUtils, -177.0, 0.0, 180.0, 0.0, 68.0, 0.0, 65.0, 0.0);
                this.aminoAcidRotamerCache[n][36] = new Rotamer(AminoAcidUtils.AminoAcid3.LYS, titrationUtils, -177.0, 0.0, 180.0, 0.0, 68.0, 0.0, 180.0, 0.0);
                this.aminoAcidRotamerCache[n][37] = new Rotamer(AminoAcidUtils.AminoAcid3.LYS, titrationUtils, -177.0, 0.0, 180.0, 0.0, 180.0, 0.0, 65.0, 0.0);
                this.aminoAcidRotamerCache[n][38] = new Rotamer(AminoAcidUtils.AminoAcid3.LYS, titrationUtils, -177.0, 0.0, 180.0, 0.0, 180.0, 0.0, 180.0, 0.0);
                this.aminoAcidRotamerCache[n][39] = new Rotamer(AminoAcidUtils.AminoAcid3.LYS, titrationUtils, -177.0, 0.0, 180.0, 0.0, 180.0, 0.0, -65.0, 0.0);
                this.aminoAcidRotamerCache[n][40] = new Rotamer(AminoAcidUtils.AminoAcid3.LYS, titrationUtils, -177.0, 0.0, 180.0, 0.0, -68.0, 0.0, 180.0, 0.0);
                this.aminoAcidRotamerCache[n][41] = new Rotamer(AminoAcidUtils.AminoAcid3.LYS, titrationUtils, -177.0, 0.0, 180.0, 0.0, -68.0, 0.0, -65.0, 0.0);
                this.aminoAcidRotamerCache[n][42] = new Rotamer(AminoAcidUtils.AminoAcid3.LYS, titrationUtils, -90.0, 0.0, 68.0, 0.0, 180.0, 0.0, 180.0);
                this.aminoAcidRotamerCache[n][43] = new Rotamer(AminoAcidUtils.AminoAcid3.LYS, titrationUtils, -67.0, 0.0, 180.0, 0.0, 68.0, 0.0, -65.0, 0.0);
                this.aminoAcidRotamerCache[n][44] = new Rotamer(AminoAcidUtils.AminoAcid3.LYS, titrationUtils, -67.0, 0.0, 180.0, 0.0, 68.0, 0.0, 180.0, 0.0);
                this.aminoAcidRotamerCache[n][45] = new Rotamer(AminoAcidUtils.AminoAcid3.LYS, titrationUtils, -67.0, 0.0, 180.0, 0.0, 180.0, 0.0, 65.0, 0.0);
                this.aminoAcidRotamerCache[n][46] = new Rotamer(AminoAcidUtils.AminoAcid3.LYS, titrationUtils, -67.0, 0.0, 180.0, 0.0, 180.0, 0.0, 180.0, 0.0);
                this.aminoAcidRotamerCache[n][47] = new Rotamer(AminoAcidUtils.AminoAcid3.LYS, titrationUtils, -67.0, 0.0, 180.0, 0.0, 180.0, 0.0, -65.0, 0.0);
                this.aminoAcidRotamerCache[n][48] = new Rotamer(AminoAcidUtils.AminoAcid3.LYS, titrationUtils, -67.0, 0.0, 180.0, 0.0, -68.0, 0.0, 180.0, 0.0);
                this.aminoAcidRotamerCache[n][49] = new Rotamer(AminoAcidUtils.AminoAcid3.LYS, titrationUtils, -67.0, 0.0, 180.0, 0.0, -68.0, 0.0, -65.0, 0.0);
                this.aminoAcidRotamerCache[n][50] = new Rotamer(AminoAcidUtils.AminoAcid3.LYS, titrationUtils, -62.0, 0.0, -68.0, 0.0, 180.0, 0.0, 65.0, 0.0);
                this.aminoAcidRotamerCache[n][51] = new Rotamer(AminoAcidUtils.AminoAcid3.LYS, titrationUtils, -62.0, 0.0, -68.0, 0.0, 180.0, 0.0, 180.0, 0.0);
                this.aminoAcidRotamerCache[n][52] = new Rotamer(AminoAcidUtils.AminoAcid3.LYS, titrationUtils, -62.0, 0.0, -68.0, 0.0, 180.0, 0.0, -65.0, 0.0);
                this.aminoAcidRotamerCache[n][53] = new Rotamer(AminoAcidUtils.AminoAcid3.LYS, titrationUtils, -62.0, 0.0, -68.0, 0.0, -68.0, 0.0, 180.0, 0.0);
                break;
            }
            case ARG: {
                this.aminoAcidRotamerCache[n] = new Rotamer[34];
                this.aminoAcidRotamerCache[n][0] = new Rotamer(name, 62.0, 0.0, 180.0, 0.0, 65.0, 0.0, 85.0, 0.0);
                this.aminoAcidRotamerCache[n][1] = new Rotamer(name, 62.0, 0.0, 180.0, 0.0, 65.0, 0.0, -175.0, 0.0);
                this.aminoAcidRotamerCache[n][2] = new Rotamer(name, 62.0, 0.0, 180.0, 0.0, 180.0, 0.0, 85.0, 0.0);
                this.aminoAcidRotamerCache[n][3] = new Rotamer(name, 62.0, 0.0, 180.0, 0.0, 180.0, 0.0, 180.0, 0.0);
                this.aminoAcidRotamerCache[n][4] = new Rotamer(name, 62.0, 0.0, 180.0, 0.0, 180.0, 0.0, -85.0, 0.0);
                this.aminoAcidRotamerCache[n][5] = new Rotamer(name, 62.0, 0.0, 180.0, 0.0, -65.0, 0.0, 175.0, 0.0);
                this.aminoAcidRotamerCache[n][6] = new Rotamer(name, 62.0, 0.0, 180.0, 0.0, -65.0, 0.0, -85.0, 0.0);
                this.aminoAcidRotamerCache[n][7] = new Rotamer(name, -177.0, 0.0, 65.0, 0.0, 65.0, 0.0, 85.0, 0.0);
                this.aminoAcidRotamerCache[n][8] = new Rotamer(name, -177.0, 0.0, 65.0, 0.0, 65.0, 0.0, -175.0, 0.0);
                this.aminoAcidRotamerCache[n][9] = new Rotamer(name, -177.0, 0.0, 65.0, 0.0, 180.0, 0.0, 85.0, 0.0);
                this.aminoAcidRotamerCache[n][10] = new Rotamer(name, -177.0, 0.0, 65.0, 0.0, 180.0, 0.0, 180.0, 0.0);
                this.aminoAcidRotamerCache[n][11] = new Rotamer(name, -177.0, 0.0, 180.0, 0.0, 65.0, 0.0, 85.0, 0.0);
                this.aminoAcidRotamerCache[n][12] = new Rotamer(name, -177.0, 0.0, 180.0, 0.0, 65.0, 0.0, -175.0, 0.0);
                this.aminoAcidRotamerCache[n][13] = new Rotamer(name, -177.0, 0.0, 180.0, 0.0, 65.0, 0.0, -105.0, 0.0);
                this.aminoAcidRotamerCache[n][14] = new Rotamer(name, -177.0, 0.0, 180.0, 0.0, 180.0, 0.0, 85.0, 0.0);
                this.aminoAcidRotamerCache[n][15] = new Rotamer(name, -177.0, 0.0, 180.0, 0.0, 180.0, 0.0, 180.0, 0.0);
                this.aminoAcidRotamerCache[n][16] = new Rotamer(name, -177.0, 0.0, 180.0, 0.0, 180.0, 0.0, -85.0, 0.0);
                this.aminoAcidRotamerCache[n][17] = new Rotamer(name, -177.0, 0.0, 180.0, 0.0, -65.0, 0.0, 105.0, 0.0);
                this.aminoAcidRotamerCache[n][18] = new Rotamer(name, -177.0, 0.0, 180.0, 0.0, -65.0, 0.0, 175.0, 0.0);
                this.aminoAcidRotamerCache[n][19] = new Rotamer(name, -177.0, 0.0, 180.0, 0.0, -65.0, 0.0, -85.0, 0.0);
                this.aminoAcidRotamerCache[n][20] = new Rotamer(name, -67.0, 0.0, 180.0, 0.0, 65.0, 0.0, 85.0, 0.0);
                this.aminoAcidRotamerCache[n][21] = new Rotamer(name, -67.0, 0.0, 180.0, 0.0, 65.0, 0.0, -175.0, 0.0);
                this.aminoAcidRotamerCache[n][22] = new Rotamer(name, -67.0, 0.0, 180.0, 0.0, 65.0, 0.0, -105.0, 0.0);
                this.aminoAcidRotamerCache[n][23] = new Rotamer(name, -67.0, 0.0, 180.0, 0.0, 180.0, 0.0, 85.0, 0.0);
                this.aminoAcidRotamerCache[n][24] = new Rotamer(name, -67.0, 0.0, 180.0, 0.0, 180.0, 0.0, 180.0, 0.0);
                this.aminoAcidRotamerCache[n][25] = new Rotamer(name, -67.0, 0.0, 180.0, 0.0, 180.0, 0.0, -85.0, 0.0);
                this.aminoAcidRotamerCache[n][26] = new Rotamer(name, -67.0, 0.0, 180.0, 0.0, -65.0, 0.0, 105.0, 0.0);
                this.aminoAcidRotamerCache[n][27] = new Rotamer(name, -67.0, 0.0, 180.0, 0.0, -65.0, 0.0, 175.0, 0.0);
                this.aminoAcidRotamerCache[n][28] = new Rotamer(name, -67.0, 0.0, -167.0, 0.0, -65.0, 0.0, -85.0, 0.0);
                this.aminoAcidRotamerCache[n][29] = new Rotamer(name, -62.0, 0.0, -68.0, 0.0, 180.0, 0.0, 85.0, 0.0);
                this.aminoAcidRotamerCache[n][30] = new Rotamer(name, -62.0, 0.0, -68.0, 0.0, 180.0, 0.0, 180.0, 0.0);
                this.aminoAcidRotamerCache[n][31] = new Rotamer(name, -62.0, 0.0, -68.0, 0.0, 180.0, 0.0, -85.0, 0.0);
                this.aminoAcidRotamerCache[n][32] = new Rotamer(name, -62.0, 0.0, -68.0, 0.0, -65.0, 0.0, 175.0, 0.0);
                this.aminoAcidRotamerCache[n][33] = new Rotamer(name, -62.0, 0.0, -68.0, 0.0, -65.0, 0.0, -85.0, 0.0);
                break;
            }
        }
        return this.aminoAcidRotamerCache[n];
    }

    private Rotamer[] getRichardsonRNARotamers(NucleicAcidUtils.NucleicAcid3 name) {
        int n = name.ordinal();
        if (this.nucleicAcidRotamerCache[n] != null) {
            return this.nucleicAcidRotamerCache[n];
        }
        switch (name) {
            case GUA: 
            case ADE: 
            case DGU: 
            case DAD: 
            case CYT: 
            case URI: 
            case THY: 
            case DCY: 
            case DTY: {
                this.nucleicAcidRotamerCache[n] = new Rotamer[46];
                this.nucleicAcidRotamerCache[n][0] = new Rotamer(name, 81.0, 4.0, -148.0, 10.0, -71.0, 7.0, -65.0, 8.0, 174.0, 8.0, 54.0, 6.0, 81.0, 3.0);
                this.nucleicAcidRotamerCache[n][1] = new Rotamer(name, 84.0, 5.0, -142.0, 16.0, -68.0, 15.0, -68.0, 16.0, -138.0, 12.0, 58.0, 10.0, 86.0, 7.0);
                this.nucleicAcidRotamerCache[n][2] = new Rotamer(name, 86.0, 4.0, -115.0, 6.0, -92.0, 13.0, -56.0, 8.0, 138.0, 4.0, 62.0, 10.0, 79.0, 5.0);
                this.nucleicAcidRotamerCache[n][3] = new Rotamer(name, 82.0, 5.0, -169.0, 7.0, -95.0, 6.0, -64.0, 9.0, -178.0, 10.0, 51.0, 7.0, 82.0, 5.0);
                this.nucleicAcidRotamerCache[n][4] = new Rotamer(name, 83.0, 4.0, -143.0, 23.0, -138.0, 14.0, -57.0, 9.0, 161.0, 15.0, 49.0, 6.0, 82.0, 3.0);
                this.nucleicAcidRotamerCache[n][5] = new Rotamer(name, 85.0, 4.0, -144.0, 24.0, 173.0, 14.0, -71.0, 12.0, 164.0, 16.0, 46.0, 7.0, 85.0, 6.0);
                this.nucleicAcidRotamerCache[n][6] = new Rotamer(name, 83.0, 2.0, -150.0, 15.0, 121.0, 13.0, -71.0, 12.0, 157.0, 23.0, 49.0, 6.0, 81.0, 3.0);
                this.nucleicAcidRotamerCache[n][7] = new Rotamer(name, 81.0, 3.0, -141.0, 8.0, -69.0, 9.0, 167.0, 8.0, 160.0, 16.0, 51.0, 5.0, 85.0, 3.0);
                this.nucleicAcidRotamerCache[n][8] = new Rotamer(name, 84.0, 4.0, -121.0, 16.0, -103.0, 12.0, 70.0, 10.0, 170.0, 23.0, 53.0, 6.0, 85.0, 3.0);
                this.nucleicAcidRotamerCache[n][9] = new Rotamer(name, 85.0, 4.0, -116.0, 15.0, -156.0, 15.0, 66.0, 19.0, -179.0, 23.0, 55.0, 6.0, 86.0, 4.0);
                this.nucleicAcidRotamerCache[n][10] = new Rotamer(name, 80.0, 4.0, -158.0, 7.0, 63.0, 14.0, 68.0, 12.0, 143.0, 30.0, 50.0, 7.0, 83.0, 2.0);
                this.nucleicAcidRotamerCache[n][11] = new Rotamer(name, 81.0, 3.0, -159.0, 8.0, -79.0, 6.0, -111.0, 9.0, 83.0, 11.0, 168.0, 6.0, 86.0, 4.0);
                this.nucleicAcidRotamerCache[n][12] = new Rotamer(name, 80.0, 3.0, -163.0, 9.0, -69.0, 10.0, 153.0, 12.0, -166.0, 12.0, 179.0, 10.0, 84.0, 3.0);
                this.nucleicAcidRotamerCache[n][13] = new Rotamer(name, 81.0, 2.0, -157.0, 14.0, -66.0, 11.0, 172.0, 11.0, 139.0, 13.0, 176.0, 10.0, 84.0, 3.0);
                this.nucleicAcidRotamerCache[n][14] = new Rotamer(name, 87.0, 7.0, -136.0, 23.0, 80.0, 15.0, 67.0, 9.0, 109.0, 10.0, 176.0, 6.0, 84.0, 4.0);
                this.nucleicAcidRotamerCache[n][15] = new Rotamer(name, 84.0, 4.0, -145.0, 10.0, -71.0, 10.0, -60.0, 9.0, 177.0, 12.0, 58.0, 7.0, 145.0, 7.0);
                this.nucleicAcidRotamerCache[n][16] = new Rotamer(name, 83.0, 4.0, -140.0, 10.0, -71.0, 10.0, -63.0, 8.0, -138.0, 9.0, 54.0, 7.0, 144.0, 8.0);
                this.nucleicAcidRotamerCache[n][17] = new Rotamer(name, 85.0, 3.0, -134.0, 18.0, 168.0, 17.0, -67.0, 15.0, 178.0, 22.0, 49.0, 5.0, 148.0, 3.0);
                this.nucleicAcidRotamerCache[n][18] = new Rotamer(name, 83.0, 3.0, -154.0, 18.0, -82.0, 19.0, -164.0, 14.0, 162.0, 25.0, 51.0, 5.0, 145.0, 5.0);
                this.nucleicAcidRotamerCache[n][19] = new Rotamer(name, 83.0, 3.0, -154.0, 5.0, 53.0, 7.0, 164.0, 5.0, 148.0, 10.0, 50.0, 5.0, 148.0, 4.0);
                this.nucleicAcidRotamerCache[n][20] = new Rotamer(name, 84.0, 3.0, -123.0, 24.0, -140.0, 15.0, 68.0, 12.0, -160.0, 30.0, 54.0, 7.0, 146.0, 6.0);
                this.nucleicAcidRotamerCache[n][21] = new Rotamer(name, 81.0, 3.0, -161.0, 20.0, -71.0, 8.0, 180.0, 17.0, -165.0, 14.0, 178.0, 9.0, 147.0, 5.0);
                this.nucleicAcidRotamerCache[n][22] = new Rotamer(name, 82.0, 8.0, -155.0, 6.0, 69.0, 14.0, 63.0, 9.0, 115.0, 17.0, 176.0, 6.0, 146.0, 4.0);
                this.nucleicAcidRotamerCache[n][23] = new Rotamer(name, 84.0, 4.0, -143.0, 17.0, -73.0, 15.0, -63.0, 7.0, -135.0, 39.0, -66.0, 7.0, 151.0, 13.0);
                this.nucleicAcidRotamerCache[n][24] = new Rotamer(name, 85.0, 4.0, -127.0, 13.0, -112.0, 19.0, 63.0, 13.0, -178.0, 27.0, -64.0, 4.0, 150.0, 7.0);
                this.nucleicAcidRotamerCache[n][25] = new Rotamer(name, 145.0, 8.0, -100.0, 12.0, -71.0, 18.0, -72.0, 13.0, -167.0, 17.0, 53.0, 7.0, 84.0, 5.0);
                this.nucleicAcidRotamerCache[n][26] = new Rotamer(name, 146.0, 7.0, -100.0, 15.0, 170.0, 14.0, -62.0, 19.0, 170.0, 34.0, 51.0, 8.0, 84.0, 5.0);
                this.nucleicAcidRotamerCache[n][27] = new Rotamer(name, 149.0, 7.0, -137.0, 11.0, 139.0, 25.0, -75.0, 11.0, 158.0, 20.0, 48.0, 6.0, 84.0, 4.0);
                this.nucleicAcidRotamerCache[n][28] = new Rotamer(name, 148.0, 3.0, -168.0, 5.0, 146.0, 6.0, -71.0, 7.0, 151.0, 12.0, 42.0, 4.0, 85.0, 3.0);
                this.nucleicAcidRotamerCache[n][29] = new Rotamer(name, 148.0, 8.0, -103.0, 14.0, 165.0, 21.0, -155.0, 14.0, 165.0, 15.0, 49.0, 7.0, 83.0, 4.0);
                this.nucleicAcidRotamerCache[n][30] = new Rotamer(name, 145.0, 7.0, -97.0, 18.0, 80.0, 16.0, -156.0, 29.0, -170.0, 23.0, 58.0, 5.0, 85.0, 7.0);
                this.nucleicAcidRotamerCache[n][31] = new Rotamer(name, 149.0, 6.0, -89.0, 10.0, -119.0, 17.0, 62.0, 10.0, 176.0, 23.0, 54.0, 4.0, 87.0, 3.0);
                this.nucleicAcidRotamerCache[n][32] = new Rotamer(name, 150.0, 6.0, -110.0, 26.0, -172.0, 7.0, 80.0, 20.0, -162.0, 20.0, 61.0, 8.0, 89.0, 4.0);
                this.nucleicAcidRotamerCache[n][33] = new Rotamer(name, 147.0, 6.0, -119.0, 23.0, 89.0, 16.0, 59.0, 14.0, 161.0, 23.0, 52.0, 7.0, 83.0, 4.0);
                this.nucleicAcidRotamerCache[n][34] = new Rotamer(name, 148.0, 4.0, -99.0, 8.0, -70.0, 12.0, -64.0, 10.0, 177.0, 17.0, 176.0, 14.0, 87.0, 4.0);
                this.nucleicAcidRotamerCache[n][35] = new Rotamer(name, 144.0, 7.0, -133.0, 14.0, -156.0, 14.0, 74.0, 12.0, -143.0, 20.0, -166.0, 9.0, 81.0, 3.0);
                this.nucleicAcidRotamerCache[n][36] = new Rotamer(name, 149.0, 2.0, -85.0, 20.0, 100.0, 13.0, 81.0, 11.0, -112.0, 12.0, -178.0, 3.0, 83.0, 2.0);
                this.nucleicAcidRotamerCache[n][37] = new Rotamer(name, 150.0, 6.0, -92.0, 11.0, 85.0, 8.0, 64.0, 5.0, -169.0, 8.0, 177.0, 9.0, 86.0, 5.0);
                this.nucleicAcidRotamerCache[n][38] = new Rotamer(name, 142.0, 8.0, -116.0, 28.0, 66.0, 15.0, 72.0, 8.0, 122.0, 22.0, -178.0, 6.0, 84.0, 3.0);
                this.nucleicAcidRotamerCache[n][39] = new Rotamer(name, 146.0, 8.0, -101.0, 16.0, -69.0, 17.0, -68.0, 12.0, -150.0, 21.0, 54.0, 7.0, 148.0, 7.0);
                this.nucleicAcidRotamerCache[n][40] = new Rotamer(name, 145.0, 7.0, -115.0, 20.0, 163.0, 13.0, -66.0, 6.0, 172.0, 14.0, 46.0, 6.0, 146.0, 6.0);
                this.nucleicAcidRotamerCache[n][41] = new Rotamer(name, 148.0, 4.0, -112.0, 20.0, 112.0, 14.0, -85.0, 17.0, 165.0, 16.0, 57.0, 12.0, 146.0, 6.0);
                this.nucleicAcidRotamerCache[n][42] = new Rotamer(name, 150.0, 10.0, -100.0, 16.0, -146.0, 19.0, 72.0, 13.0, -152.0, 27.0, 57.0, 14.0, 148.0, 4.0);
                this.nucleicAcidRotamerCache[n][43] = new Rotamer(name, 146.0, 7.0, -102.0, 21.0, 90.0, 15.0, 68.0, 12.0, 173.0, 18.0, 56.0, 8.0, 148.0, 4.0);
                this.nucleicAcidRotamerCache[n][44] = new Rotamer(name, 150.0, 2.0, -112.0, 16.0, 170.0, 12.0, -82.0, 13.0, 84.0, 7.0, 176.0, 6.0, 148.0, 2.0);
                this.nucleicAcidRotamerCache[n][45] = new Rotamer(name, 147.0, 6.0, -104.0, 15.0, -64.0, 16.0, -73.0, 4.0, -165.0, 26.0, -66.0, 7.0, 150.0, 3.0);
                break;
            }
        }
        return this.nucleicAcidRotamerCache[n];
    }

    public static enum ProteinLibrary {
        PonderAndRichards(1),
        Richardson(2),
        None(-1);

        private final int oldIntegerConstant;

        private ProteinLibrary(int oldConst) {
            this.oldIntegerConstant = oldConst;
        }

        public static ProteinLibrary getProteinLibrary(String input) throws IllegalArgumentException {
            if (input.matches("^\\d+$")) {
                return ProteinLibrary.int2Library(Integer.parseInt(input));
            }
            return Arrays.stream(ProteinLibrary.values()).filter(pl -> pl.toString().equalsIgnoreCase(input)).findAny().orElseThrow(() -> new IllegalArgumentException(" No protein library found that corresponds to " + input));
        }

        public static ProteinLibrary intToProteinLibrary(int library) throws IllegalArgumentException {
            return ProteinLibrary.int2Library(library);
        }

        private static ProteinLibrary int2Library(int library) throws IllegalArgumentException {
            for (ProteinLibrary lib : ProteinLibrary.values()) {
                if (library != lib.oldIntegerConstant) continue;
                return lib;
            }
            throw new IllegalArgumentException(String.format(" Could not find a protein rotamer library to correspond with %d!", library));
        }
    }

    public static enum NucleicAcidLibrary {
        RICHARDSON;

    }

    public static enum NucleicSugarPucker {
        C2_ENDO("south"),
        C3_ENDO("north"),
        C3_EXO;


        private NucleicSugarPucker() {
        }

        private NucleicSugarPucker(String aName) {
        }

        public static NucleicSugarPucker checkPucker(double delta, boolean isDeoxy) {
            if ((delta = ScalarMath.mod((double)delta, (double)360.0)) <= 115.0 || delta > 295.0) {
                return C3_ENDO;
            }
            return C2_ENDO;
        }
    }

    private static class NonstandardRotLibrary {
        private final String resName;
        private final String[] placeRecords;
        private final List<Rotamer> stdRotamers;

        NonstandardRotLibrary(String resname, String[] placeRecords, Rotamer[] stdRotamers) {
            this.resName = resname.toUpperCase();
            this.placeRecords = Arrays.copyOf(placeRecords, placeRecords.length);
            this.stdRotamers = new ArrayList<Rotamer>(Arrays.asList(stdRotamers));
        }

        Rotamer[] getRotamers() {
            return this.stdRotamers.toArray(new Rotamer[0]);
        }

        int measureNonstdRot(Residue residue, double[] chi, boolean print) {
            for (String placeRec : this.placeRecords) {
                String[] toks = placeRec.split("\\s+");
                if (!toks[0].equalsIgnoreCase("PLACECHI")) continue;
                int chiNum = Integer.parseInt(toks[5]) - 1;
                Atom at1 = (Atom)residue.getAtomNode(toks[1]);
                Atom at2 = (Atom)residue.getAtomNode(toks[2]);
                Atom at3 = (Atom)residue.getAtomNode(toks[3]);
                Atom at4 = (Atom)residue.getAtomNode(toks[4]);
                Torsion tors = at1.getTorsion(at2, at3, at4);
                chi[chiNum] = tors.getValue();
                if (!print) continue;
                logger.info(tors.toString());
            }
            return this.placeRecords.length;
        }

        void applyNonstdRotamer(Residue residue, Rotamer rotamer) {
            if (!residue.getName().equalsIgnoreCase(this.resName)) {
                throw new IllegalArgumentException(String.format(" Residue %s is not of type %s", residue, this.resName));
            }
            for (String record : this.placeRecords) {
                double dtors;
                String[] toks = record.split("\\s+");
                Atom at1 = (Atom)residue.getAtomNode(toks[1]);
                Atom at2 = (Atom)residue.getAtomNode(toks[2]);
                Atom at3 = (Atom)residue.getAtomNode(toks[3]);
                Atom at4 = (Atom)residue.getAtomNode(toks[4]);
                Bond b12 = at1.getBond(at2);
                double dbond = b12.bondType.distance;
                Angle a123 = at1.getAngle(at2, at3);
                double dang = a123.angleType.angle[a123.nh];
                if (toks[0].equalsIgnoreCase("PLACECHI")) {
                    int chiNum = Integer.parseInt(toks[5]);
                    dtors = rotamer.angles[chiNum - 1];
                } else {
                    dtors = Double.parseDouble(toks[5]);
                }
                int chirality = Integer.parseInt(toks[6]);
                BondedUtils.intxyz(at1, at2, dbond, at3, dang, at4, dtors, chirality);
            }
        }
    }

    public static class RotamerGuess {
        private final Residue residue;
        private final Rotamer rotamer;
        private final int rotIndex;
        private final double rmsd;

        RotamerGuess(Residue res, Rotamer rot, int index, double rmsDev) {
            this.residue = res;
            this.rotamer = rot;
            this.rotIndex = index;
            this.rmsd = rmsDev;
        }

        public double getRMSD() {
            return this.rmsd;
        }

        public Residue getResidue() {
            return this.residue;
        }

        public Rotamer getRotamer() {
            return this.rotamer;
        }

        public String toString() {
            return String.format(" Residue %7s is most likely in rotamer %2d (%s), with an RMSD of %9.5f degrees.", this.residue, this.rotIndex, this.rotamer, this.rmsd);
        }
    }
}

