/*
 * Decompiled with CFR 0.152.
 */
package ffx.numerics.optimization;

import ffx.numerics.OptimizationInterface;
import ffx.numerics.optimization.LineSearch;
import ffx.numerics.optimization.OptimizationListener;
import java.util.Arrays;
import java.util.logging.Logger;
import javax.annotation.Nullable;
import org.apache.commons.math3.util.FastMath;

public class LBFGS {
    public static final double DEFAULT_CAPPA = 0.9;
    public static final double DEFAULT_STEPMIN = 1.0E-16;
    public static final double DEFAULT_STEPMAX = 5.0;
    public static final double DEFAULT_SLOPEMAX = 10000.0;
    public static final double DEFAULT_ANGLEMAX = 180.0;
    public static final int DEFAULT_INTMAX = 5;
    private static final Logger logger = Logger.getLogger(LBFGS.class.getName());

    private LBFGS() {
    }

    public static int minimize(int n, int mSave, double[] x, double f, double[] g, double eps, int maxIterations, OptimizationInterface potential, @Nullable OptimizationListener listener) {
        if (n <= 0) {
            logger.severe("Number of variables must be positive.");
            return -1;
        }
        if (mSave < 0) {
            logger.severe("Number of correction vectors must be non-negative.");
            return -1;
        }
        if (maxIterations <= 0) {
            logger.severe("Maximum number of iterations must be positive.");
            return -1;
        }
        if (x == null || x.length < n) {
            logger.severe("Coordinate array is null or too small.");
            return -1;
        }
        if (g == null || g.length < n) {
            logger.severe("Gradient array is null or too small.");
            return -1;
        }
        if (potential == null) {
            logger.severe("Potential interface cannot be null.");
            return -1;
        }
        if (eps <= 0.0) {
            logger.warning("Convergence criterion (eps) should be positive.");
        }
        if (mSave > n) {
            logger.fine(String.format(" Resetting the number of saved L-BFGS vectors to %d.", n));
            mSave = n;
        }
        int iterations = 0;
        int evaluations = 1;
        int nErrors = 0;
        int maxErrors = 2;
        double rms = FastMath.sqrt((double)n);
        double[] scaling = potential.getScaling();
        if (scaling == null) {
            scaling = new double[n];
            Arrays.fill(scaling, 1.0);
        }
        double[][] s = new double[mSave][n];
        double[][] y = new double[mSave][n];
        if (mSave > 0) {
            for (int i = 0; i < n; ++i) {
                s[0][i] = -g[i];
            }
        }
        double grms = 0.0;
        double gnorm = 0.0;
        for (int i = 0; i < n; ++i) {
            double gi = g[i];
            if (Double.isNaN(gi) || Double.isInfinite(gi)) {
                logger.warning(String.format("The gradient of variable %d is %8.3f.", i, gi));
                return 1;
            }
            double gis = gi * scaling[i];
            gnorm += gi * gi;
            grms += gis * gis;
        }
        gnorm = FastMath.sqrt((double)gnorm);
        grms = FastMath.sqrt((double)grms) / rms;
        if (listener != null) {
            if (!listener.optimizationUpdate(iterations, mSave, evaluations, grms, 0.0, f, 0.0, 0.0, null)) {
                return 1;
            }
        } else {
            LBFGS.log(iterations, evaluations, grms, 0.0, f, 0.0, 0.0, null);
        }
        if (grms <= eps) {
            return 0;
        }
        double[] prevX = new double[n];
        double[] prevG = new double[n];
        double[] r = new double[n];
        double[] p = new double[n];
        double[] h0 = new double[n];
        double[] q = new double[n];
        double[] alpha = new double[mSave];
        double[] rho = new double[mSave];
        double gamma = 1.0;
        LineSearch lineSearch = new LineSearch(n);
        LineSearch.LineSearchResult[] info = new LineSearch.LineSearchResult[]{LineSearch.LineSearchResult.Success};
        int[] nFunctionEvals = new int[]{0};
        double[] angle = new double[]{0.0};
        double df = 2.5 * gnorm;
        int m = -1;
        do {
            int i;
            if (++iterations > maxIterations) {
                logger.info(String.format(" Maximum number of iterations reached: %d.", maxIterations));
                return 1;
            }
            if (mSave > 0) {
                int j;
                int muse = FastMath.min((int)(iterations - 1), (int)mSave);
                if (++m > mSave - 1) {
                    m = 0;
                }
                Arrays.fill(h0, gamma);
                System.arraycopy(g, 0, q, 0, n);
                int k = m;
                for (j = 0; j < muse; ++j) {
                    if (--k < 0) {
                        k = mSave - 1;
                    }
                    alpha[k] = LBFGS.v1DotV2(n, s[k], 0, 1, q, 0, 1);
                    int n2 = k;
                    alpha[n2] = alpha[n2] * rho[k];
                    LBFGS.aV1PlusV2(n, -alpha[k], y[k], 0, 1, q, 0, 1);
                }
                for (i = 0; i < n; ++i) {
                    r[i] = h0[i] * q[i];
                }
                for (j = 0; j < muse; ++j) {
                    double beta = LBFGS.v1DotV2(n, r, 0, 1, y[k], 0, 1);
                    LBFGS.aV1PlusV2(n, alpha[k] - (beta *= rho[k]), s[k], 0, 1, r, 0, 1);
                    if (++k <= mSave - 1) continue;
                    k = 0;
                }
                for (i = 0; i < n; ++i) {
                    p[i] = -r[i];
                }
            } else {
                for (int i2 = 0; i2 < n; ++i2) {
                    p[i2] = -g[i2];
                }
            }
            System.arraycopy(x, 0, prevX, 0, n);
            System.arraycopy(g, 0, prevG, 0, n);
            nFunctionEvals[0] = 0;
            double prevF = f;
            f = lineSearch.search(n, x, f, g, p, angle, df, info, nFunctionEvals, potential);
            evaluations += nFunctionEvals[0];
            for (i = 0; i < n; ++i) {
                if (!Double.isNaN(g[i]) && !Double.isInfinite(g[i])) continue;
                logger.warning(String.format("The gradient of variable %d is %8.3f after line search. Terminating optimization.", i, g[i]));
                return -1;
            }
            if (mSave > 0) {
                for (i = 0; i < n; ++i) {
                    s[m][i] = x[i] - prevX[i];
                    y[m][i] = g[i] - prevG[i];
                }
                double ys = LBFGS.v1DotV2(n, y[m], 0, 1, s[m], 0, 1);
                double yy = LBFGS.v1DotV2(n, y[m], 0, 1, y[m], 0, 1);
                gamma = FastMath.abs((double)(ys / yy));
                rho[m] = 1.0 / ys;
            }
            df = prevF - f;
            double xrms = 0.0;
            grms = 0.0;
            for (int i3 = 0; i3 < n; ++i3) {
                double dx = (x[i3] - prevX[i3]) / scaling[i3];
                xrms += dx * dx;
                double gx = g[i3] * scaling[i3];
                grms += gx * gx;
            }
            xrms = FastMath.sqrt((double)xrms) / rms;
            grms = FastMath.sqrt((double)grms) / rms;
            boolean done = false;
            if (info[0] == LineSearch.LineSearchResult.BadIntpln || info[0] == LineSearch.LineSearchResult.IntplnErr) {
                if (++nErrors >= maxErrors) {
                    done = true;
                }
            } else {
                nErrors = 0;
            }
            if (listener != null) {
                if (!listener.optimizationUpdate(iterations, mSave, evaluations, grms, xrms, f, df, angle[0], info[0])) {
                    return 1;
                }
            } else {
                LBFGS.log(iterations, evaluations, grms, xrms, f, df, angle[0], info[0]);
            }
            if (!done) continue;
            return -1;
        } while (!(grms <= eps));
        return 0;
    }

    public static int minimize(int n, int mSave, double[] x, double f, double[] g, double eps, OptimizationInterface potential, OptimizationListener listener) {
        return LBFGS.minimize(n, mSave, x, f, g, eps, 0x7FFFFFFE, potential, listener);
    }

    static void aV1PlusV2(int n, double a, double[] v1, int v1Start, int v1Step, double[] v2, int v2Start, int v2Step) {
        if (n <= 0 || a == 0.0) {
            return;
        }
        if (v1Step == 1 && v2Step == 1 && v1Start == 0 && v2Start == 0) {
            for (int i = 0; i < n; ++i) {
                v2[i] = Math.fma(a, v1[i], v2[i]);
            }
            return;
        }
        int stop = v1Start + v1Step * n;
        int i = v1Start;
        int j = v2Start;
        while (i != stop) {
            v2[j] = Math.fma(a, v1[i], v2[j]);
            i += v1Step;
            j += v2Step;
        }
    }

    private static void log(int iter, int nfun, double grms, double xrms, double f, double df, double angle, LineSearch.LineSearchResult info) {
        if (iter == 0) {
            logger.info("\n Limited Memory BFGS Quasi-Newton Optimization: \n");
            logger.info(" QN Iter    F Value      G RMS     F Move    X Move    Angle  FG Call  Comment\n");
        }
        if (info == null) {
            logger.info(String.format("%6d%13.4f%11.4f%11.4f%10.4f%9.2f%7d", iter, f, grms, df, xrms, angle, nfun));
        } else {
            logger.info(String.format("%6d%13.4f%11.4f%11.4f%10.4f%9.2f%7d   %8s", new Object[]{iter, f, grms, df, xrms, angle, nfun, info}));
        }
    }

    static double v1DotV2(int n, double[] v1, int v1Start, int v1Step, double[] v2, int v2Start, int v2Step) {
        if (n <= 0) {
            return 0.0;
        }
        if (v1Step == 1 && v2Step == 1 && v1Start == 0 && v2Start == 0) {
            double sum = 0.0;
            for (int i = 0; i < n; ++i) {
                sum = Math.fma(v1[i], v2[i], sum);
            }
            return sum;
        }
        double sum = 0.0;
        int stop = v1Start + v1Step * n;
        int i = v1Start;
        int j = v2Start;
        while (i != stop) {
            sum = Math.fma(v1[i], v2[j], sum);
            i += v1Step;
            j += v2Step;
        }
        return sum;
    }
}

