/*
 * Decompiled with CFR 0.152.
 */
package ffx.algorithms.misc;

import ffx.algorithms.AlgorithmListener;
import ffx.numerics.Potential;
import ffx.potential.MolecularAssembly;
import ffx.potential.bonded.AminoAcidUtils;
import ffx.potential.bonded.Atom;
import ffx.potential.bonded.NucleicAcidUtils;
import ffx.potential.bonded.Residue;
import ffx.potential.bonded.Rotamer;
import ffx.potential.bonded.RotamerLibrary;
import ffx.potential.parsers.PDBFilter;
import java.io.BufferedWriter;
import java.io.File;
import java.io.FileWriter;
import java.io.IOException;
import java.util.Arrays;
import java.util.logging.Level;
import java.util.logging.Logger;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

public class GenerateRotamers {
    private static final Logger logger = Logger.getLogger(GenerateRotamers.class.getName());
    private static final Pattern atRangePatt = Pattern.compile("(\\d+)-(\\d+)");
    private final Residue residue;
    private final int nChi;
    private final double[] currentChi;
    private final File outFile;
    private final MolecularAssembly molecularAssembly;
    private final AlgorithmListener listener;
    private final RotamerLibrary library;
    private final Potential potential;
    private double[] x;
    private AminoAcidUtils.AminoAcid3 baselineAAres;
    private Rotamer[] baselineRotamers;
    private double incr = 10.0;
    private boolean aroundLibrary = false;
    private double width = 180.0;
    private int startDepth = 0;
    private int endDepth;
    private boolean print = false;
    private int nEval = 0;
    private boolean writeVideo = false;
    private File videoFile;
    private PDBFilter videoFilter;

    public GenerateRotamers(MolecularAssembly molecularAssembly, Potential potential, Residue residue, File file, int nChi, AlgorithmListener listener) {
        this(molecularAssembly, potential, residue, file, nChi, listener, RotamerLibrary.getDefaultLibrary());
    }

    public GenerateRotamers(MolecularAssembly molecularAssembly, Potential potential, Residue residue, File file, int nChi, AlgorithmListener listener, RotamerLibrary library) {
        this.residue = residue;
        this.nChi = nChi;
        this.endDepth = nChi - 1;
        this.potential = potential;
        this.molecularAssembly = molecularAssembly;
        this.listener = listener;
        this.currentChi = new double[nChi];
        this.library = library;
        Arrays.fill(this.currentChi, 0.0);
        File outputFile = file;
        if (outputFile.exists()) {
            String outName = outputFile.getName();
            for (int i = 1; i < 1000 && (outputFile = new File(String.format("%s_%d", outName, i))).exists(); ++i) {
            }
            if (outputFile.exists()) {
                logger.severe(String.format(" Could not version file %s", outName));
            }
        }
        this.outFile = outputFile;
        this.baselineAAres = residue.getAminoAcid3();
        this.baselineRotamers = library.getRotamers(this.baselineAAres);
    }

    public void applyAndSaveTorsions(String[] torSets) {
        for (String torSet : torSets) {
            String[] torsions = torSet.split(",");
            double[] values = new double[this.nChi * 2];
            Arrays.fill(values, 0.0);
            for (int i = 0; i < Math.min(torsions.length, this.nChi); ++i) {
                double chival;
                this.currentChi[i] = chival = Double.parseDouble(torsions[i]);
                values[2 * i] = chival;
            }
            Rotamer newRot = this.generateRotamer(values);
            RotamerLibrary.applyRotamer((Residue)this.residue, (Rotamer)newRot);
            this.writeSnapshot();
        }
    }

    public void setBaselineAARes(AminoAcidUtils.AminoAcid3 aa3) {
        this.baselineAAres = aa3;
        if (aa3 == AminoAcidUtils.AminoAcid3.UNK) {
            boolean orig = this.library.getUsingOrigCoordsRotamer();
            this.library.setUseOrigCoordsRotamer(false);
            this.baselineRotamers = this.residue.setRotamers(this.library);
            this.library.setUseOrigCoordsRotamer(orig);
        } else {
            this.baselineRotamers = this.library.getRotamers(aa3);
        }
        this.aroundLibrary = true;
    }

    public void setDepth(int start, int end) {
        this.startDepth = start;
        this.endDepth = end < 1 || end > this.nChi ? this.nChi - 1 : end;
    }

    public void setElectrostatics(String electrostatics) {
        if (electrostatics != null) {
            String[] tokens = electrostatics.split(",");
            Atom[] atoms = this.molecularAssembly.getAtomArray();
            for (String tok : tokens) {
                Matcher m = atRangePatt.matcher(tok);
                if (m.matches()) {
                    int begin = Integer.parseInt(m.group(1));
                    int end = Integer.parseInt(m.group(2));
                    logger.info(String.format(" Inactivating electrostatics for atoms %d-%d", begin, end));
                    for (int i = begin; i <= end; ++i) {
                        Atom ai = atoms[i - 1];
                        ai.setElectrostatics(false);
                        ai.print(Level.FINE);
                    }
                    continue;
                }
                logger.info(String.format(" Discarding electrostatics input %s", tok));
            }
        }
    }

    public void setInactiveAtoms(String iatoms) {
        if (iatoms != null) {
            String[] tokens = iatoms.split(",");
            Atom[] atoms = this.molecularAssembly.getAtomArray();
            for (String tok : tokens) {
                Matcher m = atRangePatt.matcher(tok);
                if (m.matches()) {
                    int begin = Integer.parseInt(m.group(1));
                    int end = Integer.parseInt(m.group(2));
                    logger.info(String.format(" Inactivating atoms %d-%d", begin, end));
                    for (int i = begin; i <= end; ++i) {
                        Atom ai = atoms[i - 1];
                        ai.setUse(false);
                        ai.print(Level.FINE);
                    }
                    continue;
                }
                logger.info(String.format(" Discarding inactive atoms input %s", tok));
            }
        }
    }

    public void setIncrement(double incr) {
        this.incr = incr;
    }

    public void setPrint(boolean print) {
        this.print = print;
    }

    public void setSearchWidth(double width) {
        this.width = width;
    }

    public void setVideo(String videoFile) {
        if (videoFile != null) {
            File vidFile = new File(videoFile);
            if (vidFile.exists()) {
                for (int i = 0; i < 1000; ++i) {
                    vidFile = new File(String.format("%s_%d", videoFile, i));
                    if (vidFile.exists()) continue;
                    this.videoFile = vidFile;
                    this.writeVideo = true;
                    this.videoFilter = new PDBFilter(this.videoFile, this.molecularAssembly, this.molecularAssembly.getForceField(), null);
                    this.videoFilter.setLogWrites(false);
                    break;
                }
                if (vidFile.exists()) {
                    logger.warning(String.format(" Could not version video file %s", videoFile));
                }
            } else {
                this.videoFile = vidFile;
                this.writeVideo = true;
                this.videoFilter = new PDBFilter(this.videoFile, this.molecularAssembly, this.molecularAssembly.getForceField(), null);
                this.videoFilter.setLogWrites(false);
            }
        } else {
            this.writeVideo = false;
        }
    }

    public void tryRotamers() {
        try (BufferedWriter bw = new BufferedWriter(new FileWriter(this.outFile));){
            if (this.aroundLibrary) {
                for (Rotamer rotamer : this.baselineRotamers) {
                    this.applyLibRotamer(rotamer);
                    this.turnChi(this.startDepth, bw);
                }
            } else {
                RotamerLibrary.measureRotamer((Residue)this.residue, (double[])this.currentChi, (boolean)false);
                this.turnChi(this.startDepth, bw);
            }
        }
        catch (IOException ex) {
            logger.warning(String.format(" IO exception in rotamer generation: %s", ex));
        }
    }

    private void applyLibRotamer(Rotamer rotamer) {
        double[] rotValues = new double[this.nChi * 2];
        Arrays.fill(rotValues, 0.0);
        Arrays.fill(this.currentChi, 0.0);
        int fillTo = Math.min(this.startDepth, rotamer.length - 1);
        for (int i = 0; i < fillTo; ++i) {
            int ii = 2 * i;
            rotValues[ii] = rotamer.angles[i];
            this.currentChi[i] = rotamer.angles[i];
            rotValues[ii + 1] = rotamer.sigmas[i];
        }
        Rotamer newRot = this.generateRotamer(rotValues);
        RotamerLibrary.applyRotamer((Residue)this.residue, (Rotamer)newRot);
    }

    private Rotamer generateRotamer(double[] values) {
        switch (this.residue.getResidueType()) {
            case AA: {
                AminoAcidUtils.AminoAcid3 aa3 = this.residue.getAminoAcid3();
                return new Rotamer(aa3, values);
            }
            case NA: {
                NucleicAcidUtils.NucleicAcid3 na3 = this.residue.getNucleicAcid3();
                return new Rotamer(na3, values);
            }
        }
        return new Rotamer(values);
    }

    private void turnChi(int depth, BufferedWriter bw) throws IOException {
        double chi = this.currentChi[depth];
        for (double i = chi - this.width; i <= chi + this.width; i += this.incr) {
            this.currentChi[depth] = i;
            if (depth == this.endDepth) {
                this.evaluateChi(bw);
                if (this.listener != null) {
                    this.listener.algorithmUpdate(this.molecularAssembly);
                }
                if (!this.writeVideo) continue;
                this.writeSnapshot();
                continue;
            }
            this.turnChi(depth + 1, bw);
        }
        this.currentChi[depth] = chi;
    }

    private void writeSnapshot() {
        try (BufferedWriter bw = new BufferedWriter(new FileWriter(this.videoFile, true));){
            bw.write(String.format("MODEL %d", this.nEval));
            bw.newLine();
            bw.write(String.format("REMARK 301 TORSIONS %s", this.formatChi()));
            bw.newLine();
            bw.flush();
            this.videoFilter.writeFile(this.videoFile, true);
            bw.write("ENDMDL");
            bw.newLine();
            bw.flush();
        }
        catch (IOException ex) {
            logger.warning(String.format(" Exception writing to video file %s: %s", this.videoFile.getName(), ex));
        }
    }

    private void evaluateChi(BufferedWriter bw) throws IOException {
        try {
            this.applyChi();
            double e = this.currentEnergy();
            String result = String.format("%s,%12f", this.formatChi(), e);
            ++this.nEval;
            if (this.print) {
                logger.info(String.format(" Evaluation %10d %s, energy %10.5f kcal/mol", this.nEval, this.formatChi(), e));
            }
            bw.write(result);
            bw.newLine();
            if (this.nEval % 1000 == 0) {
                logger.info(String.format(" %12.7e states evaluated", this.nEval));
            }
        }
        catch (ArithmeticException ex) {
            logger.info(String.format(" Force field exception at chi %s", this.formatChi()));
        }
    }

    private void applyChi() {
        double[] rotValues = new double[this.nChi * 2];
        for (int i = 0; i < this.nChi; ++i) {
            int ii = 2 * i;
            rotValues[ii] = this.currentChi[i];
            rotValues[ii + 1] = 0.0;
        }
        RotamerLibrary.applyRotamer((Residue)this.residue, (Rotamer)this.generateRotamer(rotValues));
    }

    private String formatChi() {
        StringBuilder sb = new StringBuilder(String.format("%8f", this.currentChi[0]));
        for (int i = 1; i < this.nChi; ++i) {
            sb.append(String.format(",%8f", this.currentChi[i]));
        }
        return sb.toString();
    }

    private double currentEnergy() throws ArithmeticException {
        if (this.x == null) {
            int nVar = this.potential.getNumberOfVariables();
            this.x = new double[nVar * 3];
        }
        this.potential.getCoordinates(this.x);
        return this.potential.energy(this.x);
    }
}

