/*
 * Decompiled with CFR 0.152.
 */
package ffx.algorithms.optimize.manybody;

import ffx.algorithms.AlgorithmListener;
import ffx.algorithms.optimize.RotamerOptimization;
import ffx.algorithms.optimize.manybody.DistanceMatrix;
import ffx.algorithms.optimize.manybody.EliminatedRotamers;
import ffx.numerics.Potential;
import ffx.potential.MolecularAssembly;
import ffx.potential.bonded.Atom;
import ffx.potential.bonded.Residue;
import ffx.potential.bonded.Rotamer;
import ffx.potential.bonded.RotamerLibrary;
import ffx.potential.openmm.OpenMMEnergy;
import ffx.potential.utils.EnergyException;
import java.io.File;
import java.io.IOException;
import java.nio.charset.StandardCharsets;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.logging.Level;
import java.util.logging.Logger;
import org.apache.commons.configuration2.CompositeConfiguration;

public class EnergyExpansion {
    private static final Logger logger = Logger.getLogger(EnergyExpansion.class.getName());
    private static final double DEFAULT_OMM_RECALCULATE_THRESHOLD = -200.0;
    private static final double DEFAULT_SINGULARITY_THRESHOLD = -1000.0;
    private final Map<Integer, Integer[]> selfEnergyMap = new HashMap<Integer, Integer[]>();
    private final Map<Integer, Integer[]> twoBodyEnergyMap = new HashMap<Integer, Integer[]>();
    private final Map<Integer, Integer[]> threeBodyEnergyMap = new HashMap<Integer, Integer[]>();
    private final Map<Integer, Integer[]> fourBodyEnergyMap = new HashMap<Integer, Integer[]>();
    private final boolean master;
    private final double ommRecalculateThreshold;
    private final double singularityThreshold;
    private final boolean potentialIsOpenMM;
    private final RotamerOptimization rO;
    private final DistanceMatrix dM;
    private final EliminatedRotamers eR;
    private final MolecularAssembly molecularAssembly;
    private final AlgorithmListener algorithmListener;
    private final List<Residue> allResiduesList;
    private final int[][] resNeighbors;
    private final boolean threeBodyTerm;
    private final boolean decomposeOriginal;
    private final boolean usingBoxOptimization;
    private final boolean verbose;
    private final boolean pruneClashes;
    private final boolean prunePairClashes;
    private final int max4BodyCount;
    private double backboneEnergy;
    private double[][] selfEnergy;
    private double[][][][] twoBodyEnergy;
    private double[][][][][][] threeBodyEnergy;

    public EnergyExpansion(RotamerOptimization rO, DistanceMatrix dM, EliminatedRotamers eR, MolecularAssembly molecularAssembly, Potential potential, AlgorithmListener algorithmListener, List<Residue> allResiduesList, int[][] resNeighbors, boolean threeBodyTerm, boolean decomposeOriginal, boolean usingBoxOptimization, boolean verbose, boolean pruneClashes, boolean prunePairClashes, boolean master) {
        this.rO = rO;
        this.dM = dM;
        this.eR = eR;
        this.molecularAssembly = molecularAssembly;
        this.algorithmListener = algorithmListener;
        this.allResiduesList = allResiduesList;
        this.resNeighbors = resNeighbors;
        this.threeBodyTerm = threeBodyTerm;
        this.decomposeOriginal = decomposeOriginal;
        this.usingBoxOptimization = usingBoxOptimization;
        this.verbose = verbose;
        this.pruneClashes = pruneClashes;
        this.prunePairClashes = prunePairClashes;
        this.master = master;
        CompositeConfiguration properties = molecularAssembly.getProperties();
        this.max4BodyCount = properties.getInt("ro-max4BodyCount", Integer.MAX_VALUE);
        if (this.max4BodyCount != Integer.MAX_VALUE) {
            logger.info(String.format(" Max 4Body Count: %d", this.max4BodyCount));
        }
        this.singularityThreshold = properties.getDouble("ro-singularityThreshold", -1000.0);
        this.potentialIsOpenMM = potential instanceof OpenMMEnergy;
        this.ommRecalculateThreshold = this.potentialIsOpenMM ? properties.getDouble("ro-ommRecalculateThreshold", -200.0) : -1.0E200;
    }

    private static void turnOffAtoms(Residue residue) {
        switch (residue.getResidueType()) {
            case NA: 
            case AA: {
                List atomList = residue.getVariableAtoms();
                for (Atom atom : atomList) {
                    atom.setUse(false);
                }
                break;
            }
            default: {
                List atomList = residue.getAtomList();
                for (Atom atom : atomList) {
                    atom.setUse(false);
                }
            }
        }
    }

    private static void turnOnAtoms(Residue residue) {
        switch (residue.getResidueType()) {
            case NA: 
            case AA: {
                List atomList = residue.getVariableAtoms();
                for (Atom atom : atomList) {
                    atom.setUse(true);
                }
                break;
            }
            default: {
                List atomList = residue.getAtomList();
                for (Atom atom : atomList) {
                    atom.setUse(true);
                }
            }
        }
    }

    public HashMap<String, Integer> allocate2BodyJobMap(Residue[] residues, int nResidues, boolean reverseMap) {
        this.twoBodyEnergyMap.clear();
        HashMap<String, Integer> reverseJobMapPairs = new HashMap<String, Integer>();
        int pairJobIndex = 0;
        this.twoBodyEnergy = new double[nResidues][][][];
        for (int i = 0; i < nResidues; ++i) {
            Residue resi = residues[i];
            int indexI = this.allResiduesList.indexOf(resi);
            Rotamer[] roti = resi.getRotamers();
            int[] nI = this.resNeighbors[i];
            int lenNI = nI.length;
            this.twoBodyEnergy[i] = new double[roti.length][lenNI][];
            for (int ri = 0; ri < roti.length; ++ri) {
                if (this.eR.check(i, ri)) continue;
                for (int indJ = 0; indJ < lenNI; ++indJ) {
                    int j = nI[indJ];
                    if (!this.rO.checkNeighboringPair(i, j)) continue;
                    Residue resj = residues[j];
                    int indexJ = this.allResiduesList.indexOf(resj);
                    Rotamer[] rotj = resj.getRotamers();
                    this.twoBodyEnergy[i][ri][indJ] = new double[rotj.length];
                    for (int rj = 0; rj < rotj.length; ++rj) {
                        if (this.eR.checkToJ(i, ri, j, rj) || this.dM.checkPairDistThreshold(indexI, ri, indexJ, rj)) continue;
                        Integer[] pairJob = new Integer[]{i, ri, j, rj};
                        if (this.decomposeOriginal && (ri != 0 || rj != 0)) continue;
                        this.twoBodyEnergyMap.put(pairJobIndex, pairJob);
                        if (reverseMap) {
                            String revKey = String.format("%d %d %d %d", i, ri, j, rj);
                            reverseJobMapPairs.put(revKey, pairJobIndex);
                        }
                        ++pairJobIndex;
                    }
                }
            }
        }
        return reverseJobMapPairs;
    }

    public HashMap<String, Integer> allocate3BodyJobMap(Residue[] residues, int nResidues, boolean reverseMap) {
        HashMap<String, Integer> reverseJobMapTrimers = new HashMap<String, Integer>();
        this.threeBodyEnergyMap.clear();
        this.threeBodyEnergy = new double[nResidues][][][][][];
        int trimerJobIndex = 0;
        for (int i = 0; i < nResidues; ++i) {
            Residue resi = residues[i];
            int indexI = this.allResiduesList.indexOf(resi);
            Rotamer[] roti = resi.getRotamers();
            int lenri = roti.length;
            int[] nI = this.resNeighbors[i];
            int lenNI = nI.length;
            this.threeBodyEnergy[i] = new double[lenri][lenNI][][][];
            for (int ri = 0; ri < lenri; ++ri) {
                if (this.eR.check(i, ri)) continue;
                for (int indJ = 0; indJ < lenNI; ++indJ) {
                    int j = nI[indJ];
                    Residue resj = residues[j];
                    int indexJ = this.allResiduesList.indexOf(resj);
                    Rotamer[] rotj = resj.getRotamers();
                    int lenrj = rotj.length;
                    int[] nJ = this.resNeighbors[j];
                    int lenNJ = nJ.length;
                    this.threeBodyEnergy[i][ri][indJ] = new double[lenrj][lenNJ][];
                    for (int rj = 0; rj < lenrj; ++rj) {
                        if (this.eR.checkToJ(i, ri, j, rj)) continue;
                        for (int indK = 0; indK < lenNJ; ++indK) {
                            int k = nJ[indK];
                            Residue resk = residues[k];
                            int indexK = this.allResiduesList.indexOf(resk);
                            Rotamer[] rotk = resk.getRotamers();
                            int lenrk = rotk.length;
                            this.threeBodyEnergy[i][ri][indJ][rj][indK] = new double[lenrk];
                            for (int rk = 0; rk < lenrk; ++rk) {
                                if (this.eR.checkToK(i, ri, j, rj, k, rk) || this.dM.checkTriDistThreshold(indexI, ri, indexJ, rj, indexK, rk)) continue;
                                Integer[] trimerJob = new Integer[]{i, ri, j, rj, k, rk};
                                if (this.decomposeOriginal && (ri != 0 || rj != 0 || rk != 0)) continue;
                                this.threeBodyEnergyMap.put(trimerJobIndex, trimerJob);
                                if (reverseMap) {
                                    String revKey = String.format("%d %d %d %d %d %d", i, ri, j, rj, k, rk);
                                    reverseJobMapTrimers.put(revKey, trimerJobIndex);
                                }
                                ++trimerJobIndex;
                            }
                        }
                    }
                }
            }
        }
        return reverseJobMapTrimers;
    }

    public void allocate4BodyJobMap(Residue[] residues, int nResidues) {
        logger.info(" Creating 4-Body jobs...");
        this.fourBodyEnergyMap.clear();
        boolean maxedOut = false;
        int quadJobIndex = 0;
        for (int i = 0; i < nResidues; ++i) {
            Residue resi = residues[i];
            Rotamer[] roti = resi.getRotamers();
            for (int ri = 0; ri < roti.length; ++ri) {
                if (this.eR.check(i, ri)) continue;
                for (int j = i + 1; j < nResidues; ++j) {
                    Residue resj = residues[j];
                    Rotamer[] rotj = resj.getRotamers();
                    for (int rj = 0; rj < rotj.length; ++rj) {
                        if (this.eR.checkToJ(i, ri, j, rj)) continue;
                        for (int k = j + 1; k < nResidues; ++k) {
                            Residue resk = residues[k];
                            Rotamer[] rotk = resk.getRotamers();
                            for (int rk = 0; rk < rotk.length; ++rk) {
                                if (this.eR.checkToK(i, ri, j, rj, k, rk)) continue;
                                for (int l = k + 1; l < nResidues; ++l) {
                                    Residue resl = residues[l];
                                    Rotamer[] rotl = resl.getRotamers();
                                    for (int rl = 0; rl < rotl.length; ++rl) {
                                        if (this.eR.checkToL(i, ri, j, rj, k, rk, l, rl)) continue;
                                        Integer[] quadJob = new Integer[]{i, ri, j, rj, k, rk, l, rl};
                                        if (this.decomposeOriginal && (ri != 0 || rj != 0 || rk != 0 || rl != 0)) continue;
                                        this.fourBodyEnergyMap.put(quadJobIndex++, quadJob);
                                        if (this.fourBodyEnergyMap.size() <= this.max4BodyCount) continue;
                                        maxedOut = true;
                                        break;
                                    }
                                    if (maxedOut) break;
                                }
                                if (maxedOut) break;
                            }
                            if (maxedOut) break;
                        }
                        if (maxedOut) break;
                    }
                    if (maxedOut) break;
                }
                if (maxedOut) break;
            }
            if (maxedOut) break;
        }
    }

    public HashMap<String, Integer> allocateSelfJobMap(Residue[] residues, int nResidues, boolean reverseMap) {
        this.selfEnergyMap.clear();
        HashMap<String, Integer> reverseJobMapSingles = new HashMap<String, Integer>();
        int singleJobIndex = 0;
        this.selfEnergy = new double[nResidues][];
        for (int i = 0; i < nResidues; ++i) {
            Residue resi = residues[i];
            Rotamer[] roti = resi.getRotamers();
            this.selfEnergy[i] = new double[roti.length];
            for (int ri = 0; ri < roti.length; ++ri) {
                if (this.eR.check(i, ri)) continue;
                Integer[] selfJob = new Integer[]{i, ri};
                if (this.decomposeOriginal && ri != 0) continue;
                this.selfEnergyMap.put(singleJobIndex, selfJob);
                if (reverseMap) {
                    String revKey = String.format("%d %d", i, ri);
                    reverseJobMapSingles.put(revKey, singleJobIndex);
                }
                ++singleJobIndex;
            }
        }
        return reverseJobMapSingles;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public double compute2BodyEnergy(Residue[] residues, int i, int ri, int j, int rj) {
        double energy;
        this.rO.turnOffAllResidues(residues);
        this.turnOnResidue(residues[i], ri);
        this.turnOnResidue(residues[j], rj);
        try {
            if (this.algorithmListener != null) {
                this.algorithmListener.algorithmUpdate(this.molecularAssembly);
            }
            Rotamer[] rot_i = residues[i].getRotamers();
            Rotamer[] rot_j = residues[j].getRotamers();
            double subtract = -this.backboneEnergy - this.getSelf(i, ri, rot_i[ri], true) - this.getSelf(j, rj, rot_j[rj], true);
            energy = this.rO.currentEnergy(residues) + subtract;
            if (logger.isLoggable(Level.FINE)) {
                logger.fine(String.format(" %s Pair-Energy %16.8f = FF %16.8f - BB %16.8f + Self Ri %16.8f + Self Rj %16.8f ", rot_i[ri].getName() + "-" + rot_j[rj].getName(), energy, energy + this.backboneEnergy, this.backboneEnergy, this.getSelf(i, ri, rot_i[ri], true), this.getSelf(j, rj, rot_j[rj], true)));
            }
            if (this.potentialIsOpenMM && energy < this.ommRecalculateThreshold) {
                logger.warning(String.format(" Experimental: re-computing pair energy %s-%d %s-%d using Force Field X", residues[i], ri, residues[j], rj));
                energy = this.rO.currentFFXPE() + subtract;
            }
            if (energy < this.singularityThreshold) {
                String message = String.format(" Rejecting pair energy for %s-%d %s-%d is %10.5g << %10f, likely in error.", residues[i], ri, residues[j], rj, energy, this.singularityThreshold);
                logger.warning(message);
                throw new EnergyException(message, false, energy);
            }
        }
        finally {
            this.turnOffResidue(residues[i]);
            this.turnOffResidue(residues[j]);
        }
        return energy;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public double compute3BodyEnergy(Residue[] residues, int i, int ri, int j, int rj, int k, int rk) {
        double energy;
        this.turnOffAllResidues(residues);
        this.turnOnResidue(residues[i], ri);
        this.turnOnResidue(residues[j], rj);
        this.turnOnResidue(residues[k], rk);
        Rotamer[] rot_i = residues[i].getRotamers();
        Rotamer[] rot_j = residues[j].getRotamers();
        Rotamer[] rot_k = residues[k].getRotamers();
        try {
            if (this.algorithmListener != null) {
                this.algorithmListener.algorithmUpdate(this.molecularAssembly);
            }
            double subtract = -this.backboneEnergy - this.getSelf(i, ri, rot_i[ri], true) - this.getSelf(j, rj, rot_j[rj], true) - this.getSelf(k, rk, rot_k[rk], true) - this.get2Body(i, ri, j, rj) - this.get2Body(i, ri, k, rk) - this.get2Body(j, rj, k, rk);
            energy = this.rO.currentEnergy(residues) + subtract;
            if (this.potentialIsOpenMM && energy < this.ommRecalculateThreshold) {
                logger.warning(String.format(" Experimental: re-computing triple energy %s-%d %s-%d %s-%d using Force Field X", residues[i], ri, residues[j], rj, residues[k], rk));
                energy = this.rO.currentFFXPE() + subtract;
            }
            if (energy < this.singularityThreshold) {
                String message = String.format(" Rejecting triple energy for %s-%d %s-%d %s-%d is %10.5g << %10f, likely in error.", residues[i], ri, residues[j], rj, residues[k], rk, energy, this.singularityThreshold);
                logger.warning(message);
                throw new EnergyException(message);
            }
        }
        finally {
            this.turnOffResidue(residues[i]);
            this.turnOffResidue(residues[j]);
            this.turnOffResidue(residues[k]);
        }
        return energy;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public double compute4BodyEnergy(Residue[] residues, int i, int ri, int j, int rj, int k, int rk, int l, int rl) {
        double energy;
        this.turnOffAllResidues(residues);
        this.turnOnResidue(residues[i], ri);
        this.turnOnResidue(residues[j], rj);
        this.turnOnResidue(residues[k], rk);
        this.turnOnResidue(residues[l], rl);
        try {
            if (this.algorithmListener != null) {
                this.algorithmListener.algorithmUpdate(this.molecularAssembly);
            }
            double subtract = -this.backboneEnergy - this.getSelf(i, ri) - this.getSelf(j, rj) - this.getSelf(k, rk) - this.getSelf(l, rl) - this.get2Body(i, ri, j, rj) - this.get2Body(i, ri, k, rk) - this.get2Body(i, ri, l, rl) - this.get2Body(j, rj, k, rk) - this.get2Body(j, rj, l, rl) - this.get2Body(k, rk, l, rl) - this.get3Body(residues, i, ri, j, rj, k, rk) - this.get3Body(residues, i, ri, j, rj, l, rl) - this.get3Body(residues, i, ri, k, rk, l, rl) - this.get3Body(residues, j, rj, k, rk, l, rl);
            energy = this.rO.currentEnergy(residues) + subtract;
            if (this.potentialIsOpenMM && energy < this.ommRecalculateThreshold) {
                logger.warning(String.format(" Experimental: re-computing quad energy %s-%d %s-%d %s-%d %s-%d using Force Field X", residues[i], ri, residues[j], rj, residues[k], rk, residues[l], rl));
                energy = this.rO.currentFFXPE() + subtract;
            }
            if (energy < this.singularityThreshold) {
                String message = String.format(" Rejecting quad energy for %s-%d %s-%d %s-%d %s-%d is %10.5g << %10f, likely in error.", residues[i], ri, residues[j], rj, residues[k], rk, residues[l], rl, energy, this.singularityThreshold);
                logger.warning(message);
                throw new EnergyException(message);
            }
        }
        finally {
            this.turnOffResidue(residues[i]);
            this.turnOffResidue(residues[j]);
            this.turnOffResidue(residues[k]);
            this.turnOffResidue(residues[l]);
        }
        return energy;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public double computeSelfEnergy(Residue[] residues, int i, int ri) {
        double energy;
        this.rO.turnOffAllResidues(residues);
        this.rO.turnOnResidue(residues[i], ri);
        double KpH = this.rO.getPHRestraint();
        try {
            if (this.algorithmListener != null) {
                this.algorithmListener.algorithmUpdate(this.molecularAssembly);
            }
            energy = this.rO.currentEnergy(residues) - this.backboneEnergy;
            if (this.potentialIsOpenMM && energy < this.ommRecalculateThreshold) {
                logger.warning(String.format(" Experimental: re-computing self energy %s-%d using Force Field X", residues[i], ri));
                energy = this.rO.currentFFXPE() - this.backboneEnergy;
            }
            if (energy < this.singularityThreshold) {
                String message = String.format(" Rejecting self energy for %s-%d is %10.5g << %10f, likely in error.", residues[i], ri, energy, this.singularityThreshold);
                logger.warning(message);
                throw new EnergyException(message);
            }
        }
        finally {
            this.rO.turnOffResidue(residues[i]);
        }
        Rotamer[] rotamers = residues[i].getRotamers();
        if (rotamers[ri].isTitrating) {
            double bias = rotamers[ri].getRotamerPhBias();
            double pH = this.rO.getPH();
            String name = rotamers[ri].getName();
            double pHrestraint = 0.0;
            switch (name) {
                case "ASP": {
                    if (!(pH <= 3.94)) break;
                    pHrestraint = KpH * Math.pow(pH - 3.94, 2.0);
                    break;
                }
                case "ASH": {
                    if (!(pH >= 3.94)) break;
                    pHrestraint = KpH * Math.pow(pH - 3.94, 2.0);
                    break;
                }
                case "GLU": {
                    if (!(pH <= 4.25)) break;
                    pHrestraint = KpH * Math.pow(pH - 4.25, 2.0);
                    break;
                }
                case "GLH": {
                    if (!(pH >= 4.25)) break;
                    pHrestraint = KpH * Math.pow(pH - 4.25, 2.0);
                    break;
                }
                case "HIS": {
                    if (!(pH >= 6.6)) break;
                    pHrestraint = KpH * Math.pow(pH - 6.6, 2.0);
                    break;
                }
                case "HID": {
                    if (!(pH <= 7.0)) break;
                    pHrestraint = KpH * Math.pow(pH - 7.0, 2.0);
                    break;
                }
                case "HIE": {
                    if (!(pH <= 6.6)) break;
                    pHrestraint = KpH * Math.pow(pH - 6.6, 2.0);
                    break;
                }
                case "LYD": {
                    if (!(pH <= 10.4)) break;
                    pHrestraint = KpH * Math.pow(pH - 10.4, 2.0);
                    break;
                }
                case "LYS": {
                    if (!(pH >= 10.4)) break;
                    pHrestraint = KpH * Math.pow(pH - 10.4, 2.0);
                    break;
                }
                case "CYD": {
                    if (!(pH - 8.55 <= 0.0)) break;
                    pHrestraint = KpH * Math.pow(pH - 8.55, 2.0);
                    break;
                }
                case "CYS": {
                    if (!(pH - 8.55 >= 0.0)) break;
                    pHrestraint = KpH * Math.pow(pH - 8.55, 2.0);
                    break;
                }
            }
            if (logger.isLoggable(Level.FINE)) {
                logger.fine(String.format(" %s Self-Energy %16.8f = FF %16.8f - BB %16.8f + Ph Bias %16.8f + pHrestraint %16.8f ", rotamers[ri].getName(), energy + bias + pHrestraint, energy + this.backboneEnergy, this.backboneEnergy, bias, pHrestraint));
            }
            energy += bias + pHrestraint;
        }
        return energy;
    }

    public double getTotalRotamerPhBias(List<Residue> residues, int[] rotamers, double pH, double KpH) {
        double total = 0.0;
        int n = residues.size();
        for (int i = 0; i < n; ++i) {
            Rotamer[] rot = residues.get(i).getRotamers();
            int ri = rotamers[i];
            double pHrestraint = 0.0;
            if (!rot[ri].isTitrating) continue;
            switch (rot[ri].getName()) {
                case "ASP": {
                    if (!(pH <= 3.94)) break;
                    pHrestraint = KpH * Math.pow(pH - 3.94, 2.0);
                    break;
                }
                case "ASH": {
                    if (!(pH >= 3.94)) break;
                    pHrestraint = KpH * Math.pow(pH - 3.94, 2.0);
                    break;
                }
                case "GLU": {
                    if (!(pH <= 4.25)) break;
                    pHrestraint = KpH * Math.pow(pH - 4.25, 2.0);
                    break;
                }
                case "GLH": {
                    if (!(pH >= 4.25)) break;
                    pHrestraint = KpH * Math.pow(pH - 4.25, 2.0);
                    break;
                }
                case "HIS": {
                    if (!(pH >= 6.6)) break;
                    pHrestraint = KpH * Math.pow(pH - 6.6, 2.0);
                    break;
                }
                case "HID": {
                    if (!(pH <= 7.0)) break;
                    pHrestraint = KpH * Math.pow(pH - 7.0, 2.0);
                    break;
                }
                case "HIE": {
                    if (!(pH <= 6.6)) break;
                    pHrestraint = KpH * Math.pow(pH - 6.6, 2.0);
                    break;
                }
                case "LYD": {
                    if (!(pH <= 10.4)) break;
                    pHrestraint = KpH * Math.pow(pH - 10.4, 2.0);
                    break;
                }
                case "LYS": {
                    if (!(pH >= 10.4)) break;
                    pHrestraint = KpH * Math.pow(pH - 10.4, 2.0);
                    break;
                }
                case "CYD": {
                    if (!(pH - 8.55 <= 0.0)) break;
                    pHrestraint = KpH * Math.pow(pH - 8.55, 2.0);
                    break;
                }
                case "CYS": {
                    if (!(pH - 8.55 >= 0.0)) break;
                    pHrestraint = KpH * Math.pow(pH - 8.55, 2.0);
                    break;
                }
            }
            total += rot[ri].getRotamerPhBias() + pHrestraint;
        }
        return total;
    }

    public double get2Body(int i, int ri, int j, int rj) {
        if (j < i) {
            int ii = i;
            int iri = ri;
            i = j;
            ri = rj;
            j = ii;
            rj = iri;
        }
        try {
            int[] nI = this.resNeighbors[i];
            int indJ = -1;
            for (int l = 0; l < nI.length; ++l) {
                if (nI[l] != j) continue;
                indJ = l;
                break;
            }
            if (indJ == -1) {
                logger.fine(String.format(" Residue %d not found in neighbors of %d; assumed past cutoff.", j, i));
                return 0.0;
            }
            return this.twoBodyEnergy[i][ri][indJ][rj];
        }
        catch (NullPointerException npe) {
            logger.info(String.format(" NPE for 2-body energy (%3d,%2d) (%3d,%2d).", i, ri, j, rj));
            throw npe;
        }
    }

    public double get3Body(Residue[] residues, int i, int ri, int j, int rj, int k, int rk) {
        int indexK;
        int indexJ;
        int indexI;
        int iri;
        int ii;
        if (!this.threeBodyTerm) {
            return 0.0;
        }
        if (j < i) {
            ii = i;
            iri = ri;
            i = j;
            ri = rj;
            j = ii;
            rj = iri;
        }
        if (k < i) {
            ii = i;
            iri = ri;
            i = k;
            ri = rk;
            k = ii;
            rk = iri;
        }
        if (k < j) {
            int jj = j;
            int jrj = rj;
            j = k;
            rj = rk;
            k = jj;
            rk = jrj;
        }
        int[] nI = this.resNeighbors[i];
        int indJ = -1;
        for (int l = 0; l < nI.length; ++l) {
            if (nI[l] != j) continue;
            indJ = l;
            break;
        }
        int[] nJ = this.resNeighbors[j];
        int indK = -1;
        for (int l = 0; l < nJ.length; ++l) {
            if (nJ[l] != k) continue;
            indK = l;
            break;
        }
        if (this.dM.checkTriDistThreshold(indexI = this.allResiduesList.indexOf(residues[i]), ri, indexJ = this.allResiduesList.indexOf(residues[j]), rj, indexK = this.allResiduesList.indexOf(residues[k]), rk)) {
            return 0.0;
        }
        try {
            return this.threeBodyEnergy[i][ri][indJ][rj][indK][rk];
        }
        catch (ArrayIndexOutOfBoundsException | NullPointerException ex) {
            String message = String.format(" Could not find an energy for 3-body energy (%3d,%2d) (%3d,%2d) (%3d,%2d)", i, ri, j, rj, k, rk);
            logger.warning(message);
            throw ex;
        }
    }

    public double getBackboneEnergy() {
        return this.backboneEnergy;
    }

    public void setBackboneEnergy(double backboneEnergy) {
        this.backboneEnergy = backboneEnergy;
    }

    public Map<Integer, Integer[]> getFourBodyEnergyMap() {
        return this.fourBodyEnergyMap;
    }

    public double getSelf(int i, int ri) {
        try {
            return this.selfEnergy[i][ri];
        }
        catch (NullPointerException npe) {
            logger.info(String.format(" NPE for self energy (%3d,%2d).", i, ri));
            throw npe;
        }
    }

    public double getSelf(int i, int ri, Rotamer rot, boolean excludeFMod) {
        try {
            double totalSelf;
            if (rot.isTitrating && excludeFMod) {
                String name = rot.getName();
                double pHRestraint = 0.0;
                switch (name) {
                    case "ASP": {
                        if (!(this.rO.getPH() <= 3.94)) break;
                        pHRestraint = this.rO.getPHRestraint() * Math.pow(this.rO.getPH() - 3.94, 2.0);
                        break;
                    }
                    case "ASH": {
                        if (!(this.rO.getPH() >= 3.94)) break;
                        pHRestraint = this.rO.getPHRestraint() * Math.pow(this.rO.getPH() - 3.94, 2.0);
                        break;
                    }
                    case "GLU": {
                        if (!(this.rO.getPH() <= 4.25)) break;
                        pHRestraint = this.rO.getPHRestraint() * Math.pow(this.rO.getPH() - 4.25, 2.0);
                        break;
                    }
                    case "GLH": {
                        if (!(this.rO.getPH() >= 4.25)) break;
                        pHRestraint = this.rO.getPHRestraint() * Math.pow(this.rO.getPH() - 4.25, 2.0);
                        break;
                    }
                    case "HIS": {
                        if (!(this.rO.getPH() >= 6.6)) break;
                        pHRestraint = this.rO.getPHRestraint() * Math.pow(this.rO.getPH() - 6.6, 2.0);
                        break;
                    }
                    case "HID": {
                        if (!(this.rO.getPH() <= 7.0)) break;
                        pHRestraint = this.rO.getPHRestraint() * Math.pow(this.rO.getPH() - 7.0, 2.0);
                        break;
                    }
                    case "HIE": {
                        if (!(this.rO.getPH() <= 6.6)) break;
                        pHRestraint = this.rO.getPHRestraint() * Math.pow(this.rO.getPH() - 6.6, 2.0);
                        break;
                    }
                    case "LYD": {
                        if (!(this.rO.getPH() <= 10.4)) break;
                        pHRestraint = this.rO.getPHRestraint() * Math.pow(this.rO.getPH() - 10.4, 2.0);
                        break;
                    }
                    case "LYS": {
                        if (!(this.rO.getPH() >= 10.4)) break;
                        pHRestraint = this.rO.getPHRestraint() * Math.pow(this.rO.getPH() - 10.4, 2.0);
                        break;
                    }
                    case "CYD": {
                        if (!(this.rO.getPH() - 8.55 <= 0.0)) break;
                        pHRestraint = this.rO.getPHRestraint() * Math.pow(this.rO.getPH() - 8.55, 2.0);
                        break;
                    }
                    case "CYS": {
                        if (!(this.rO.getPH() - 8.55 >= 0.0)) break;
                        pHRestraint = this.rO.getPHRestraint() * Math.pow(this.rO.getPH() - 8.55, 2.0);
                        break;
                    }
                }
                totalSelf = this.selfEnergy[i][ri] - rot.getRotamerPhBias() - pHRestraint;
            } else {
                totalSelf = this.selfEnergy[i][ri];
            }
            return totalSelf;
        }
        catch (NullPointerException npe) {
            logger.info(String.format(" NPE for self energy (%3d,%2d).", i, ri));
            throw npe;
        }
    }

    public Map<Integer, Integer[]> getSelfEnergyMap() {
        return this.selfEnergyMap;
    }

    public Map<Integer, Integer[]> getThreeBodyEnergyMap() {
        return this.threeBodyEnergyMap;
    }

    public Map<Integer, Integer[]> getTwoBodyEnergyMap() {
        return this.twoBodyEnergyMap;
    }

    public int loadEnergyRestart(File restartFile, Residue[] residues) {
        return this.loadEnergyRestart(restartFile, residues, -1, null);
    }

    public int loadEnergyRestart(File restartFile, Residue[] residues, int boxIteration, int[] cellIndices) {
        try {
            int rj;
            int ri;
            boolean reverseMap;
            int nResidues = residues.length;
            Path path = Paths.get(restartFile.getCanonicalPath(), new String[0]);
            List<String> lines = Files.readAllLines(path, StandardCharsets.UTF_8);
            ArrayList<String> linesThisBox = new ArrayList<String>();
            try {
                this.backboneEnergy = this.rO.computeBackboneEnergy(residues);
            }
            catch (ArithmeticException ex) {
                logger.severe(String.format(" Exception %s in calculating backbone energy; FFX shutting down.", ex));
            }
            this.rO.logIfRank0(String.format("\n Backbone energy:  %s\n", this.rO.formatEnergy(this.backboneEnergy)));
            if (this.usingBoxOptimization && boxIteration >= 0) {
                boolean foundBox = false;
                for (int i = 0; i < lines.size(); ++i) {
                    String l;
                    String line = lines.get(i);
                    if (!line.startsWith("Box")) continue;
                    String[] tok = line.replaceAll("Box", "").replaceAll(":", ",").replaceAll(" ", "").split(",");
                    int readIteration = Integer.parseInt(tok[0]);
                    int readCellIndexX = Integer.parseInt(tok[1]);
                    int readCellIndexY = Integer.parseInt(tok[2]);
                    int readCellIndexZ = Integer.parseInt(tok[3]);
                    if (readIteration != boxIteration || readCellIndexX != cellIndices[0] || readCellIndexY != cellIndices[1] || readCellIndexZ != cellIndices[2]) continue;
                    foundBox = true;
                    for (int j = i + 1; j < lines.size() && !(l = lines.get(j)).startsWith("Box"); ++j) {
                        linesThisBox.add(l);
                    }
                    break;
                }
                if (!foundBox) {
                    this.rO.logIfRank0(String.format(" Didn't find restart energies for Box %d: %d,%d,%d", boxIteration, cellIndices[0], cellIndices[1], cellIndices[2]));
                    return 0;
                }
                if (linesThisBox.isEmpty()) {
                    return 0;
                }
                lines = linesThisBox;
            }
            ArrayList<String> singleLines = new ArrayList<String>();
            ArrayList<String> pairLines = new ArrayList<String>();
            ArrayList<String> tripleLines = new ArrayList<String>();
            for (String line : lines) {
                String[] tok = line.split("\\s");
                if (tok[0].startsWith("Self")) {
                    singleLines.add(line);
                    continue;
                }
                if (tok[0].startsWith("Pair")) {
                    pairLines.add(line);
                    continue;
                }
                if (!tok[0].startsWith("Triple")) continue;
                tripleLines.add(line);
            }
            int loaded = 0;
            if (!tripleLines.isEmpty()) {
                loaded = 3;
            } else if (!pairLines.isEmpty()) {
                loaded = 2;
            } else if (!singleLines.isEmpty()) {
                loaded = 1;
            } else {
                logger.warning(String.format(" Empty or unreadable energy restart file: %s.", restartFile.getCanonicalPath()));
            }
            if (loaded >= 1) {
                reverseMap = true;
                HashMap<String, Integer> reverseJobMapSingles = this.allocateSelfJobMap(residues, nResidues, reverseMap);
                for (String line : singleLines) {
                    try {
                        int i;
                        block55: {
                            String[] tok = line.replace(",", "").replace(":", "").split("\\s+");
                            i = tok[1].contains("-") ? this.nameToNumber(tok[1], residues) : Integer.parseInt(tok[1]);
                            ri = Integer.parseInt(tok[2]);
                            double energy = Double.parseDouble(tok[3]);
                            try {
                                this.setSelf(i, ri, energy);
                                if (this.verbose) {
                                    this.rO.logIfRank0(String.format(" From restart file: Self energy %3d (%8s,%2d): %s", i, residues[i].toFormattedString(false, true), ri, this.rO.formatEnergy(energy)));
                                }
                            }
                            catch (Exception e) {
                                if (!this.verbose) break block55;
                                this.rO.logIfRank0(String.format(" Restart file out-of-bounds index: %s", line));
                            }
                        }
                        String revKey = String.format("%d %d", i, ri);
                        this.selfEnergyMap.remove(reverseJobMapSingles.get(revKey));
                    }
                    catch (NumberFormatException ex) {
                        logger.log(Level.WARNING, String.format(" Unparsable line in energy restart file: \n%s", line), ex);
                    }
                }
                this.rO.logIfRank0(" Loaded self energies from restart file.");
                this.eR.prePruneSelves(residues);
                if (this.pruneClashes) {
                    this.eR.pruneSingleClashes(residues);
                }
            }
            this.condenseEnergyMap(this.selfEnergyMap);
            if (loaded >= 2) {
                if (!this.selfEnergyMap.isEmpty()) {
                    this.rO.logIfRank0(" Double-check that parameters match original run due to missing self-energies.");
                }
                reverseMap = true;
                HashMap<String, Integer> reverseJobMapPairs = this.allocate2BodyJobMap(residues, nResidues, reverseMap);
                for (String line : pairLines) {
                    try {
                        int j;
                        int i;
                        block56: {
                            String[] tok = line.replace(",", "").replace(":", "").split("\\s+");
                            i = tok[1].contains("-") ? this.nameToNumber(tok[1], residues) : Integer.parseInt(tok[1]);
                            ri = Integer.parseInt(tok[2]);
                            j = tok[3].contains("-") ? this.nameToNumber(tok[3], residues) : Integer.parseInt(tok[3]);
                            rj = Integer.parseInt(tok[4]);
                            double energy = Double.parseDouble(tok[5]);
                            try {
                                if (this.rO.checkNeighboringPair(i, j)) {
                                    int indexJ;
                                    Residue residueI = residues[i];
                                    Residue residueJ = residues[j];
                                    int indexI = this.allResiduesList.indexOf(residueI);
                                    if (!this.dM.checkPairDistThreshold(indexI, ri, indexJ = this.allResiduesList.indexOf(residueJ), rj)) {
                                        this.set2Body(i, ri, j, rj, energy);
                                        double resDist = this.dM.getResidueDistance(indexI, ri, indexJ, rj);
                                        String resDistString = "large";
                                        if (resDist < Double.MAX_VALUE) {
                                            resDistString = String.format("%5.3f", resDist);
                                        }
                                        double dist = this.dM.checkDistMatrix(indexI, ri, indexJ, rj);
                                        String distString = "     large";
                                        if (dist < Double.MAX_VALUE) {
                                            distString = String.format("%10.3f", dist);
                                        }
                                        logger.fine(String.format(" Pair %8s %-2d, %8s %-2d: %s at %s Ang (%s Ang by residue).", residueI.toFormattedString(false, true), ri, residueJ.toFormattedString(false, true), rj, this.rO.formatEnergy(this.get2Body(i, ri, j, rj)), distString, resDistString));
                                    }
                                } else {
                                    logger.fine(String.format("Ignoring a pair-energy from outside the cutoff: 2-energy [(%8s,%2d),(%8s,%2d)]: %12.4f", residues[i].toFormattedString(false, true), ri, residues[j].toFormattedString(false, true), rj, energy));
                                }
                                if (this.verbose) {
                                    this.rO.logIfRank0(String.format(" From restart file: Pair energy [(%8s,%2d),(%8s,%2d)]: %12.4f", residues[i].toFormattedString(false, true), ri, residues[j].toFormattedString(false, true), rj, energy));
                                }
                            }
                            catch (Exception e) {
                                if (!this.verbose) break block56;
                                this.rO.logIfRank0(String.format(" Restart file out-of-bounds index: %s", line));
                            }
                        }
                        String revKey = String.format("%d %d %d %d", i, ri, j, rj);
                        this.twoBodyEnergyMap.remove(reverseJobMapPairs.get(revKey));
                    }
                    catch (NumberFormatException ex) {
                        logger.log(Level.WARNING, String.format("Unparsable line in energy restart file: \n%s", line), ex);
                    }
                }
                this.rO.logIfRank0(" Loaded 2-body energies from restart file.");
                this.eR.prePrunePairs(residues);
                if (this.prunePairClashes) {
                    this.eR.prunePairClashes(residues);
                }
            }
            this.condenseEnergyMap(this.twoBodyEnergyMap);
            if (loaded >= 3) {
                if (!this.twoBodyEnergyMap.isEmpty() && this.master) {
                    logger.warning("Double-check that parameters match original run!  Found trimers in restart file, but pairs job queue is non-empty.");
                }
                reverseMap = true;
                HashMap<String, Integer> reverseJobMapTrimers = this.allocate3BodyJobMap(residues, nResidues, reverseMap);
                for (String line : tripleLines) {
                    try {
                        double energy;
                        int rk;
                        int k;
                        int j;
                        int i;
                        block57: {
                            String[] tok = line.replace(",", "").replace(":", "").split("\\s+");
                            i = tok[1].contains("-") ? this.nameToNumber(tok[1], residues) : Integer.parseInt(tok[1]);
                            ri = Integer.parseInt(tok[2]);
                            j = tok[3].contains("-") ? this.nameToNumber(tok[3], residues) : Integer.parseInt(tok[3]);
                            rj = Integer.parseInt(tok[4]);
                            k = tok[5].contains("-") ? this.nameToNumber(tok[5], residues) : Integer.parseInt(tok[5]);
                            rk = Integer.parseInt(tok[6]);
                            energy = Double.parseDouble(tok[7]);
                            try {
                                if (this.rO.checkNeighboringTriple(i, j, k)) {
                                    int indexK;
                                    int indexJ;
                                    Residue residueI = residues[i];
                                    Residue residueJ = residues[j];
                                    Residue residueK = residues[k];
                                    int indexI = this.allResiduesList.indexOf(residueI);
                                    if (!this.dM.checkTriDistThreshold(indexI, ri, indexJ = this.allResiduesList.indexOf(residueJ), rj, indexK = this.allResiduesList.indexOf(residueK), rk)) {
                                        this.set3Body(residues, i, ri, j, rj, k, rk, energy);
                                        double resDist = this.dM.get3BodyResidueDistance(indexI, ri, indexJ, rj, indexK, rk);
                                        String resDistString = "     large";
                                        if (resDist < Double.MAX_VALUE) {
                                            resDistString = String.format("%5.3f", resDist);
                                        }
                                        double rawDist = this.dM.getRawNBodyDistance(indexI, ri, indexJ, rj, indexK, rk);
                                        String distString = "     large";
                                        if (rawDist < Double.MAX_VALUE) {
                                            distString = String.format("%10.3f", rawDist);
                                        }
                                        logger.fine(String.format(" 3-Body %8s %-2d, %8s %-2d, %8s %-2d: %s at %s Ang (%s Ang by residue).", residueI.toFormattedString(false, true), ri, residueJ.toFormattedString(false, true), rj, residueK.toFormattedString(false, true), rk, this.rO.formatEnergy(this.get3Body(residues, i, ri, j, rj, k, rk)), distString, resDistString));
                                    }
                                } else {
                                    logger.fine(String.format("Ignoring a triple-energy from outside the cutoff: 3-Body %8s %-2d, %8s %-2d, %8s %-2d: %s", residues[i].toFormattedString(false, true), ri, residues[j].toFormattedString(false, true), rj, residues[k].toFormattedString(false, true), rk, this.rO.formatEnergy(this.get3Body(residues, i, ri, j, rj, k, rk))));
                                }
                            }
                            catch (ArrayIndexOutOfBoundsException ex) {
                                if (this.verbose) {
                                    this.rO.logIfRank0(String.format(" Restart file out-of-bounds index: %s", line));
                                }
                            }
                            catch (NullPointerException npe) {
                                if (!this.verbose) break block57;
                                this.rO.logIfRank0(String.format(" NPE in loading 3-body energies: pruning likely changed! 3-body %s-%d %s-%d %s-%d", residues[i].toFormattedString(false, true), ri, residues[j], rj, residues[k], rk));
                            }
                        }
                        if (this.verbose) {
                            this.rO.logIfRank0(String.format(" From restart file: Trimer energy %3d %-2d, %3d %-2d, %3d %-2d: %s", i, ri, j, rj, k, rk, this.rO.formatEnergy(energy)));
                        }
                        String revKey = String.format("%d %d %d %d %d %d", i, ri, j, rj, k, rk);
                        this.threeBodyEnergyMap.remove(reverseJobMapTrimers.get(revKey));
                    }
                    catch (NumberFormatException ex) {
                        logger.log(Level.WARNING, String.format("Unparsable line in energy restart file: \n%s", line), ex);
                    }
                }
                this.rO.logIfRank0(" Loaded trimer energies from restart file.");
            }
            this.condenseEnergyMap(this.threeBodyEnergyMap);
            return loaded;
        }
        catch (IOException ex) {
            logger.log(Level.WARNING, "Exception while loading energy restart file.", ex);
            return 0;
        }
    }

    public double lowestPairEnergy(Residue[] residues, int i, int ri, int j) {
        if (residues == null) {
            return 0.0;
        }
        int n = residues.length;
        if (i < 0 || i >= n) {
            return 0.0;
        }
        if (j < 0 || j >= n) {
            return 0.0;
        }
        Rotamer[] rotamers = residues[j].getRotamers();
        int nr = rotamers.length;
        double energy = Double.MAX_VALUE;
        for (int jr = 0; jr < nr; ++jr) {
            try {
                double e = this.get2Body(i, ri, j, jr);
                if (!(e < energy)) continue;
                energy = e;
                continue;
            }
            catch (Exception exception) {
                // empty catch block
            }
        }
        return energy;
    }

    public double lowestSelfEnergy(Residue[] residues, int i) {
        if (residues == null) {
            return 0.0;
        }
        int n = residues.length;
        if (i < 0 || i >= n) {
            return 0.0;
        }
        Rotamer[] rotamers = residues[i].getRotamers();
        int nr = rotamers.length;
        double energy = Double.MAX_VALUE;
        for (int ni = 0; ni < nr; ++ni) {
            try {
                double e = this.getSelf(i, ni);
                if (!(e < energy)) continue;
                energy = e;
                continue;
            }
            catch (Exception exception) {
                // empty catch block
            }
        }
        return energy;
    }

    public boolean minMaxE2(Residue[] residues, double[] minMax, int i, int ri, int j, int rj) throws IllegalArgumentException {
        Residue resi = residues[i];
        Residue resj = residues[j];
        if (this.eR.check(i, ri) || this.eR.check(j, rj) || this.eR.check(i, ri, j, rj)) {
            throw new IllegalArgumentException(String.format(" Called for minMaxE2 on an eliminated pair %s-%d %s-%d", resi.toFormattedString(false, true), ri, resj.toFormattedString(false, true), rj));
        }
        minMax[0] = 0.0;
        minMax[1] = 0.0;
        int nRes = residues.length;
        for (int k = 0; k < nRes; ++k) {
            if (k == i || k == j) continue;
            Residue resk = residues[k];
            Rotamer[] rotsk = resk.getRotamers();
            int lenrk = rotsk.length;
            double[] minMaxK = new double[]{Double.MAX_VALUE, Double.MIN_VALUE};
            for (int rk = 0; rk < lenrk; ++rk) {
                double currentMin;
                if (this.eR.check(k, rk)) continue;
                if (this.eR.check(i, ri, k, rk) || this.eR.check(j, rj, k, rk)) {
                    minMaxK[1] = Double.NaN;
                    continue;
                }
                double currentMax = currentMin = this.get2Body(i, ri, k, rk) + this.get2Body(j, rj, k, rk);
                if (this.threeBodyTerm) {
                    currentMax = currentMin += this.get3Body(residues, i, ri, j, rj, k, rk);
                    double[] minMaxTriple = new double[2];
                    if (this.minMaxE3(residues, minMaxTriple, i, ri, j, rj, k, rk)) {
                        assert (Double.isFinite(minMaxTriple[0]) && minMaxTriple[0] != Double.MAX_VALUE);
                        currentMin += minMaxTriple[0];
                        currentMax = Double.isFinite(currentMax) && Double.isFinite(minMaxTriple[1]) ? (currentMax += minMaxTriple[1]) : Double.NaN;
                    } else {
                        currentMin = Double.NaN;
                        currentMax = Double.NaN;
                    }
                }
                assert (this.threeBodyTerm || currentMax == currentMin);
                if (Double.isFinite(currentMin) && currentMin < minMaxK[0]) {
                    minMaxK[0] = currentMin;
                }
                minMaxK[1] = Double.isFinite(currentMax) && Double.isFinite(minMaxK[1]) ? Math.max(currentMax, minMaxK[1]) : Double.NaN;
            }
            if (!Double.isFinite(minMaxK[0])) {
                minMax[0] = Double.NaN;
                minMax[1] = Double.NaN;
                return false;
            }
            minMax[0] = minMax[0] + minMaxK[0];
            minMax[1] = Double.isFinite(minMaxK[1]) && Double.isFinite(minMax[1]) ? minMax[1] + minMaxK[1] : Double.NaN;
        }
        return Double.isFinite(minMax[0]);
    }

    public boolean minMaxPairEnergy(Residue[] residues, double[] minMax, int i, int ri, int j) {
        Residue residuej = residues[j];
        Rotamer[] rotamersj = residuej.getRotamers();
        int lenrj = rotamersj.length;
        boolean valid = false;
        minMax[0] = Double.MAX_VALUE;
        minMax[1] = Double.MIN_VALUE;
        for (int rj = 0; rj < lenrj; ++rj) {
            double currMax;
            if (this.eR.check(i, ri) || this.eR.check(j, rj) || this.eR.check(i, ri, j, rj)) continue;
            double currMin = currMax = this.get2Body(i, ri, j, rj);
            if (this.threeBodyTerm) {
                double[] minMaxTriple = new double[2];
                boolean validPair = this.minMax2BodySum(residues, minMaxTriple, i, ri, j, rj);
                if (!validPair) {
                    Residue residuei = residues[i];
                    this.rO.logIfRank0(String.format(" Inconsistent Pair: %8s %2d, %8s %2d.", residuei.toFormattedString(false, true), ri, residuej.toFormattedString(false, true), rj), Level.INFO);
                    continue;
                }
                currMin = Double.isFinite(currMin) && Double.isFinite(minMaxTriple[0]) ? (currMin += minMaxTriple[0]) : Double.NaN;
                currMax = Double.isFinite(currMax) && Double.isFinite(minMaxTriple[1]) ? (currMax += minMaxTriple[1]) : Double.NaN;
            }
            valid = true;
            if (Double.isFinite(currMin) && currMin < minMax[0]) {
                minMax[0] = currMin;
            }
            if (Double.isFinite(currMax) && Double.isFinite(minMax[1])) {
                if (!(currMax > minMax[1])) continue;
                minMax[1] = currMax;
                continue;
            }
            minMax[1] = Double.NaN;
        }
        minMax[0] = minMax[0] == Double.MAX_VALUE ? Double.NaN : minMax[0];
        return valid;
    }

    public void set2Body(int i, int ri, int j, int rj, double e) {
        this.set2Body(i, ri, j, rj, e, false);
    }

    public void set2Body(int i, int ri, int j, int rj, double e, boolean quiet) {
        if (j < i) {
            int ii = i;
            int iri = ri;
            i = j;
            ri = rj;
            j = ii;
            rj = iri;
        }
        try {
            int[] nI = this.resNeighbors[i];
            int indJ = -1;
            for (int l = 0; l < nI.length; ++l) {
                if (nI[l] != j) continue;
                indJ = l;
                break;
            }
            if (indJ == -1) {
                throw new IllegalArgumentException(String.format(" Residue %d not found in neighbors of %d; assumed past cutoff.", j, i));
            }
            this.twoBodyEnergy[i][ri][indJ][rj] = e;
        }
        catch (NullPointerException npe) {
            if (!quiet) {
                logger.info(String.format(" NPE for 2-body energy (%3d,%2d) (%3d,%2d).", i, ri, j, rj));
            }
            throw npe;
        }
    }

    public void set3Body(Residue[] residues, int i, int ri, int j, int rj, int k, int rk, double e) {
        this.set3Body(residues, i, ri, j, rj, k, rk, e, false);
    }

    public void set3Body(Residue[] residues, int i, int ri, int j, int rj, int k, int rk, double e, boolean quiet) throws IllegalStateException {
        int indexK;
        int indexJ;
        int indexI;
        int iri;
        int ii;
        if (!this.threeBodyTerm) {
            throw new IllegalStateException(" Attempting to set a 3-body energy when threeBodyTerm is false!");
        }
        if (j < i) {
            ii = i;
            iri = ri;
            i = j;
            ri = rj;
            j = ii;
            rj = iri;
        }
        if (k < i) {
            ii = i;
            iri = ri;
            i = k;
            ri = rk;
            k = ii;
            rk = iri;
        }
        if (k < j) {
            int jj = j;
            int jrj = rj;
            j = k;
            rj = rk;
            k = jj;
            rk = jrj;
        }
        int[] nI = this.resNeighbors[i];
        int indJ = -1;
        for (int l = 0; l < nI.length; ++l) {
            if (nI[l] != j) continue;
            indJ = l;
            break;
        }
        int[] nJ = this.resNeighbors[j];
        int indK = -1;
        for (int l = 0; l < nJ.length; ++l) {
            if (nJ[l] != k) continue;
            indK = l;
            break;
        }
        if (this.dM.checkTriDistThreshold(indexI = this.allResiduesList.indexOf(residues[i]), ri, indexJ = this.allResiduesList.indexOf(residues[j]), rj, indexK = this.allResiduesList.indexOf(residues[k]), rk)) {
            throw new IllegalArgumentException(String.format(" Residue %d not found in neighbors of %d; assumed past cutoff.", j, i));
        }
        try {
            this.threeBodyEnergy[i][ri][indJ][rj][indK][rk] = e;
        }
        catch (ArrayIndexOutOfBoundsException | NullPointerException ex) {
            if (!quiet) {
                String message = String.format(" Could not access threeBodyEnergy array for 3-body energy (%3d,%2d) (%3d,%2d) (%3d,%2d)", i, ri, j, rj, k, rk);
                logger.warning(message);
            }
            throw ex;
        }
    }

    public void setSelf(int i, int ri, double e) {
        this.setSelf(i, ri, e, false);
    }

    public void setSelf(int i, int ri, double e, boolean quiet) {
        try {
            this.selfEnergy[i][ri] = e;
        }
        catch (ArrayIndexOutOfBoundsException | NullPointerException ex) {
            if (!quiet) {
                logger.warning(String.format(" NPE or array index error for (%3d,%2d)", i, ri));
            }
            throw ex;
        }
    }

    public void turnOffAllResidues(Residue[] residues) {
        if (residues == null) {
            return;
        }
        for (Residue residue : residues) {
            this.turnOffResidue(residue);
        }
    }

    public void turnOffResidue(Residue residue) {
        EnergyExpansion.turnOffAtoms(residue);
        this.applyDefaultRotamer(residue);
    }

    public void turnOnAllResidues(Residue[] residues) {
        if (residues == null) {
            return;
        }
        for (Residue residue : residues) {
            EnergyExpansion.turnOnAtoms(residue);
        }
    }

    public void turnOnResidue(Residue residue, int ri) {
        EnergyExpansion.turnOnAtoms(residue);
        Rotamer[] rotamers = residue.getRotamers();
        RotamerLibrary.applyRotamer((Residue)residue, (Rotamer)rotamers[ri]);
    }

    private boolean minMax2BodySum(Residue[] residues, double[] minMax, int i, int ri, int j, int rj) {
        int nres = residues.length;
        double minSum = 0.0;
        double maxSum = 0.0;
        for (int k = 0; k < nres; ++k) {
            if (k == i || k == j) continue;
            Residue residuek = residues[k];
            Rotamer[] romatersk = residuek.getRotamers();
            int lenrk = romatersk.length;
            double currentMin = Double.MAX_VALUE;
            double currentMax = Double.MIN_VALUE;
            for (int rk = 0; rk < lenrk; ++rk) {
                if (this.eR.check(k, rk)) continue;
                if (this.eR.check(i, ri, k, rk) || this.eR.check(j, rj, k, rk)) {
                    currentMax = Double.NaN;
                    continue;
                }
                double current = this.get3Body(residues, i, ri, j, rj, k, rk);
                if (Double.isFinite(current) && current < currentMin) {
                    currentMin = current;
                }
                if (Double.isFinite(current) && Double.isFinite(currentMax)) {
                    if (!(current > currentMax)) continue;
                    currentMax = current;
                    continue;
                }
                currentMax = Double.NaN;
            }
            if (currentMin == Double.MAX_VALUE || !Double.isFinite(minSum)) {
                minMax[0] = Double.NaN;
                minMax[1] = Double.NaN;
                return false;
            }
            minSum += currentMin;
            if (Double.isFinite(maxSum) && Double.isFinite(currentMax)) {
                maxSum += currentMax;
                continue;
            }
            maxSum = Double.NaN;
        }
        minMax[0] = minSum;
        minMax[1] = maxSum;
        return true;
    }

    private boolean minMaxE3(Residue[] residues, double[] minMax, int i, int ri, int j, int rj, int k, int rk) throws IllegalArgumentException {
        Residue resi = residues[i];
        Residue resj = residues[j];
        Residue resk = residues[k];
        if (this.eR.check(i, ri) || this.eR.check(j, rj) || this.eR.check(k, rk) || this.eR.check(i, ri, j, rj) || this.eR.check(i, ri, k, rk) || this.eR.check(j, rj, k, rk)) {
            throw new IllegalArgumentException(String.format(" Called for minMaxE2 on an eliminated triple %s-%d %s-%d %s-%d", resi.toFormattedString(false, true), ri, resj.toFormattedString(false, true), rj, resk.toFormattedString(false, true), rk));
        }
        minMax[0] = 0.0;
        minMax[1] = 0.0;
        int nRes = residues.length;
        for (int l = 0; l < nRes; ++l) {
            if (l == i || l == j || l == k) continue;
            Residue resl = residues[l];
            Rotamer[] rotsl = resl.getRotamers();
            int lenrl = rotsl.length;
            double currentMax = Double.MIN_VALUE;
            double currentMin = Double.MAX_VALUE;
            for (int rl = 0; rl < lenrl; ++rl) {
                if (this.eR.check(l, rl) || this.eR.check(k, rk, l, rl)) continue;
                double current = this.eR.check(i, ri, l, rl) || this.eR.check(j, rj, l, rl) ? Double.NaN : this.get3Body(residues, i, ri, k, rk, l, rl) + this.get3Body(residues, j, rj, k, rk, l, rl);
                if (Double.isFinite(current) && current < currentMin) {
                    currentMin = current;
                }
                if (Double.isFinite(current) && Double.isFinite(currentMax)) {
                    if (!(current > currentMax)) continue;
                    currentMax = current;
                    continue;
                }
                currentMax = Double.NaN;
            }
            if (!Double.isFinite(currentMin)) {
                minMax[0] = Double.NaN;
                minMax[1] = Double.NaN;
                return false;
            }
            minMax[0] = minMax[0] + currentMin;
            minMax[1] = Double.isFinite(currentMax) && Double.isFinite(minMax[1]) ? minMax[1] + currentMax : Double.NaN;
        }
        return Double.isFinite(minMax[0]);
    }

    private void applyDefaultRotamer(Residue residue) {
        RotamerLibrary.applyRotamer((Residue)residue, (Rotamer)residue.getRotamers()[0]);
    }

    private int nameToNumber(String residueString, Residue[] residues) throws NumberFormatException {
        int ret = -1;
        for (int x = 0; x < residues.length; ++x) {
            if (!residueString.equals(residues[x].toString())) continue;
            ret = x;
            break;
        }
        if (ret == -1) {
            throw new NumberFormatException();
        }
        return ret;
    }

    private void condenseEnergyMap(Map<Integer, Integer[]> energyMap) {
        Set<Integer> keys = energyMap.keySet();
        HashMap<Integer, Integer[]> tempMap = new HashMap<Integer, Integer[]>();
        int count = 0;
        for (int key : keys) {
            tempMap.put(count, energyMap.get(key));
            ++count;
        }
        energyMap.clear();
        energyMap.putAll(tempMap);
    }
}

