/*
 * Decompiled with CFR 0.152.
 */
package ffx.algorithms.optimize;

import com.google.common.collect.MinMaxPriorityQueue;
import ffx.algorithms.AlgorithmListener;
import ffx.algorithms.optimize.Minimize;
import ffx.algorithms.optimize.TorsionSearch;
import ffx.numerics.Potential;
import ffx.potential.AssemblyState;
import ffx.potential.ForceFieldEnergy;
import ffx.potential.MolecularAssembly;
import ffx.potential.bonded.Atom;
import ffx.potential.bonded.Bond;
import ffx.potential.bonded.Molecule;
import ffx.potential.bonded.RestrainDistance;
import ffx.potential.nonbonded.pme.Polarization;
import ffx.potential.parameters.BondType;
import ffx.potential.parsers.XYZFilter;
import ffx.potential.utils.Superpose;
import java.io.File;
import java.util.AbstractQueue;
import java.util.ArrayList;
import java.util.concurrent.ArrayBlockingQueue;
import java.util.logging.Logger;
import org.apache.commons.math3.util.FastMath;

public class ConformationScan {
    private static final Logger logger = Logger.getLogger(ConformationScan.class.getName());
    private final MolecularAssembly mola;
    private final ForceFieldEnergy forceFieldEnergy;
    private final AssemblyState initState;
    private double[] x;
    private AlgorithmListener algorithmListener;
    private final Molecule[] s1;
    private final Molecule[] s2;
    private final Atom[] s1Atoms;
    private final Atom[] s2Atoms;
    private final ArrayList<Atom> s1TargetAtoms;
    private final ArrayList<Atom> s2TargetAtoms;
    private AbstractQueue<StateContainer> statesQueue;
    private final double eps;
    private final int maxIter;
    private double hBondDist;
    private double flatBottomRadius;
    private final boolean tScan;
    private final boolean excludeH;
    private final boolean minimize;
    private double m1MinEnergy;
    private double m2MinEnergy;
    private double totalMonomerMinimizedEnergy;
    private double minimumEnergy;
    private double averageEnergy;
    private double averageEnergyNoOutlier;
    private double stdOfEnergies;
    private double stdOfEnergiesNoOutlier;

    public ConformationScan(MolecularAssembly molecularAssembly, Molecule[] systemOne, Molecule[] systemTwo, double minimizeEps, int minimizeMaxIterations, boolean staticTorsionScan, boolean excludeExtraHydrogen, boolean minimize) {
        this.mola = molecularAssembly;
        this.forceFieldEnergy = molecularAssembly.getPotentialEnergy();
        this.initState = new AssemblyState(this.mola);
        this.s1 = systemOne;
        this.s2 = systemTwo;
        this.eps = minimizeEps;
        this.maxIter = minimizeMaxIterations;
        this.tScan = staticTorsionScan;
        this.excludeH = excludeExtraHydrogen;
        this.minimize = minimize;
        this.s1Atoms = ConformationScan.getAtomsFromMoleculeArray(this.s1);
        this.s2Atoms = ConformationScan.getAtomsFromMoleculeArray(this.s2);
        this.s1TargetAtoms = new ArrayList();
        this.s2TargetAtoms = new ArrayList();
        this.setTargetAtoms(this.mola.getAtomArray());
        this.statesQueue = MinMaxPriorityQueue.maximumSize((int)(this.s1TargetAtoms.size() * this.s2TargetAtoms.size())).create();
        if (!minimize) {
            this.statesQueue = new ArrayBlockingQueue<StateContainer>(this.s1TargetAtoms.size() * this.s2TargetAtoms.size() + 1);
        }
    }

    public void scan() {
        this.systemEnergies();
        double[] zAxis = new double[]{0.0, 0.0, 1.0};
        int loopCounter = 1;
        for (Atom a : this.s1TargetAtoms) {
            for (Atom b : this.s2TargetAtoms) {
                zAxis[2] = 1.0;
                this.initState.revertState();
                ConformationScan.alignSystemCOMtoAtomVecWithAxis(a, zAxis, this.s1Atoms);
                zAxis[2] = -1.0;
                logger.info("\n ----- Trial " + loopCounter + " out of " + this.s2TargetAtoms.size() * this.s1TargetAtoms.size() + " -----");
                ++loopCounter;
                ConformationScan.alignSystemCOMtoAtomVecWithAxis(b, zAxis, this.s2Atoms);
                this.hBondDist = 2.0;
                double[] hBondVector = new double[]{0.0, 0.0, a.getZ() - b.getZ() + this.hBondDist};
                logger.info(" Initial H-bond distance: " + hBondVector[2]);
                this.forceFieldEnergy.getPmeNode().setPolarization(Polarization.NONE);
                hBondVector[2] = hBondVector[2] + this.minimizeVector(hBondVector, -15, 15)[2];
                this.forceFieldEnergy.getPmeNode().setPolarization(Polarization.MUTUAL);
                logger.info(" Best H-bond distance: " + hBondVector[2]);
                this.hBondDist = hBondVector[2] - (a.getZ() - b.getZ());
                this.flatBottomRadius = Math.abs(this.hBondDist / 2.0);
                logger.info(" Flat bottom radius: " + this.flatBottomRadius);
                try {
                    this.mola.update();
                    this.forceFieldEnergy.getCoordinates(this.x);
                    int status = 0;
                    if (this.minimize) {
                        status = this.minimizeSystem(a, b);
                    }
                    if (status != -1) {
                        this.forceFieldEnergy.getCoordinates(this.x);
                        double e = this.forceFieldEnergy.energy(this.x, true) - this.totalMonomerMinimizedEnergy;
                        logger.info("\n Binding energy of trial " + (loopCounter - 1) + ": " + e);
                        this.statesQueue.add(new StateContainer(new AssemblyState(this.mola), e));
                        continue;
                    }
                    logger.warning(" Minimization failed. No state will be saved.");
                }
                catch (Exception ignored) {
                    logger.warning(" Minimization failed. No state will be saved.");
                }
            }
        }
        logger.info("\n ------------------------- End of Trials -------------------------");
        if (this.statesQueue.peek() == null) {
            logger.warning(" No states were saved. No minimum energy found.");
            this.minimumEnergy = Double.NaN;
            return;
        }
        this.minimumEnergy = ((StateContainer)this.statesQueue.peek()).getEnergy();
        this.calculateMeanStd();
    }

    public void logAllEnergyInformation() {
        logger.info(String.format(" Minimum energy of monomer 1:                 %12.5f", this.m1MinEnergy));
        logger.info(String.format(" Minimum energy of monomer 2:                 %12.5f", this.m2MinEnergy));
        logger.info(String.format(" Sum of minimum monomers:                     %12.5f", this.totalMonomerMinimizedEnergy));
        logger.info(String.format(" Minimum binding energy of system:            %12.5f", this.minimumEnergy));
        logger.info(String.format(" Average binding energy:                      %12.5f +- %6.3f", this.averageEnergy, this.stdOfEnergies));
        logger.info(String.format(" Average binding energy (no outlier):         %12.5f +- %6.3f", this.averageEnergyNoOutlier, this.stdOfEnergiesNoOutlier));
    }

    public boolean writeStructuresToXYZ(File outputFile) {
        XYZFilter xyzFilter = new XYZFilter(outputFile, this.mola, this.mola.getForceField(), this.mola.getForceField().getProperties());
        logger.info("\n Writing structures to " + outputFile.getAbsolutePath());
        int count = 1;
        StateContainer[] states = new StateContainer[this.statesQueue.size()];
        if (this.minimize) {
            int index;
            for (index = 0; !this.statesQueue.isEmpty() && index < states.length; ++index) {
                states[index] = (StateContainer)this.statesQueue.poll();
            }
            if (index < states.length) {
                logger.warning(" Not all states will be saved. ");
                return false;
            }
        } else {
            this.statesQueue.toArray(states);
        }
        if (states.length == 0) {
            logger.warning(" No states were saved. No structures will be written.");
            return false;
        }
        for (StateContainer s : states) {
            AssemblyState state = s.getState();
            double energy = s.getEnergy();
            logger.info(" Writing configuration #" + count + " with energy: " + energy);
            state.revertState();
            xyzFilter.writeFile(outputFile, count != 1);
            ++count;
        }
        return true;
    }

    public static ArrayList<Double> logBindingEnergyCalculation(ConformationScan m1, ConformationScan m2, ConformationScan dimer) {
        ArrayList<Double> bindingEnergies = new ArrayList<Double>();
        logger.info("\n ------------------------- Binding Energy -------------------------");
        logger.info(" ddE (of binding) = dE (of coformer dimer binding) - dE (average of both monomer dimer bindings summed)");
        logger.info("\n Calculation using minimized energies: ");
        double averageMonomerEnergy = (m1.getMinimumEnergy() + m2.getMinimumEnergy()) / 2.0;
        double bindingE = dimer.getMinimumEnergy() - averageMonomerEnergy;
        logger.info(String.format(" Minimal Structure    ddE = %8.3f - %8.3f = %12.5f", dimer.getMinimumEnergy(), averageMonomerEnergy, bindingE));
        bindingEnergies.add(bindingE);
        logger.info("\n Calculation using average energies: ");
        averageMonomerEnergy = (m1.getAverageEnergy() + m2.getAverageEnergy()) / 2.0;
        bindingE = dimer.getAverageEnergy() - averageMonomerEnergy;
        logger.info(String.format(" Average              ddE = %8.3f - %8.3f = %12.5f", dimer.getAverageEnergy(), averageMonomerEnergy, bindingE));
        bindingEnergies.add(bindingE);
        logger.info("\n Calculation using average energies (no outliers): ");
        averageMonomerEnergy = (m1.getAverageEnergyNoOutlier() + m2.getAverageEnergyNoOutlier()) / 2.0;
        bindingE = dimer.getAverageEnergyNoOutlier() - averageMonomerEnergy;
        logger.info(String.format(" Average (no outlier) ddE = %8.3f - %8.3f = %12.5f", dimer.getAverageEnergyNoOutlier(), averageMonomerEnergy, bindingE));
        bindingEnergies.add(bindingE);
        logger.info("\n N.B. -- Monomer energies are calculated using the minimized structures in all cases.");
        logger.info(" ------------------------- End of Binding Energy -------------------------\n");
        return bindingEnergies;
    }

    public ArrayList<AssemblyState> getStates() {
        ArrayList<AssemblyState> states = new ArrayList<AssemblyState>();
        for (StateContainer s : this.statesQueue) {
            states.add(s.getState());
        }
        return states;
    }

    public ArrayList<AssemblyState> getStatesWithinEnergy(double energy) {
        ArrayList<AssemblyState> states = new ArrayList<AssemblyState>();
        StateContainer minState = (StateContainer)this.statesQueue.peek();
        if (minState != null) {
            double minEnergy = minState.getEnergy();
            for (StateContainer s : this.statesQueue) {
                if (!(s.getEnergy() - minEnergy < energy)) continue;
                states.add(s.getState());
            }
        }
        return states;
    }

    public ArrayList<AssemblyState> getStatesFilteredByRMSD(double rmsdCutoff) {
        return null;
    }

    public ArrayList<AssemblyState> getStatesAroundAverage(double averageE, double radialCutoff) {
        return null;
    }

    public ArrayList<Double> getEnergies() {
        ArrayList<Double> energies = new ArrayList<Double>();
        for (StateContainer s : this.statesQueue) {
            energies.add(s.getEnergy());
        }
        return energies;
    }

    public ArrayList<Double> getEnergiesWithinEnergy(double energy) {
        ArrayList<Double> energies = new ArrayList<Double>();
        StateContainer minState = (StateContainer)this.statesQueue.peek();
        if (minState != null) {
            double minEnergy = minState.getEnergy();
            for (StateContainer s : this.statesQueue) {
                if (!(s.getEnergy() - minEnergy < energy)) continue;
                energies.add(s.getEnergy());
            }
        }
        return energies;
    }

    public double getM1MinEnergy() {
        return this.m1MinEnergy;
    }

    public double getM2MinEnergy() {
        return this.m2MinEnergy;
    }

    public double getTotalMonomerMinimizedEnergy() {
        return this.totalMonomerMinimizedEnergy;
    }

    public double getMinimumEnergy() {
        return this.minimumEnergy;
    }

    public double getAverageEnergy() {
        return this.averageEnergy;
    }

    public double getAverageEnergyNoOutlier() {
        return this.averageEnergyNoOutlier;
    }

    public double getStdOfEnergies() {
        return this.stdOfEnergies;
    }

    public double getStdOfEnergiesNoOutlier() {
        return this.stdOfEnergiesNoOutlier;
    }

    private double[] minimizeVector(double[] hBondVector, int lowBound, int highBound) {
        double aPotential;
        double a;
        double[] coarsePotentialSurface = new double[Math.abs(++highBound - lowBound)];
        double[] zSearched = new double[Math.abs(highBound - lowBound)];
        double[] coarseVector = new double[3];
        int minIndex = -1;
        double minE = Double.MAX_VALUE;
        coarseVector[2] = hBondVector[2] + (double)lowBound;
        for (int i = 0; i < Math.abs(highBound - lowBound); ++i) {
            zSearched[i] = i == 0 ? (double)lowBound : zSearched[i - 1] + 1.0;
            Atom[] atomArray = this.s2Atoms;
            int n = atomArray.length;
            for (int j = 0; j < n; ++j) {
                Atom a2 = atomArray[j];
                a2.move(coarseVector);
            }
            this.forceFieldEnergy.getCoordinates(this.x);
            coarsePotentialSurface[i] = this.forceFieldEnergy.energy(this.x, false);
            if (coarsePotentialSurface[i] < minE) {
                minE = coarsePotentialSurface[i];
                minIndex = i;
            }
            coarseVector[2] = 1.0;
        }
        for (Atom s2Atom : this.s2Atoms) {
            s2Atom.move(new double[]{0.0, 0.0, hBondVector[2] - zSearched[zSearched.length - 1]});
        }
        double[] refinedVector = new double[3];
        if (minIndex == 0) {
            a = zSearched[minIndex + 1];
            aPotential = coarsePotentialSurface[minIndex + 1];
        } else if (minIndex == coarsePotentialSurface.length - 1) {
            a = zSearched[minIndex - 1];
            aPotential = coarsePotentialSurface[minIndex - 1];
        } else {
            a = coarsePotentialSurface[minIndex - 1] < coarsePotentialSurface[minIndex + 1] ? zSearched[minIndex - 1] : zSearched[minIndex + 1];
            aPotential = Math.min(coarsePotentialSurface[minIndex - 1], coarsePotentialSurface[minIndex + 1]);
        }
        double b = zSearched[minIndex];
        double bPotential = coarsePotentialSurface[minIndex];
        double c = 0.0;
        double convergence = 1.0E-5;
        while (Math.abs(aPotential - bPotential) > convergence) {
            refinedVector[2] = (a + b) / 2.0 - c;
            for (Atom a1 : this.s2Atoms) {
                a1.move(refinedVector);
            }
            this.forceFieldEnergy.getCoordinates(this.x);
            double e = this.forceFieldEnergy.energy(this.x, false);
            minE = Math.min(e, minE);
            if (aPotential > bPotential && e < aPotential) {
                a = (a + b) / 2.0;
                aPotential = e;
                c = a;
                continue;
            }
            if (e < bPotential) {
                b = (a + b) / 2.0;
                bPotential = e;
                c = b;
                continue;
            }
            c = (a + b) / 2.0;
            if (aPotential < bPotential) {
                refinedVector[2] = a - c;
                for (Atom a1 : this.s2Atoms) {
                    a1.move(refinedVector);
                }
            } else {
                refinedVector[2] = b - c;
                for (Atom a1 : this.s2Atoms) {
                    a1.move(refinedVector);
                }
            }
            break;
        }
        refinedVector[2] = aPotential < bPotential ? a : b;
        return refinedVector;
    }

    private void systemEnergies() {
        for (Atom a : this.s2Atoms) {
            a.setUse(false);
        }
        logger.info("\n --------- System 1 Energy Breakdown --------- ");
        double monomerEnergy = this.forceFieldEnergy.energy(this.x, true);
        for (Atom a : this.s2Atoms) {
            a.setUse(true);
        }
        for (Atom a : this.s1Atoms) {
            a.setUse(false);
        }
        logger.info("\n --------- System 2 Energy Breakdown --------- ");
        double monomerEnergy2 = this.forceFieldEnergy.energy(this.x, true);
        for (Atom a : this.s1Atoms) {
            a.setUse(true);
        }
        logger.info(String.format("\n %-29s%12.7f kcal/mol", "System energy 1:", monomerEnergy));
        logger.info(String.format(" %-29s%12.7f kcal/mol", "System energy 2:", monomerEnergy2));
        this.m1MinEnergy = monomerEnergy;
        this.m2MinEnergy = monomerEnergy2;
        this.totalMonomerMinimizedEnergy = monomerEnergy + monomerEnergy2;
    }

    private int minimizeMolecule(Molecule m) {
        if (this.tScan) {
            this.staticScanMolecule(m);
        }
        Minimize minimize = new Minimize(this.mola, (Potential)this.forceFieldEnergy, this.algorithmListener);
        minimize.minimize(this.eps, this.maxIter).getCoordinates(this.x);
        return minimize.getStatus();
    }

    private void staticScanMolecule(Molecule m) {
        TorsionSearch torsionSearch = new TorsionSearch(this.mola, m, 32, 1);
        torsionSearch.staticAnalysis(0, 100.0);
        if (!torsionSearch.getStates().isEmpty()) {
            AssemblyState minState = torsionSearch.getStates().get(0);
            minState.revertState();
        }
    }

    private int minimizeSystem(Atom a, Atom b) throws Exception {
        if (this.tScan) {
            logger.info("\n --------- System 1 Static Torsion Scan --------- ");
            this.forceFieldEnergy.getCoordinates(this.x);
            double tscanE = this.forceFieldEnergy.energy(this.x, false);
            for (Molecule m : this.s1) {
                this.staticScanMolecule(m);
            }
            this.forceFieldEnergy.getCoordinates(this.x);
            double tscanEAfter = this.forceFieldEnergy.energy(this.x, false);
            logger.info("\n Energy before static torsion scan of system 1: " + tscanE);
            logger.info(" Energy after static torsion scan of system 1: " + tscanEAfter);
            logger.info("\n --------- System 2 Static Torsion Scan --------- ");
            this.forceFieldEnergy.getCoordinates(this.x);
            tscanE = this.forceFieldEnergy.energy(this.x, false);
            for (Molecule m : this.s2) {
                this.staticScanMolecule(m);
            }
            this.forceFieldEnergy.getCoordinates(this.x);
            tscanEAfter = this.forceFieldEnergy.energy(this.x, false);
            logger.info("\n Energy before static torsion scan of system 2: " + tscanE);
            logger.info(" Energy after static torsion scan of system 2: " + tscanEAfter);
        }
        this.forceFieldEnergy.getCoordinates(this.x);
        double e = this.forceFieldEnergy.energy(this.x, true);
        RestrainDistance restrainDistance = this.getRestraintBond(a, b, e);
        Minimize minEngine = new Minimize(this.mola, (Potential)this.forceFieldEnergy, null);
        try {
            minEngine.minimize(this.eps, this.maxIter);
        }
        catch (Exception ex) {
            a.getBonds().remove(restrainDistance);
            b.getBonds().remove(restrainDistance);
            a.update();
            b.update();
            this.mola.getBondList().remove(restrainDistance);
            this.mola.update();
            return -1;
        }
        a.getBonds().remove(restrainDistance);
        b.getBonds().remove(restrainDistance);
        a.update();
        b.update();
        this.mola.getBondList().remove(restrainDistance);
        this.mola.update();
        return minEngine.getStatus();
    }

    private RestrainDistance getRestraintBond(Atom a, Atom b, double e) throws Exception {
        if (e > 1000000.0) {
            throw new Exception(" Energy too high to minimize.");
        }
        BondType restraint = new BondType(new int[]{a.getAtomicNumber(), b.getAtomicNumber()}, 100.0, this.hBondDist, BondType.BondFunction.FLAT_BOTTOM_QUARTIC, this.flatBottomRadius);
        RestrainDistance restrainDistance = new RestrainDistance(a, b, null, false, 0.0, 0.0, null);
        restrainDistance.setBondType(restraint);
        return restrainDistance;
    }

    private void setTargetAtoms(Atom[] atoms) {
        for (Atom a : atoms) {
            if (a.getAtomType().atomicNumber == 7 || a.getAtomType().atomicNumber == 8 || a.getAtomType().atomicNumber == 9 || a.getAtomType().atomicNumber == 15 || a.getAtomType().atomicNumber == 16 || a.getAtomType().atomicNumber == 17) {
                if (a.getMoleculeNumber() == this.mola.getMoleculeNumbers()[0]) {
                    this.s1TargetAtoms.add(a);
                } else {
                    this.s2TargetAtoms.add(a);
                }
                if (!this.excludeH) continue;
                for (Bond b : a.getBonds()) {
                    int num = b.get1_2((Atom)a).getAtomType().atomicNumber;
                    if (num != 1) continue;
                    if (a.getMoleculeNumber() == this.mola.getMoleculeNumbers()[0]) {
                        this.s1TargetAtoms.add(a);
                        continue;
                    }
                    this.s2TargetAtoms.add(a);
                }
                continue;
            }
            if (a.getAtomType().atomicNumber != 1 || this.excludeH) continue;
            if (a.getMoleculeNumber() == this.mola.getMoleculeNumbers()[0]) {
                this.s1TargetAtoms.add(a);
                continue;
            }
            this.s2TargetAtoms.add(a);
        }
    }

    private void calculateMeanStd() {
        ArrayList<Double> energies = this.getEnergies();
        double highOutlierCutoff = Double.MAX_VALUE;
        double lowOutlierCutoff = Double.MIN_VALUE;
        if (energies.size() > 4) {
            double iqr = energies.get(energies.size() / 4) - energies.get(3 * energies.size() / 4);
            double q3 = energies.get(3 * energies.size() / 4);
            double q1 = energies.get(energies.size() / 4);
            highOutlierCutoff = q3 + 1.5 * iqr;
            lowOutlierCutoff = q1 - 1.5 * iqr;
        }
        int count = 0;
        double sum = 0.0;
        double sumNoOutlier = 0.0;
        for (double e : energies) {
            if (e < highOutlierCutoff && e > lowOutlierCutoff) {
                sumNoOutlier += e;
                ++count;
            }
            sum += e;
        }
        this.averageEnergy = sum / (double)energies.size();
        this.averageEnergyNoOutlier = sumNoOutlier / (double)count;
        double sumOfSquares = 0.0;
        double sumOfSquaresNoOutlier = 0.0;
        for (double e : energies) {
            sumOfSquares += (e - this.averageEnergy) * (e - this.averageEnergy);
            if (!(e < highOutlierCutoff) || !(e > lowOutlierCutoff)) continue;
            sumOfSquaresNoOutlier += (e - this.averageEnergyNoOutlier) * (e - this.averageEnergyNoOutlier);
        }
        this.stdOfEnergies = Math.sqrt(sumOfSquares / (double)energies.size());
        this.stdOfEnergiesNoOutlier = Math.sqrt(sumOfSquaresNoOutlier / (double)count);
    }

    private static void alignSystemCOMtoAtomVecWithAxis(Atom a, double[] axis, Atom[] mAtoms) {
        int i;
        double[] moleculeOneCOM = ConformationScan.getCOM(mAtoms);
        for (Atom mAtom : mAtoms) {
            mAtom.move(new double[]{-moleculeOneCOM[0], -moleculeOneCOM[1], -moleculeOneCOM[2]});
        }
        double[] aCoords = a.getXYZ().copy().get();
        double[][] rotation = ConformationScan.getRotationBetween(aCoords, axis);
        double[] moleculeAtomicPositions = new double[mAtoms.length * 3];
        for (i = 0; i < mAtoms.length; ++i) {
            moleculeAtomicPositions[i * 3] = mAtoms[i].getX();
            moleculeAtomicPositions[i * 3 + 1] = mAtoms[i].getY();
            moleculeAtomicPositions[i * 3 + 2] = mAtoms[i].getZ();
        }
        Superpose.applyRotation((double[])moleculeAtomicPositions, (double[][])rotation);
        for (i = 0; i < mAtoms.length; ++i) {
            mAtoms[i].setXYZ(new double[]{moleculeAtomicPositions[i * 3], moleculeAtomicPositions[i * 3 + 1], moleculeAtomicPositions[i * 3 + 2]});
        }
    }

    private static double[] getCOM(Atom[] atoms) {
        double[] COM = new double[3];
        double totalMass = 0.0;
        for (Atom s : atoms) {
            double[] pos = s.getXYZ().get();
            COM[0] = COM[0] + pos[0] * s.getMass();
            COM[1] = COM[1] + pos[1] * s.getMass();
            COM[2] = COM[2] + pos[2] * s.getMass();
            totalMass += s.getMass();
        }
        totalMass = 1.0 / totalMass;
        COM[0] = COM[0] * totalMass;
        COM[1] = COM[1] * totalMass;
        COM[2] = COM[2] * totalMass;
        return COM;
    }

    private static double[][] getRotationBetween(double[] v1, double[] v2) {
        double v1Norm = 1.0 / Math.sqrt(v1[0] * v1[0] + v1[1] * v1[1] + v1[2] * v1[2]);
        double v2Norm = 1.0 / Math.sqrt(v2[0] * v2[0] + v2[1] * v2[1] + v2[2] * v2[2]);
        int i = 0;
        while (i < 3) {
            int n = i;
            v1[n] = v1[n] * v1Norm;
            int n2 = i++;
            v2[n2] = v2[n2] * v2Norm;
        }
        double[] crossProduct = new double[]{v1[1] * v2[2] - v1[2] * v2[1], v1[2] * v2[0] - v1[0] * v2[2], v1[0] * v2[1] - v1[1] * v2[0]};
        double crossProductNorm = 1.0 / Math.sqrt(crossProduct[0] * crossProduct[0] + crossProduct[1] * crossProduct[1] + crossProduct[2] * crossProduct[2]);
        int i2 = 0;
        while (i2 < 3) {
            int n = i2++;
            crossProduct[n] = crossProduct[n] * crossProductNorm;
        }
        double dotProduct = v1[0] * v2[0] + v1[1] * v2[1] + v1[2] * v2[2];
        double theta = Math.acos(dotProduct);
        double[] quaternion = new double[]{FastMath.cos((double)(theta / 2.0)), crossProduct[0] * FastMath.sin((double)(theta / 2.0)), crossProduct[1] * FastMath.sin((double)(theta / 2.0)), crossProduct[2] * FastMath.sin((double)(theta / 2.0))};
        double quaternionNorm = 1.0 / Math.sqrt(quaternion[0] * quaternion[0] + quaternion[1] * quaternion[1] + quaternion[2] * quaternion[2] + quaternion[3] * quaternion[3]);
        int i3 = 0;
        while (i3 < 4) {
            int n = i3++;
            quaternion[n] = quaternion[n] * quaternionNorm;
        }
        double q1q1 = quaternion[1] * quaternion[1];
        double q2q2 = quaternion[2] * quaternion[2];
        double q3q3 = quaternion[3] * quaternion[3];
        double q0q1 = quaternion[0] * quaternion[1];
        double q0q2 = quaternion[0] * quaternion[2];
        double q0q3 = quaternion[0] * quaternion[3];
        double q1q2 = quaternion[1] * quaternion[2];
        double q1q3 = quaternion[1] * quaternion[3];
        double q2q3 = quaternion[2] * quaternion[3];
        double[][] rotation = new double[3][3];
        rotation[0][0] = 1.0 - 2.0 * (q2q2 + q3q3);
        rotation[0][1] = 2.0 * (q1q2 - q0q3);
        rotation[0][2] = 2.0 * (q1q3 + q0q2);
        rotation[1][0] = 2.0 * (q1q2 + q0q3);
        rotation[1][1] = 1.0 - 2.0 * (q1q1 + q3q3);
        rotation[1][2] = 2.0 * (q2q3 - q0q1);
        rotation[2][0] = 2.0 * (q1q3 - q0q2);
        rotation[2][1] = 2.0 * (q2q3 + q0q1);
        rotation[2][2] = 1.0 - 2.0 * (q1q1 + q2q2);
        return rotation;
    }

    private static Atom[] getAtomsFromMoleculeArray(Molecule[] system) {
        ArrayList atoms = new ArrayList();
        for (Molecule m : system) {
            atoms.addAll(m.getAtomList());
        }
        return atoms.toArray(new Atom[0]);
    }

    private record StateContainer(AssemblyState state, double e) implements Comparable<StateContainer>
    {
        AssemblyState getState() {
            return this.state;
        }

        double getEnergy() {
            return this.e;
        }

        @Override
        public int compareTo(StateContainer o) {
            return Double.compare(this.e, o.getEnergy());
        }
    }
}

