/*
 * Decompiled with CFR 0.152.
 */
package ffx.algorithms.optimize;

import ffx.algorithms.optimize.RotamerOptimization;
import ffx.algorithms.optimize.manybody.ManyBodyCell;
import ffx.crystal.Crystal;
import ffx.crystal.SymOp;
import ffx.potential.bonded.Atom;
import ffx.potential.bonded.Residue;
import ffx.potential.bonded.ResidueState;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashSet;
import java.util.List;
import java.util.logging.Level;
import java.util.logging.Logger;

class BoxOptimization {
    private final RotamerOptimization rotamerOptimization;
    private static final Logger logger = Logger.getLogger(RotamerOptimization.class.getName());
    public final int[] numXYZCells = new int[]{3, 3, 3};
    public double cellBorderSize = 0.0;
    public double approxBoxLength = 0.0;
    public int boxInclusionCriterion = 1;
    public int cellStart = 0;
    public int cellEnd = -1;
    public boolean manualSuperbox = false;
    public double[] boxDimensions;
    public double superboxBuffer = 8.0;
    public int boxLoadIndex = -1;
    public int[] boxLoadCellIndices;
    public boolean titrationBoxes;
    public double titrationBoxSize;

    public BoxOptimization(RotamerOptimization rotamerOptimization) {
        this.rotamerOptimization = rotamerOptimization;
    }

    public double boxOptimization(List<Residue> residueList) throws Exception {
        this.rotamerOptimization.usingBoxOptimization = true;
        long beginTime = -System.nanoTime();
        Residue[] residues = residueList.toArray(new Residue[0]);
        Crystal crystal = this.generateSuperbox(residueList);
        int totalCells = this.getTotalCellCount(crystal);
        if (this.cellStart > totalCells - 1) {
            this.rotamerOptimization.logIfRank0(String.format(" First cell out of range (%d) -- reset to first cell.", this.cellStart + 1));
            this.cellStart = 0;
        }
        if (this.cellEnd > totalCells - 1) {
            if (this.cellEnd != -1 && this.cellEnd != Integer.MAX_VALUE) {
                this.rotamerOptimization.logIfRank0(String.format(" Final cell out of range (%d) -- reset to last cell.", this.cellEnd + 1));
            }
            this.cellEnd = totalCells - 1;
        } else if (this.cellEnd < 0) {
            this.cellEnd = totalCells - 1;
        } else if (!crystal.aperiodic()) {
            this.cellEnd = totalCells - 1;
        }
        ManyBodyCell[] cells = this.titrationBoxes ? this.loadTitrationCells(crystal, residues) : this.loadCells(crystal, residues);
        int numCells = cells.length;
        this.rotamerOptimization.logIfRank0(String.format(" Optimizing cells %d to %d", this.cellStart + 1, this.cellEnd + 1));
        for (int i = 0; i < numCells; ++i) {
            ManyBodyCell manyBodyCell = cells[i];
            ArrayList<Residue> residueSubsetList = manyBodyCell.getResiduesAsList();
            int[] cellIndices = manyBodyCell.getABCIndices();
            this.rotamerOptimization.logIfRank0(String.format("\n Iteration %d of cell based optimization.", i + 1));
            this.rotamerOptimization.logIfRank0(manyBodyCell.toString());
            int nResidueSubset = residueSubsetList.size();
            if (nResidueSubset > 0) {
                if (this.rotamerOptimization.rank0 && this.rotamerOptimization.writeEnergyRestart && this.rotamerOptimization.printFiles) {
                    String boxHeader = String.format(" Box %d: %d,%d,%d", i + 1, cellIndices[0], cellIndices[1], cellIndices[2]);
                    try {
                        this.rotamerOptimization.energyWriter.append(boxHeader);
                        this.rotamerOptimization.energyWriter.newLine();
                    }
                    catch (IOException ex) {
                        logger.log(Level.SEVERE, " Exception writing box header to energy restart file.", ex);
                    }
                }
                if (this.rotamerOptimization.loadEnergyRestart) {
                    this.boxLoadIndex = i + 1;
                    this.boxLoadCellIndices = new int[3];
                    this.boxLoadCellIndices[0] = cellIndices[0];
                    this.boxLoadCellIndices[1] = cellIndices[1];
                    this.boxLoadCellIndices[2] = cellIndices[2];
                }
                long boxTime = -System.nanoTime();
                Residue firstResidue = (Residue)residueSubsetList.getFirst();
                Residue lastResidue = (Residue)residueSubsetList.get(nResidueSubset - 1);
                Residue[] residueSubsetArray = new Residue[residueSubsetList.size()];
                residueSubsetList.toArray(residueSubsetArray);
                if (this.rotamerOptimization.revert) {
                    ResidueState[] coordinates = ResidueState.storeAllCoordinates(residueSubsetList);
                    double startingEnergy = 0.0;
                    double finalEnergy = 0.0;
                    try {
                        startingEnergy = this.rotamerOptimization.currentEnergy(residueSubsetArray);
                    }
                    catch (ArithmeticException ex) {
                        logger.severe(String.format(" Exception %s in calculating starting energy of a box; FFX shutting down", ex));
                    }
                    this.rotamerOptimization.globalOptimization(residueSubsetList);
                    try {
                        finalEnergy = this.rotamerOptimization.currentEnergy(residueSubsetArray);
                    }
                    catch (ArithmeticException ex) {
                        logger.severe(String.format(" Exception %s in calculating starting energy of a box; FFX shutting down", ex));
                    }
                    if (startingEnergy <= finalEnergy) {
                        logger.info("Optimization did not yield a better energy. Reverting to original coordinates.");
                        ResidueState.revertAllCoordinates(residueSubsetList, (ResidueState[])coordinates);
                    } else {
                        int r = 0;
                        for (Residue residue : residueSubsetList) {
                            int index = residueList.indexOf(residue);
                            this.rotamerOptimization.optimum[index] = this.rotamerOptimization.optimumSubset[r++];
                        }
                    }
                    long currentTime = System.nanoTime();
                    boxTime += currentTime;
                    if (this.rotamerOptimization.genZ) {
                        int[] currentRotamers = new int[this.rotamerOptimization.optimumSubset.length];
                        this.rotamerOptimization.getFractions(residueSubsetArray, 0, currentRotamers, true);
                        if (this.titrationBoxes) {
                            int currentResidueNum = manyBodyCell.getABCIndices()[0] + 1;
                            for (Residue residue : residueSubsetList) {
                                if (residue.getResidueNumber() != currentResidueNum) continue;
                                Residue[] titratationResidue = new Residue[]{residue};
                                this.rotamerOptimization.getProtonationPopulations(titratationResidue);
                            }
                        } else {
                            this.rotamerOptimization.getProtonationPopulations(residueSubsetArray);
                        }
                    }
                    this.rotamerOptimization.logIfRank0(String.format(" Time elapsed for this iteration: %11.3f sec", (double)boxTime * 1.0E-9));
                    this.rotamerOptimization.logIfRank0(String.format(" Overall time elapsed: %11.3f sec", (double)(currentTime + beginTime) * 1.0E-9));
                } else {
                    this.rotamerOptimization.globalOptimization(residueSubsetList);
                    int r = 0;
                    for (Residue residue : residueSubsetList) {
                        int index = residueList.indexOf(residue);
                        this.rotamerOptimization.optimum[index] = this.rotamerOptimization.optimumSubset[r++];
                    }
                    long currentTime = System.nanoTime();
                    boxTime += currentTime;
                    if (this.rotamerOptimization.genZ) {
                        int[] currentRotamers = new int[this.rotamerOptimization.optimumSubset.length];
                        this.rotamerOptimization.getFractions(residueSubsetArray, 0, currentRotamers, true);
                        if (this.titrationBoxes) {
                            int currentResidueNum = manyBodyCell.getABCIndices()[0] + 1;
                            for (Residue residue : residueSubsetList) {
                                if (residue.getResidueNumber() != currentResidueNum) continue;
                                Residue[] titratationResidue = new Residue[]{residue};
                                this.rotamerOptimization.getProtonationPopulations(titratationResidue);
                            }
                        } else {
                            this.rotamerOptimization.getProtonationPopulations(residueSubsetArray);
                        }
                    }
                    this.rotamerOptimization.logIfRank0(String.format(" Time elapsed for this iteration: %11.3f sec", (double)boxTime * 1.0E-9));
                    this.rotamerOptimization.logIfRank0(String.format(" Overall time elapsed: %11.3f sec", (double)(currentTime + beginTime) * 1.0E-9));
                }
                if (!this.rotamerOptimization.rank0 || !this.rotamerOptimization.printFiles || i == numCells - 1) continue;
                try {
                    if (firstResidue != lastResidue) {
                        this.rotamerOptimization.logIfRank0(String.format(" File with residues %s ... %s in window written.", firstResidue, lastResidue));
                        continue;
                    }
                    this.rotamerOptimization.logIfRank0(String.format(" File with residue %s in window written.", firstResidue));
                }
                catch (Exception e) {
                    logger.warning("Exception writing to file.");
                }
                continue;
            }
            this.rotamerOptimization.logIfRank0(" Empty box: no residues found.");
        }
        return 0.0;
    }

    public void update(String boxDim) {
        try {
            String[] bdTokens = boxDim.split(",+");
            this.boxDimensions = new double[6];
            if (bdTokens.length != 7) {
                logger.warning(" Improper number of arguments to boxDimensions; default settings used.");
            } else {
                for (int i = 1; i < 7; i += 2) {
                    this.boxDimensions[i - 1] = Double.parseDouble(bdTokens[i]);
                    this.boxDimensions[i] = Double.parseDouble(bdTokens[i + 1]);
                    if (!(this.boxDimensions[i] < this.boxDimensions[i - 1])) continue;
                    logger.info(String.format(" Improper dimension min %8.5f > max %8.5f; max/min reversed.", this.boxDimensions[i - 1], this.boxDimensions[i]));
                    double temp = this.boxDimensions[i];
                    this.boxDimensions[i] = this.boxDimensions[i - 1];
                    this.boxDimensions[i - 1] = temp;
                }
                this.superboxBuffer = Double.parseDouble(bdTokens[0]);
                this.manualSuperbox = true;
            }
        }
        catch (Exception ex) {
            logger.warning(String.format(" Error in parsing box dimensions: input discarded and defaults used: %s.", ex));
            this.manualSuperbox = false;
        }
    }

    private Crystal generateSuperbox(List<Residue> residueList) {
        double[] maxXYZ = new double[3];
        double[] minXYZ = new double[3];
        Crystal originalCrystal = this.rotamerOptimization.molecularAssembly.getCrystal();
        if (this.manualSuperbox) {
            for (int i = 0; i < maxXYZ.length; ++i) {
                int ii = 2 * i;
                minXYZ[i] = this.boxDimensions[ii] - this.superboxBuffer;
                maxXYZ[i] = this.boxDimensions[ii + 1] + this.superboxBuffer;
            }
        } else if (originalCrystal.aperiodic()) {
            if (residueList == null || residueList.isEmpty()) {
                throw new IllegalArgumentException(" Null or empty residue list when generating superbox.");
            }
            Atom initializerAtom = residueList.get(0).getReferenceAtom();
            initializerAtom.getXYZ(minXYZ);
            initializerAtom.getXYZ(maxXYZ);
            for (Residue residue : residueList) {
                Atom refAtom = residue.getReferenceAtom();
                double[] refAtomCoords = new double[3];
                refAtom.getXYZ(refAtomCoords);
                for (int i = 0; i < 3; ++i) {
                    maxXYZ[i] = Math.max(refAtomCoords[i], maxXYZ[i]);
                    minXYZ[i] = Math.min(refAtomCoords[i], minXYZ[i]);
                }
            }
            int i = 0;
            while (i < 3) {
                int n = i;
                minXYZ[n] = minXYZ[n] - this.superboxBuffer;
                int n2 = i++;
                maxXYZ[n2] = maxXYZ[n2] + this.superboxBuffer;
            }
        } else {
            return originalCrystal.getUnitCell();
        }
        double newA = maxXYZ[0] - minXYZ[0];
        double newB = maxXYZ[1] - minXYZ[1];
        double newC = maxXYZ[2] - minXYZ[2];
        if (this.manualSuperbox) {
            logger.info(String.format(" Manual superbox set over (minX, maxX, minY, maxY, minZ, maxZ): %f, %f, %f, %f, %f, %f", minXYZ[0], maxXYZ[0], minXYZ[1], maxXYZ[1], minXYZ[2], maxXYZ[2]));
        } else {
            logger.info(" System is aperiodic: protein box generated over these coordinates (minX, maxX, minY, maxY, minZ, maxZ):");
            String message = " Aperiodic box dimensions: ";
            for (int i = 0; i < minXYZ.length; ++i) {
                message = message.concat(String.format("%f,%f,", minXYZ[i], maxXYZ[i]));
            }
            message = message.substring(0, message.length() - 1);
            logger.info(message);
        }
        logger.info(String.format(" Buffer size (included in dimensions): %f\n", this.superboxBuffer));
        return new Crystal(newA, newB, newC, 90.0, 90.0, 90.0, "P1");
    }

    private int getTotalCellCount(Crystal crystal) {
        int numCells = 1;
        if (this.approxBoxLength > 0.0) {
            double[] boxes = new double[]{crystal.a / this.approxBoxLength, crystal.b / this.approxBoxLength, crystal.c / this.approxBoxLength};
            for (int i = 0; i < boxes.length; ++i) {
                this.numXYZCells[i] = boxes[i] < 1.0 ? 1 : (int)boxes[i];
            }
        }
        for (int numXYZBox : this.numXYZCells) {
            numCells *= numXYZBox;
        }
        return numCells;
    }

    private ManyBodyCell[] loadCells(Crystal crystal, Residue[] residues) {
        double aCellBorderFracSize = this.cellBorderSize / crystal.a;
        double bCellBorderFracSize = this.cellBorderSize / crystal.b;
        double cCellBorderFracSize = this.cellBorderSize / crystal.c;
        int numCells = this.cellEnd - this.cellStart + 1;
        this.rotamerOptimization.logIfRank0(String.format(" Number of fractional cells: %d = %d x %d x %d", numCells, this.numXYZCells[0], this.numXYZCells[1], this.numXYZCells[2]));
        ManyBodyCell[] cells = new ManyBodyCell[numCells];
        int currentIndex = 0;
        int filledCells = 0;
        int[] xyzIndices = new int[3];
        boolean doBreak = false;
        for (int i = 0; i < this.numXYZCells[0] && !doBreak; ++i) {
            xyzIndices[0] = i;
            block4: for (int j = 0; j < this.numXYZCells[1] && !doBreak; ++j) {
                xyzIndices[1] = j;
                for (int k = 0; k < this.numXYZCells[2]; ++k) {
                    if (currentIndex < this.cellStart) {
                        ++currentIndex;
                        continue;
                    }
                    if (currentIndex > this.cellEnd) {
                        doBreak = true;
                        continue block4;
                    }
                    xyzIndices[2] = k;
                    double[] fracCoords = new double[]{1.0 * (double)i / (double)this.numXYZCells[0] - aCellBorderFracSize, 1.0 * (double)j / (double)this.numXYZCells[1] - bCellBorderFracSize, 1.0 * (double)k / (double)this.numXYZCells[2] - cCellBorderFracSize, (1.0 + (double)i) / (double)this.numXYZCells[0] + aCellBorderFracSize, (1.0 + (double)j) / (double)this.numXYZCells[1] + bCellBorderFracSize, (1.0 + (double)k) / (double)this.numXYZCells[2] + cCellBorderFracSize};
                    cells[filledCells++] = new ManyBodyCell(fracCoords, xyzIndices, currentIndex);
                    ++currentIndex;
                }
            }
        }
        this.assignResiduesToCells(crystal, residues, cells);
        for (ManyBodyCell cell : cells) {
            cell.sortCellResidues();
        }
        switch (this.rotamerOptimization.direction) {
            case BACKWARD: {
                ManyBodyCell[] tempCells = new ManyBodyCell[numCells];
                for (int i = 0; i < numCells; ++i) {
                    tempCells[i] = cells[numCells - (i + 1)];
                }
                cells = tempCells;
            }
        }
        return cells;
    }

    private ManyBodyCell[] loadTitrationCells(Crystal crystal, Residue[] residues) {
        String[] titratableResidues = new String[]{"HIS", "HIE", "HID", "GLU", "GLH", "ASP", "ASH", "LYS", "LYD", "CYS", "CYD"};
        List<String> titratableResiduesList = Arrays.asList(titratableResidues);
        ArrayList<Residue> centerResidues = new ArrayList<Residue>();
        for (Residue residue : residues) {
            if (!titratableResiduesList.contains(residue.getName())) continue;
            logger.info(" Adding residue: " + residue.getName() + residue.getResidueNumber() + " to center residue list");
            centerResidues.add(residue);
        }
        int numCells = centerResidues.size();
        ManyBodyCell[] cells = new ManyBodyCell[numCells];
        int currentIndex = 0;
        int filledCells = 0;
        boolean doBreak = false;
        for (Residue centerResidue : centerResidues) {
            double[] center = new double[3];
            center = centerResidue.getAtomByName("CA", true).getXYZ(center);
            double[] fracCenter = new double[3];
            crystal.toFractionalCoordinates(center, fracCenter);
            double[] fracCoords = new double[]{fracCenter[0] - this.titrationBoxSize / crystal.a, fracCenter[1] - this.titrationBoxSize / crystal.b, fracCenter[2] - this.titrationBoxSize / crystal.c, fracCenter[0] + this.titrationBoxSize / crystal.a, fracCenter[1] + this.titrationBoxSize / crystal.b, fracCenter[2] + this.titrationBoxSize / crystal.c};
            currentIndex = centerResidues.indexOf(centerResidue);
            int[] xyzIndices = new int[]{centerResidue.getResidueNumber() - 1, centerResidue.getResidueNumber() - 1, centerResidue.getResidueNumber() - 1};
            cells[filledCells++] = new ManyBodyCell(fracCoords, xyzIndices, currentIndex);
        }
        this.assignResiduesToCells(crystal, residues, cells);
        for (ManyBodyCell cell : cells) {
            cell.sortCellResidues();
        }
        switch (this.rotamerOptimization.direction) {
            case BACKWARD: {
                ManyBodyCell[] tempCells = new ManyBodyCell[numCells];
                for (int i = 0; i < numCells; ++i) {
                    tempCells[i] = cells[numCells - (i + 1)];
                }
                cells = tempCells;
            }
        }
        return cells;
    }

    private void assignResiduesToCells(Crystal crystal, Residue[] residues, ManyBodyCell[] cells) {
        int nSymm = crystal.spaceGroup.getNumberOfSymOps();
        for (ManyBodyCell cell : cells) {
            HashSet<Residue> toAdd = new HashSet<Residue>();
            for (int iSymm = 0; iSymm < nSymm; ++iSymm) {
                SymOp symOp = crystal.spaceGroup.getSymOp(iSymm);
                for (Residue residue : residues) {
                    if (!(switch (this.boxInclusionCriterion) {
                        default -> cell.atomInsideCell(residue.getReferenceAtom(), crystal, symOp);
                        case 2 -> cell.residueInsideCell(residue, crystal, symOp, true);
                        case 3 -> cell.anyRotamerInsideCell(residue, crystal, symOp, true);
                    })) continue;
                    toAdd.add(residue);
                }
                if (toAdd.isEmpty()) break;
            }
            toAdd.forEach(cell::addResidue);
        }
    }

    public void setTitrationBoxSize(double titrationBoxSize) {
        this.titrationBoxSize = titrationBoxSize;
    }
}

