/*
 * Decompiled with CFR 0.152.
 */
package ffx.crystal;

import ffx.crystal.CrystalSystem;
import ffx.crystal.HKL;
import ffx.crystal.LatticeSystem;
import ffx.crystal.SpaceGroup;
import ffx.crystal.SpaceGroupConversions;
import ffx.crystal.SpaceGroupDefinitions;
import ffx.crystal.SpaceGroupInfo;
import ffx.crystal.SymOp;
import ffx.numerics.math.DoubleMath;
import ffx.numerics.math.MatrixMath;
import ffx.numerics.math.ScalarMath;
import ffx.utilities.FFXProperty;
import ffx.utilities.PropertyGroup;
import ffx.utilities.StringUtils;
import java.util.ArrayList;
import java.util.List;
import java.util.Objects;
import java.util.logging.Level;
import java.util.logging.Logger;
import org.apache.commons.configuration2.CompositeConfiguration;
import org.apache.commons.math3.linear.Array2DRowRealMatrix;
import org.apache.commons.math3.linear.LUDecomposition;
import org.apache.commons.math3.linear.RealMatrix;
import org.apache.commons.math3.util.FastMath;

public class Crystal {
    private static final Logger logger = Logger.getLogger(Crystal.class.getName());
    @FFXProperty(name="SpaceGroup", propertyGroup=PropertyGroup.UnitCellAndSpaceGroup, clazz=SpaceGroup.class, defaultValue="P1", description="This property defines the crystal space group used during structural manipulations and force field calculations.\n")
    public final SpaceGroup spaceGroup;
    @FFXProperty(name="a-axis", propertyGroup=PropertyGroup.UnitCellAndSpaceGroup, defaultValue="None", description="Sets the value of the a-axis length for a crystal unit cell, or, equivalently,\nthe X-axis length for a periodic box (Angstroms).\n")
    public double a;
    @FFXProperty(name="b-axis", propertyGroup=PropertyGroup.UnitCellAndSpaceGroup, defaultValue="A-Axis", description="Sets the value of the b-axis length for a crystal unit cell, or, equivalently,\nthe Y-axis length for a periodic box (Angstroms).\n")
    public double b;
    @FFXProperty(name="c-axis", propertyGroup=PropertyGroup.UnitCellAndSpaceGroup, defaultValue="A-Axis", description="Sets the value of the c-axis length for a crystal unit cell, or, equivalently,\nthe Z-axis length for a periodic box (Angstroms).\n")
    public double c;
    @FFXProperty(name="alpha", propertyGroup=PropertyGroup.UnitCellAndSpaceGroup, defaultValue="90.0", description="Sets the value of the \u03b1-angle of a crystal unit cell, i.e.,\nthe angle between the b-axis and c-axis of a unit cell, or, equivalently,\nthe angle between the Y-axis and Z-axis of a periodic box.\n")
    public double alpha;
    @FFXProperty(name="beta", propertyGroup=PropertyGroup.UnitCellAndSpaceGroup, defaultValue="Alpha", description="Sets the value of the \u03b2-angle of a crystal unit cell, i.e.,\nthe angle between the a-axis and c-axis of a unit cell, or, equivalently,\nthe angle between the X-axis and Z-axis of a periodic box.\n")
    public double beta;
    @FFXProperty(name="gamma", propertyGroup=PropertyGroup.UnitCellAndSpaceGroup, defaultValue="Alpha", description="Sets the value of the \u03b3-angle of a crystal unit cell, i.e.,\nthe angle between the a-axis and b-axis of a unit cell, or, equivalently,\nthe angle between the X-axis and Y-axis of a periodic box.\n")
    public double gamma;
    private static final int XX = 0;
    private static final int YY = 1;
    private static final int ZZ = 2;
    public final double[][] Ai = new double[3][3];
    public final double[][] G = new double[3][3];
    private final CrystalSystem crystalSystem;
    private final LatticeSystem latticeSystem;
    public double volume;
    public double[][] A;
    public double A00;
    public double A10;
    public double A20;
    public double A11;
    public double A21;
    public double A22;
    public double interfacialRadiusA;
    public double interfacialRadiusB;
    public double interfacialRadiusC;
    public int[] scaleB = new int[6];
    public int scaleN;
    public double Ai00;
    public double Ai10;
    public double Ai11;
    public double Ai20;
    public double Ai21;
    public double Ai22;
    public double dVdA;
    public double dVdB;
    public double dVdC;
    public double dVdAlpha;
    public double dVdBeta;
    public double dVdGamma;
    boolean checkRestrictions = true;
    private double specialPositionCutoff = 0.3;
    private List<SymOp> symOpsCartesian;
    private double[][] Gstar;
    private double specialPositionCutoff2 = this.specialPositionCutoff * this.specialPositionCutoff;
    private boolean aperiodic;

    public Crystal(double a, double b, double c, double alpha, double beta, double gamma, int sgNumber) {
        this(a, b, c, alpha, beta, gamma, SpaceGroupDefinitions.spaceGroupFactory((int)sgNumber).pdbName);
    }

    public Crystal(double a, double b, double c, double alpha, double beta, double gamma, String sg) {
        SpaceGroup tempSG = SpaceGroupDefinitions.spaceGroupFactory(sg);
        LatticeSystem tempLS = tempSG.latticeSystem;
        this.a = a;
        this.b = b;
        this.c = c;
        this.alpha = alpha;
        this.beta = beta;
        this.gamma = gamma;
        this.aperiodic = false;
        if (!tempLS.validParameters(a, b, c, alpha, beta, gamma)) {
            StringBuilder sb = new StringBuilder(String.format(" The %s lattice parameters do not satisfy the %s lattice system restrictions.\n", new Object[]{tempSG.pdbName, tempLS}));
            sb.append(String.format("  A-axis:                              %18.15e\n", a));
            sb.append(String.format("  B-axis:                              %18.15e\n", b));
            sb.append(String.format("  C-axis:                              %18.15e\n", c));
            sb.append(String.format("  Alpha:                               %18.15e\n", alpha));
            sb.append(String.format("  Beta:                                %18.15e\n", beta));
            sb.append(String.format("  Gamma:                               %18.15e\n", gamma));
            logger.info(sb.toString());
            sb = new StringBuilder();
            if (tempLS == LatticeSystem.HEXAGONAL_LATTICE || tempLS == LatticeSystem.RHOMBOHEDRAL_LATTICE) {
                Crystal convertedCrystal = SpaceGroupConversions.hrConversion(a, b, c, alpha, beta, gamma, tempSG);
                this.a = convertedCrystal.a;
                this.b = convertedCrystal.b;
                this.c = convertedCrystal.c;
                this.alpha = convertedCrystal.alpha;
                this.beta = convertedCrystal.beta;
                this.gamma = convertedCrystal.gamma;
                this.spaceGroup = convertedCrystal.spaceGroup;
                this.crystalSystem = this.spaceGroup.crystalSystem;
                this.latticeSystem = this.spaceGroup.latticeSystem;
                sb.append(" Converted ").append(tempSG.pdbName).append(" to ").append(this.spaceGroup.pdbName);
                if (!this.latticeSystem.validParameters(this.a, this.b, this.c, this.alpha, this.beta, this.gamma)) {
                    sb.append(String.format(" The %s lattice parameters do not satisfy the %s lattice system restrictions.\n", new Object[]{this.spaceGroup.pdbName, this.latticeSystem}));
                    sb.append(String.format("  A-axis:                              %18.15e\n", this.a));
                    sb.append(String.format("  B-axis:                              %18.15e\n", this.b));
                    sb.append(String.format("  C-axis:                              %18.15e\n", this.c));
                    sb.append(String.format("  Alpha:                               %18.15e\n", this.alpha));
                    sb.append(String.format("  Beta:                                %18.15e\n", this.beta));
                    sb.append(String.format("  Gamma:                               %18.15e\n", this.gamma));
                    logger.severe(sb.toString());
                } else {
                    logger.info(sb.toString());
                }
            } else {
                this.a = a;
                this.b = b;
                this.c = c;
                this.alpha = alpha;
                this.beta = beta;
                this.gamma = gamma;
                this.spaceGroup = tempSG;
                this.crystalSystem = this.spaceGroup.crystalSystem;
                this.latticeSystem = this.spaceGroup.latticeSystem;
                logger.severe(sb.toString());
            }
        } else {
            this.a = a;
            this.b = b;
            this.c = c;
            this.alpha = alpha;
            this.beta = beta;
            this.gamma = gamma;
            this.spaceGroup = tempSG;
            this.crystalSystem = this.spaceGroup.crystalSystem;
            this.latticeSystem = this.spaceGroup.latticeSystem;
        }
        for (int i = 0; i < 6; ++i) {
            this.scaleB[i] = -1;
        }
        int index = 0;
        switch (this.crystalSystem) {
            case TRICLINIC: {
                for (int i = 0; i < 6; ++i) {
                    this.scaleB[i] = index++;
                }
                break;
            }
            case MONOCLINIC: {
                this.scaleB[0] = index++;
                this.scaleB[1] = index++;
                this.scaleB[2] = index++;
                SymOp symop = this.spaceGroup.symOps.get(1);
                double[][] rot = symop.rot;
                if (rot[0][0] > 0.0) {
                    this.scaleB[5] = index++;
                    break;
                }
                if (rot[1][1] > 0.0) {
                    this.scaleB[4] = index++;
                    break;
                }
                this.scaleB[3] = index++;
                break;
            }
            case ORTHORHOMBIC: {
                this.scaleB[0] = index++;
                this.scaleB[1] = index++;
                this.scaleB[2] = index++;
                break;
            }
            case TETRAGONAL: {
                this.scaleB[0] = index++;
                this.scaleB[1] = this.scaleB[0];
                this.scaleB[2] = index++;
                break;
            }
            case TRIGONAL: 
            case HEXAGONAL: {
                double[][] rot;
                boolean hexagonal = false;
                for (int i = 1; i < this.spaceGroup.symOps.size(); ++i) {
                    SymOp symop = this.spaceGroup.symOps.get(i);
                    rot = symop.rot;
                    index = 0;
                    if (rot[1][1] * rot[1][2] == -1.0 || rot[2][1] * rot[2][2] == -1.0 || rot[1][1] * rot[1][2] == 1.0 || rot[2][1] * rot[2][2] == 1.0) {
                        this.scaleB[0] = index++;
                        this.scaleB[1] = index++;
                        this.scaleB[2] = this.scaleB[1];
                        hexagonal = true;
                    } else if (rot[0][0] * rot[0][2] == -1.0 || rot[2][0] * rot[2][2] == -1.0 || rot[0][0] * rot[0][2] == 1.0 || rot[2][0] * rot[2][2] == 1.0) {
                        this.scaleB[0] = index++;
                        this.scaleB[1] = index++;
                        this.scaleB[2] = this.scaleB[0];
                        hexagonal = true;
                    } else if (rot[0][0] * rot[0][1] == -1.0 || rot[1][0] * rot[1][1] == -1.0 || rot[0][0] * rot[0][1] == 1.0 || rot[1][0] * rot[1][1] == 1.0) {
                        this.scaleB[0] = index++;
                        this.scaleB[1] = this.scaleB[0];
                        this.scaleB[2] = index++;
                        hexagonal = true;
                    }
                    if (hexagonal) break;
                }
                if (hexagonal) break;
                this.scaleB[3] = index++;
                this.scaleB[4] = this.scaleB[3];
                this.scaleB[5] = this.scaleB[3];
                break;
            }
        }
        this.scaleN = index;
        this.updateCrystal();
    }

    public static Crystal checkProperties(CompositeConfiguration properties) {
        double a = properties.getDouble("a-axis", -1.0);
        double b = properties.getDouble("b-axis", -1.0);
        double c = properties.getDouble("c-axis", -1.0);
        double alpha = properties.getDouble("alpha", -1.0);
        double beta = properties.getDouble("beta", -1.0);
        double gamma = properties.getDouble("gamma", -1.0);
        String sg = properties.getString("spacegroup", null);
        sg = SpaceGroupInfo.pdb2ShortName(sg);
        if (a < 0.0 || b < 0.0 || c < 0.0 || alpha < 0.0 || beta < 0.0 || gamma < 0.0 || sg == null) {
            return null;
        }
        SpaceGroup spaceGroup = SpaceGroupDefinitions.spaceGroupFactory(sg);
        if (spaceGroup == null && (spaceGroup = SpaceGroupDefinitions.spaceGroupFactory(sg = sg.replaceAll(" ", ""))) == null) {
            return null;
        }
        return new Crystal(a, b, c, alpha, beta, gamma, sg);
    }

    public double invressq(HKL hkl) {
        return hkl.quadForm(this.Gstar);
    }

    public double res(HKL hkl) {
        return 1.0 / FastMath.sqrt((double)hkl.quadForm(this.Gstar));
    }

    public boolean aperiodic() {
        return this.aperiodic;
    }

    public void applySymOp(int n, double[] x, double[] y, double[] z, double[] mateX, double[] mateY, double[] mateZ, SymOp symOp) {
        if (x == null || y == null || z == null) {
            throw new IllegalArgumentException("The input arrays x, y and z must not be null.");
        }
        if (x.length < n || y.length < n || z.length < n) {
            throw new IllegalArgumentException("The input arrays x, y and z must be of length n: " + n);
        }
        if (mateX == null || mateY == null || mateZ == null) {
            throw new IllegalArgumentException("The output arrays mateX, mateY and mateZ must not be null.");
        }
        if (mateX.length < n || mateY.length < n || mateZ.length < n) {
            throw new IllegalArgumentException("The output arrays mateX, mateY and mateZ must be of length n: " + n);
        }
        double[][] rot = symOp.rot;
        double[] trans = symOp.tr;
        double rot00 = rot[0][0];
        double rot10 = rot[1][0];
        double rot20 = rot[2][0];
        double rot01 = rot[0][1];
        double rot11 = rot[1][1];
        double rot21 = rot[2][1];
        double rot02 = rot[0][2];
        double rot12 = rot[1][2];
        double rot22 = rot[2][2];
        double t0 = trans[0];
        double t1 = trans[1];
        double t2 = trans[2];
        for (int i = 0; i < n; ++i) {
            double xc = x[i];
            double yc = y[i];
            double zc = z[i];
            double xi = xc * this.A00 + yc * this.A10 + zc * this.A20;
            double yi = yc * this.A11 + zc * this.A21;
            double zi = zc * this.A22;
            double fx = rot00 * xi + rot01 * yi + rot02 * zi + t0;
            double fy = rot10 * xi + rot11 * yi + rot12 * zi + t1;
            double fz = rot20 * xi + rot21 * yi + rot22 * zi + t2;
            mateX[i] = fx * this.Ai00 + fy * this.Ai10 + fz * this.Ai20;
            mateY[i] = fy * this.Ai11 + fz * this.Ai21;
            mateZ[i] = fz * this.Ai22;
        }
    }

    public void applySymOp(double[] xyz, double[] mate, SymOp symOp) {
        assert (xyz.length % 3 == 0);
        assert (xyz.length == mate.length);
        double[][] rot = symOp.rot;
        double r00 = rot[0][0];
        double r01 = rot[0][1];
        double r02 = rot[0][2];
        double r10 = rot[1][0];
        double r11 = rot[1][1];
        double r12 = rot[1][2];
        double r20 = rot[2][0];
        double r21 = rot[2][1];
        double r22 = rot[2][2];
        double[] trans = symOp.tr;
        double t0 = trans[0];
        double t1 = trans[1];
        double t2 = trans[2];
        int len = xyz.length / 3;
        for (int i = 0; i < len; ++i) {
            int index = i * 3;
            double xc = xyz[index + 0];
            double yc = xyz[index + 1];
            double zc = xyz[index + 2];
            double xi = xc * this.A00 + yc * this.A10 + zc * this.A20;
            double yi = yc * this.A11 + zc * this.A21;
            double zi = zc * this.A22;
            double fx = r00 * xi + r01 * yi + r02 * zi + t0;
            double fy = r10 * xi + r11 * yi + r12 * zi + t1;
            double fz = r20 * xi + r21 * yi + r22 * zi + t2;
            mate[index + 0] = fx * this.Ai00 + fy * this.Ai10 + fz * this.Ai20;
            mate[index + 1] = fy * this.Ai11 + fz * this.Ai21;
            mate[index + 2] = fz * this.Ai22;
        }
    }

    public void applySymRot(double[] xyz, double[] mate, SymOp symOp) {
        double[][] rot = symOp.rot;
        double xc = xyz[0];
        double yc = xyz[1];
        double zc = xyz[2];
        double xi = xc * this.A00 + yc * this.A10 + zc * this.A20;
        double yi = yc * this.A11 + zc * this.A21;
        double zi = zc * this.A22;
        double fx = rot[0][0] * xi + rot[0][1] * yi + rot[0][2] * zi;
        double fy = rot[1][0] * xi + rot[1][1] * yi + rot[1][2] * zi;
        double fz = rot[2][0] * xi + rot[2][1] * yi + rot[2][2] * zi;
        mate[0] = fx * this.Ai00 + fy * this.Ai10 + fz * this.Ai20;
        mate[1] = fy * this.Ai11 + fz * this.Ai21;
        mate[2] = fz * this.Ai22;
    }

    public void applyTransSymRot(int n, double[] x, double[] y, double[] z, double[] mateX, double[] mateY, double[] mateZ, SymOp symOp, double[][] rotmat) {
        if (x == null || y == null || z == null) {
            throw new IllegalArgumentException("The input arrays x, y and z must not be null.");
        }
        if (x.length < n || y.length < n || z.length < n) {
            throw new IllegalArgumentException("The input arrays x, y and z must be of length n: " + n);
        }
        if (mateX == null || mateY == null || mateZ == null) {
            throw new IllegalArgumentException("The output arrays mateX, mateY and mateZ must not be null.");
        }
        if (mateX.length < n || mateY.length < n || mateZ.length < n) {
            throw new IllegalArgumentException("The output arrays mateX, mateY and mateZ must be of length n: " + n);
        }
        this.getTransformationOperator(symOp, rotmat);
        for (int i = 0; i < n; ++i) {
            double xc = x[i];
            double yc = y[i];
            double zc = z[i];
            mateX[i] = xc * rotmat[0][0] + yc * rotmat[1][0] + zc * rotmat[2][0];
            mateY[i] = xc * rotmat[0][1] + yc * rotmat[1][1] + zc * rotmat[2][1];
            mateZ[i] = xc * rotmat[0][2] + yc * rotmat[1][2] + zc * rotmat[2][2];
        }
    }

    public void averageTensor(double[][] m, double[][] r) {
        int n = this.spaceGroup.symOps.size();
        for (int i = 0; i < n; ++i) {
            SymOp symop = this.spaceGroup.symOps.get(i);
            double[][] rot = symop.rot;
            double[][] rt = MatrixMath.mat3Transpose((double[][])rot);
            double[][] rmrt = MatrixMath.mat3Mat3Multiply((double[][])MatrixMath.mat3Mat3Multiply((double[][])rot, (double[][])m), (double[][])rt);
            for (int j = 0; j < 3; ++j) {
                for (int k = 0; k < 3; ++k) {
                    double[] dArray = r[j];
                    int n2 = k;
                    dArray[n2] = dArray[n2] + rmrt[j][k];
                }
            }
        }
        for (int j = 0; j < 3; ++j) {
            int k = 0;
            while (k < 3) {
                double[] dArray = r[j];
                int n3 = k++;
                dArray[n3] = dArray[n3] / 6.0;
            }
        }
    }

    public void averageTensor(double[] v, double[][] r) {
        int n = this.spaceGroup.symOps.size();
        for (int i = 0; i < n; ++i) {
            SymOp symop = this.spaceGroup.symOps.get(i);
            double[][] rot = symop.rot;
            double[][] rt = MatrixMath.mat3Transpose((double[][])rot);
            double[][] rmrt = MatrixMath.mat3Mat3Multiply((double[][])MatrixMath.mat3SymVec6((double[][])rot, (double[])v), (double[][])rt);
            for (int j = 0; j < 3; ++j) {
                for (int k = 0; k < 3; ++k) {
                    double[] dArray = r[j];
                    int n2 = k;
                    dArray[n2] = dArray[n2] + rmrt[j][k];
                }
            }
        }
        for (int j = 0; j < 3; ++j) {
            int k = 0;
            while (k < 3) {
                double[] dArray = r[j];
                int n3 = k++;
                dArray[n3] = dArray[n3] / 6.0;
            }
        }
    }

    public boolean changeUnitCellParameters(double a, double b, double c, double alpha, double beta, double gamma) {
        if (this.checkRestrictions && !this.latticeSystem.validParameters(a, b, c, alpha, beta, gamma)) {
            if (logger.isLoggable(Level.FINE)) {
                StringBuilder sb = new StringBuilder(" The proposed lattice parameters for " + this.spaceGroup.pdbName + " do not satisfy the " + String.valueOf((Object)this.latticeSystem) + " lattice system restrictions and were ignored.\n");
                sb.append(String.format("  A-axis:                              %18.15e\n", a));
                sb.append(String.format("  B-axis:                              %18.15e\n", b));
                sb.append(String.format("  C-axis:                              %18.15e\n", c));
                sb.append(String.format("  Alpha:                               %18.15e\n", alpha));
                sb.append(String.format("  Beta:                                %18.15e\n", beta));
                sb.append(String.format("  Gamma:                               %18.15e\n", gamma));
                logger.fine(sb.toString());
            }
            return false;
        }
        this.a = a;
        this.b = b;
        this.c = c;
        this.alpha = alpha;
        this.beta = beta;
        this.gamma = gamma;
        this.updateCrystal();
        return true;
    }

    public boolean changeUnitCellParametersAndVolume(double a, double b, double c, double alpha, double beta, double gamma, double targetAUVolume) {
        if (this.changeUnitCellParameters(a, b, c, alpha, beta, gamma)) {
            double currentAUVolume = this.volume / (double)this.getNumSymOps();
            double scale = FastMath.cbrt((double)(targetAUVolume / currentAUVolume));
            return this.changeUnitCellParameters(scale * a, scale * b, scale * c, alpha, beta, gamma);
        }
        return false;
    }

    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || this.getClass() != o.getClass()) {
            return false;
        }
        Crystal crystal = (Crystal)o;
        return this.a == crystal.a && this.b == crystal.b && this.c == crystal.c && this.alpha == crystal.alpha && this.beta == crystal.beta && this.gamma == crystal.gamma && this.spaceGroup.number == crystal.spaceGroup.number;
    }

    public boolean getCheckRestrictions() {
        return this.checkRestrictions;
    }

    public void setCheckRestrictions(boolean checkRestrictions) {
        this.checkRestrictions = checkRestrictions;
    }

    public double getDensity(double mass) {
        int nSymm = this.spaceGroup.symOps.size();
        return mass * (double)nSymm / 6.02214076E23 * (1.0E24 / this.volume);
    }

    public int getNumSymOps() {
        return this.spaceGroup.getNumberOfSymOps();
    }

    public double[] getRandomCartTranslation() {
        double[] coords = new double[]{FastMath.random(), FastMath.random(), FastMath.random()};
        this.toCartesianCoordinates(coords, coords);
        return coords;
    }

    public double getSpecialPositionCutoff() {
        return this.specialPositionCutoff;
    }

    public double getSpecialPositionCutoff2() {
        return this.specialPositionCutoff2;
    }

    public void setSpecialPositionCutoff(double cutoff) {
        this.specialPositionCutoff = cutoff;
        this.specialPositionCutoff2 = cutoff * cutoff;
    }

    public void getTransformationOperator(SymOp symOp, double[][] rotmat) {
        double[][] rot = symOp.rot;
        for (int i = 0; i < 3; ++i) {
            for (int j = 0; j < 3; ++j) {
                rotmat[i][j] = 0.0;
                for (int k = 0; k < 3; ++k) {
                    for (int l = 0; l < 3; ++l) {
                        double[] dArray = rotmat[i];
                        int n = j;
                        dArray[n] = dArray[n] + this.Ai[k][i] * rot[k][l] * this.A[j][l];
                    }
                }
            }
        }
    }

    public Crystal getUnitCell() {
        return this;
    }

    public int hashCode() {
        return Objects.hash(this.a, this.b, this.c, this.alpha, this.beta, this.gamma, this.spaceGroup.number);
    }

    public double image(double[] xyz, double[] xyz2) {
        double dx = xyz[0] - xyz2[0];
        double dy = xyz[1] - xyz2[1];
        double dz = xyz[2] - xyz2[2];
        return this.image(dx, dy, dz);
    }

    public double image(double[] xyz) {
        double x = xyz[0];
        double y = xyz[1];
        double z = xyz[2];
        if (this.aperiodic) {
            return x * x + y * y + z * z;
        }
        double xf = x * this.A00 + y * this.A10 + z * this.A20;
        double yf = y * this.A11 + z * this.A21;
        double zf = z * this.A22;
        xf = FastMath.floor((double)(FastMath.abs((double)xf) + 0.5)) * FastMath.signum((double)(-xf)) + xf;
        yf = FastMath.floor((double)(FastMath.abs((double)yf) + 0.5)) * FastMath.signum((double)(-yf)) + yf;
        zf = FastMath.floor((double)(FastMath.abs((double)zf) + 0.5)) * FastMath.signum((double)(-zf)) + zf;
        x = xf * this.Ai00 + yf * this.Ai10 + zf * this.Ai20;
        y = yf * this.Ai11 + zf * this.Ai21;
        z = zf * this.Ai22;
        xyz[0] = x;
        xyz[1] = y;
        xyz[2] = z;
        return x * x + y * y + z * z;
    }

    public double image(double dx, double dy, double dz) {
        if (this.aperiodic) {
            return dx * dx + dy * dy + dz * dz;
        }
        double xf = dx * this.A00 + dy * this.A10 + dz * this.A20;
        double yf = dy * this.A11 + dz * this.A21;
        double zf = dz * this.A22;
        xf = FastMath.floor((double)(FastMath.abs((double)xf) + 0.5)) * FastMath.signum((double)(-xf)) + xf;
        yf = FastMath.floor((double)(FastMath.abs((double)yf) + 0.5)) * FastMath.signum((double)(-yf)) + yf;
        zf = FastMath.floor((double)(FastMath.abs((double)zf) + 0.5)) * FastMath.signum((double)(-zf)) + zf;
        dx = xf * this.Ai00 + yf * this.Ai10 + zf * this.Ai20;
        dy = yf * this.Ai11 + zf * this.Ai21;
        dz = zf * this.Ai22;
        return dx * dx + dy * dy + dz * dz;
    }

    public double minDistOverSymOps(double[] xyzA, double[] xyzB) {
        double dist = 0.0;
        for (int i = 0; i < 3; ++i) {
            double dx = xyzA[i] - xyzB[i];
            dist += dx * dx;
        }
        double[] symB = new double[3];
        for (SymOp symOp : this.spaceGroup.symOps) {
            this.applySymOp(xyzB, symB, symOp);
            for (int i = 0; i < 3; ++i) {
                int n = i;
                symB[n] = symB[n] - xyzA[i];
            }
            double d = this.image(symB);
            dist = FastMath.min((double)d, (double)dist);
        }
        return FastMath.sqrt((double)dist);
    }

    public boolean perturbCellVectors(double[][] dStrain) {
        int j;
        int i;
        double[][] AA = new double[3][3];
        double[][] newAi = new double[3][3];
        for (i = 0; i < 3; ++i) {
            for (j = 0; j < 3; ++j) {
                AA[i][j] = this.getUnitCell().Ai[i][j];
            }
        }
        for (i = 0; i < 3; ++i) {
            for (j = 0; j < 3; ++j) {
                for (int k = 0; k < 3; ++k) {
                    double Kronecker = 0.0;
                    if (j == k) {
                        Kronecker = 1.0;
                    }
                    double[] dArray = newAi[i];
                    int n = j;
                    dArray[n] = dArray[n] + (Kronecker + dStrain[j][k]) * AA[i][k];
                }
            }
        }
        return this.setCellVectors(newAi);
    }

    public boolean randomParameters(double density, double mass) {
        double[] params = this.latticeSystem.resetUnitCellParams();
        boolean succeed = this.changeUnitCellParameters(params[0], params[1], params[2], params[3], params[4], params[5]);
        if (succeed) {
            this.setDensity(density, mass);
        }
        return succeed;
    }

    public void setAperiodic(boolean aperiodic) {
        this.aperiodic = aperiodic;
    }

    public double[] getCellParametersFromVectors(double[][] cellVectors) {
        double aa = DoubleMath.length((double[])cellVectors[0]);
        double bb = DoubleMath.length((double[])cellVectors[1]);
        double cc = DoubleMath.length((double[])cellVectors[2]);
        double aalpha = FastMath.toDegrees((double)FastMath.acos((double)(DoubleMath.dot((double[])cellVectors[1], (double[])cellVectors[2]) / (bb * cc))));
        double bbeta = FastMath.toDegrees((double)FastMath.acos((double)(DoubleMath.dot((double[])cellVectors[0], (double[])cellVectors[2]) / (aa * cc))));
        double ggamma = FastMath.toDegrees((double)FastMath.acos((double)(DoubleMath.dot((double[])cellVectors[0], (double[])cellVectors[1]) / (aa * bb))));
        double[] params = new double[]{aa, bb, cc, aalpha, bbeta, ggamma};
        return params;
    }

    public boolean setCellVectors(double[][] cellVectors) {
        double[] p = this.getCellParametersFromVectors(cellVectors);
        return this.changeUnitCellParameters(p[0], p[1], p[2], p[3], p[4], p[5]);
    }

    public boolean setCellVectorsAndVolume(double[][] cellVectors, double targetAUVolume) {
        if (this.setCellVectors(cellVectors)) {
            double currentAUVolume = this.volume / (double)this.getNumSymOps();
            double scale = FastMath.cbrt((double)(targetAUVolume / currentAUVolume));
            return this.changeUnitCellParameters(scale * this.a, scale * this.b, scale * this.c, this.alpha, this.beta, this.gamma);
        }
        return false;
    }

    public void setDensity(double density, double mass) {
        double currentDensity = this.getDensity(mass);
        double scale = FastMath.cbrt((double)(currentDensity / density));
        this.changeUnitCellParameters(this.a * scale, this.b * scale, this.c * scale, this.alpha, this.beta, this.gamma);
        currentDensity = this.getDensity(mass);
        if (logger.isLoggable(Level.FINE)) {
            logger.fine(String.format(" Updated density %6.3f (g/cc) with unit cell %s.", currentDensity, this.toShortString()));
        }
    }

    public String toCRYST1() {
        return String.format("CRYST1%9.3f%9.3f%9.3f%7.2f%7.2f%7.2f %10s\n", this.a, this.b, this.c, this.alpha, this.beta, this.gamma, StringUtils.padRight((String)this.spaceGroup.pdbName, (int)10));
    }

    public void toCartesianCoordinates(int n, double[] xf, double[] yf, double[] zf, double[] x, double[] y, double[] z) {
        for (int i = 0; i < n; ++i) {
            double xi = xf[i];
            double yi = yf[i];
            double zi = zf[i];
            x[i] = xi * this.Ai00 + yi * this.Ai10 + zi * this.Ai20;
            y[i] = yi * this.Ai11 + zi * this.Ai21;
            z[i] = zi * this.Ai22;
        }
    }

    public void toCartesianCoordinates(int n, double[] frac, double[] cart) {
        int i3 = 0;
        for (int i = 0; i < n; ++i) {
            int iX = i3 + 0;
            int iY = i3 + 1;
            int iZ = i3 + 2;
            i3 += 3;
            double xf = frac[iX];
            double yf = frac[iY];
            double zf = frac[iZ];
            cart[iX] = xf * this.Ai00 + yf * this.Ai10 + zf * this.Ai20;
            cart[iY] = yf * this.Ai11 + zf * this.Ai21;
            cart[iZ] = zf * this.Ai22;
        }
    }

    public void toCartesianCoordinates(double[] xf, double[] x) {
        double fx = xf[0];
        double fy = xf[1];
        double fz = xf[2];
        x[0] = fx * this.Ai00 + fy * this.Ai10 + fz * this.Ai20;
        x[1] = fy * this.Ai11 + fz * this.Ai21;
        x[2] = fz * this.Ai22;
    }

    public void toFractionalCoordinates(int n, double[] x, double[] y, double[] z, double[] xf, double[] yf, double[] zf) {
        for (int i = 0; i < n; ++i) {
            double xc = x[i];
            double yc = y[i];
            double zc = z[i];
            xf[i] = xc * this.A00 + yc * this.A10 + zc * this.A20;
            yf[i] = yc * this.A11 + zc * this.A21;
            zf[i] = zc * this.A22;
        }
    }

    public void toFractionalCoordinates(int n, double[] cart, double[] frac) {
        int i3 = 0;
        for (int i = 0; i < n; ++i) {
            int iX = i3 + 0;
            int iY = i3 + 1;
            int iZ = i3 + 2;
            i3 += 3;
            double xc = cart[iX];
            double yc = cart[iY];
            double zc = cart[iZ];
            frac[iX] = xc * this.A00 + yc * this.A10 + zc * this.A20;
            frac[iY] = yc * this.A11 + zc * this.A21;
            frac[iZ] = zc * this.A22;
        }
    }

    public void toFractionalCoordinates(double[] x, double[] xf) {
        double xc = x[0];
        double yc = x[1];
        double zc = x[2];
        xf[0] = xc * this.A00 + yc * this.A10 + zc * this.A20;
        xf[1] = yc * this.A11 + zc * this.A21;
        xf[2] = zc * this.A22;
    }

    public void toPrimaryCell(double[] in, double[] out) {
        this.toFractionalCoordinates(in, out);
        out[0] = ScalarMath.mod((double)out[0], (double)1.0);
        out[1] = ScalarMath.mod((double)out[1], (double)1.0);
        out[2] = ScalarMath.mod((double)out[2], (double)1.0);
        this.toCartesianCoordinates(out, out);
    }

    public String toShortString() {
        return String.format("%6.2f %6.2f %6.2f %6.2f %6.2f %6.2f", this.a, this.b, this.c, this.alpha, this.beta, this.gamma);
    }

    public String toString() {
        StringBuilder sb = new StringBuilder("\n Unit Cell\n");
        sb.append(String.format("  A-axis length:                       %8.3f\n", this.a));
        sb.append(String.format("  B-axis length:                       %8.3f\n", this.b));
        sb.append(String.format("  C-axis length:                       %8.3f\n", this.c));
        sb.append(String.format("  Alpha:                               %8.3f\n", this.alpha));
        sb.append(String.format("  Beta:                                %8.3f\n", this.beta));
        sb.append(String.format("  Gamma:                               %8.3f\n", this.gamma));
        sb.append("  Space group\n");
        sb.append(String.format("   Number:                                  %3d\n", this.spaceGroup.number));
        sb.append(String.format("   Symbol:                             %8s\n", this.spaceGroup.shortName));
        sb.append(String.format("   Number of Symmetry Operators:            %3d\n", this.spaceGroup.getNumberOfSymOps()));
        sb.append("  Lattice Vectors\n");
        sb.append(String.format("   A:              %8.3f, %8.3f, %8.3f\n", this.Ai00, 0.0, 0.0));
        sb.append(String.format("   B:              %8.3f, %8.3f, %8.3f\n", this.Ai10, this.Ai11, 0.0));
        sb.append(String.format("   C:              %8.3f, %8.3f, %8.3f\n", this.Ai20, this.Ai21, this.Ai22));
        sb.append("  Reciprocal Lattice Vectors\n");
        sb.append(String.format("   A*:             %8.3f, %8.3f, %8.3f\n", this.A00, this.A10, this.A20));
        sb.append(String.format("   B*:             %8.3f, %8.3f, %8.3f\n", 0.0, this.A11, this.A21));
        sb.append(String.format("   C*:             %8.3f, %8.3f, %8.3f", 0.0, 0.0, this.A22));
        return sb.toString();
    }

    public final void updateCrystal() {
        double gamma_term;
        double beta_term;
        double cos_gamma;
        double sin_gamma;
        double cos_beta;
        double cos_alpha;
        switch (this.crystalSystem) {
            default: {
                cos_alpha = 0.0;
                cos_beta = 0.0;
                sin_gamma = 1.0;
                cos_gamma = 0.0;
                beta_term = 0.0;
                gamma_term = 1.0;
                this.volume = this.a * this.b * this.c;
                this.dVdA = this.b * this.c;
                this.dVdB = this.a * this.c;
                this.dVdC = this.a * this.b;
                this.dVdAlpha = 0.0;
                this.dVdBeta = 0.0;
                this.dVdGamma = 0.0;
                break;
            }
            case MONOCLINIC: {
                cos_alpha = 0.0;
                double sin_beta = FastMath.sin((double)FastMath.toRadians((double)this.beta));
                cos_beta = FastMath.cos((double)FastMath.toRadians((double)this.beta));
                sin_gamma = 1.0;
                cos_gamma = 0.0;
                beta_term = 0.0;
                gamma_term = sin_beta;
                this.volume = sin_beta * this.a * this.b * this.c;
                this.dVdA = sin_beta * this.b * this.c;
                this.dVdB = sin_beta * this.a * this.c;
                this.dVdC = sin_beta * this.a * this.b;
                this.dVdAlpha = 0.0;
                this.dVdBeta = cos_beta * this.a * this.b * this.c;
                this.dVdGamma = 0.0;
                break;
            }
            case TRICLINIC: 
            case TRIGONAL: 
            case HEXAGONAL: {
                double sin_alpha = FastMath.sin((double)FastMath.toRadians((double)this.alpha));
                cos_alpha = FastMath.cos((double)FastMath.toRadians((double)this.alpha));
                double sin_beta = FastMath.sin((double)FastMath.toRadians((double)this.beta));
                cos_beta = FastMath.cos((double)FastMath.toRadians((double)this.beta));
                sin_gamma = FastMath.sin((double)FastMath.toRadians((double)this.gamma));
                cos_gamma = FastMath.cos((double)FastMath.toRadians((double)this.gamma));
                beta_term = (cos_alpha - cos_beta * cos_gamma) / sin_gamma;
                gamma_term = FastMath.sqrt((double)(sin_beta * sin_beta - beta_term * beta_term));
                this.volume = sin_gamma * gamma_term * this.a * this.b * this.c;
                this.dVdA = sin_gamma * gamma_term * this.b * this.c;
                this.dVdB = sin_gamma * gamma_term * this.a * this.c;
                this.dVdC = sin_gamma * gamma_term * this.a * this.b;
                double dbeta = 2.0 * sin_beta * cos_beta - 2.0 * (cos_alpha - cos_beta * cos_gamma) * sin_beta * cos_gamma / (sin_gamma * sin_gamma);
                double dgamma1 = -2.0 * (cos_alpha - cos_beta * cos_gamma) * cos_beta / sin_gamma;
                double dgamma2 = cos_alpha - cos_beta * cos_gamma;
                dgamma2 *= dgamma2 * 2.0 * cos_gamma / (sin_gamma * sin_gamma * sin_gamma);
                this.dVdAlpha = (cos_alpha - cos_beta * cos_gamma) * sin_alpha / (sin_gamma * gamma_term) * this.a * this.b * this.c;
                this.dVdBeta = 0.5 * sin_gamma * dbeta / gamma_term * this.a * this.b * this.c;
                this.dVdGamma = (cos_gamma * gamma_term + 0.5 * sin_gamma * (dgamma1 + dgamma2) / gamma_term) * this.a * this.b * this.c;
            }
        }
        this.G[0][0] = this.a * this.a;
        this.G[0][1] = this.a * this.b * cos_gamma;
        this.G[0][2] = this.a * this.c * cos_beta;
        this.G[1][0] = this.G[0][1];
        this.G[1][1] = this.b * this.b;
        this.G[1][2] = this.b * this.c * cos_alpha;
        this.G[2][0] = this.G[0][2];
        this.G[2][1] = this.G[1][2];
        this.G[2][2] = this.c * this.c;
        Array2DRowRealMatrix m = new Array2DRowRealMatrix(this.G, true);
        m = new LUDecomposition((RealMatrix)m).getSolver().getInverse();
        this.Gstar = m.getData();
        this.Ai[0][0] = this.a;
        this.Ai[0][1] = 0.0;
        this.Ai[0][2] = 0.0;
        this.Ai[1][0] = this.b * cos_gamma;
        this.Ai[1][1] = this.b * sin_gamma;
        this.Ai[1][2] = 0.0;
        this.Ai[2][0] = this.c * cos_beta;
        this.Ai[2][1] = this.c * beta_term;
        this.Ai[2][2] = this.c * gamma_term;
        this.Ai00 = this.Ai[0][0];
        this.Ai10 = this.Ai[1][0];
        this.Ai11 = this.Ai[1][1];
        this.Ai20 = this.Ai[2][0];
        this.Ai21 = this.Ai[2][1];
        this.Ai22 = this.Ai[2][2];
        m = new Array2DRowRealMatrix(this.Ai, true);
        m = new LUDecomposition((RealMatrix)m).getSolver().getInverse();
        this.A = m.getData();
        this.A00 = this.A[0][0];
        this.A10 = this.A[1][0];
        this.A20 = this.A[2][0];
        this.A11 = this.A[1][1];
        this.A21 = this.A[2][1];
        this.A22 = this.A[2][2];
        double aStar = 1.0 / FastMath.sqrt((double)(this.A00 * this.A00 + this.A10 * this.A10 + this.A20 * this.A20));
        double bStar = 1.0 / FastMath.sqrt((double)(this.A11 * this.A11 + this.A21 * this.A21));
        double cStar = 1.0 / FastMath.sqrt((double)(this.A22 * this.A22));
        this.interfacialRadiusA = aStar / 2.0;
        this.interfacialRadiusB = bStar / 2.0;
        this.interfacialRadiusC = cStar / 2.0;
        if (logger.isLoggable(Level.FINEST)) {
            logger.finest(String.format(" Half lattice lengths: (%12.6f, %12.6f, %12.6f)", this.a / 2.0, this.b / 2.0, this.c / 2.0));
            logger.finest(String.format(" Interfacial radii:    (%12.6f, %12.6f, %12.6f)", this.interfacialRadiusA, this.interfacialRadiusB, this.interfacialRadiusC));
        }
        List<SymOp> symOps = this.spaceGroup.symOps;
        int nSymm = symOps.size();
        if (this.symOpsCartesian == null) {
            this.symOpsCartesian = new ArrayList<SymOp>(nSymm);
        } else {
            this.symOpsCartesian.clear();
        }
        Array2DRowRealMatrix toFrac = new Array2DRowRealMatrix(this.A);
        Array2DRowRealMatrix toCart = new Array2DRowRealMatrix(this.Ai);
        for (SymOp symOp : symOps) {
            m = new Array2DRowRealMatrix(symOp.rot);
            RealMatrix rotMat = m.preMultiply(toCart.transpose()).multiply(toFrac.transpose());
            double[] tr = toCart.preMultiply(symOp.tr);
            this.symOpsCartesian.add(new SymOp(rotMat.getData(), tr));
        }
    }
}

