/*
 * Decompiled with CFR 0.152.
 */
package ffx.potential.parsers;

import ffx.crystal.Crystal;
import ffx.crystal.LatticeSystem;
import ffx.crystal.SpaceGroup;
import ffx.crystal.SpaceGroupConversions;
import ffx.crystal.SpaceGroupDefinitions;
import ffx.numerics.math.DoubleMath;
import ffx.potential.MolecularAssembly;
import ffx.potential.Utilities;
import ffx.potential.bonded.Angle;
import ffx.potential.bonded.Bond;
import ffx.potential.bonded.BondedUtils;
import ffx.potential.bonded.MSNode;
import ffx.potential.bonded.Molecule;
import ffx.potential.bonded.Polymer;
import ffx.potential.bonded.Residue;
import ffx.potential.parameters.ForceField;
import ffx.potential.parsers.PDBFilter;
import ffx.potential.parsers.SystemFilter;
import ffx.potential.parsers.XYZFilter;
import java.io.BufferedReader;
import java.io.BufferedWriter;
import java.io.File;
import java.io.FileWriter;
import java.io.IOException;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Iterator;
import java.util.List;
import java.util.Objects;
import java.util.logging.Level;
import java.util.logging.Logger;
import javax.vecmath.Point3d;
import org.apache.commons.configuration2.CompositeConfiguration;
import org.apache.commons.io.FilenameUtils;
import org.apache.commons.math3.util.FastMath;
import org.openscience.cdk.Atom;
import org.openscience.cdk.AtomContainer;
import org.openscience.cdk.config.AtomTypeFactory;
import org.openscience.cdk.graph.rebond.RebondTool;
import org.openscience.cdk.interfaces.IAtom;
import org.openscience.cdk.interfaces.IAtomContainer;
import org.openscience.cdk.interfaces.IAtomType;
import org.openscience.cdk.interfaces.IBond;
import org.openscience.cdk.interfaces.IChemObjectBuilder;
import org.openscience.cdk.isomorphism.AtomMatcher;
import org.openscience.cdk.isomorphism.BondMatcher;
import org.openscience.cdk.isomorphism.VentoFoggia;
import org.openscience.cdk.tools.periodictable.PeriodicTable;
import org.rcsb.cif.CifIO;
import org.rcsb.cif.model.FloatColumn;
import org.rcsb.cif.model.StrColumn;
import org.rcsb.cif.schema.StandardSchemata;
import org.rcsb.cif.schema.core.AtomSite;
import org.rcsb.cif.schema.core.Cell;
import org.rcsb.cif.schema.core.Chemical;
import org.rcsb.cif.schema.core.CifCoreBlock;
import org.rcsb.cif.schema.core.CifCoreFile;
import org.rcsb.cif.schema.core.Symmetry;

public class CIFFilter
extends SystemFilter {
    private static final Logger logger = Logger.getLogger(CIFFilter.class.getName());
    private double bondTolerance = 0.2;
    private final BufferedReader bufferedReader = null;
    private CifCoreFile cifFile;
    private final List<String> createdFiles = new ArrayList<String>();
    private int snapShot;
    private String sgName = null;
    private int sgNum = -1;
    private Crystal crystal;
    private boolean fixLattice = false;
    private boolean usePDB = false;
    private int zPrime = -1;
    public static final double MAX_COVALENT_RADIUS = 2.0;
    public static final double MIN_BOND_DISTANCE = 0.5;

    public CIFFilter(File file, MolecularAssembly molecularAssembly, ForceField forceField, CompositeConfiguration properties, boolean saveCIF) {
        super(file, molecularAssembly, forceField, properties);
        this.fileType = Utilities.FileType.CIF;
        if (!saveCIF) {
            String dir = FilenameUtils.getFullPath((String)molecularAssembly.getFile().getAbsolutePath()) + File.separator;
            String cifName = FilenameUtils.removeExtension((String)file.getName()) + ".cif";
            Path path = Paths.get(dir + cifName, new String[0]);
            try {
                this.cifFile = (CifCoreFile)CifIO.readFromPath((Path)path).as(StandardSchemata.CIF_CORE);
            }
            catch (Exception ex) {
                logger.info(" Failed to create CIF file object.\n " + String.valueOf(ex));
            }
            String ext = FilenameUtils.getExtension((String)file.getName());
            this.currentFile = new File(FilenameUtils.removeExtension((String)this.currentFile.getAbsolutePath()) + ext);
            if (ext.equalsIgnoreCase("pdb")) {
                this.usePDB = true;
            }
        }
    }

    public CIFFilter(File file, List<MolecularAssembly> molecularAssemblies, ForceField forceField, CompositeConfiguration properties, boolean saveCIF) {
        super(file, molecularAssemblies, forceField, properties);
        this.fileType = Utilities.FileType.CIF;
        if (!saveCIF) {
            String dir = FilenameUtils.getFullPath((String)molecularAssemblies.get(0).getFile().getAbsolutePath()) + File.separator;
            String cifName = FilenameUtils.removeExtension((String)file.getName()) + ".cif";
            Path path = Paths.get(dir + cifName, new String[0]);
            try {
                this.cifFile = (CifCoreFile)CifIO.readFromPath((Path)path).as(StandardSchemata.CIF_CORE);
            }
            catch (Exception ex) {
                logger.info(" Failed to create CIF file object.\n" + String.valueOf(ex));
            }
            this.currentFile = new File(FilenameUtils.removeExtension((String)this.currentFile.getAbsolutePath()) + ".xyz");
        }
    }

    public CIFFilter(List<File> files, MolecularAssembly molecularAssembly, ForceField forceField, CompositeConfiguration properties, boolean saveCIF) {
        super(files, molecularAssembly, forceField, properties);
        this.fileType = Utilities.FileType.CIF;
        if (!saveCIF) {
            String dir = FilenameUtils.getFullPath((String)molecularAssembly.getFile().getAbsolutePath()) + File.separator;
            String cifName = FilenameUtils.removeExtension((String)files.get(0).getName()) + ".cif";
            Path path = Paths.get(dir + cifName, new String[0]);
            try {
                this.cifFile = (CifCoreFile)CifIO.readFromPath((Path)path).as(StandardSchemata.CIF_CORE);
            }
            catch (Exception ex) {
                logger.info(" Failed to create CIF file object.\n" + String.valueOf(ex));
            }
            this.currentFile = new File(FilenameUtils.removeExtension((String)this.currentFile.getAbsolutePath()) + ".xyz");
        }
    }

    public static int bondAtoms(ffx.potential.bonded.Atom[] atoms, double bondTolerance) {
        int bondCount = 0;
        int nAtoms = atoms.length;
        for (int i = 0; i < nAtoms; ++i) {
            double radiusI;
            ffx.potential.bonded.Atom atom1 = atoms[i];
            String atomIelement = CIFFilter.getAtomElement(atom1);
            double d = radiusI = PeriodicTable.getCovalentRadius((String)atomIelement) == null ? 0.0 : PeriodicTable.getCovalentRadius((String)atomIelement);
            if (radiusI == 0.0) {
                logger.warning(String.format(" Atom %d (%s) element (%s) not determined.", i + 1, atom1.getName(), atomIelement));
                return -1;
            }
            double[] xyz = new double[]{atom1.getX(), atom1.getY(), atom1.getZ()};
            for (int j = i + 1; j < nAtoms; ++j) {
                double bondLength;
                double radiusJ;
                ffx.potential.bonded.Atom atom2 = atoms[j];
                String atomJelement = CIFFilter.getAtomElement(atom2);
                double d2 = radiusJ = PeriodicTable.getCovalentRadius((String)atomJelement) == null ? 0.0 : PeriodicTable.getCovalentRadius((String)atomJelement);
                if (radiusJ == 0.0) {
                    logger.warning(String.format(" Atom %d element (%s) not determined.", j + 1, atomJelement));
                    return -1;
                }
                double[] dArray = new double[]{atom2.getX(), atom2.getY(), atom2.getZ()};
                double[] xyz2 = dArray;
                double length = DoubleMath.dist((double[])xyz, (double[])xyz2);
                if (!(length < (bondLength = radiusI + radiusJ + bondTolerance))) continue;
                ++bondCount;
                Bond bond = new Bond(atom1, atom2);
                atom1.setBond(bond);
                atom2.setBond(bond);
                logger.finest(String.format(" Bonded atom %d (%s) with atom %d (%s): bond length (%4.4f \u00c5) < tolerance (%4.4f \u00c5)", i + 1, atomIelement, j + 1, atomJelement, length, bondLength));
            }
        }
        return bondCount;
    }

    @Override
    public void closeReader() {
        if (this.bufferedReader != null) {
            try {
                this.bufferedReader.close();
            }
            catch (IOException ex) {
                logger.warning(String.format(" Exception in closing CIF filter: %s", ex));
            }
        }
    }

    public static void collectAtoms(ffx.potential.bonded.Atom seed, ArrayList<ffx.potential.bonded.Atom> atoms) {
        logger.finest(String.format(" Atom: %s", seed.getName()));
        atoms.add(seed);
        for (Bond b : seed.getBonds()) {
            ffx.potential.bonded.Atom nextAtom = b.get1_2(seed);
            if (atoms.contains(nextAtom)) continue;
            CIFFilter.collectAtoms(nextAtom, atoms);
        }
    }

    public static String getAtomElement(ffx.potential.bonded.Atom atom) {
        return CIFFilter.getAtomElement(atom.getName());
    }

    private static String getAtomElement(String name) {
        String value = "";
        try {
            value = name.replaceAll("[()]", "").replaceAll("_", "").replaceAll("-", "").replaceAll(" +", "").split("[0-9]")[0];
        }
        catch (Exception e) {
            logger.severe(" Error extracting atom element. Please ensure the CIF is formatted correctly.\n" + String.valueOf(e));
        }
        return value;
    }

    public String[] getCreatedFileNames() {
        String[] files = new String[this.createdFiles.size()];
        return this.createdFiles.toArray(files);
    }

    private ffx.potential.bonded.Atom[] parseBlock(CifCoreBlock block) {
        Cell cell;
        logger.info("\n Block ID: " + block.getBlockHeader());
        Chemical chemical = block.getChemical();
        StrColumn nameCommon = chemical.getNameCommon();
        StrColumn nameSystematic = chemical.getNameSystematic();
        int rowCount = nameCommon.getRowCount();
        if (rowCount > 1) {
            logger.info(" Chemical components");
            for (int i = 0; i < rowCount; ++i) {
                logger.info(String.format("  %s", nameCommon.getColumnName()));
            }
        } else if (rowCount > 0) {
            logger.info(String.format(" Chemical component: %s", nameCommon.getColumnName()));
        }
        Symmetry symmetry = block.getSymmetry();
        if (this.sgNum == -1 && this.sgName == null || this.sgName.isEmpty()) {
            if (symmetry.getIntTablesNumber().getRowCount() > 0) {
                this.sgNum = symmetry.getIntTablesNumber().get(0);
                logger.info(String.format(" CIF International Tables Number: %d", this.sgNum));
            }
            if (symmetry.getSpaceGroupNameH_M().getRowCount() > 0) {
                this.sgName = symmetry.getSpaceGroupNameH_M().get(0);
                logger.info(String.format(" CIF Hermann\u2013Mauguin Space Group: %s", this.sgName));
            } else if (block.getSpaceGroup().getNameH_mFull().getRowCount() > 0) {
                this.sgName = block.getSpaceGroup().getNameH_mFull().get(0);
                logger.info(String.format(" CIF Hermann\u2013Mauguin Space Group: %s", this.sgName));
            } else if (block.getSpaceGroup().getNameH_mAlt().getRowCount() > 0) {
                this.sgName = block.getSpaceGroup().getNameH_mAlt().get(0);
                logger.info(String.format(" CIF Hermann\u2013Mauguin Space Group: %s", this.sgName));
            }
        } else if (this.sgNum != -1) {
            logger.info(String.format(" Command line International Tables Number: %d", this.sgNum));
        } else {
            logger.info(String.format(" Command line space group name: %s", this.sgName));
        }
        SpaceGroup sg = null;
        if (this.sgName != null) {
            sg = SpaceGroupDefinitions.spaceGroupFactory((String)this.sgName);
        }
        if (sg == null) {
            logger.finer(" Space group name not found. Attempting to use space group number.");
            sg = SpaceGroupDefinitions.spaceGroupFactory((int)this.sgNum);
        }
        if (sg == null) {
            logger.warning(" The space group could not be determined from the CIF file (using P1).");
            sg = SpaceGroupDefinitions.spaceGroupFactory((int)1);
        }
        if ((cell = block.getCell()).isDefined()) {
            double a = cell.getLengthA().get(0);
            double b = cell.getLengthB().get(0);
            double c = cell.getLengthC().get(0);
            double alpha = cell.getAngleAlpha().get(0);
            double beta = cell.getAngleBeta().get(0);
            double gamma = cell.getAngleGamma().get(0);
            assert (sg != null);
            LatticeSystem latticeSystem = sg.latticeSystem;
            double[] latticeParameters = new double[]{a, b, c, alpha, beta, gamma};
            if (latticeSystem.validParameters(a, b, c, alpha, beta, gamma)) {
                this.crystal = new Crystal(a, b, c, alpha, beta, gamma, sg.pdbName);
            } else if (this.fixLattice) {
                logger.info(" Attempting to patch disagreement between lattice system and lattice parameters.");
                boolean fixed = false;
                if (latticeSystem == LatticeSystem.HEXAGONAL_LATTICE && LatticeSystem.RHOMBOHEDRAL_LATTICE.validParameters(a, b, c, alpha, beta, gamma)) {
                    this.crystal = SpaceGroupConversions.hrConversion((double)a, (double)b, (double)c, (double)alpha, (double)beta, (double)gamma, (SpaceGroup)sg);
                    latticeSystem = this.crystal.spaceGroup.latticeSystem;
                    if (latticeSystem.validParameters(this.crystal.a, this.crystal.b, this.crystal.c, this.crystal.alpha, this.crystal.beta, this.crystal.gamma)) {
                        fixed = true;
                    }
                } else if (latticeSystem == LatticeSystem.RHOMBOHEDRAL_LATTICE && LatticeSystem.HEXAGONAL_LATTICE.validParameters(a, b, c, alpha, beta, gamma)) {
                    this.crystal = SpaceGroupConversions.hrConversion((double)a, (double)b, (double)c, (double)alpha, (double)beta, (double)gamma, (SpaceGroup)sg);
                    latticeSystem = this.crystal.spaceGroup.latticeSystem;
                    if (latticeSystem.validParameters(this.crystal.a, this.crystal.b, this.crystal.c, this.crystal.alpha, this.crystal.beta, this.crystal.gamma)) {
                        fixed = true;
                    }
                }
                if (!fixed) {
                    double[] newLatticeParameters = latticeSystem.fixParameters(a, b, c, alpha, beta, gamma);
                    if (newLatticeParameters == latticeParameters) {
                        logger.warning(" Conversion Failed: The proposed lattice parameters for " + sg.pdbName + " do not satisfy the " + String.valueOf(latticeSystem) + ".");
                        return null;
                    }
                    a = newLatticeParameters[0];
                    b = newLatticeParameters[1];
                    c = newLatticeParameters[2];
                    alpha = newLatticeParameters[3];
                    beta = newLatticeParameters[4];
                    gamma = newLatticeParameters[5];
                    this.crystal = new Crystal(a, b, c, alpha, beta, gamma, sg.pdbName);
                }
            } else {
                logger.warning(" Conversion Failed: The proposed lattice parameters for " + sg.pdbName + " do not satisfy the " + String.valueOf(latticeSystem) + ".");
                logger.info(" Use \"--fixLattice\" or \"--fl\" flag to attempt to fix automatically.");
                return null;
            }
            this.activeMolecularAssembly.setName(block.getBlockHeader());
            logger.info(" New Crystal: " + String.valueOf(this.crystal));
            this.activeMolecularAssembly.setCrystal(this.crystal);
        }
        AtomSite atomSite = block.getAtomSite();
        StrColumn label = atomSite.getLabel();
        StrColumn typeSymbol = atomSite.getTypeSymbol();
        FloatColumn fractX = atomSite.getFractX();
        FloatColumn fractY = atomSite.getFractY();
        FloatColumn fractZ = atomSite.getFractZ();
        FloatColumn cartX = atomSite.getCartnX();
        FloatColumn cartY = atomSite.getCartnY();
        FloatColumn cartZ = atomSite.getCartnZ();
        int nAtoms = label.getRowCount();
        if (nAtoms < 1) {
            logger.warning(" CIF file did not contain coordinates.");
            return null;
        }
        if (logger.isLoggable(Level.FINE)) {
            logger.fine(String.format("\n Number of Atoms in CIF: %d", nAtoms));
        }
        ffx.potential.bonded.Atom[] atoms = new ffx.potential.bonded.Atom[nAtoms];
        String resName = "CIF";
        double bfactor = 1.0;
        char altLoc = ' ';
        char chain = 'A';
        String[] symbols = new String[nAtoms];
        for (int i = 0; i < nAtoms; ++i) {
            double occupancy = 1.0;
            if (atomSite.getOccupancy().isDefined()) {
                occupancy = atomSite.getOccupancy().get(i);
            }
            symbols[i] = typeSymbol.getRowCount() > 0 ? typeSymbol.getStringData(i) : CIFFilter.getAtomElement(label.getStringData(i));
            double x = fractX.isDefined() ? fractX.get(i) : cartX.get(i);
            double y = fractY.isDefined() ? fractY.get(i) : cartY.get(i);
            double z = fractZ.isDefined() ? fractZ.get(i) : cartZ.get(i);
            double[] xyz = new double[]{x, y, z};
            if (fractX.isDefined() && this.crystal != null) {
                this.crystal.toCartesianCoordinates(xyz, xyz);
            }
            atoms[i] = new ffx.potential.bonded.Atom(i + 1, label.getStringData(i), Character.valueOf(altLoc), xyz, resName, i, Character.valueOf(chain), occupancy, bfactor, symbols[i]);
            atoms[i].setHetero(true);
            if (!logger.isLoggable(Level.FINE)) continue;
            logger.fine(String.format(" Atom (%2d) Name: " + atoms[i].getName() + " Label: " + label.getStringData(i) + " Symbol: " + symbols[i], i));
        }
        return atoms;
    }

    /*
     * Could not resolve type clashes
     * Unable to fully structure code
     */
    @Override
    public boolean readFile() {
        System.setProperty("mpoleterm", "false");
        inputAtoms = this.activeMolecularAssembly.getAtomArray();
        nInputAtoms = inputAtoms.length;
        numHydrogen = 0;
        for (ffx.potential.bonded.Atom atom : inputAtoms) {
            if (!atom.isHydrogen()) continue;
            ++numHydrogen;
        }
        entitiesInput = this.activeMolecularAssembly.getNodeList();
        numEntitiesInput = entitiesInput.size();
        if (CIFFilter.logger.isLoggable(Level.FINE)) {
            CIFFilter.logger.fine(String.format(" Number of entities in input: %d", new Object[]{numEntitiesInput}));
            for (MSNode entity : entitiesInput) {
                CIFFilter.logger.fine(String.format(" Entity: " + entity.getName(), new Object[0]));
                size = entity.getAtomList().size();
                CIFFilter.logger.fine(String.format("   Entity Size: %3d", new Object[]{size}));
                if (size > 0) {
                    CIFFilter.logger.fine(String.format("   Entity First Atom: " + entity.getAtomList().get(0).toString(), new Object[0]));
                    continue;
                }
                CIFFilter.logger.warning(" Entity did not contain atoms.");
            }
        }
        for (CifCoreBlock block : this.cifFile.getBlocks()) {
            atoms = this.parseBlock(block);
            CIFFilter.logger.info(this.crystal.toString());
            if (atoms == null || atoms.length == 0) {
                CIFFilter.logger.warning(" Atoms could not be obtained from CIF.");
                return false;
            }
            nAtoms = atoms.length;
            outputAssembly = new MolecularAssembly(block.getBlockHeader());
            zPrime = this.zPrime > 0 ? this.zPrime : (nAtoms % nInputAtoms == 0 ? nAtoms / nInputAtoms : (nAtoms % (nInputAtoms - numHydrogen) == 0 ? nAtoms / (nInputAtoms - numHydrogen) : 1));
            if (zPrime > 1) {
                CIFFilter.logger.info(String.format(" Detected more than one copy in asymmetric unit of CIF file (Z'=%d). -- attempting to separate.", new Object[]{zPrime}));
            }
            xyzatoms = new ArrayList<ArrayList>();
            atomIndex = 1;
            moleculeNumbers = this.activeMolecularAssembly.getMoleculeNumbers();
            if (CIFFilter.logger.isLoggable(Level.FINER)) {
                for (i = 0; i < moleculeNumbers.length; ++i) {
                    CIFFilter.logger.finer(String.format(" Molecule Number (atom %2d): %2d", new Object[]{i, moleculeNumbers[i]}));
                }
            }
            for (i = 0; i < numEntitiesInput; ++i) {
                mol = entitiesInput.get(i);
                shiftIndex = mol.getAtomList().get(0).getXyzIndex() - 1;
                CIFFilter.logger.info(String.format(" Size of Entity: %2d", new Object[]{mol.getAtomList().size()}));
                xyzatoms.add((ArrayList)mol.getAtomList());
                currentAtoms = (ArrayList)xyzatoms.get(i);
                xyzCDKAtoms = new AtomContainer();
                numMolHydrogen = 0;
                atomCount = 0;
                for (ffx.potential.bonded.Atom atom : currentAtoms) {
                    if (atom.isHydrogen()) {
                        ++numMolHydrogen;
                    }
                    atomName = PeriodicTable.getSymbol((int)atom.getAtomType().atomicNumber);
                    xyzCDKAtoms.addAtom((IAtom)new Atom(atomName, new Point3d(atom.getXYZ(null))));
                    ++atomCount;
                }
                CIFFilter.logger.info(String.format(" Number of atoms in XYZ CDK: %2d", new Object[]{atomCount}));
                numInputMolAtoms = currentAtoms.size();
                if (CIFFilter.logger.isLoggable(Level.FINE)) {
                    CIFFilter.logger.fine(String.format(" Current entity number of atoms: %d (%d + %dH)", new Object[]{numInputMolAtoms, numInputMolAtoms - numMolHydrogen, numMolHydrogen}));
                }
                bonds = mol.getBondList();
                order = IBond.Order.SINGLE;
                xyzBonds = bonds.size();
                if (xyzBonds == 0) {
                    CIFFilter.logger.warning(" No bonds detected in input structure. Please check input.\n If correct, separate non-bonded entities into multiple CIFs.");
                }
                for (Object xyzBond : bonds) {
                    atom0Index = xyzBond.getAtom(0).getXyzIndex();
                    atom1Index = xyzBond.getAtom(1).getXyzIndex();
                    if (CIFFilter.logger.isLoggable(Level.FINER)) {
                        CIFFilter.logger.finer(String.format(" Bonded atom 1: %d, Bonded atom 2: %d", new Object[]{atom0Index, atom1Index}));
                    }
                    CIFFilter.logger.fine(String.format(" Atom 0 index: %2d Atom 1 Index: %2d shiftIndex: %2d Value 0: %2d Value 1: %2d", new Object[]{atom0Index, atom1Index, shiftIndex, atom0Index - shiftIndex - 1, atom1Index - shiftIndex - 1}));
                    xyzCDKAtoms.addBond(atom0Index - shiftIndex - 1, atom1Index - shiftIndex - 1, order);
                }
                factory = AtomTypeFactory.getInstance((String)"org/openscience/cdk/config/data/jmol_atomtypes.txt", (IChemObjectBuilder)xyzCDKAtoms.getBuilder());
                xyzBond = xyzCDKAtoms.atoms().iterator();
                while (xyzBond.hasNext()) {
                    atom = (IAtom)xyzBond.next();
                    CIFFilter.setAtomTypes(factory, atom);
                    try {
                        factory.configure(atom);
                    }
                    catch (Exception ex) {
                        CIFFilter.logger.info(" Failed to configure atoms from CIF.\n" + String.valueOf(ex) + "\n" + Arrays.toString(ex.getStackTrace()));
                    }
                }
                zindices = new ArrayList<ArrayList<E>>();
                counter = 0;
                cifBonds = CIFFilter.bondAtoms(atoms, this.bondTolerance);
                if (CIFFilter.logger.isLoggable(Level.FINE)) {
                    CIFFilter.logger.fine(String.format(" Created %d bonds between CIF atoms (%d in input).", new Object[]{cifBonds, xyzBonds}));
                }
                atomPool = new ArrayList<ffx.potential.bonded.Atom>(Arrays.asList(atoms));
                try {
                    while (!atomPool.isEmpty()) {
                        molecule = new ArrayList<ffx.potential.bonded.Atom>();
                        CIFFilter.collectAtoms((ffx.potential.bonded.Atom)atomPool.get(0), molecule);
                        if (CIFFilter.logger.isLoggable(Level.FINER)) {
                            CIFFilter.logger.finer(String.format(" Molecule (%d) Size: %d", new Object[]{counter, molecule.size()}));
                        }
                        indices = new ArrayList<Integer>();
                        while (!molecule.isEmpty()) {
                            atom = molecule.remove(0);
                            indices.add(atom.getIndex());
                            atomPool.remove(atom);
                        }
                        if (CIFFilter.logger.isLoggable(Level.FINER)) {
                            CIFFilter.logger.finer(String.format(" Molecule %d: %d atoms are ready and %d remain (%d atoms in input, %d atoms in CIF). ", new Object[]{counter + 1, indices.size(), atomPool.size(), nInputAtoms, nAtoms}));
                        }
                        zindices.add(indices);
                        ++counter;
                    }
                }
                catch (Exception e) {
                    CIFFilter.logger.info(String.valueOf(e) + "\n" + Arrays.toString(e.getStackTrace()));
                    CIFFilter.logger.warning(" Failed to separate copies within the asymmetric unit.");
                    return false;
                }
                zPrime = zindices.size();
                cifCDKAtomsArr = new AtomContainer[zPrime];
                cifCDKAtoms = new AtomContainer();
                for (j = 0; j < zPrime; ++j) {
                    block85: {
                        block88: {
                            block86: {
                                block87: {
                                    if (zPrime > 1) {
                                        CIFFilter.logger.info(String.format("\n Attempting entity %d of %d", new Object[]{j + 1, zPrime}));
                                    }
                                    currentList = (ArrayList)zindices.get(j);
                                    cifMolAtoms = currentList.size();
                                    if (CIFFilter.logger.isLoggable(Level.FINE)) {
                                        CIFFilter.logger.fine(String.format(" CIF atoms in current: %d", new Object[]{cifMolAtoms}));
                                    }
                                    if (cifMolAtoms % numInputMolAtoms != 0 && cifMolAtoms % (numInputMolAtoms - numMolHydrogen) != 0) break block85;
                                    cifCDKAtomsArr[j] = new AtomContainer();
                                    for (Iterator<T> integer : currentList) {
                                        cifCDKAtomsArr[j].addAtom((IAtom)new Atom(atoms[integer.intValue() - 1].getSegID(), new Point3d(atoms[integer.intValue() - 1].getXYZ(null))));
                                    }
                                    nullCDKAtoms = new AtomContainer();
                                    for (IAtom atom : cifCDKAtomsArr[j].atoms()) {
                                        if (atom.toString() != null) continue;
                                        nullCDKAtoms.addAtom(atom);
                                    }
                                    cifCDKAtomsArr[j].remove((IAtomContainer)nullCDKAtoms);
                                    factory = AtomTypeFactory.getInstance((String)"org/openscience/cdk/config/data/jmol_atomtypes.txt", (IChemObjectBuilder)cifCDKAtomsArr[j].getBuilder());
                                    integer = cifCDKAtomsArr[j].atoms().iterator();
                                    while (integer.hasNext()) {
                                        atom = (IAtom)integer.next();
                                        CIFFilter.setAtomTypes(factory, atom);
                                        try {
                                            factory.configure(atom);
                                        }
                                        catch (Exception ex) {
                                            CIFFilter.logger.info(" Failed to configure CIF atoms.\n" + String.valueOf(ex) + "\n" + Arrays.toString(ex.getStackTrace()));
                                        }
                                    }
                                    rebonder = new RebondTool(2.0, 0.5, this.bondTolerance);
                                    try {
                                        rebonder.rebond((IAtomContainer)cifCDKAtomsArr[j]);
                                    }
                                    catch (Exception ex) {
                                        CIFFilter.logger.info("Failed to rebond CIF atoms.\n" + String.valueOf(ex) + "\n" + Arrays.toString(ex.getStackTrace()));
                                    }
                                    cifMolBonds = cifCDKAtomsArr[j].getBondCount();
                                    if (CIFFilter.logger.isLoggable(Level.FINE)) {
                                        CIFFilter.logger.fine(String.format(" Number of CIF bonds: %d (%d in input)", new Object[]{cifMolBonds, xyzBonds}));
                                    }
                                    if (cifMolBonds == 0 || cifMolBonds % xyzBonds != 0) break block86;
                                    pattern = VentoFoggia.findIdentical((IAtomContainer)xyzCDKAtoms, (AtomMatcher)AtomMatcher.forElement(), (BondMatcher)BondMatcher.forAny());
                                    p = pattern.match((IAtomContainer)cifCDKAtomsArr[j]);
                                    pLength = p.length;
                                    if (p == null || pLength != numInputMolAtoms) break block87;
                                    for (k = 0; k < pLength; ++k) {
                                        if (CIFFilter.logger.isLoggable(Level.FINEST)) {
                                            CIFFilter.logger.finest(String.format(" %d input %s -> CIF %s", new Object[]{k, xyzCDKAtoms.getAtom(k).getSymbol(), cifCDKAtomsArr[j].getAtom(p[k]).getSymbol()}));
                                        }
                                        point3d = cifCDKAtomsArr[j].getAtom(p[k]).getPoint3d();
                                        ((ffx.potential.bonded.Atom)currentAtoms.get(k)).setXYZ(new double[]{point3d.x, point3d.y, point3d.z});
                                    }
                                    break block88;
                                }
                                if (!CIFFilter.logger.isLoggable(Level.FINE)) continue;
                                CIFFilter.logger.fine(String.format(" Atoms from CIF (%d) and input (%d) structures don't match.", new Object[]{p.length, nAtoms}));
                                continue;
                            }
                            if (xyzBonds - numMolHydrogen != 0 && cifMolBonds % (xyzBonds - numMolHydrogen) != 0) ** GOTO lbl268
                            if (CIFFilter.logger.isLoggable(Level.FINE)) {
                                CIFFilter.logger.info(" CIF may contain implicit hydrogen -- attempting to patch.");
                            }
                            pattern = VentoFoggia.findSubstructure((IAtomContainer)cifCDKAtomsArr[j], (AtomMatcher)AtomMatcher.forElement(), (BondMatcher)BondMatcher.forAny());
                            p = pattern.match((IAtomContainer)xyzCDKAtoms);
                            pLength = p.length;
                            if (p != null && pLength == numInputMolAtoms - numMolHydrogen) {
                                for (k = 0; k < pLength; ++k) {
                                    point3d = cifCDKAtomsArr[j].getAtom(k).getPoint3d();
                                    ((ffx.potential.bonded.Atom)currentAtoms.get(p[k])).setXYZ(new double[]{point3d.x, point3d.y, point3d.z});
                                }
                                lastKnownAtom1 = null;
                                knownCount = 0;
                                for (ffx.potential.bonded.Atom hydrogen : currentAtoms) {
                                    if (!hydrogen.isHydrogen()) continue;
                                    bond0 = hydrogen.getBonds().get(0);
                                    atom1 = bond0.get1_2(hydrogen);
                                    angle0_2 = 0.0;
                                    anglesList = hydrogen.getAngles();
                                    atom2 = null;
                                    switch (anglesList.size()) {
                                        case 0: {
                                            hydrogen.moveTo(new double[]{atom1.getX() - 0.6, atom1.getY() - 0.6, atom1.getZ() - 0.6});
                                            break;
                                        }
                                        case 1: {
                                            for (Angle angle : anglesList) {
                                                atom2 = angle.get1_3(hydrogen);
                                                if (atom2 != null) {
                                                    angle0_2 = angle.angleType.angle[0];
                                                }
                                                if (angle0_2 == 0.0) continue;
                                                break;
                                            }
                                            if (!CIFFilter.$assertionsDisabled && atom2 == null) {
                                                throw new AssertionError();
                                            }
                                            bonds2 = atom2.getBonds();
                                            atom3 = bonds2.size() > 1 && atom1 == bonds2.get(0).get1_2(atom2) ? bonds2.get(1).get1_2(atom2) : bonds2.get(0).get1_2(atom2);
                                            diAng = DoubleMath.dihedralAngle((double[])hydrogen.getXYZ(null), (double[])atom1.getXYZ(null), (double[])atom2.getXYZ(null), (double[])atom3.getXYZ(null));
                                            if (atom1 != atom3) {
                                                BondedUtils.intxyz(hydrogen, atom1, bond0.bondType.distance, atom2, angle0_2, atom3, Math.toDegrees(diAng), 0);
                                                break;
                                            }
                                            coord = new double[]{atom2.getX(), atom2.getY(), atom3.getZ()};
                                            mag = FastMath.sqrt((double)(coord[0] * coord[0] + coord[1] * coord[1] + coord[2] * coord[2]));
                                            coord[0] = coord[0] / mag;
                                            coord[1] = coord[1] / mag;
                                            coord[2] = coord[2] / mag;
                                            hydrogen.moveTo(atom1.getX() - coord[0], atom1.getY() - coord[1], atom1.getZ() - coord[2]);
                                            break;
                                        }
                                        default: {
                                            atom2B = null;
                                            angle0_2B = 0.0;
                                            proposedAngle = 0.0;
                                            chiral = 1;
                                            for (Angle angle : anglesList) {
                                                proposedAtom = angle.get1_3(hydrogen);
                                                if (proposedAtom != null && !proposedAtom.isHydrogen()) {
                                                    proposedAngle = angle.angleType.angle[0];
                                                }
                                                if (proposedAngle != 0.0) {
                                                    if (angle0_2 != 0.0) {
                                                        atom2B = proposedAtom;
                                                        angle0_2B = proposedAngle;
                                                    } else {
                                                        atom2 = proposedAtom;
                                                        angle0_2 = proposedAngle;
                                                    }
                                                    proposedAngle = 0.0;
                                                }
                                                if (angle0_2 == 0.0 || angle0_2B == 0.0) continue;
                                                break;
                                            }
                                            if (lastKnownAtom1 == null || lastKnownAtom1 != atom1) {
                                                lastKnownAtom1 = atom1;
                                                knownCount = 0;
                                            } else {
                                                ++knownCount;
                                            }
                                            if (angle0_2B == 0.0) {
                                                chiral = 0;
                                                if (!CIFFilter.$assertionsDisabled && atom2 == null) {
                                                    throw new AssertionError();
                                                }
                                                bonds2 = atom2.getBonds();
                                                atom2B = atom1 == bonds2.get(0).get1_2(atom2) ? bonds2.get(1).get1_2(atom2) : bonds2.get(0).get1_2(atom2);
                                                angle0_2B = 180.0 - 120.0 * (double)knownCount;
                                            } else if (anglesList.size() == 2) {
                                                chiral = 3;
                                            } else if (knownCount == 1) {
                                                chiral = -1;
                                            }
                                            BondedUtils.intxyz(hydrogen, atom1, bond0.bondType.distance, atom2, angle0_2, atom2B, angle0_2B, chiral);
                                        }
                                    }
                                }
                            } else {
                                if (CIFFilter.logger.isLoggable(Level.FINE)) {
                                    CIFFilter.logger.fine(" Could not match heavy atoms between CIF and input.");
                                }
                                if (p == null || !CIFFilter.logger.isLoggable(Level.FINE)) continue;
                                CIFFilter.logger.fine(String.format(" Matched %d atoms out of %d in CIF (%d in input)", new Object[]{pLength, nAtoms, nInputAtoms - numMolHydrogen}));
                                continue;
lbl268:
                                // 1 sources

                                CIFFilter.logger.info(String.format(" CIF (%d) and input ([%d+%dH=]%d) have a different number of bonds.", new Object[]{cifMolBonds, xyzBonds - numMolHydrogen, numMolHydrogen, xyzBonds}));
                                continue;
                            }
                        }
                        cifCDKAtoms.add((IAtomContainer)cifCDKAtomsArr[j]);
                        molecule = mol instanceof Polymer != false ? new Polymer(((Polymer)mol).getChainID(), mol.getName(), true) : new Molecule(mol.getName());
                        atomList = new ArrayList<ffx.potential.bonded.Atom>();
                        for (ffx.potential.bonded.Atom atom : currentAtoms) {
                            if (CIFFilter.logger.isLoggable(Level.FINER)) {
                                CIFFilter.logger.finer(String.format(" Atom Residue: %s %2d", new Object[]{atom.getResidueName(), atom.getResidueNumber()}));
                            }
                            if (molecule instanceof Polymer) {
                                molAtom = new ffx.potential.bonded.Atom(atomIndex++, atom.getName(), atom.getAltLoc(), atom.getXYZ(null), atom.getResidueName(), atom.getResidueNumber(), atom.getChainID(), atom.getOccupancy(), atom.getTempFactor(), atom.getSegID());
                                molAtom.setAtomType(atom.getAtomType());
                            } else {
                                molAtom = new ffx.potential.bonded.Atom(atomIndex++, atom.getName(), atom.getAtomType(), atom.getXYZ(null));
                                if (atom.getResidueName() != null) {
                                    molAtom.setResName(atom.getResidueName());
                                }
                            }
                            atomList.add(molAtom);
                        }
                        bondList = mol.getBondList();
                        for (Bond bond : bondList) {
                            a1 = bond.getAtom(0);
                            a2 = bond.getAtom(1);
                            if (CIFFilter.logger.isLoggable(Level.FINE)) {
                                CIFFilter.logger.fine(String.format(" Bonded atom 1: %d, Bonded atom 2: %d", new Object[]{a1.getXyzIndex(), a2.getXyzIndex()}));
                            }
                            newA1 = (ffx.potential.bonded.Atom)atomList.get(a1.getIndex() - shiftIndex - 1);
                            newA2 = (ffx.potential.bonded.Atom)atomList.get(a2.getIndex() - shiftIndex - 1);
                            bond2 = new Bond(newA1, newA2);
                            bond2.setBondType(bond.getBondType());
                        }
                        res = null;
                        for (ffx.potential.bonded.Atom atom : atomList) {
                            if (molecule instanceof Polymer) {
                                if (res == null) {
                                    res = new Residue(atom.getResidueName(), atom.getResidueNumber(), ((Polymer)mol).getResidue(atom.getResidueNumber()).getResidueType());
                                } else if (res.getResidueNumber() != atom.getResidueNumber()) {
                                    if (CIFFilter.logger.isLoggable(Level.FINER)) {
                                        CIFFilter.logger.finer(" Added Residue " + res.getResidueNumber() + " " + res.getName());
                                    }
                                    molecule.addMSNode(res);
                                    res = new Residue(atom.getResidueName(), atom.getResidueNumber(), ((Polymer)mol).getResidue(atom.getResidueNumber()).getResidueType());
                                }
                                res.addMSNode(atom);
                                continue;
                            }
                            molecule.addMSNode(atom);
                        }
                        if (molecule instanceof Polymer && res != null) {
                            if (CIFFilter.logger.isLoggable(Level.FINER)) {
                                CIFFilter.logger.finer(" Added Final Residue " + res.getResidueNumber() + " " + res.getName());
                            }
                            molecule.addMSNode(res);
                        }
                        outputAssembly.addMSNode(molecule);
                        continue;
                    }
                    if (!CIFFilter.logger.isLoggable(Level.INFO)) continue;
                    CIFFilter.logger.info(String.format(" Number of atoms in CIF (%d) molecule do not match input (%d + %dH = %d).", new Object[]{cifMolAtoms, nInputAtoms - numMolHydrogen, numMolHydrogen, nInputAtoms}));
                }
            }
            if (CIFFilter.logger.isLoggable(Level.FINE)) {
                CIFFilter.logger.fine(String.format("\n Output Assembly Atoms: %d", new Object[]{outputAssembly.getAtomList().size()}));
            }
            outputAssembly.setPotential(this.activeMolecularAssembly.getPotentialEnergy());
            outputAssembly.setCrystal(this.crystal);
            outputAssembly.setForceField(this.activeMolecularAssembly.getForceField());
            outputAssembly.setFile(this.activeMolecularAssembly.getFile());
            outputAssembly.setName(this.activeMolecularAssembly.getName());
            this.setMolecularSystem(outputAssembly);
            if (outputAssembly.getAtomList().isEmpty()) {
                CIFFilter.logger.info(" Atom types could not be matched. File could not be written.");
            } else if (!this.writeOutputFile()) {
                CIFFilter.logger.info(" Input File could not be written.");
            }
            outputAssembly.destroy();
            this.sgNum = -1;
            this.sgName = null;
        }
        return true;
    }

    @Override
    public boolean readNext() {
        return this.readNext(false, true);
    }

    @Override
    public boolean readNext(boolean resetPosition) {
        return this.readNext(resetPosition, true);
    }

    @Override
    public boolean readNext(boolean resetPosition, boolean print) {
        return this.readNext(resetPosition, print, true);
    }

    @Override
    public boolean readNext(boolean resetPosition, boolean print, boolean parse) {
        CifCoreBlock currentBlock;
        List blocks = this.cifFile.getBlocks();
        if (!parse) {
            ++this.snapShot;
            if (print) {
                logger.info(String.format(" Skipped Block: %d", this.snapShot));
            }
            return true;
        }
        if (resetPosition) {
            currentBlock = (CifCoreBlock)blocks.get(0);
            this.snapShot = 0;
        } else if (++this.snapShot < blocks.size()) {
            currentBlock = (CifCoreBlock)blocks.get(this.snapShot);
        } else {
            if (print) {
                logger.info(" Reached end of available blocks in CIF file.");
            }
            return false;
        }
        if (print) {
            logger.info(" Current Block: " + currentBlock.getBlockHeader());
        }
        return true;
    }

    public static void setAtomTypes(AtomTypeFactory factory, IAtom atom) {
        String atomTypeName = atom.getAtomTypeName();
        if (atomTypeName == null || atomTypeName.isEmpty()) {
            IAtomType[] types = factory.getAtomTypes(atom.getSymbol());
            if (types.length > 0) {
                IAtomType atomType = types[0];
                atom.setAtomTypeName(atomType.getAtomTypeName());
            } else {
                logger.info(" No atom type found for " + String.valueOf(atom));
            }
        }
    }

    public void setBondTolerance(double bondTolerance) {
        this.bondTolerance = bondTolerance;
    }

    public void setFixLattice(boolean fixLattice) {
        this.fixLattice = fixLattice;
    }

    public void setSgName(String sgName) {
        this.sgName = sgName;
    }

    public void setSgNum(int sgNum) {
        this.sgNum = sgNum;
    }

    public void setZprime(int zPrime) {
        this.zPrime = zPrime;
    }

    public boolean writeFiles() {
        for (File file : this.files) {
            File saveFile = new File(FilenameUtils.removeExtension((String)file.getAbsolutePath()) + ".cif");
            if (this.writeFile(saveFile, true, null)) continue;
            return false;
        }
        return true;
    }

    @Override
    public boolean writeFile(File saveFile, boolean append) {
        return this.writeFile(saveFile, append, null);
    }

    @Override
    public boolean writeFile(File saveFile, boolean append, String[] extraLines) {
        try {
            int numEntities;
            if (!append) {
                SystemFilter.setVersioning(SystemFilter.Versioning.PREFIX);
                saveFile = CIFFilter.version(saveFile);
            }
            BufferedWriter bw = new BufferedWriter(new FileWriter(saveFile, append));
            if (extraLines != null) {
                for (String line : extraLines) {
                    line = line.replaceAll("\n", " ");
                    bw.write("### " + line + "\n");
                }
            }
            bw.write("\ndata_" + this.activeMolecularAssembly.getName());
            Crystal xtal = this.activeMolecularAssembly.getCrystal().getUnitCell();
            bw.write("\n_symmetry_cell_setting\t" + xtal.spaceGroup.latticeSystem.name().toLowerCase().replaceAll("_lattice", ""));
            bw.write("\n_symmetry_space_group_name_H-M\t'" + xtal.spaceGroup.shortName + "'");
            bw.write("\n_symmetry_Int_Tables_number\t" + xtal.spaceGroup.number);
            bw.write("\nloop_\n_symmetry_equiv_pos_site_id\n_symmetry_equiv_pos_as_xyz");
            int numSymOps = xtal.spaceGroup.getNumberOfSymOps();
            for (int i = 0; i < numSymOps; ++i) {
                bw.write(String.format("\n%d " + xtal.spaceGroup.getSymOp(i).toXYZString().toLowerCase().replaceAll(" +", ""), i));
            }
            bw.write(String.format("\n_cell_length_a\t%4.4f", xtal.a));
            bw.write(String.format("\n_cell_length_b\t%4.4f", xtal.b));
            bw.write(String.format("\n_cell_length_c\t%4.4f", xtal.c));
            bw.write(String.format("\n_cell_angle_alpha\t%4.4f", xtal.alpha));
            bw.write(String.format("\n_cell_angle_beta\t%4.4f", xtal.beta));
            bw.write(String.format("\n_cell_angle_gamma\t%4.4f", xtal.gamma));
            bw.write(String.format("\n_cell_volume\t%4.4f", xtal.volume));
            int n = numEntities = this.zPrime < 1 ? this.activeMolecularAssembly.getMolecules().size() : this.zPrime;
            if (numEntities > 1) {
                if (this.zPrime < 1) {
                    logger.info(String.format(" Molecules detected, guessing a Z' of %d. Set manually using --zp.", numEntities));
                }
                bw.write(String.format("\n_cell_formula_units_Z\t%3d", numEntities));
            }
            bw.write("\nloop_");
            bw.write("\n_atom_site_label");
            bw.write("\n_atom_site_type_symbol");
            bw.write("\n_atom_site_fract_x");
            bw.write("\n_atom_site_fract_y");
            bw.write("\n_atom_site_fract_z");
            ffx.potential.bonded.Atom[] atoms = this.activeMolecularAssembly.getAtomArray();
            int count = 1;
            for (ffx.potential.bonded.Atom atom : atoms) {
                String symbol;
                Object name = atom.getName();
                if (Objects.equals(name, symbol = PeriodicTable.getSymbol((int)atom.getAtomicNumber()))) {
                    name = (String)name + count++;
                }
                double[] xyzC = new double[]{atom.getX(), atom.getY(), atom.getZ()};
                double[] xyzF = new double[3];
                xtal.toFractionalCoordinates(xyzC, xyzF);
                bw.write(String.format("\n%-3s %2s %8.6f %8.6f %8.6f", name, symbol, xyzF[0], xyzF[1], xyzF[2]));
            }
            bw.write("\n#END\n");
            bw.close();
            logger.info(String.format("\n Wrote CIF file: %s \n", saveFile.getAbsolutePath()));
        }
        catch (Exception ex) {
            logger.info(String.format("\n Failed to write out CIF file: %s \n" + String.valueOf(ex), saveFile.getAbsolutePath()));
            return false;
        }
        if (!this.createdFiles.contains(saveFile.getAbsolutePath())) {
            this.createdFiles.add(saveFile.getAbsolutePath());
        }
        return true;
    }

    private boolean writeOutputFile() {
        File saveFile;
        String dir = FilenameUtils.getFullPath((String)((File)this.files.get(0)).getAbsolutePath()) + File.separator;
        Object fileName = FilenameUtils.removeExtension((String)FilenameUtils.getName((String)((File)this.files.get(0)).getAbsolutePath()));
        String spacegroup = this.activeMolecularAssembly.getCrystal() != null ? this.activeMolecularAssembly.getCrystal().getUnitCell().spaceGroup.shortName : null;
        List<MSNode> entities = this.activeMolecularAssembly.getAllBondedEntities();
        if (this.usePDB) {
            if (entities.size() > 1) {
                fileName = (String)fileName + "_z" + entities.size();
            }
            saveFile = new File(dir + (String)fileName + ".pdb");
        } else if (this.cifFile.getBlocks().size() > 1) {
            if (entities.size() > 1) {
                fileName = (String)fileName + "_z" + entities.size();
            }
            fileName = (String)fileName + "_" + spacegroup.replaceAll("\\/", "");
            saveFile = new File(dir + (String)fileName + ".arc");
        } else {
            saveFile = XYZFilter.version(new File(dir + (String)fileName + ".xyz"));
        }
        ForceField ff = this.activeMolecularAssembly.getForceField();
        CompositeConfiguration properties = this.activeMolecularAssembly.getProperties();
        if (this.usePDB) {
            PDBFilter pdbFilter = new PDBFilter(saveFile, this.activeMolecularAssembly, ff, properties);
            pdbFilter.writeFile(saveFile, true);
        } else {
            XYZFilter xyzFilter = new XYZFilter(saveFile, this.activeMolecularAssembly, ff, properties);
            xyzFilter.writeFile(saveFile, true);
        }
        logger.info("\n Saved output file:        " + saveFile.getAbsolutePath());
        this.writePropertiesFile(dir, (String)fileName, spacegroup, entities);
        if (!this.createdFiles.contains(saveFile.getAbsolutePath())) {
            this.createdFiles.add(saveFile.getAbsolutePath());
        }
        return true;
    }

    public boolean writeOutputFile(MolecularAssembly ma) {
        this.setMolecularSystem(ma);
        return this.writeOutputFile();
    }

    public void writePropertiesFile(String dir, String fileName, String spacegroup, List<MSNode> entities) {
        File propertyFile = new File(dir + fileName + ".properties");
        if (!propertyFile.exists()) {
            try {
                String patch;
                String forceFieldProperty;
                FileWriter fw = new FileWriter(propertyFile, false);
                BufferedWriter bw = new BufferedWriter(fw);
                ForceField forceField = this.activeMolecularAssembly.getForceField();
                String parameters = forceField.getString("parameters", "none");
                if (parameters != null && !parameters.equalsIgnoreCase("none")) {
                    bw.write(String.format("parameters %s\n", parameters));
                }
                if ((forceFieldProperty = forceField.getString("forcefield", "none")) != null && !forceFieldProperty.equalsIgnoreCase("none")) {
                    bw.write(String.format("forcefield %s\n", forceFieldProperty));
                }
                if ((patch = forceField.getString("patch", "none")) != null && !patch.equalsIgnoreCase("none")) {
                    bw.write(String.format("patch %s\n", patch));
                }
                if (spacegroup != null) {
                    bw.write(String.format("spacegroup %s\n", spacegroup));
                }
                if (entities.size() > 1) {
                    bw.write("intermolecular-softcore true\n");
                }
                logger.info("\n Saved properties file: " + propertyFile.getAbsolutePath() + "\n");
                bw.close();
            }
            catch (Exception ex) {
                logger.info("Failed to write files.\n" + String.valueOf(ex));
            }
        } else {
            logger.info("\n Property file already exists:  " + propertyFile.getAbsolutePath() + "\n");
        }
    }
}

