/*
 * Decompiled with CFR 0.152.
 */
package ffx.potential.parameters;

import ffx.potential.parameters.AtomType;
import ffx.potential.parameters.BaseType;
import ffx.potential.parameters.ForceField;
import ffx.utilities.FFXProperties;
import ffx.utilities.FFXProperty;
import ffx.utilities.PropertyGroup;
import java.util.Arrays;
import java.util.Comparator;
import java.util.HashMap;
import java.util.Map;
import java.util.logging.Level;
import java.util.logging.Logger;
import javax.annotation.Nullable;
import org.apache.commons.math3.util.FastMath;
import org.w3c.dom.Document;
import org.w3c.dom.Element;

@FFXProperties(value={@FFXProperty(name="improper", clazz=String.class, propertyGroup=PropertyGroup.PotentialFunctionParameter, description="[4 integers and 2 reals]\"\nProvides the values for a single CHARMM-style improper dihedral angle parameter.\nThe integer modifiers give the atom class numbers for the four kinds of atoms involved in the torsion which is to be defined.\nThe real number modifiers give the force constant value for the deviation from the target improper torsional angle, and the target value for the torsional angle, respectively.\nThe default units for the improper force constant are kcal/mole/radian^2, but this can be controlled via the impropunit keyword.\n"), @FFXProperty(name="torsion", clazz=String.class, propertyGroup=PropertyGroup.PotentialFunctionParameter, description="[4 integers and up to 6 real/real/integer triples]\nProvides the values for a single torsional angle parameter.\nThe first four integer modifiers give the atom class numbers for the atoms involved in the torsional angle to be defined.\nEach of the remaining triples of real/real/integer modifiers give the amplitude,\nphase offset in degrees and periodicity of a particular torsional function term, respectively.\nPeriodicities through 6-fold are allowed for torsional parameters.\n")})
public final class TorsionType
extends BaseType
implements Comparator<String> {
    private static final Logger logger = Logger.getLogger(TorsionType.class.getName());
    public final int[] atomClasses;
    public final int terms;
    public final double[] amplitude;
    public final double[] phase;
    public final double[] cosine;
    public final double[] sine;
    public final int[] periodicity;
    private final TorsionMode torsionMode;
    public static final double DEFAULT_TORSION_UNIT = 1.0;
    @FFXProperty(name="torsionunit", propertyGroup=PropertyGroup.EnergyUnitConversion, defaultValue="1.0", description="Sets the scale factor needed to convert the energy value computed by the torsional angle potential into units of kcal/mole.\nThe correct value is force field dependent and typically provided in the header of the master force field parameter file.\n")
    public double torsionUnit = 1.0;

    public TorsionType(int[] atomClasses, double[] amplitude, double[] phase, int[] periodicity) {
        this(atomClasses, amplitude, phase, periodicity, TorsionMode.NORMAL);
    }

    public TorsionType(int[] atomClasses, double[] amplitude, double[] phase, int[] periodicity, TorsionMode torsionMode) {
        super(ForceField.ForceFieldType.TORSION, TorsionType.sortKey(atomClasses));
        int i;
        this.atomClasses = atomClasses;
        int max = 1;
        for (int i1 : periodicity) {
            if (i1 <= max) continue;
            max = i1;
        }
        this.terms = max;
        this.amplitude = new double[this.terms];
        this.phase = new double[this.terms];
        this.periodicity = new int[this.terms];
        for (i = 0; i < this.terms; ++i) {
            this.periodicity[i] = i + 1;
        }
        for (i = 0; i < amplitude.length; ++i) {
            int j = periodicity[i] - 1;
            if (j < 0 || j >= this.terms) continue;
            this.amplitude[j] = amplitude[i];
            this.phase[j] = phase[i];
        }
        this.cosine = new double[this.terms];
        this.sine = new double[this.terms];
        for (i = 0; i < this.terms; ++i) {
            double angle = FastMath.toRadians((double)this.phase[i]);
            this.cosine[i] = FastMath.cos((double)angle);
            this.sine[i] = FastMath.sin((double)angle);
        }
        this.torsionMode = torsionMode;
        if (torsionMode == TorsionMode.IMPROPER) {
            this.forceFieldType = ForceField.ForceFieldType.IMPROPER;
        }
    }

    public static TorsionType average(@Nullable TorsionType torsionType1, @Nullable TorsionType torsionType2, @Nullable int[] atomClasses) {
        if (torsionType1 == null || torsionType2 == null || atomClasses == null) {
            return null;
        }
        int len = torsionType1.amplitude.length;
        if (len != torsionType2.amplitude.length) {
            return null;
        }
        double[] amplitude = new double[len];
        double[] phase = new double[len];
        int[] periodicity = new int[len];
        for (int i = 0; i < len; ++i) {
            amplitude[i] = (torsionType1.amplitude[i] + torsionType2.amplitude[i]) / 2.0;
            phase[i] = (torsionType1.phase[i] + torsionType2.phase[i]) / 2.0;
            periodicity[i] = (torsionType1.periodicity[i] + torsionType2.periodicity[i]) / 2;
        }
        return new TorsionType(atomClasses, amplitude, phase, periodicity);
    }

    public static TorsionType parse(String input, String[] tokens) {
        if (tokens.length < 5) {
            logger.log(Level.WARNING, "Invalid TORSION type:\n{0}", input);
        } else {
            try {
                int[] atomClasses = new int[]{Integer.parseInt(tokens[1]), Integer.parseInt(tokens[2]), Integer.parseInt(tokens[3]), Integer.parseInt(tokens[4])};
                int terms = (tokens.length - 5) / 3;
                double[] amplitude = new double[terms];
                double[] phase = new double[terms];
                int[] periodicity = new int[terms];
                int index = 5;
                for (int i = 0; i < terms; ++i) {
                    amplitude[i] = Double.parseDouble(tokens[index++]);
                    phase[i] = Double.parseDouble(tokens[index++]);
                    periodicity[i] = Integer.parseInt(tokens[index++]);
                }
                return new TorsionType(atomClasses, amplitude, phase, periodicity);
            }
            catch (NumberFormatException e) {
                String message = "NumberFormatException parsing TORSION type:\n" + input + "\n";
                logger.log(Level.SEVERE, message, e);
            }
            catch (Exception e) {
                String message = "Exception parsing TORSION type:\n" + input + "\n";
                logger.log(Level.SEVERE, message, e);
            }
        }
        return null;
    }

    public static TorsionType parseImproper(String input, String[] tokens) {
        if (tokens.length < 5) {
            logger.log(Level.WARNING, "Invalid IMPROPER type:\n{0}", input);
        } else {
            try {
                int[] atomClasses = new int[]{Integer.parseInt(tokens[1]), Integer.parseInt(tokens[2]), Integer.parseInt(tokens[3]), Integer.parseInt(tokens[4])};
                double[] amplitude = new double[1];
                double[] phase = new double[1];
                int[] periodicity = new int[1];
                int index = 5;
                amplitude[0] = Double.parseDouble(tokens[index++]);
                phase[0] = Double.parseDouble(tokens[index]);
                periodicity[0] = 1;
                return new TorsionType(atomClasses, amplitude, phase, periodicity, TorsionMode.IMPROPER);
            }
            catch (NumberFormatException e) {
                String message = "Exception parsing IMPROPER type:\n" + input + "\n";
                logger.log(Level.SEVERE, message, e);
            }
        }
        return null;
    }

    public static String sortKey(int[] c) {
        if (c == null || c.length != 4) {
            return null;
        }
        if (c[1] >= c[2]) {
            if (c[2] < c[1]) {
                int temp = c[0];
                c[0] = c[3];
                c[3] = temp;
                temp = c[1];
                c[1] = c[2];
                c[2] = temp;
            } else if (c[1] == c[2]) {
                if (c[0] > c[3]) {
                    int temp = c[0];
                    c[0] = c[3];
                    c[3] = temp;
                }
            } else if (c[0] > c[3]) {
                int temp = c[0];
                c[0] = c[3];
                c[3] = temp;
                temp = c[1];
                c[1] = c[2];
                c[2] = temp;
            }
        }
        return c[0] + " " + c[1] + " " + c[2] + " " + c[3];
    }

    @Override
    public int compare(String s1, String s2) {
        String[] keys1 = s1.split(" ");
        String[] keys2 = s2.split(" ");
        int[] c1 = new int[4];
        int[] c2 = new int[4];
        for (int i = 0; i < 4; ++i) {
            c1[i] = Integer.parseInt(keys1[i]);
            c2[i] = Integer.parseInt(keys2[i]);
        }
        if (c1[1] < c2[1]) {
            return -1;
        }
        if (c1[1] > c2[1]) {
            return 1;
        }
        if (c1[2] < c2[2]) {
            return -1;
        }
        if (c1[2] > c2[2]) {
            return 1;
        }
        if (c1[0] < c2[0]) {
            return -1;
        }
        if (c1[0] > c2[0]) {
            return 1;
        }
        if (c1[3] < c2[3]) {
            return -1;
        }
        if (c1[3] > c2[3]) {
            return 1;
        }
        return 0;
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || this.getClass() != o.getClass()) {
            return false;
        }
        TorsionType torsionType = (TorsionType)o;
        return Arrays.equals(this.atomClasses, torsionType.atomClasses);
    }

    public int hashCode() {
        return Arrays.hashCode(this.atomClasses);
    }

    public void incrementClasses(int increment) {
        for (int i = 0; i < this.atomClasses.length; ++i) {
            if (this.atomClasses[i] == 0) continue;
            int n = i;
            this.atomClasses[n] = this.atomClasses[n] + increment;
        }
        this.setKey(TorsionType.sortKey(this.atomClasses));
    }

    public TorsionType patchClasses(HashMap<AtomType, AtomType> typeMap) {
        int count = 0;
        int len = this.atomClasses.length;
        for (AtomType newType : typeMap.keySet()) {
            for (int atomClass : this.atomClasses) {
                if (atomClass != newType.atomClass) continue;
                ++count;
            }
        }
        if (count == 1 || count == 2 || count == 3) {
            int[] newClasses = Arrays.copyOf(this.atomClasses, len);
            for (AtomType newType : typeMap.keySet()) {
                for (int i = 0; i < len; ++i) {
                    if (this.atomClasses[i] != newType.atomClass) continue;
                    AtomType knownType = typeMap.get(newType);
                    newClasses[i] = knownType.atomClass;
                }
            }
            return new TorsionType(newClasses, this.amplitude, this.phase, this.periodicity);
        }
        return null;
    }

    @Override
    public String toString() {
        StringBuilder torsionBuffer = this.torsionMode == TorsionMode.IMPROPER ? new StringBuilder("improper") : new StringBuilder("torsion");
        for (int i : this.atomClasses) {
            torsionBuffer.append(String.format(" %5d", i));
        }
        boolean nonZero = false;
        for (double v : this.amplitude) {
            if (v == 0.0) continue;
            nonZero = true;
            break;
        }
        for (int i = 0; i < this.amplitude.length; ++i) {
            if (this.amplitude[i] == 0.0 && (i > 0 || nonZero)) continue;
            if (this.torsionMode == TorsionMode.NORMAL) {
                torsionBuffer.append(String.format(" %7.3f %7.3f %1d", this.amplitude[i], this.phase[i], this.periodicity[i]));
                continue;
            }
            torsionBuffer.append(String.format(" %7.3f %7.3f", this.amplitude[i], this.phase[i]));
        }
        return torsionBuffer.toString();
    }

    public static Element getXMLForce(Document doc, ForceField forceField) {
        Map<String, TorsionType> types = forceField.getTorsionTypes();
        if (!types.values().isEmpty()) {
            Element node = doc.createElement("PeriodicTorsionForce");
            double torsionUnit = forceField.getDouble("torsionunit", 1.0);
            for (TorsionType torsionType : types.values()) {
                node.appendChild(torsionType.toXML(doc, torsionUnit));
            }
            return node;
        }
        return null;
    }

    public Element toXML(Document doc, Double torsUnit) {
        Element node = doc.createElement("Proper");
        node.setAttribute("class1", String.format("%d", this.atomClasses[0]));
        node.setAttribute("class2", String.format("%d", this.atomClasses[1]));
        node.setAttribute("class3", String.format("%d", this.atomClasses[2]));
        node.setAttribute("class4", String.format("%d", this.atomClasses[3]));
        for (int i = 0; i < this.amplitude.length; ++i) {
            node.setAttribute(String.format("k%d", i + 1), String.format("%.17f", this.amplitude[i] * 4.184 * torsUnit));
            node.setAttribute(String.format("phase%d", i + 1), String.format("%.17f", this.phase[i] / 57.29577951308232));
            node.setAttribute(String.format("periodicity%d", i + 1), String.format("%d", this.periodicity[i]));
        }
        return node;
    }

    public static enum TorsionMode {
        NORMAL,
        IMPROPER;

    }
}

