/*
 * Decompiled with CFR 0.152.
 */
package ffx.potential.parameters;

import ffx.numerics.math.DoubleMath;
import ffx.potential.bonded.Atom;
import ffx.potential.parameters.AtomType;
import ffx.potential.parameters.BaseType;
import ffx.potential.parameters.ForceField;
import ffx.potential.parameters.PolarizeType;
import ffx.utilities.FFXProperty;
import ffx.utilities.PropertyGroup;
import java.io.BufferedReader;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Comparator;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.logging.Level;
import java.util.logging.Logger;
import org.apache.commons.math3.util.FastMath;
import org.w3c.dom.Document;
import org.w3c.dom.Element;

@FFXProperty(name="multipole", clazz=String[].class, propertyGroup=PropertyGroup.PotentialFunctionParameter, description="[5 lines with: 3 or 4 integers and 1 real; 3 reals; 1 real; 2 reals; 3 reals]\nProvides the values for a set of atomic multipole parameters at a single site.\nA complete keyword entry consists of five consecutive lines, the first line containing the multipole keyword and the four following lines.\nThe first line contains three integers which define the atom type on which the multipoles are centered,\nand the Z-axis and X-axis defining atom types for this center.\nThe optional fourth integer contains the Y-axis defining atom type, and is only required for locally chiral multipole sites.\nThe real number on the first line gives the monopole (atomic charge) in electrons.\nThe second line contains three real numbers which give the X-, Y- and Z-components of the atomic dipole in electron-Ang.\nThe final three lines, consisting of one, two and three real numbers give the upper triangle of the\ntraceless atomic quadrupole tensor in electron-Ang^2.\n")
public final class MultipoleType
extends BaseType
implements Comparator<String> {
    public static final double[] zeroM = new double[10];
    public static final double[] zeroD = new double[3];
    public static final double[][] zeroQ = new double[3][3];
    public static final int t000 = 0;
    public static final int t100 = 1;
    public static final int t010 = 2;
    public static final int t001 = 3;
    public static final int t200 = 4;
    public static final int t020 = 5;
    public static final int t002 = 6;
    public static final int t110 = 7;
    public static final int t101 = 8;
    public static final int t011 = 9;
    public static final int t300 = 10;
    public static final int t030 = 11;
    public static final int t003 = 12;
    public static final int t210 = 13;
    public static final int t201 = 14;
    public static final int t120 = 15;
    public static final int t021 = 16;
    public static final int t102 = 17;
    public static final int t012 = 18;
    public static final int t111 = 19;
    private static final Logger logger = Logger.getLogger(MultipoleType.class.getName());
    public static final double DEFAULT_MPOLE_12_SCALE = 0.0;
    public static final double DEFAULT_MPOLE_13_SCALE = 0.0;
    public static final double DEFAULT_MPOLE_14_SCALE = 1.0;
    public static final double DEFAULT_MPOLE_15_SCALE = 1.0;
    public final double charge;
    public final double[] dipole;
    public final double[][] quadrupole;
    public final MultipoleFrameDefinition frameDefinition;
    public final int[] frameAtomTypes;
    private final double[] multipole;

    public MultipoleType(double[] multipole, int[] frameAtomTypes, MultipoleFrameDefinition frameDefinition, boolean convertFromBohr) {
        super(ForceField.ForceFieldType.MULTIPOLE, frameAtomTypes);
        this.multipole = convertFromBohr ? MultipoleType.bohrToElectronAngstroms(multipole) : multipole;
        this.frameAtomTypes = frameAtomTypes;
        this.frameDefinition = frameDefinition;
        this.charge = multipole[0];
        this.dipole = MultipoleType.unpackDipole(multipole);
        this.quadrupole = MultipoleType.unpackQuad(multipole);
        this.checkMultipole();
    }

    public MultipoleType(MultipoleType multipoleType, double[] multipole) {
        this(multipole, Arrays.copyOf(multipoleType.frameAtomTypes, multipoleType.frameAtomTypes.length), multipoleType.frameDefinition, false);
    }

    public MultipoleType(double charge, double[] dipole, double[][] quadrupole, int[] frameAtomTypes, MultipoleFrameDefinition frameDefinition, boolean convertFromBohr) {
        super(ForceField.ForceFieldType.MULTIPOLE, frameAtomTypes);
        this.charge = charge;
        if (convertFromBohr) {
            this.multipole = MultipoleType.bohrToElectronAngstroms(MultipoleType.pack(charge, dipole, quadrupole));
            this.dipole = MultipoleType.unpackDipole(this.multipole);
            this.quadrupole = MultipoleType.unpackQuad(this.multipole);
        } else {
            this.multipole = MultipoleType.pack(charge, dipole, quadrupole);
            this.dipole = dipole;
            this.quadrupole = quadrupole;
        }
        this.frameAtomTypes = frameAtomTypes;
        this.frameDefinition = frameDefinition;
        this.checkMultipole();
    }

    public static boolean assignMultipole(ForceField.ELEC_FORM elecForm, Atom atom, ForceField forceField, double[] multipole, int i, int[][] axisAtom, MultipoleFrameDefinition[] frame) {
        MultipoleType type = MultipoleType.multipoleTypeFactory(elecForm, atom, forceField);
        if (type == null) {
            return false;
        }
        System.arraycopy(type.getMultipole(), 0, multipole, 0, 10);
        axisAtom[i] = atom.getAxisAtomIndices();
        frame[i] = atom.getMultipoleType().frameDefinition;
        return true;
    }

    public static MultipoleType averageTypes(MultipoleType multipoleType1, MultipoleType multipoleType2, int[] multipoleFrameTypes) {
        if (multipoleType1.frameDefinition != multipoleType2.frameDefinition) {
            return null;
        }
        MultipoleType[] types = new MultipoleType[]{multipoleType1, multipoleType2};
        double[] weights = new double[]{0.5, 0.5};
        double[] averagedMultipole = MultipoleType.weightMultipole(types, weights);
        if (averagedMultipole == null) {
            return null;
        }
        return new MultipoleType(averagedMultipole, multipoleFrameTypes, multipoleType1.frameDefinition, false);
    }

    public static boolean checkMultipoleChirality(MultipoleFrameDefinition frame, double[] localOrigin, double[][] frameCoords) {
        if (frame != MultipoleFrameDefinition.ZTHENX) {
            return false;
        }
        double[] zAxis = new double[3];
        double[] xAxis = new double[3];
        double[] yAxis = new double[3];
        double[] yMinOrigin = new double[3];
        zAxis[0] = frameCoords[0][0];
        zAxis[1] = frameCoords[0][1];
        zAxis[2] = frameCoords[0][2];
        xAxis[0] = frameCoords[1][0];
        xAxis[1] = frameCoords[1][1];
        xAxis[2] = frameCoords[1][2];
        yAxis[0] = frameCoords[2][0];
        yAxis[1] = frameCoords[2][1];
        yAxis[2] = frameCoords[2][2];
        DoubleMath.sub((double[])localOrigin, (double[])yAxis, (double[])yMinOrigin);
        DoubleMath.sub((double[])zAxis, (double[])yAxis, (double[])zAxis);
        DoubleMath.sub((double[])xAxis, (double[])yAxis, (double[])xAxis);
        double c1 = zAxis[1] * xAxis[2] - zAxis[2] * xAxis[1];
        double c2 = xAxis[1] * yMinOrigin[2] - xAxis[2] * yMinOrigin[1];
        double c3 = yMinOrigin[1] * zAxis[2] - yMinOrigin[2] * zAxis[1];
        double vol = yMinOrigin[0] * c1 + zAxis[0] * c2 + xAxis[0] * c3;
        return vol < 0.0;
    }

    public static double[][] getRotationMatrix(MultipoleFrameDefinition frame, double[] localOrigin, double[][] frameCoords) {
        double[][] rotMat = new double[3][3];
        MultipoleType.getRotationMatrix(frame, localOrigin, frameCoords, rotMat);
        return rotMat;
    }

    public static void getRotationMatrix(MultipoleFrameDefinition frame, double[] localOrigin, double[][] frameCoords, double[][] rotMat) {
        double[] zAxis = new double[3];
        double[] xAxis = new double[3];
        double[] yAxis = new double[3];
        rotMat[0][0] = 1.0;
        rotMat[1][0] = 0.0;
        rotMat[2][0] = 0.0;
        rotMat[0][1] = 0.0;
        rotMat[1][1] = 1.0;
        rotMat[2][1] = 0.0;
        rotMat[0][2] = 0.0;
        rotMat[1][2] = 0.0;
        rotMat[2][2] = 1.0;
        switch (frame.ordinal()) {
            case 0: {
                return;
            }
            case 3: {
                zAxis[0] = frameCoords[0][0];
                zAxis[1] = frameCoords[0][1];
                zAxis[2] = frameCoords[0][2];
                xAxis[0] = frameCoords[1][0];
                xAxis[1] = frameCoords[1][1];
                xAxis[2] = frameCoords[1][2];
                DoubleMath.sub((double[])zAxis, (double[])localOrigin, (double[])zAxis);
                DoubleMath.normalize((double[])zAxis, (double[])zAxis);
                DoubleMath.sub((double[])xAxis, (double[])localOrigin, (double[])xAxis);
                DoubleMath.normalize((double[])xAxis, (double[])xAxis);
                DoubleMath.add((double[])xAxis, (double[])zAxis, (double[])zAxis);
                DoubleMath.normalize((double[])zAxis, (double[])zAxis);
                rotMat[0][2] = zAxis[0];
                rotMat[1][2] = zAxis[1];
                rotMat[2][2] = zAxis[2];
                double dot = DoubleMath.dot((double[])xAxis, (double[])zAxis);
                DoubleMath.scale((double[])zAxis, (double)dot, (double[])zAxis);
                DoubleMath.sub((double[])xAxis, (double[])zAxis, (double[])xAxis);
                DoubleMath.normalize((double[])xAxis, (double[])xAxis);
                break;
            }
            case 4: {
                zAxis[0] = frameCoords[0][0];
                zAxis[1] = frameCoords[0][1];
                zAxis[2] = frameCoords[0][2];
                xAxis[0] = frameCoords[1][0];
                xAxis[1] = frameCoords[1][1];
                xAxis[2] = frameCoords[1][2];
                yAxis[0] = frameCoords[2][0];
                yAxis[1] = frameCoords[2][1];
                yAxis[2] = frameCoords[2][2];
                DoubleMath.sub((double[])zAxis, (double[])localOrigin, (double[])zAxis);
                DoubleMath.normalize((double[])zAxis, (double[])zAxis);
                rotMat[0][2] = zAxis[0];
                rotMat[1][2] = zAxis[1];
                rotMat[2][2] = zAxis[2];
                DoubleMath.sub((double[])xAxis, (double[])localOrigin, (double[])xAxis);
                DoubleMath.normalize((double[])xAxis, (double[])xAxis);
                DoubleMath.sub((double[])yAxis, (double[])localOrigin, (double[])yAxis);
                DoubleMath.normalize((double[])yAxis, (double[])yAxis);
                DoubleMath.add((double[])xAxis, (double[])yAxis, (double[])xAxis);
                DoubleMath.normalize((double[])xAxis, (double[])xAxis);
                double dot = DoubleMath.dot((double[])xAxis, (double[])zAxis);
                DoubleMath.scale((double[])zAxis, (double)dot, (double[])zAxis);
                DoubleMath.sub((double[])xAxis, (double[])zAxis, (double[])xAxis);
                DoubleMath.normalize((double[])xAxis, (double[])xAxis);
                break;
            }
            case 1: {
                zAxis[0] = frameCoords[0][0];
                zAxis[1] = frameCoords[0][1];
                zAxis[2] = frameCoords[0][2];
                DoubleMath.sub((double[])zAxis, (double[])localOrigin, (double[])zAxis);
                DoubleMath.normalize((double[])zAxis, (double[])zAxis);
                rotMat[0][2] = zAxis[0];
                rotMat[1][2] = zAxis[1];
                rotMat[2][2] = zAxis[2];
                xAxis[0] = 1.0;
                xAxis[1] = 0.0;
                xAxis[2] = 0.0;
                double dot = rotMat[0][2];
                if (FastMath.abs((double)dot) > 0.866) {
                    xAxis[0] = 0.0;
                    xAxis[1] = 1.0;
                    dot = rotMat[1][2];
                }
                DoubleMath.scale((double[])zAxis, (double)dot, (double[])zAxis);
                DoubleMath.sub((double[])xAxis, (double[])zAxis, (double[])xAxis);
                DoubleMath.normalize((double[])xAxis, (double[])xAxis);
                break;
            }
            case 5: {
                zAxis[0] = frameCoords[0][0];
                zAxis[1] = frameCoords[0][1];
                zAxis[2] = frameCoords[0][2];
                DoubleMath.sub((double[])zAxis, (double[])localOrigin, (double[])zAxis);
                DoubleMath.normalize((double[])zAxis, (double[])zAxis);
                xAxis[0] = frameCoords[1][0];
                xAxis[1] = frameCoords[1][1];
                xAxis[2] = frameCoords[1][2];
                DoubleMath.sub((double[])xAxis, (double[])localOrigin, (double[])xAxis);
                DoubleMath.normalize((double[])xAxis, (double[])xAxis);
                yAxis[0] = frameCoords[2][0];
                yAxis[1] = frameCoords[2][1];
                yAxis[2] = frameCoords[2][2];
                DoubleMath.sub((double[])yAxis, (double[])localOrigin, (double[])yAxis);
                DoubleMath.normalize((double[])yAxis, (double[])yAxis);
                double[] sum = new double[3];
                DoubleMath.add((double[])zAxis, (double[])xAxis, (double[])sum);
                DoubleMath.add((double[])sum, (double[])yAxis, (double[])sum);
                DoubleMath.normalize((double[])sum, (double[])sum);
                rotMat[0][2] = sum[0];
                rotMat[1][2] = sum[1];
                rotMat[2][2] = sum[2];
                double dot = DoubleMath.dot((double[])zAxis, (double[])sum);
                DoubleMath.scale((double[])sum, (double)dot, (double[])sum);
                DoubleMath.sub((double[])zAxis, (double[])sum, (double[])xAxis);
                DoubleMath.normalize((double[])xAxis, (double[])xAxis);
                break;
            }
            case 2: {
                zAxis[0] = frameCoords[0][0];
                zAxis[1] = frameCoords[0][1];
                zAxis[2] = frameCoords[0][2];
                xAxis[0] = frameCoords[1][0];
                xAxis[1] = frameCoords[1][1];
                xAxis[2] = frameCoords[1][2];
                DoubleMath.sub((double[])zAxis, (double[])localOrigin, (double[])zAxis);
                DoubleMath.normalize((double[])zAxis, (double[])zAxis);
                rotMat[0][2] = zAxis[0];
                rotMat[1][2] = zAxis[1];
                rotMat[2][2] = zAxis[2];
                DoubleMath.sub((double[])xAxis, (double[])localOrigin, (double[])xAxis);
                double dot = DoubleMath.dot((double[])xAxis, (double[])zAxis);
                DoubleMath.scale((double[])zAxis, (double)dot, (double[])zAxis);
                DoubleMath.sub((double[])xAxis, (double[])zAxis, (double[])xAxis);
                DoubleMath.normalize((double[])xAxis, (double[])xAxis);
                break;
            }
            default: {
                throw new IllegalStateException("Unexpected value: " + String.valueOf((Object)frame));
            }
        }
        rotMat[0][0] = xAxis[0];
        rotMat[1][0] = xAxis[1];
        rotMat[2][0] = xAxis[2];
        rotMat[0][1] = rotMat[2][0] * rotMat[1][2] - rotMat[1][0] * rotMat[2][2];
        rotMat[1][1] = rotMat[0][0] * rotMat[2][2] - rotMat[2][0] * rotMat[0][2];
        rotMat[2][1] = rotMat[1][0] * rotMat[0][2] - rotMat[0][0] * rotMat[1][2];
    }

    public static MultipoleType multipoleTypeFactory(ForceField.ELEC_FORM elecForm, Atom atom, ForceField forceField) {
        String key3;
        String key2;
        String key1;
        MultipoleType multipoleType;
        AtomType atomType = atom.getAtomType();
        if (atomType == null) {
            String message = " Multipoles can only be assigned to atoms that have been typed.";
            logger.severe(message);
            return null;
        }
        if (elecForm != ForceField.ELEC_FORM.FIXED_CHARGE) {
            PolarizeType polarizeType = forceField.getPolarizeType(atomType.getKey());
            if (polarizeType != null) {
                atom.setPolarizeType(polarizeType);
            } else {
                String message = " No polarization type was found for " + String.valueOf(atom);
                logger.info(message);
                double polarizability = 0.0;
                double thole = 0.0;
                double ddp = 0.0;
                polarizeType = new PolarizeType(atomType.type, polarizability, thole, ddp, null);
                forceField.addForceFieldType(polarizeType);
                atom.setPolarizeType(polarizeType);
            }
        }
        if ((multipoleType = forceField.getMultipoleType(key1 = atomType.getKey())) != null) {
            atom.setMultipoleType(multipoleType);
            atom.setAxisAtoms(null);
            return multipoleType;
        }
        List<Atom> n12 = atom.get12List();
        if (n12 == null || n12.isEmpty()) {
            String message = "Multipoles can only be assigned after bonded relationships are defined.\n";
            logger.severe(message);
            return null;
        }
        for (Atom atom2 : n12) {
            String key = key1 + " " + atom2.getAtomType().getKey();
            multipoleType = forceField.getMultipoleType(key);
            if (multipoleType == null) continue;
            atom.setMultipoleType(multipoleType);
            atom.setAxisAtoms(atom2);
            return multipoleType;
        }
        for (Atom atom2 : n12) {
            key2 = atom2.getAtomType().getKey();
            for (Atom atom3 : n12) {
                Object key;
                if (atom2 == atom3 || (multipoleType = forceField.getMultipoleType((String)(key = key1 + " " + key2 + " " + (key3 = atom3.getAtomType().getKey())))) == null) continue;
                atom.setMultipoleType(multipoleType);
                atom.setAxisAtoms(atom2, atom3);
                return multipoleType;
            }
        }
        for (Atom atom2 : n12) {
            key2 = atom2.getAtomType().getKey();
            for (Atom atom3 : n12) {
                if (atom2 == atom3) continue;
                key3 = atom3.getAtomType().getKey();
                for (Atom atom4 : n12) {
                    String key4;
                    String key;
                    if (atom2 == atom4 || atom3 == atom4 || (multipoleType = forceField.getMultipoleType(key = key1 + " " + key2 + " " + key3 + " " + (key4 = atom4.getAtomType().getKey()))) == null) continue;
                    atom.setMultipoleType(multipoleType);
                    atom.setAxisAtoms(atom2, atom3, atom4);
                    return multipoleType;
                }
            }
        }
        List<Atom> n13 = atom.get13List();
        for (Atom atom2 : n12) {
            String key22 = atom2.getAtomType().getKey();
            for (Atom atom3 : n13) {
                String key32 = atom3.getAtomType().getKey();
                String key = key1 + " " + key22 + " " + key32;
                multipoleType = forceField.getMultipoleType(key);
                if (multipoleType != null) {
                    atom.setMultipoleType(multipoleType);
                    atom.setAxisAtoms(atom2, atom3);
                    return multipoleType;
                }
                for (Atom atom4 : n13) {
                    String key4;
                    if (atom4 == null || atom4 == atom3 || (multipoleType = forceField.getMultipoleType(key = key1 + " " + key22 + " " + key32 + " " + (key4 = atom4.getAtomType().getKey()))) == null) continue;
                    atom.setMultipoleType(multipoleType);
                    atom.setAxisAtoms(atom2, atom3, atom4);
                    return multipoleType;
                }
            }
        }
        return null;
    }

    public static void assignAxisAtoms(Atom atom) {
        String key3;
        String key2;
        MultipoleType multipoleType = atom.getMultipoleType();
        String mutipoleKey = multipoleType.getKey();
        String atomKey = atom.getAtomType().getKey();
        String[] types = mutipoleKey.split(" +");
        int numAxisAtoms = types.length - 1;
        if (numAxisAtoms == 0) {
            atom.setAxisAtoms(null);
            return;
        }
        List<Atom> n12 = atom.get12List();
        if (n12 == null || n12.isEmpty()) {
            String message = "Multipoles can only be assigned after bonded relationships are defined.\n";
            logger.severe(message);
            return;
        }
        for (Atom atom2 : n12) {
            String key = atomKey + " " + atom2.getAtomType().getKey();
            if (!key.equalsIgnoreCase(mutipoleKey)) continue;
            atom.setAxisAtoms(atom2);
            return;
        }
        for (Atom atom3 : n12) {
            key2 = atom3.getAtomType().getKey();
            for (Atom atom32 : n12) {
                Object key;
                if (atom3 == atom32 || !((String)(key = atomKey + " " + key2 + " " + (key3 = atom32.getAtomType().getKey()))).equalsIgnoreCase(mutipoleKey)) continue;
                atom.setAxisAtoms(atom3, atom32);
                return;
            }
        }
        for (Atom atom4 : n12) {
            key2 = atom4.getAtomType().getKey();
            for (Atom atom32 : n12) {
                if (atom4 == atom32) continue;
                key3 = atom32.getAtomType().getKey();
                for (Atom atom42 : n12) {
                    String key4;
                    String key;
                    if (atom4 == atom42 || atom32 == atom42 || !(key = atomKey + " " + key2 + " " + key3 + " " + (key4 = atom42.getAtomType().getKey())).equalsIgnoreCase(mutipoleKey)) continue;
                    atom.setAxisAtoms(atom4, atom32);
                    return;
                }
            }
        }
        List<Atom> n13 = atom.get13List();
        for (Atom atom2 : n12) {
            String key22 = atom2.getAtomType().getKey();
            for (Atom atom3 : n13) {
                String key32 = atom3.getAtomType().getKey();
                String key = atomKey + " " + key22 + " " + key32;
                if (key.equalsIgnoreCase(mutipoleKey)) {
                    atom.setAxisAtoms(atom2, atom3);
                    return;
                }
                for (Atom atom4 : n13) {
                    String key4;
                    if (atom4 == null || atom4 == atom3 || !(key = atomKey + " " + key22 + " " + key32 + " " + (key4 = atom4.getAtomType().getKey())).equalsIgnoreCase(mutipoleKey)) continue;
                    atom.setAxisAtoms(atom2, atom3, atom4);
                    return;
                }
            }
        }
        String string = String.format(" Assignment of axis atoms failed for %s  %s.", atom, multipoleType);
        logger.severe(string);
    }

    public static MultipoleType parse(String input, String[] tokens, BufferedReader br) {
        if (tokens == null || tokens.length < 3 || tokens.length > 6) {
            logger.log(Level.WARNING, "Invalid MULTIPOLE type:\n{0}", input);
            return null;
        }
        try {
            int nTokens = tokens.length;
            ArrayList<Integer> frameAtoms = new ArrayList<Integer>();
            for (int i = 1; i < nTokens - 1; ++i) {
                int frameType = Integer.parseInt(tokens[i]);
                if (frameType == 0) continue;
                frameAtoms.add(frameType);
            }
            int nAtomTypes = frameAtoms.size();
            int[] atomTypes = new int[nAtomTypes];
            for (int i = 0; i < nAtomTypes; ++i) {
                atomTypes[i] = (Integer)frameAtoms.get(i);
            }
            double charge = Double.parseDouble(tokens[nTokens - 1]);
            MultipoleFrameDefinition frameDefinition = null;
            if (nAtomTypes == 1) {
                frameDefinition = MultipoleFrameDefinition.NONE;
            } else if (nAtomTypes == 2) {
                frameDefinition = MultipoleFrameDefinition.ZONLY;
            } else if (nAtomTypes == 3) {
                frameDefinition = atomTypes[1] < 0 || atomTypes[2] < 0 ? MultipoleFrameDefinition.BISECTOR : MultipoleFrameDefinition.ZTHENX;
            } else if (nAtomTypes == 4 && atomTypes[2] < 0 && atomTypes[3] < 0) {
                frameDefinition = MultipoleFrameDefinition.ZTHENBISECTOR;
                if (atomTypes[1] < 0) {
                    frameDefinition = MultipoleFrameDefinition.THREEFOLD;
                }
            }
            if (frameDefinition == null) {
                logger.log(Level.FINE, "Ambiguous MULTIPOLE type:\n{0}", input);
                frameDefinition = MultipoleFrameDefinition.ZTHENX;
            }
            for (int i = 0; i < nAtomTypes; ++i) {
                atomTypes[i] = FastMath.abs((int)atomTypes[i]);
            }
            input = br.readLine().split("#")[0];
            tokens = input.trim().split(" +");
            if (tokens.length != 3) {
                logger.log(Level.WARNING, "Invalid MULTIPOLE type:\n{0}", input);
                return null;
            }
            double[] dipole = new double[]{Double.parseDouble(tokens[0]), Double.parseDouble(tokens[1]), Double.parseDouble(tokens[2])};
            input = br.readLine().split("#")[0];
            tokens = input.trim().split(" +");
            if (tokens.length != 1) {
                logger.log(Level.WARNING, "Invalid MULTIPOLE type:\n{0}", input);
                return null;
            }
            double[][] quadrupole = new double[3][3];
            quadrupole[0][0] = Double.parseDouble(tokens[0]);
            input = br.readLine().split("#")[0];
            tokens = input.trim().split(" +");
            if (tokens.length != 2) {
                logger.log(Level.WARNING, "Invalid MULTIPOLE type:\n{0}", input);
                return null;
            }
            quadrupole[1][0] = Double.parseDouble(tokens[0]);
            quadrupole[1][1] = Double.parseDouble(tokens[1]);
            input = br.readLine().split("#")[0];
            tokens = input.trim().split(" +");
            if (tokens.length != 3) {
                logger.log(Level.WARNING, "Invalid MULTIPOLE type:\n{0}", input);
                return null;
            }
            quadrupole[2][0] = Double.parseDouble(tokens[0]);
            quadrupole[2][1] = Double.parseDouble(tokens[1]);
            quadrupole[2][2] = Double.parseDouble(tokens[2]);
            quadrupole[0][1] = quadrupole[1][0];
            quadrupole[0][2] = quadrupole[2][0];
            quadrupole[1][2] = quadrupole[2][1];
            return new MultipoleType(charge, dipole, quadrupole, atomTypes, frameDefinition, true);
        }
        catch (Exception e) {
            String message = "Exception parsing MULTIPOLE type:\n" + input + "\n";
            logger.log(Level.SEVERE, message, e);
            return null;
        }
    }

    public static MultipoleType parse(String input, String[] tokens) {
        if (tokens == null || tokens.length < 12 || tokens.length > 15) {
            logger.log(Level.WARNING, "Invalid MULTIPOLE type:\n{0}", input);
            return null;
        }
        try {
            int nTokens = tokens.length;
            ArrayList<Integer> frameAtoms = new ArrayList<Integer>();
            for (int i = 1; i < nTokens - 10; ++i) {
                int frameType = Integer.parseInt(tokens[i]);
                if (frameType == 0) continue;
                frameAtoms.add(frameType);
            }
            int nAtomTypes = frameAtoms.size();
            int[] atomTypes = new int[nAtomTypes];
            for (int i = 0; i < nAtomTypes; ++i) {
                atomTypes[i] = (Integer)frameAtoms.get(i);
            }
            MultipoleFrameDefinition frameDefinition = null;
            if (nAtomTypes == 1) {
                frameDefinition = MultipoleFrameDefinition.NONE;
            } else if (nAtomTypes == 2) {
                frameDefinition = MultipoleFrameDefinition.ZONLY;
            } else if (nAtomTypes == 3) {
                frameDefinition = atomTypes[1] < 0 || atomTypes[2] < 0 ? MultipoleFrameDefinition.BISECTOR : MultipoleFrameDefinition.ZTHENX;
            } else if (nAtomTypes == 4 && atomTypes[2] < 0 && atomTypes[3] < 0) {
                frameDefinition = MultipoleFrameDefinition.ZTHENBISECTOR;
                if (atomTypes[1] < 0) {
                    frameDefinition = MultipoleFrameDefinition.THREEFOLD;
                }
            }
            if (frameDefinition == null) {
                logger.log(Level.FINE, "Ambiguous MULTIPOLE type:\n{0}", input);
                frameDefinition = MultipoleFrameDefinition.ZTHENX;
            }
            for (int i = 0; i < nAtomTypes; ++i) {
                atomTypes[i] = FastMath.abs((int)atomTypes[i]);
            }
            double[] dipole = new double[3];
            double[][] quadrupole = new double[3][3];
            double charge = Double.parseDouble(tokens[nTokens - 10]);
            dipole[0] = Double.parseDouble(tokens[nTokens - 9]);
            dipole[1] = Double.parseDouble(tokens[nTokens - 8]);
            dipole[2] = Double.parseDouble(tokens[nTokens - 7]);
            quadrupole[0][0] = Double.parseDouble(tokens[nTokens - 6]);
            quadrupole[1][0] = Double.parseDouble(tokens[nTokens - 5]);
            quadrupole[1][1] = Double.parseDouble(tokens[nTokens - 4]);
            quadrupole[2][0] = Double.parseDouble(tokens[nTokens - 3]);
            quadrupole[2][1] = Double.parseDouble(tokens[nTokens - 2]);
            quadrupole[2][2] = Double.parseDouble(tokens[nTokens - 1]);
            quadrupole[0][1] = quadrupole[1][0];
            quadrupole[0][2] = quadrupole[2][0];
            quadrupole[1][2] = quadrupole[2][1];
            return new MultipoleType(charge, dipole, quadrupole, atomTypes, frameDefinition, true);
        }
        catch (Exception e) {
            String message = "Exception parsing MULTIPOLE type:\n" + input + "\n";
            logger.log(Level.SEVERE, message, e);
            return null;
        }
    }

    public static MultipoleType parseChargeType(String input, String[] tokens) {
        if (tokens.length < 3) {
            logger.log(Level.WARNING, "Invalid CHARGE type:\n{0}", input);
        } else {
            try {
                int[] atomTypes = new int[]{Integer.parseInt(tokens[1])};
                double charge = Double.parseDouble(tokens[2]);
                double[] dipole = new double[3];
                double[][] quadrupole = new double[3][3];
                MultipoleFrameDefinition frameDefinition = MultipoleFrameDefinition.NONE;
                return new MultipoleType(charge, dipole, quadrupole, atomTypes, frameDefinition, true);
            }
            catch (NumberFormatException e) {
                String message = "Exception parsing CHARGE type:\n" + input + "\n";
                logger.log(Level.SEVERE, message, e);
            }
        }
        return null;
    }

    public static void rotateDipole(double[][] rotMat, double[] dipole, double[] rotatedDipole) {
        for (int i = 0; i < 3; ++i) {
            double[] rotmati = rotMat[i];
            for (int j = 0; j < 3; ++j) {
                int n = i;
                rotatedDipole[n] = rotatedDipole[n] + rotmati[j] * dipole[j];
            }
        }
    }

    public static void rotateMultipole(double[][] rotmat, double[] dipole, double[][] quadrupole, double[] rotatedDipole, double[][] rotatedQuadrupole) {
        int i;
        Arrays.fill(rotatedDipole, 0.0);
        for (i = 0; i < 3; ++i) {
            Arrays.fill(rotatedQuadrupole[i], 0.0);
        }
        for (i = 0; i < 3; ++i) {
            double[] rotmati = rotmat[i];
            double[] quadrupolei = rotatedQuadrupole[i];
            rotatedDipole[i] = 0.0;
            for (int j = 0; j < 3; ++j) {
                double[] rotmatj = rotmat[j];
                int n = i;
                rotatedDipole[n] = rotatedDipole[n] + rotmati[j] * dipole[j];
                if (j < i) {
                    quadrupolei[j] = rotatedQuadrupole[j][i];
                    continue;
                }
                for (int k = 0; k < 3; ++k) {
                    double[] localQuadrupolek = quadrupole[k];
                    int n2 = j;
                    quadrupolei[n2] = quadrupolei[n2] + rotmati[k] * (rotmatj[0] * localQuadrupolek[0] + rotmatj[1] * localQuadrupolek[1] + rotmatj[2] * localQuadrupolek[2]);
                }
            }
        }
    }

    public static double[] scale(MultipoleType type, double[] scaleFactors) {
        return MultipoleType.scale(type.getMultipole(), scaleFactors);
    }

    public static double[] scale(double[] multipole, double[] scaleFactors) {
        double chargeScale = scaleFactors[0];
        double dipoleScale = scaleFactors[1];
        double quadScale = scaleFactors[2];
        return new double[]{multipole[0] * chargeScale, multipole[1] * dipoleScale, multipole[2] * dipoleScale, multipole[3] * dipoleScale, multipole[4] * quadScale, multipole[5] * quadScale, multipole[6] * quadScale, multipole[7] * quadScale, multipole[8] * quadScale, multipole[9] * quadScale};
    }

    public static MultipoleType weightMultipoleTypes(MultipoleType[] types, double[] weights, int[] frameAtomTypes) {
        double[] weightedMultipole = MultipoleType.weightMultipole(types, weights);
        if (weightedMultipole == null) {
            return null;
        }
        return new MultipoleType(weightedMultipole, frameAtomTypes, types[0].frameDefinition, false);
    }

    private static double[] bohrToElectronAngstroms(double[] multipole) {
        multipole[1] = multipole[1] * 0.529177210903;
        multipole[2] = multipole[2] * 0.529177210903;
        multipole[3] = multipole[3] * 0.529177210903;
        multipole[4] = multipole[4] * 0.2800285205390781;
        multipole[5] = multipole[5] * 0.2800285205390781;
        multipole[6] = multipole[6] * 0.2800285205390781;
        multipole[7] = multipole[7] * 0.2800285205390781;
        multipole[8] = multipole[8] * 0.2800285205390781;
        multipole[9] = multipole[9] * 0.2800285205390781;
        return new double[]{multipole[0], multipole[1], multipole[2], multipole[3], multipole[4], multipole[5], multipole[6], multipole[7], multipole[8], multipole[9]};
    }

    private static double[] pack(double charge, double[] dipole, double[][] quad) {
        return new double[]{charge, dipole[0], dipole[1], dipole[2], quad[0][0], quad[1][1], quad[2][2], quad[0][1], quad[0][2], quad[1][2]};
    }

    private static double[] unpackDipole(double[] mpole) {
        return new double[]{mpole[1], mpole[2], mpole[3]};
    }

    private static double[][] unpackQuad(double[] mpole) {
        return new double[][]{{mpole[4], mpole[7], mpole[8]}, {mpole[7], mpole[5], mpole[9]}, {mpole[8], mpole[9], mpole[6]}};
    }

    private static double[] weightMultipole(MultipoleType[] types, double[] weights) {
        if (types == null || weights == null || types.length != weights.length) {
            throw new IllegalArgumentException();
        }
        if (Arrays.asList(types).contains(null)) {
            return null;
        }
        for (MultipoleType type : types) {
            if (type.frameDefinition == types[0].frameDefinition) continue;
            logger.warning(String.format("Multipole frame definition mismatch during weighting:\n\t%s->%s,\n\t%s->%s", types[0], types[0].frameDefinition.toString(), type, type.frameDefinition.toString()));
            throw new IllegalArgumentException();
        }
        double[] weightedMultipole = new double[10];
        Arrays.fill(weightedMultipole, 0.0);
        for (int idx = 0; idx < types.length; ++idx) {
            double[] multipole = types[idx].getMultipole();
            for (int comp = 0; comp < 10; ++comp) {
                int n = comp;
                weightedMultipole[n] = weightedMultipole[n] + weights[idx] * multipole[comp];
            }
        }
        return weightedMultipole;
    }

    @Override
    public int compare(String s1, String s2) {
        String[] keys1 = s1.split(" ");
        String[] keys2 = s2.split(" ");
        int len = keys1.length;
        if (keys1.length > keys2.length) {
            len = keys2.length;
        }
        int[] c1 = new int[len];
        int[] c2 = new int[len];
        for (int i = 0; i < len; ++i) {
            c1[i] = FastMath.abs((int)Integer.parseInt(keys1[i]));
            c2[i] = FastMath.abs((int)Integer.parseInt(keys2[i]));
            if (c1[i] < c2[i]) {
                return -1;
            }
            if (c1[i] <= c2[i]) continue;
            return 1;
        }
        if (keys1.length < keys2.length) {
            return -1;
        }
        if (keys1.length > keys2.length) {
            return 1;
        }
        return 0;
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || this.getClass() != o.getClass()) {
            return false;
        }
        MultipoleType multipoleType = (MultipoleType)o;
        return Arrays.equals(this.frameAtomTypes, multipoleType.frameAtomTypes);
    }

    public double getCharge() {
        return this.multipole[0];
    }

    public double[] getDipole() {
        return new double[]{this.multipole[1], this.multipole[2], this.multipole[3]};
    }

    public double[] getMultipole() {
        return new double[]{this.multipole[0], this.multipole[1], this.multipole[2], this.multipole[3], this.multipole[4], this.multipole[5], this.multipole[6], this.multipole[7], this.multipole[8], this.multipole[9]};
    }

    public double[][] getQuadrupole() {
        return new double[][]{{this.multipole[4], this.multipole[7], this.multipole[8]}, {this.multipole[7], this.multipole[5], this.multipole[9]}, {this.multipole[8], this.multipole[9], this.multipole[6]}};
    }

    public int hashCode() {
        return Arrays.hashCode(this.frameAtomTypes);
    }

    @Override
    public String toString() {
        return this.toBohrString();
    }

    void incrementType(int increment) {
        for (int i = 0; i < this.frameAtomTypes.length; ++i) {
            if (this.frameAtomTypes[i] > 0) {
                int n = i;
                this.frameAtomTypes[n] = this.frameAtomTypes[n] + increment;
                continue;
            }
            if (this.frameAtomTypes[i] >= 0) continue;
            int n = i;
            this.frameAtomTypes[n] = this.frameAtomTypes[n] - increment;
        }
        this.setKey(this.frameAtomTypes);
    }

    MultipoleType patchTypes(HashMap<AtomType, AtomType> typeMap) {
        int count = 0;
        int len = this.frameAtomTypes.length;
        for (AtomType newType : typeMap.keySet()) {
            for (int frameAtomType : this.frameAtomTypes) {
                if (frameAtomType != newType.type && frameAtomType != 0) continue;
                ++count;
            }
        }
        if (count > 0 && count < len) {
            int[] newFrame = Arrays.copyOf(this.frameAtomTypes, len);
            for (AtomType newType : typeMap.keySet()) {
                for (int i = 0; i < len; ++i) {
                    if (this.frameAtomTypes[i] != newType.type) continue;
                    AtomType knownType = typeMap.get(newType);
                    newFrame[i] = knownType.type;
                }
            }
            return new MultipoleType(this.multipole, newFrame, this.frameDefinition, false);
        }
        return null;
    }

    private void checkMultipole() {
        double[][] quadrupole = MultipoleType.unpackQuad(this.multipole);
        if (FastMath.abs((double)(quadrupole[0][1] - quadrupole[1][0])) > 1.0E-6) {
            logger.warning("Multipole component Qxy != Qyx");
            logger.info(this.toString());
        }
        if (FastMath.abs((double)(quadrupole[0][2] - quadrupole[2][0])) > 1.0E-6) {
            logger.warning("Multipole component Qxz != Qzx");
            logger.info(this.toString());
        }
        if (FastMath.abs((double)(quadrupole[1][2] - quadrupole[2][1])) > 1.0E-6) {
            logger.warning("Multipole component Qyz != Qzy");
            logger.info(this.toString());
        }
        if (FastMath.abs((double)(quadrupole[0][0] + quadrupole[1][1] + quadrupole[2][2])) > 1.0E-5) {
            logger.log(Level.WARNING, String.format("Multipole is not traceless: %12.8f", FastMath.abs((double)(quadrupole[0][0] + quadrupole[1][1] + quadrupole[2][2]))));
            logger.info(this.toString());
        }
    }

    private String toBohrString() {
        StringBuilder multipoleBuffer = new StringBuilder("multipole");
        switch (this.frameDefinition.ordinal()) {
            case 0: {
                multipoleBuffer.append(String.format("  %5d  %5s  %5s  %5s", this.frameAtomTypes[0], "", "", ""));
                break;
            }
            case 1: {
                multipoleBuffer.append(String.format("  %5d  %5d  %5s  %5s", this.frameAtomTypes[0], this.frameAtomTypes[1], "", ""));
                break;
            }
            case 2: {
                if (this.frameAtomTypes.length == 3) {
                    multipoleBuffer.append(String.format("  %5d  %5d  %5d  %5s", this.frameAtomTypes[0], this.frameAtomTypes[1], this.frameAtomTypes[2], ""));
                    break;
                }
                multipoleBuffer.append(String.format("  %5d  %5d  %5d  %5d", this.frameAtomTypes[0], this.frameAtomTypes[1], this.frameAtomTypes[2], this.frameAtomTypes[3]));
                break;
            }
            case 3: {
                multipoleBuffer.append(String.format("  %5d  %5d  %5d  %5s", this.frameAtomTypes[0], -this.frameAtomTypes[1], -this.frameAtomTypes[2], ""));
                break;
            }
            case 4: {
                multipoleBuffer.append(String.format("  %5d  %5d  %5d  %5d", this.frameAtomTypes[0], this.frameAtomTypes[1], -this.frameAtomTypes[2], -this.frameAtomTypes[3]));
                break;
            }
            case 5: {
                multipoleBuffer.append(String.format("  %5d  %5d  %5d  %5d", this.frameAtomTypes[0], -this.frameAtomTypes[1], -this.frameAtomTypes[2], -this.frameAtomTypes[3]));
            }
        }
        multipoleBuffer.append(String.format("  % 7.5f \\\n%11$s % 7.5f % 7.5f % 7.5f \\\n%11$s % 7.5f \\\n%11$s % 7.5f % 7.5f \\\n%11$s % 7.5f % 7.5f % 7.5f", this.multipole[0], this.multipole[1] / 0.529177210903, this.multipole[2] / 0.529177210903, this.multipole[3] / 0.529177210903, this.multipole[4] / 0.2800285205390781, this.multipole[7] / 0.2800285205390781, this.multipole[5] / 0.2800285205390781, this.multipole[8] / 0.2800285205390781, this.multipole[9] / 0.2800285205390781, this.multipole[6] / 0.2800285205390781, "                                      "));
        return multipoleBuffer.toString();
    }

    public static Element getXMLForce(Document doc, ForceField forceField) {
        Map<String, MultipoleType> mpMap = forceField.getMultipoleTypes();
        Map<String, PolarizeType> polMap = forceField.getPolarizeTypes();
        if (!mpMap.values().isEmpty() || !polMap.values().isEmpty()) {
            Element node = doc.createElement("AmoebaMultipoleForce");
            node.setAttribute("mpole12Scale", String.valueOf(forceField.getDouble("mpole-12-scale", 0.0)));
            node.setAttribute("mpole13Scale", String.valueOf(forceField.getDouble("mpole-13-scale", 0.0)));
            node.setAttribute("mpole14Scale", String.valueOf(forceField.getDouble("mpole-14-scale", 1.0)));
            node.setAttribute("mpole15Scale", String.valueOf(forceField.getDouble("mpole-15-scale", 1.0)));
            PolarizeType.addXMLAttributes(node, forceField);
            for (MultipoleType multipoleType : mpMap.values()) {
                node.appendChild(multipoleType.toXML(doc));
            }
            for (PolarizeType polarizeType : polMap.values()) {
                node.appendChild(polarizeType.toXML(doc));
            }
            return node;
        }
        return null;
    }

    public Element toXML(Document doc) {
        Element node = doc.createElement("Multipole");
        switch (this.frameDefinition.ordinal()) {
            case 0: {
                node.setAttribute("type", String.format("%d", this.frameAtomTypes[0]));
                break;
            }
            case 1: {
                node.setAttribute("type", String.format("%d", this.frameAtomTypes[0]));
                node.setAttribute("kz", String.format("%d", this.frameAtomTypes[1]));
                break;
            }
            case 2: {
                if (this.frameAtomTypes.length == 3) {
                    node.setAttribute("type", String.format("%d", this.frameAtomTypes[0]));
                    node.setAttribute("kz", String.format("%d", this.frameAtomTypes[1]));
                    node.setAttribute("kx", String.format("%d", this.frameAtomTypes[2]));
                    break;
                }
                node.setAttribute("type", String.format("%d", this.frameAtomTypes[0]));
                node.setAttribute("kz", String.format("%d", this.frameAtomTypes[1]));
                node.setAttribute("kx", String.format("%d", this.frameAtomTypes[2]));
                node.setAttribute("ky", String.format("%d", this.frameAtomTypes[3]));
                break;
            }
            case 3: {
                node.setAttribute("type", String.format("%d", this.frameAtomTypes[0]));
                node.setAttribute("kz", String.format("%d", -this.frameAtomTypes[1]));
                node.setAttribute("kx", String.format("%d", -this.frameAtomTypes[2]));
                break;
            }
            case 4: {
                node.setAttribute("type", String.format("%d", this.frameAtomTypes[0]));
                node.setAttribute("kz", String.format("%d", this.frameAtomTypes[1]));
                node.setAttribute("kx", String.format("%d", -this.frameAtomTypes[2]));
                node.setAttribute("ky", String.format("%d", -this.frameAtomTypes[3]));
                break;
            }
            case 5: {
                node.setAttribute("type", String.format("%d", this.frameAtomTypes[0]));
                node.setAttribute("kz", String.format("%d", -this.frameAtomTypes[1]));
                node.setAttribute("kx", String.format("%d", -this.frameAtomTypes[2]));
                node.setAttribute("ky", String.format("%d", -this.frameAtomTypes[3]));
            }
        }
        node.setAttribute("c0", String.format("%.10f", this.multipole[0]));
        node.setAttribute("d1", String.format("%.17f", this.multipole[1] * 0.1));
        node.setAttribute("d2", String.format("%.17f", this.multipole[2] * 0.1));
        node.setAttribute("d3", String.format("%.17f", this.multipole[3] * 0.1));
        node.setAttribute("q11", String.format("%.17f", this.multipole[4] * 0.1 * 0.1 / 3.0));
        node.setAttribute("q21", String.format("%.17f", this.multipole[7] * 0.1 * 0.1 / 3.0));
        node.setAttribute("q22", String.format("%.17f", this.multipole[5] * 0.1 * 0.1 / 3.0));
        node.setAttribute("q31", String.format("%.17f", this.multipole[8] * 0.1 * 0.1 / 3.0));
        node.setAttribute("q32", String.format("%.17f", this.multipole[9] * 0.1 * 0.1 / 3.0));
        node.setAttribute("q33", String.format("%.17f", this.multipole[6] * 0.1 * 0.1 / 3.0));
        return node;
    }

    public static enum MultipoleFrameDefinition {
        NONE,
        ZONLY,
        ZTHENX,
        BISECTOR,
        ZTHENBISECTOR,
        THREEFOLD;

    }
}

