/*
 * Decompiled with CFR 0.152.
 */
package ffx.potential.parameters;

import ffx.potential.parameters.AtomType;
import ffx.potential.parameters.BaseType;
import ffx.potential.parameters.ForceField;
import ffx.utilities.FFXProperty;
import ffx.utilities.PropertyGroup;
import java.util.Arrays;
import java.util.Comparator;
import java.util.HashMap;
import java.util.Map;
import java.util.logging.Level;
import java.util.logging.Logger;
import javax.annotation.Nullable;
import org.w3c.dom.Document;
import org.w3c.dom.Element;

@FFXProperty(name="bond", clazz=String.class, propertyGroup=PropertyGroup.PotentialFunctionParameter, description="[2 integers and 2 reals]\nProvides the values for a single bond stretching parameter.\nThe integer modifiers give the atom class numbers for the two kinds of atoms involved in the bond which is to be defined.\nThe real number modifiers give the force constant value for the bond and the ideal bond length in Angstroms.\nThe default value of 1.0 is used, if the bondunit keyword is not given in the force field parameter file or the keyfile.\n")
public final class BondType
extends BaseType
implements Comparator<String> {
    public static final double DEFAULT_BOND_UNIT = 1.0;
    public static final double DEFAULT_BOND_CUBIC = 0.0;
    public static final double DEFAULT_BOND_QUARTIC = 0.0;
    @FFXProperty(name="bondunit", propertyGroup=PropertyGroup.EnergyUnitConversion, defaultValue="1.0", description="Sets the scale factor needed to convert the energy value computed by the bond stretching potential into units of kcal/mole.\nThe correct value is force field dependent and typically provided in the header of the master force field parameter file.\n")
    public double bondUnit = 1.0;
    @FFXProperty(name="bond-cubic", propertyGroup=PropertyGroup.LocalGeometryFunctionalForm, defaultValue="0.0", description="Sets the value of the cubic term in the Taylor series expansion form of the bond stretching potential energy.\nThe real number modifier gives the value of the coefficient as a multiple of the quadratic coefficient.\nThis term multiplied by the bond stretching energy unit conversion factor, the force constant,\nand the cube of the deviation of the bond length from its ideal value gives the cubic contribution to the bond stretching energy.\nThe default value in the absence of the bond-cubic keyword is zero; i.e., the cubic bond stretching term is omitted.\n")
    public double cubic = 0.0;
    @FFXProperty(name="bond-quartic", propertyGroup=PropertyGroup.LocalGeometryFunctionalForm, defaultValue="0.0", description="Sets the value of the quartic term in the Taylor series expansion form of the bond stretching potential energy.\nThe real number modifier gives the value of the coefficient as a multiple of the quadratic coefficient.\nThis term multiplied by the bond stretching energy unit conversion factor, the force constant,\nand the forth power of the deviation of the bond length from its ideal value gives the quartic contribution to the bond stretching energy.\nThe default value in the absence of the bond-quartic keyword is zero; i.e., the quartic bond stretching term is omitted.\n")
    public double quartic = 0.0;
    private static final Logger logger = Logger.getLogger(BondType.class.getName());
    public final int[] atomClasses;
    public final double forceConstant;
    public final double distance;
    public final double flatBottomRadius;
    public BondFunction bondFunction;

    public BondType(int[] atomClasses, double forceConstant, double distance) {
        this(atomClasses, forceConstant, distance, BondFunction.QUARTIC, 0.0);
    }

    public BondType(int[] atomClasses, double forceConstant, double distance, BondFunction bondFunction) {
        this(atomClasses, forceConstant, distance, bondFunction, 0.0);
    }

    public BondType(int[] atomClasses, double forceConstant, double distance, BondFunction bondFunction, double flatBottomRadius) {
        super(ForceField.ForceFieldType.BOND, BondType.sortKey(atomClasses));
        this.atomClasses = atomClasses;
        this.forceConstant = forceConstant;
        this.distance = distance;
        this.bondFunction = bondFunction;
        this.flatBottomRadius = flatBottomRadius;
        assert (flatBottomRadius == 0.0 || bondFunction == BondFunction.FLAT_BOTTOM_HARMONIC);
    }

    public static BondType average(@Nullable BondType bondType1, @Nullable BondType bondType2, @Nullable int[] atomClasses) {
        if (bondType1 == null || bondType2 == null || atomClasses == null) {
            return null;
        }
        BondFunction bondFunction = bondType1.bondFunction;
        if (bondFunction != bondType2.bondFunction) {
            return null;
        }
        double forceConstant = (bondType1.forceConstant + bondType2.forceConstant) / 2.0;
        double distance = (bondType1.distance + bondType2.distance) / 2.0;
        return new BondType(atomClasses, forceConstant, distance, bondFunction);
    }

    public static BondType parse(String input, String[] tokens) {
        if (tokens.length < 5) {
            logger.log(Level.WARNING, "Invalid BOND type:\n{0}", input);
        } else {
            try {
                int[] atomClasses = new int[]{Integer.parseInt(tokens[1]), Integer.parseInt(tokens[2])};
                double forceConstant = Double.parseDouble(tokens[3]);
                double distance = Double.parseDouble(tokens[4]);
                return new BondType(atomClasses, forceConstant, distance);
            }
            catch (NumberFormatException e) {
                String message = "Exception parsing BOND type:\n" + input + "\n";
                logger.log(Level.SEVERE, message, e);
            }
        }
        return null;
    }

    public static String sortKey(int[] c) {
        if (c == null || c.length != 2) {
            return null;
        }
        if (c[1] <= c[0]) {
            int temp = c[1];
            c[1] = c[0];
            c[0] = temp;
        }
        return c[0] + " " + c[1];
    }

    @Override
    public int compare(String key1, String key2) {
        String[] keys1 = key1.split(" ");
        String[] keys2 = key2.split(" ");
        int[] c1 = new int[2];
        int[] c2 = new int[2];
        for (int i = 0; i < 2; ++i) {
            c1[i] = Integer.parseInt(keys1[i]);
            c2[i] = Integer.parseInt(keys2[i]);
        }
        if (c1[0] < c2[0]) {
            return -1;
        }
        if (c1[0] > c2[0]) {
            return 1;
        }
        if (c1[1] < c2[1]) {
            return -1;
        }
        if (c1[1] > c2[1]) {
            return 1;
        }
        return 0;
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || this.getClass() != o.getClass()) {
            return false;
        }
        BondType bondType = (BondType)o;
        return Arrays.equals(this.atomClasses, bondType.atomClasses);
    }

    public int hashCode() {
        return Arrays.hashCode(this.atomClasses);
    }

    public void incrementClasses(int increment) {
        int i = 0;
        while (i < this.atomClasses.length) {
            int n = i++;
            this.atomClasses[n] = this.atomClasses[n] + increment;
        }
        this.setKey(BondType.sortKey(this.atomClasses));
    }

    public BondType patchClasses(HashMap<AtomType, AtomType> typeMap) {
        int count = 0;
        int len = this.atomClasses.length;
        for (AtomType newType : typeMap.keySet()) {
            for (int atomClass : this.atomClasses) {
                if (atomClass != newType.atomClass) continue;
                ++count;
            }
        }
        if (count == 1) {
            int[] newClasses = Arrays.copyOf(this.atomClasses, len);
            for (AtomType newType : typeMap.keySet()) {
                for (int i = 0; i < len; ++i) {
                    if (this.atomClasses[i] != newType.atomClass) continue;
                    AtomType knownType = typeMap.get(newType);
                    newClasses[i] = knownType.atomClass;
                }
            }
            return new BondType(newClasses, this.forceConstant, this.distance, this.bondFunction);
        }
        return null;
    }

    public void setBondFunction(BondFunction bondFunction) {
        this.bondFunction = bondFunction;
    }

    @Override
    public String toString() {
        return String.format("bond  %5d  %5d  %7.2f  %7.4f", this.atomClasses[0], this.atomClasses[1], this.forceConstant, this.distance);
    }

    public static Element getXMLForce(Document doc, ForceField forceField) {
        Map<String, BondType> btMap = forceField.getBondTypes();
        if (!btMap.values().isEmpty()) {
            Element node = doc.createElement("AmoebaBondForce");
            node.setAttribute("bond-cubic", String.format("%f", forceField.getDouble("bond-cubic", 0.0) * 10.0));
            node.setAttribute("bond-quartic", String.format("%f", forceField.getDouble("bond-quartic", 0.0) * 10.0 * 10.0));
            for (BondType bondType : btMap.values()) {
                node.appendChild(bondType.toXML(doc));
            }
            return node;
        }
        return null;
    }

    public Element toXML(Document doc) {
        Element node = doc.createElement("Bond");
        node.setAttribute("class1", String.format("%d", this.atomClasses[0]));
        node.setAttribute("class2", String.format("%d", this.atomClasses[1]));
        node.setAttribute("length", String.format("%f", this.distance * 0.1));
        node.setAttribute("k", String.format("%.17f", this.forceConstant * 10.0 * 10.0 * 4.184));
        return node;
    }

    public static enum BondFunction {
        HARMONIC("0.5*k*(r-r0)^2", false),
        QUARTIC("0.5*k*dv^2*((1+cubic)*dv+(1+quartic)*dv^2);dv=r-r0", false),
        FLAT_BOTTOM_HARMONIC("0.5*k*dv^2;dv=step(dv)*step(dv-fb)*(dv-fb)+step(-dv)*step(-dv-fb)*(-dv-fb);dv=r-r0", true),
        FLAT_BOTTOM_QUARTIC("0.5*k*dv^2*((1+cubic)*dv+(1+quartic)*dv^2);dv=step(dv)*step(dv-fb)*(dv-fb)+step(-dv)*step(-dv-fb)*(-dv-fb);dv=r-r0", true);

        private final String mathematicalForm;
        private final boolean hasFlatBottom;

        private BondFunction(String mathForm, boolean flatBottom) {
            this.mathematicalForm = mathForm;
            this.hasFlatBottom = flatBottom;
        }

        public boolean hasFlatBottom() {
            return this.hasFlatBottom;
        }

        public String toMathematicalForm() {
            return this.mathematicalForm;
        }
    }
}

