/*
 * Decompiled with CFR 0.152.
 */
package ffx.potential.commands;

import ffx.crystal.Crystal;
import ffx.crystal.ReplicatesCrystal;
import ffx.numerics.Potential;
import ffx.potential.ForceFieldEnergy;
import ffx.potential.MolecularAssembly;
import ffx.potential.bonded.Atom;
import ffx.potential.bonded.MSNode;
import ffx.potential.bonded.Polymer;
import ffx.potential.cli.PotentialCommand;
import ffx.potential.extended.ExtendedSystem;
import ffx.potential.utils.ConvexHullOps;
import ffx.utilities.FFXBinding;
import java.io.BufferedReader;
import java.io.File;
import java.io.FileReader;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Random;
import java.util.Set;
import java.util.logging.Logger;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import java.util.stream.Collectors;
import org.apache.commons.configuration2.Configuration;
import org.apache.commons.io.FilenameUtils;
import org.apache.commons.math3.util.FastMath;
import picocli.CommandLine;

@CommandLine.Command(description={" Creates a box of solvent around a solute."}, name="Solvator")
public class Solvator
extends PotentialCommand {
    @CommandLine.Option(names={"--sFi", "--solventFile"}, paramLabel="water", description={"A file containing the solvent box to be used. There is currently no default."})
    private String solventFileName = null;
    @CommandLine.Option(names={"--iFi", "--ionFile"}, paramLabel="ions", description={"Name of the file containing ions. Must also have a .ions file (e.g. nacl.pdb must also have nacl.ions). Default: no ions."})
    private String ionFileName = null;
    @CommandLine.Option(names={"-r", "--rectangular"}, paramLabel="false", defaultValue="false", description={"Use a rectangular prism rather than a cube for solvation."})
    private boolean rectangular = false;
    @CommandLine.Option(names={"-p", "--padding"}, paramLabel="9.0", defaultValue="9.0", description={"Sets the minimum amount of solvent padding around the solute."})
    private double padding = 9.0;
    @CommandLine.Option(names={"-b", "--boundary"}, paramLabel="2.5", defaultValue="2.5", description={"Delete solvent molecules that infringe closer than this to the solute."})
    private double boundary = 2.5;
    @CommandLine.Option(names={"-x", "--translate"}, paramLabel="true", defaultValue="true", description={"Move solute molecules to center of box. Turning off should only considered when specifying the unit cell box dimensions"})
    private boolean translate = true;
    @CommandLine.Option(names={"--abc", "--boxLengths"}, paramLabel="a,b,c", description={"Specify a comma-separated set of unit cell box lengths, instead of calculating them (a,b,c)"})
    private String manualBox = null;
    @CommandLine.Option(names={"-s", "--randomSeed"}, paramLabel="auto", description={"Specify a random seed for ion placement."})
    private String seedString = null;
    @CommandLine.Option(names={"--pH", "--constantPH"}, paramLabel="7.4", description={"pH value for the system. If set, titration states will be initialized based on this pH."})
    private Double pH = null;
    @CommandLine.Parameters(arity="1", paramLabel="file", description={"The atomic coordinate file in PDB or XYZ format."})
    String filename = null;
    private MolecularAssembly solute;
    private MolecularAssembly solvent;
    private MolecularAssembly ions;
    private File createdFile;
    private Configuration additionalProperties;

    public Solvator() {
    }

    public Solvator(FFXBinding binding) {
        super(binding);
    }

    public Solvator(String[] args) {
        super(args);
    }

    public void setProperties(Configuration additionalProps) {
        this.additionalProperties = additionalProps;
    }

    public Solvator run() {
        char c;
        double[] moments;
        if (!this.init()) {
            return this;
        }
        if (this.solventFileName == null) {
            logger.info(this.helpString());
            return this;
        }
        double twoBoundary = 2.0 * this.boundary;
        String nlistCuts = Double.toString(twoBoundary);
        System.setProperty("vdw-cutoff", nlistCuts);
        System.setProperty("ewald-cutoff", nlistCuts);
        this.activeAssembly = this.getActiveAssembly(this.filename);
        if (this.activeAssembly == null) {
            logger.info(this.helpString());
            return this;
        }
        ForceFieldEnergy forceFieldEnergy = this.activeAssembly.getPotentialEnergy();
        if (this.pH != null) {
            logger.info("\n Initializing titration states for pH " + this.pH);
            ExtendedSystem esvSystem = new ExtendedSystem(this.activeAssembly, this.pH, null);
            forceFieldEnergy.attachExtendedSystem(esvSystem);
            esvSystem.reGuessLambdas();
            logger.info(esvSystem.getLambdaList());
        }
        int nVars = forceFieldEnergy.getNumberOfVariables();
        double[] coordinates = new double[nVars];
        forceFieldEnergy.getCoordinates(coordinates);
        forceFieldEnergy.energy(coordinates, true);
        if (forceFieldEnergy.getPmeNode() == null) {
            logger.severe("PME (electrostatics) is not initialized. Check force field settings.");
            return this;
        }
        Atom[] activeAtoms = this.activeAssembly.getActiveAtomArray();
        if (activeAtoms == null || activeAtoms.length == 0) {
            logger.severe("No active atoms found in the assembly.");
            return this;
        }
        try {
            moments = forceFieldEnergy.getPmeNode().computeMoments(activeAtoms, false);
            if (moments == null || moments.length == 0) {
                logger.severe("Failed to compute moments (null or empty array returned).");
                return this;
            }
            logger.info(String.format("\n System has a total charge of %8.4g before solvation", moments[0]));
        }
        catch (Exception e) {
            logger.severe("Error computing moments: " + e.getMessage());
            return this;
        }
        this.solute = this.activeAssembly;
        ForceFieldEnergy soluteEnergy = this.activeAssembly.getPotentialEnergy();
        this.solvent = this.potentialFunctions.open(this.solventFileName);
        Atom[] baseSolventAtoms = this.solvent.getActiveAtomArray();
        Atom[] ionAtoms = null;
        File ionsFile = null;
        if (this.ionFileName != null) {
            this.ions = this.potentialFunctions.open(this.ionFileName);
            ionAtoms = this.ions.getActiveAtomArray();
            Object ionsName = FilenameUtils.removeExtension((String)this.ionFileName);
            ionsName = (String)ionsName + ".ions";
            ionsFile = new File((String)ionsName);
            assert (ionsFile != null && ionsFile.exists());
        }
        Atom[] soluteAtoms = this.activeAssembly.getAtomArray();
        int nSolute = soluteAtoms.length;
        double[][] soluteCoordinates = new double[nSolute][3];
        Crystal soluteCrystal = this.activeAssembly.getCrystal();
        Crystal solventCrystal = this.solvent.getCrystal();
        if (solventCrystal instanceof ReplicatesCrystal) {
            solventCrystal = solventCrystal.getUnitCell();
        }
        if (!soluteCrystal.aperiodic() && !soluteCrystal.spaceGroup.shortName.equalsIgnoreCase("P1")) {
            logger.severe(" Solute must be aperiodic or strictly P1 periodic");
        }
        if (solventCrystal.aperiodic() || !solventCrystal.spaceGroup.shortName.equalsIgnoreCase("P1")) {
            logger.severe(" Solvent box must be periodic (and P1)!");
        }
        for (int i = 0; i < nSolute; ++i) {
            Atom ati = soluteAtoms[i];
            double[] xyzi = new double[3];
            xyzi = ati.getXYZ(xyzi);
            System.arraycopy(xyzi, 0, soluteCoordinates[i], 0, 3);
        }
        if (!soluteCrystal.aperiodic()) {
            for (Atom ati : soluteAtoms) {
                double[] xyzi = new double[3];
                soluteCrystal.image(ati.getXYZ(xyzi));
                ati.setXYZ(xyzi);
            }
        }
        double[] minSoluteXYZ = new double[3];
        double[] maxSoluteXYZ = new double[3];
        double[] soluteBoundingBox = new double[3];
        for (int i = 0; i < 3; ++i) {
            int index = i;
            minSoluteXYZ[i] = Arrays.stream(soluteCoordinates).mapToDouble(xyz -> xyz[index]).min().getAsDouble();
            maxSoluteXYZ[i] = Arrays.stream(soluteCoordinates).mapToDouble(xyz -> xyz[index]).max().getAsDouble();
            soluteBoundingBox[i] = maxSoluteXYZ[i] - minSoluteXYZ[i];
        }
        double soluteMass = Arrays.stream(soluteAtoms).mapToDouble(Atom::getMass).sum();
        double invSolMass = 1.0 / soluteMass;
        double[] origCoM = new double[3];
        for (Atom ati : soluteAtoms) {
            double massi = ati.getMass();
            massi *= invSolMass;
            double[] xyzi = new double[3];
            xyzi = ati.getXYZ(xyzi);
            for (int j = 0; j < 3; ++j) {
                int n = j;
                origCoM[n] = origCoM[n] + xyzi[j] * massi;
            }
        }
        logger.fine(String.format(" Original CoM: %s", Arrays.toString(origCoM)));
        double[] newBox = new double[3];
        if (this.manualBox != null) {
            String[] toks = this.manualBox.split(",");
            if (toks.length == 1) {
                double len = Double.parseDouble(this.manualBox);
                Arrays.fill(newBox, len);
            } else if (toks.length == 3) {
                for (int i = 0; i < 3; ++i) {
                    newBox[i] = Double.parseDouble(toks[i]);
                    if (!(newBox[i] <= 0.0)) continue;
                    logger.severe(" Specified a box length of less than zero!");
                }
            } else {
                logger.severe(" The manualBox option requires either 1 box length, or 3 comma-separated values.");
            }
        } else if (this.rectangular) {
            newBox = Arrays.stream(soluteBoundingBox).map(x -> x + 2.0 * this.padding).toArray();
        } else {
            double soluteLinearSize = ConvexHullOps.maxDist(soluteAtoms);
            Arrays.fill(newBox, soluteLinearSize += 2.0 * this.padding);
        }
        logger.info(String.format(" Molecule will be solvated in a periodic box of size %10.4g, %10.4g, %10.4g", newBox[0], newBox[1], newBox[2]));
        if (this.translate) {
            double[] soluteTranslate = new double[3];
            for (int i = 0; i < 3; ++i) {
                soluteTranslate[i] = 0.5 * newBox[i];
                int n = i;
                soluteTranslate[n] = soluteTranslate[n] - origCoM[i];
            }
            for (Atom atom : soluteAtoms) {
                atom.move(soluteTranslate);
            }
            logger.fine(String.format(" Solute translated by %s", Arrays.toString(soluteTranslate)));
        }
        this.solvent.moveAllIntoUnitCell();
        int nSolvent = baseSolventAtoms.length;
        double[][] baseSolventCoordinates = new double[nSolvent][3];
        for (int i = 0; i < nSolvent; ++i) {
            Atom ati = baseSolventAtoms[i];
            double[] xyzi = new double[3];
            xyzi = ati.getXYZ(xyzi);
            System.arraycopy(xyzi, 0, baseSolventCoordinates[i], 0, 3);
        }
        logger.info(String.format(" Solute size: %12.4g, %12.4g, %12.4g", soluteBoundingBox[0], soluteBoundingBox[1], soluteBoundingBox[2]));
        int[] solventReplicas = new int[3];
        double[] solventBoxVectors = new double[]{solventCrystal.a, solventCrystal.b, solventCrystal.c};
        for (int i = 0; i < 3; ++i) {
            solventReplicas[i] = (int)Math.ceil(newBox[i] / solventBoxVectors[i]);
        }
        Crystal newCrystal = new Crystal(newBox[0], newBox[1], newBox[2], 90.0, 90.0, 90.0, "P1");
        forceFieldEnergy.setCrystal(newCrystal, true);
        List<MSNode> bondedNodes = this.solvent.getAllBondedEntities();
        MSNode[] solventEntities = bondedNodes.toArray(new MSNode[0]);
        int nSolventMols = solventEntities.length;
        double[][] solventCOMs = new double[nSolventMols][];
        for (int i = 0; i < nSolventMols; ++i) {
            List<Atom> moleculeAtoms = solventEntities[i].getAtomList();
            double totMass = 0.0;
            for (Atom atom : moleculeAtoms) {
                totMass += atom.getMass();
            }
            double invMass = 1.0 / totMass;
            double[] dArray = new double[3];
            Arrays.fill(dArray, 0.0);
            for (Atom atom : moleculeAtoms) {
                double[] xyz2 = new double[3];
                xyz2 = atom.getXYZ(xyz2);
                double mass = atom.getAtomType().atomicWeight;
                for (int j = 0; j < 3; ++j) {
                    int n = j;
                    dArray[n] = dArray[n] + mass * xyz2[j] * invMass;
                }
            }
            solventCrystal.toPrimaryCell(dArray, dArray);
            solventCOMs[i] = dArray;
        }
        double[] xyzOffset = new double[3];
        int currXYZIndex = Arrays.stream(soluteAtoms).mapToInt(Atom::getXyzIndex).max().getAsInt();
        int currResSeq = 1;
        char[] possibleChains = new char[]{'A', 'B', 'C', 'D', 'E', 'F', 'G', 'H', 'I', 'J', 'K', 'L', 'M', 'N', 'O', 'P', 'Q', 'R', 'S', 'T', 'U', 'V', 'W', 'X', 'Y', 'Z'};
        char solventChain = ' ';
        Set soluteChains = Arrays.stream(soluteAtoms).map(Atom::getChainID).collect(Collectors.toSet());
        for (char solvChainOpt : possibleChains) {
            if (soluteChains.contains(Character.valueOf(solvChainOpt))) continue;
            solventChain = solvChainOpt;
            break;
        }
        int n = 32;
        for (char solvChainOpt : possibleChains) {
            if (solvChainOpt == solventChain || soluteChains.contains(Character.valueOf(solvChainOpt))) continue;
            c = solvChainOpt;
            break;
        }
        DomainDecomposition ddc = new DomainDecomposition(this.boundary, soluteAtoms, newCrystal);
        if (solventChain == ' ') {
            logger.severe(" Could not find an unused character A-Z for the new solvent!");
        }
        logger.info(String.format(" New solvent molecules will be placed in chain %c", Character.valueOf(solventChain)));
        if (c == ' ') {
            logger.severe(" Could not find an unused character A-Z for the new solvent!");
        }
        logger.info(String.format(" New ions will be placed in chain %c", Character.valueOf(c)));
        ArrayList<Atom[]> newMolecules = new ArrayList<Atom[]>();
        for (int ai = 0; ai < solventReplicas[0]; ++ai) {
            xyzOffset[0] = (double)ai * solventBoxVectors[0];
            for (int bi = 0; bi < solventReplicas[1]; ++bi) {
                xyzOffset[1] = (double)bi * solventBoxVectors[1];
                for (int ci = 0; ci < solventReplicas[2]; ++ci) {
                    xyzOffset[2] = (double)ci * solventBoxVectors[2];
                    logger.info(String.format(" Tiling solvent replica %d,%d,%d", ai + 1, bi + 1, ci + 1));
                    block28: for (int i = 0; i < nSolventMols; ++i) {
                        MSNode moli = solventEntities[i];
                        double[] comi = new double[3];
                        for (int j = 0; j < 3; ++j) {
                            comi[j] = xyzOffset[j] + solventCOMs[i][j];
                            if (comi[j] < 0.0) {
                                logger.warning(String.format(" Skipping a copy of solvent molecule %d for violating minimum boundary 0,0,0. This should not occur!", i));
                                continue block28;
                            }
                            if (!(comi[j] > newBox[j])) continue;
                            logger.fine(String.format(" Skipping a copy of solvent molecule %d for violating maximum boundary %12.4g,%12.4g,%12.4g", i, newBox[0], newBox[1], newBox[2]));
                            continue block28;
                        }
                        logger.fine(String.format(" Placing a molecule at %f,%f,%f", comi[0], comi[1], comi[2]));
                        List<Atom> parentAtoms = moli.getAtomList();
                        int nMolAtoms = parentAtoms.size();
                        Atom[] newAtomArray = new Atom[nMolAtoms];
                        for (int atI = 0; atI < nMolAtoms; ++atI) {
                            Atom parentAtom = parentAtoms.get(atI);
                            double[] newXYZ = new double[3];
                            newXYZ = parentAtom.getXYZ(newXYZ);
                            for (int j = 0; j < 3; ++j) {
                                int n2 = j;
                                newXYZ[n2] = newXYZ[n2] + xyzOffset[j];
                            }
                            Atom newAtom = new Atom(++currXYZIndex, parentAtom, newXYZ, currResSeq, solventChain, Character.toString(solventChain));
                            logger.fine(String.format(" New atom %s at chain %c on residue %s-%d", newAtom, newAtom.getChainID(), newAtom.getResidueName(), newAtom.getResidueNumber()));
                            newAtom.setHetero(!(moli instanceof Polymer));
                            newAtom.setResName(moli.getName());
                            if (newAtom.getAltLoc() == null) {
                                newAtom.setAltLoc(Character.valueOf(' '));
                            }
                            newAtomArray[atI] = newAtom;
                        }
                        boolean overlapFound = false;
                        for (Atom atom : newAtomArray) {
                            if (!ddc.checkClashes(atom, this.boundary)) continue;
                            overlapFound = true;
                            break;
                        }
                        if (overlapFound) {
                            logger.fine(String.format(" Skipping a copy of molecule %d for overlapping with the solute.", i));
                            continue;
                        }
                        newMolecules.add(newAtomArray);
                        ++currResSeq;
                    }
                }
            }
        }
        Random random = this.seedString == null ? new Random() : new Random(Long.parseLong(this.seedString));
        Collections.shuffle(newMolecules, random);
        if (this.ionFileName != null) {
            logger.info(String.format(" Ions will be placed into chain %c", Character.valueOf(c)));
            double volume = newBox[0] * newBox[1] * newBox[2];
            double ionsPermM = volume * 1.0E-27 * 6.02214076E23 * 0.001;
            ArrayList byConc = new ArrayList();
            IonAddition neutAnion = null;
            IonAddition neutCation = null;
            Pattern ionicPattern = Pattern.compile("^\\s*([0-9]+) +([0-9]+) +([0-9]+(?:\\.[0-9]*)?|NEUT\\S*)");
            Pattern concPatt = Pattern.compile("^[0-9]+(?:\\.[0-9]*)?");
            try (BufferedReader br = new BufferedReader(new FileReader(ionsFile));){
                String line = br.readLine();
                while (line != null) {
                    Matcher m = ionicPattern.matcher(line);
                    if (m.matches()) {
                        boolean toNeutralize;
                        int start = Integer.parseInt(m.group(1));
                        int end = Integer.parseInt(m.group(2));
                        if (end < start) {
                            throw new IllegalArgumentException(" end < start");
                        }
                        int nAts = end - start + 1;
                        Atom[] atoms = new Atom[nAts];
                        for (int i = 0; i < nAts; ++i) {
                            int atI = start + i - 1;
                            atoms[i] = ionAtoms[atI];
                        }
                        double conc = 0.0;
                        if (concPatt.matcher(m.group(3)).matches()) {
                            conc = Double.parseDouble(m.group(3));
                            toNeutralize = false;
                        } else {
                            toNeutralize = true;
                        }
                        IonAddition ia = new IonAddition(atoms, conc, toNeutralize);
                        if (toNeutralize) {
                            if (ia.charge > 0.0) {
                                neutCation = ia;
                            } else if (ia.charge < 0.0) {
                                neutAnion = ia;
                            } else {
                                logger.severe(String.format(" Specified a neutralizing ion %s with no net charge!", ia));
                            }
                        } else {
                            byConc.add(ia);
                        }
                    }
                    line = br.readLine();
                }
            }
            catch (IOException e) {
                logger.severe("Error reading ions file: " + e.getMessage());
                throw new RuntimeException(e);
            }
            ArrayList<Atom[]> addedIons = new ArrayList<Atom[]>();
            int ionResSeq = 0;
            double initialCharge = moments[0];
            logger.info(String.format(" Charge before addition of ions is %8.4g", initialCharge));
            Iterator end = byConc.iterator();
            while (end.hasNext()) {
                IonAddition ia = (IonAddition)end.next();
                logger.info(ia.toString());
                int nIons = (int)Math.ceil(ionsPermM * ia.conc);
                if (nIons > newMolecules.size()) {
                    logger.severe(String.format(" Insufficient solvent molecules remain (%d) to add %d ions!", newMolecules.size(), nIons));
                }
                logger.info(String.format(" Number of ions to place: %d\n", nIons));
                for (int i = 0; i < nIons; ++i) {
                    Atom[] newIon = Solvator.swapIon(newMolecules, ia, currXYZIndex, c, ionResSeq++);
                    currXYZIndex += newIon.length;
                    addedIons.add(newIon);
                }
                initialCharge += (double)nIons * ia.charge;
            }
            logger.info(String.format(" Charge before neutralization is %8.4g", initialCharge));
            if (initialCharge > 0.0) {
                if (neutAnion == null) {
                    logger.info(String.format(" No counter-anion specified; system will be cationic at charge %8.4g", initialCharge));
                } else {
                    logger.info(String.format(" Neutralizing system with %s", neutAnion));
                    double charge = neutAnion.charge;
                    int nAnions = (int)FastMath.round((double)(-1.0 * (initialCharge / charge)));
                    double netCharge = initialCharge + (double)nAnions * charge;
                    if (nAnions > newMolecules.size()) {
                        logger.severe(String.format(" Insufficient solvent molecules remain (%d) to add %d counter-anions!", newMolecules.size(), nAnions));
                    }
                    for (i = 0; i < nAnions; ++i) {
                        newIon = Solvator.swapIon(newMolecules, neutAnion, currXYZIndex, c, ionResSeq++);
                        currXYZIndex += newIon.length;
                        addedIons.add(newIon);
                    }
                    logger.info(String.format(" System neutralized to %8.4g charge with %d counter-anions", netCharge, nAnions));
                }
            } else if (initialCharge < 0.0) {
                if (neutCation == null) {
                    logger.info(String.format(" No counter-cation specified; system will be anionic at charge %8.4g", initialCharge));
                } else {
                    logger.info(String.format(" Neutralizing system with %s", neutCation));
                    double charge = neutCation.charge;
                    int nCations = (int)FastMath.round((double)(-1.0 * (initialCharge / charge)));
                    double netCharge = initialCharge + (double)nCations * charge;
                    if (nCations > newMolecules.size()) {
                        logger.severe(String.format(" Insufficient solvent molecules remain (%d) to add %d counter-cations!", newMolecules.size(), nCations));
                    }
                    for (i = 0; i < nCations; ++i) {
                        newIon = Solvator.swapIon(newMolecules, neutCation, currXYZIndex, c, ionResSeq++);
                        currXYZIndex += newIon.length;
                        addedIons.add(newIon);
                    }
                    logger.info(String.format(" System neutralized to %8.4g charge with %d counter-cations", netCharge, nCations));
                }
            } else {
                logger.info(" System is neutral; no neutralizing ions needed.");
            }
            newMolecules.addAll(addedIons);
        }
        logger.info(" Adding solvent and ion atoms to system...");
        long time = -System.nanoTime();
        for (Atom[] atoms : newMolecules) {
            for (Atom atom : atoms) {
                atom.setHetero(true);
                this.activeAssembly.addMSNode(atom);
            }
        }
        logger.info(String.format(" Solvent and ions added in %12.4g sec", (double)(time += System.nanoTime()) * 1.0E-9));
        time = -System.nanoTime();
        String solvatedName = this.activeAssembly.getFile().getPath().replaceFirst("\\.[^.]+$", ".pdb");
        this.createdFile = new File(solvatedName);
        if (!soluteCrystal.aperiodic()) {
            for (int i = 0; i < nSolute; ++i) {
                Atom ati = soluteAtoms[i];
                ati.setXYZ(soluteCoordinates[i]);
            }
        }
        this.potentialFunctions.saveAsPDB(this.activeAssembly, this.createdFile);
        logger.info(String.format(" Structure written to disc in %12.4g sec", (double)(time += System.nanoTime()) * 1.0E-9));
        return this;
    }

    public File getWrittenFile() {
        return this.createdFile;
    }

    @Override
    public List<Potential> getPotentials() {
        ArrayList<Potential> potentials = new ArrayList<Potential>(3);
        if (this.ions != null) {
            potentials.add((Potential)this.ions.getPotentialEnergy());
        }
        if (this.solvent != null) {
            potentials.add((Potential)this.solvent.getPotentialEnergy());
        }
        if (this.solute != null) {
            potentials.add((Potential)this.solute.getPotentialEnergy());
        }
        return potentials;
    }

    private static double[] getCOM(Atom[] atoms) {
        double totMass = 0.0;
        double[] com = new double[3];
        double[] xyz = new double[3];
        for (Atom atom : atoms) {
            xyz = atom.getXYZ(xyz);
            double mass = atom.getMass();
            double invTotMass = 1.0 / (totMass += mass);
            for (int i = 0; i < 3; ++i) {
                int n = i;
                xyz[n] = xyz[n] - com[i];
                int n2 = i;
                xyz[n2] = xyz[n2] * (mass * invTotMass);
                int n3 = i;
                com[n3] = com[n3] + xyz[i];
            }
        }
        return com;
    }

    private static Atom[] swapIon(List<Atom[]> solvent, IonAddition ia, int currXYZIndex, char chain, int resSeq) {
        int nSolv = solvent.size() - 1;
        Atom[] lastSolvent = solvent.get(nSolv);
        solvent.remove(nSolv);
        double[] com = Solvator.getCOM(lastSolvent);
        return ia.createIon(com, currXYZIndex, chain, resSeq);
    }

    private static class DomainDecomposition {
        private static final Logger logger = Logger.getLogger(DomainDecomposition.class.getName());
        private final Atom[][][][] domains;
        private final Atom[][][][] withAdjacent;
        private final int nX;
        private final int nY;
        private final int nZ;
        private final double domainLength;
        private final Crystal crystal;

        private static Set<Integer> neighborsWithWraparound(int i, int nI) {
            ArrayList<Integer> boxesI = new ArrayList<Integer>(3);
            boxesI.add(i);
            if (i == 0) {
                boxesI.add(nI - 1);
                boxesI.add(nI - 2);
                boxesI.add(1);
            } else if (i == nI - 2) {
                boxesI.add(nI - 3);
                boxesI.add(nI - 1);
                boxesI.add(0);
            } else if (i == nI - 1) {
                boxesI.add(nI - 2);
                boxesI.add(0);
            } else {
                boxesI.add(i - 1);
                boxesI.add(i + 1);
            }
            HashSet<Integer> set = new HashSet<Integer>();
            for (Integer n : boxesI) {
                if (n < 0 || n >= nI) continue;
                set.add(n);
            }
            return set;
        }

        DomainDecomposition(double boundary, Atom[] soluteAtoms, Crystal crystal) {
            int k;
            int j;
            int i;
            double[] xyz;
            long time = -System.nanoTime();
            this.domainLength = 2.1 * boundary;
            this.nX = (int)(crystal.a / this.domainLength) + 1;
            this.nY = (int)(crystal.b / this.domainLength) + 1;
            this.nZ = (int)(crystal.c / this.domainLength) + 1;
            this.crystal = crystal;
            int[][][] nAts = new int[this.nX][this.nY][this.nZ];
            for (Atom at : soluteAtoms) {
                xyz = new double[3];
                xyz = at.getXYZ(xyz);
                i = (int)(xyz[0] / this.domainLength);
                j = (int)(xyz[1] / this.domainLength);
                k = (int)(xyz[2] / this.domainLength);
                int[] nArray = nAts[i][j];
                int n = k;
                nArray[n] = nArray[n] + 1;
            }
            this.domains = new Atom[this.nX][this.nY][this.nZ][];
            for (int i2 = 0; i2 < this.nX; ++i2) {
                for (int j2 = 0; j2 < this.nY; ++j2) {
                    for (int k2 = 0; k2 < this.nZ; ++k2) {
                        this.domains[i2][j2][k2] = new Atom[nAts[i2][j2][k2]];
                    }
                    Arrays.fill(nAts[i2][j2], 0);
                }
            }
            for (Atom at : soluteAtoms) {
                xyz = new double[3];
                xyz = at.getXYZ(xyz);
                i = (int)(xyz[0] / this.domainLength);
                j = (int)(xyz[1] / this.domainLength);
                k = (int)(xyz[2] / this.domainLength);
                int[] nArray = nAts[i][j];
                int n = k;
                int n2 = nArray[n];
                nArray[n] = n2 + 1;
                this.domains[i][j][k][n2] = at;
            }
            this.withAdjacent = new Atom[this.nX][this.nY][this.nZ][];
            for (int i3 = 0; i3 < this.nX; ++i3) {
                Set<Integer> neighborsI = DomainDecomposition.neighborsWithWraparound(i3, this.nX);
                for (int j3 = 0; j3 < this.nY; ++j3) {
                    Set<Integer> neighborsJ = DomainDecomposition.neighborsWithWraparound(j3, this.nY);
                    for (int k3 = 0; k3 < this.nZ; ++k3) {
                        Set<Integer> neighborsK = DomainDecomposition.neighborsWithWraparound(k3, this.nZ);
                        ArrayList<Atom> neighborAtoms = new ArrayList<Atom>();
                        for (int l : neighborsI) {
                            for (int m : neighborsJ) {
                                for (int n : neighborsK) {
                                    neighborAtoms.addAll(Arrays.asList(this.domains[l][m][n]));
                                }
                            }
                        }
                        logger.fine(String.format(" Constructing domain %3d-%3d-%3d with %d atoms.", i3, j3, k3, neighborAtoms.size()));
                        logger.fine(String.format(" Neighbors along X: %s", neighborsI));
                        logger.fine(String.format(" Neighbors along Y: %s", neighborsJ));
                        logger.fine(String.format(" Neighbors along Z: %s\n", neighborsK));
                        this.withAdjacent[i3][j3][k3] = new Atom[neighborAtoms.size()];
                        this.withAdjacent[i3][j3][k3] = neighborAtoms.toArray(this.withAdjacent[i3][j3][k3]);
                    }
                }
            }
            int nBoxes = this.nX * this.nY * this.nZ;
            double avgAtoms = (double)soluteAtoms.length / (double)nBoxes;
            logger.info(String.format(" Decomposed the solute into %d domains of side length %11.4g, averaging %10.3g atoms apiece in %8.3g sec.", nBoxes, this.domainLength, avgAtoms, (double)(time += System.nanoTime()) * 1.0E-9));
        }

        boolean checkClashes(Atom a, double threshold) {
            double[] xyz = new double[3];
            xyz = a.getXYZ(xyz);
            int i = (int)(xyz[0] / this.domainLength);
            int j = (int)(xyz[1] / this.domainLength);
            int k = (int)(xyz[2] / this.domainLength);
            if (i >= this.nX) {
                logger.fine(String.format(" Atom %s violates the X boundary at %12.7g Angstroms", a, xyz[0]));
                i = 0;
            }
            if (j >= this.nY) {
                logger.fine(String.format(" Atom %s violates the Y boundary at %12.7g Angstroms", a, xyz[1]));
                j = 0;
            }
            if (k >= this.nZ) {
                logger.fine(String.format(" Atom %s violates the Y boundary at %12.7g Angstroms", a, xyz[2]));
                k = 0;
            }
            double finalThreshold = threshold;
            double[] finalXyz = xyz;
            return Arrays.stream(this.withAdjacent[i][j][k]).anyMatch(s -> {
                double[] xyzS = new double[3];
                double dist = this.crystal.minDistOverSymOps(finalXyz, xyzS = s.getXYZ(xyzS));
                return dist < finalThreshold;
            });
        }
    }

    private static class IonAddition {
        private static final Logger logger = Logger.getLogger(IonAddition.class.getName());
        final double conc;
        final boolean toNeutralize;
        final Atom[] atoms;
        private final int nAts;
        final double[][] atomOffsets;
        final double charge;

        IonAddition(Atom[] atoms, double conc, boolean toNeutralize) {
            this.atoms = Arrays.copyOf(atoms, atoms.length);
            this.conc = conc;
            this.toNeutralize = toNeutralize;
            this.charge = Arrays.stream(atoms).mapToDouble(atom -> atom.getMultipoleType().getCharge()).sum();
            this.nAts = atoms.length;
            if (this.nAts > 1) {
                double[] com = new double[3];
                this.atomOffsets = new double[this.nAts][3];
                Arrays.fill(com, 0.0);
                double sumMass = 0.0;
                for (Atom atom2 : atoms) {
                    double mass = atom2.getMass();
                    sumMass += mass;
                    double[] xyz = new double[3];
                    xyz = atom2.getXYZ(xyz);
                    for (int i = 0; i < 3; ++i) {
                        int n = i;
                        xyz[n] = xyz[n] * mass;
                        int n2 = i;
                        com[n2] = com[n2] + xyz[i];
                    }
                }
                int i = 0;
                while (i < 3) {
                    int n = i++;
                    com[n] = com[n] / sumMass;
                }
                for (i = 0; i < this.nAts; ++i) {
                    Atom ai = atoms[i];
                    double[] xyz = new double[3];
                    xyz = ai.getXYZ(xyz);
                    for (int j = 0; j < 3; ++j) {
                        this.atomOffsets[i][j] = xyz[j] - com[j];
                    }
                }
            } else {
                this.atomOffsets = new double[1][3];
                Arrays.fill(this.atomOffsets[0], 0.0);
            }
        }

        Atom[] createIon(double[] com, int currXYZIndex, char chain, int resSeq) {
            Atom[] newIonAts = new Atom[this.nAts];
            for (int i = 0; i < this.nAts; ++i) {
                Atom fromAtom = this.atoms[i];
                double[] newXYZ = new double[3];
                System.arraycopy(com, 0, newXYZ, 0, 3);
                for (int j = 0; j < 3; ++j) {
                    int n = j;
                    newXYZ[n] = newXYZ[n] + this.atomOffsets[i][j];
                }
                Atom newAtom = new Atom(++currXYZIndex, fromAtom, newXYZ, resSeq, chain, Character.toString(chain));
                logger.fine(String.format(" New ion atom %s at chain %c on ion molecule %s-%d", newAtom, newAtom.getChainID(), newAtom.getResidueName(), newAtom.getResidueNumber()));
                newAtom.setHetero(true);
                newAtom.setResName(fromAtom.getResidueName());
                if (newAtom.getAltLoc() == null) {
                    newAtom.setAltLoc(Character.valueOf(' '));
                }
                newIonAts[i] = newAtom;
            }
            return newIonAts;
        }

        public String toString() {
            StringBuilder sb = new StringBuilder(String.format(" Ion addition with %d atoms per ion, concentration %10.3g mM, charge %6.2f, used to neutralize %b", this.atoms.length, this.conc, this.charge, this.toNeutralize));
            for (Atom atom : this.atoms) {
                sb.append("\n").append(" Includes atom ").append(atom.toString());
            }
            return sb.toString();
        }
    }
}

