/*
 * Decompiled with CFR 0.152.
 */
package ffx.xray.refine;

import ffx.numerics.math.ScalarMath;
import ffx.potential.MolecularAssembly;
import ffx.potential.bonded.Atom;
import ffx.potential.bonded.Bond;
import ffx.potential.bonded.MSNode;
import ffx.potential.bonded.Molecule;
import ffx.potential.bonded.Polymer;
import ffx.potential.bonded.Residue;
import ffx.xray.refine.RefinedBFactor;
import ffx.xray.refine.RefinedCoordinates;
import ffx.xray.refine.RefinedOccupancy;
import ffx.xray.refine.RefinedParameter;
import ffx.xray.refine.RefinementMode;
import java.util.ArrayList;
import java.util.IdentityHashMap;
import java.util.List;
import java.util.Map;
import java.util.logging.Logger;
import org.apache.commons.configuration2.CompositeConfiguration;

public class RefinementModel {
    private static final Logger logger = Logger.getLogger(RefinementModel.class.getName());
    private final MolecularAssembly[] molecularAssemblies;
    private RefinementMode refinementMode;
    private final boolean addAnisou;
    private final boolean byResidue;
    private final int nResiduePerBFactor;
    private final boolean ridingHydrogen;
    private final double bMass;
    private final boolean refineMolOcc;
    private final boolean resetHDOccupancy;
    private final double occMass;
    private final Atom[] scatteringAtoms;
    private final List<Atom> coordinateAtomList = new ArrayList<Atom>();
    private final Map<Atom, RefinedCoordinates> refinedCoordinates;
    private final List<Atom> bFactorAtomList = new ArrayList<Atom>();
    private final Map<Atom, RefinedBFactor> refinedBFactors;
    private final List<Atom[]> bFactorRestraints = new ArrayList<Atom[]>();
    private final List<Atom> occupancyAtomList = new ArrayList<Atom>();
    private final Map<Atom, RefinedOccupancy> refinedOccupancies;
    private final List<RefinedParameter> allParametersList;
    private final List<List<Residue>> altResidues;
    private final List<List<Molecule>> altMolecules;

    public RefinementModel(MolecularAssembly[] molecularAssemblies) {
        this(RefinementMode.COORDINATES_AND_BFACTORS_AND_OCCUPANCIES, molecularAssemblies);
    }

    public RefinementModel(RefinementMode refinementMode, MolecularAssembly[] molecularAssemblies) {
        this.refinementMode = refinementMode;
        this.molecularAssemblies = molecularAssemblies;
        MolecularAssembly rootAssembly = molecularAssemblies[0];
        CompositeConfiguration properties = rootAssembly.getProperties();
        this.addAnisou = properties.getBoolean("add-anisou", false);
        this.byResidue = properties.getBoolean("residue-bfactor", false);
        this.nResiduePerBFactor = properties.getInt("n-residue-bfactor", 1);
        this.ridingHydrogen = properties.getBoolean("riding-hydrogen-bfactor", true);
        this.bMass = properties.getDouble("bfactor-mass", 5.0);
        this.occMass = properties.getDouble("occupancy-mass", 10.0);
        this.resetHDOccupancy = properties.getBoolean("reset-hd-occupancy", false);
        this.refineMolOcc = properties.getBoolean("refine-mol-occ", false);
        if (this.addAnisou) {
            this.addAnisotropicBFactors();
        }
        if (refinementMode.includesBFactors()) {
            this.regularizeActiveAtoms();
        }
        ArrayList<Atom> scatteringList = new ArrayList<Atom>();
        this.refinedCoordinates = this.createCoordinateModel(scatteringList);
        this.scatteringAtoms = scatteringList.toArray(new Atom[0]);
        this.refinedBFactors = this.createBFactorModel();
        this.altResidues = new ArrayList<List<Residue>>();
        this.altMolecules = new ArrayList<List<Molecule>>();
        this.refinedOccupancies = this.createOccupancyModel();
        this.allParametersList = new ArrayList<RefinedParameter>();
        this.setRefinementMode(refinementMode);
    }

    public void setRefinementMode(RefinementMode mode) {
        this.refinementMode = mode;
        this.allParametersList.clear();
        if (this.refinementMode.includesCoordinates()) {
            for (Atom atom : this.coordinateAtomList) {
                this.allParametersList.add(this.refinedCoordinates.get(atom));
            }
        }
        if (this.refinementMode.includesBFactors()) {
            for (Atom atom : this.bFactorAtomList) {
                this.allParametersList.add(this.refinedBFactors.get(atom));
            }
            this.collectBFactorRestraints();
        } else {
            this.bFactorRestraints.clear();
        }
        if (this.refinementMode.includesOccupancies()) {
            for (Atom atom : this.occupancyAtomList) {
                this.allParametersList.add(this.refinedOccupancies.get(atom));
            }
        }
        this.setParameterIndices();
    }

    public Atom[] getScatteringAtoms() {
        return this.scatteringAtoms;
    }

    public Atom[] getActiveAtoms() {
        return this.coordinateAtomList.toArray(new Atom[0]);
    }

    public List<List<Molecule>> getAltMolecules() {
        return this.altMolecules;
    }

    public List<List<Residue>> getAltResidues() {
        return this.altResidues;
    }

    public MolecularAssembly[] getMolecularAssemblies() {
        return this.molecularAssemblies;
    }

    public List<Atom[]> getBFactorRestraints() {
        return this.bFactorRestraints;
    }

    public void addAssemblyGradient(int assembly, double[] gradient) {
        MolecularAssembly molecularAssembly = this.molecularAssemblies[assembly];
        Atom[] activeAtoms = molecularAssembly.getActiveAtomArray();
        double[] xyz = new double[3];
        for (Atom a : activeAtoms) {
            int index = a.getXrayCoordIndex() * 3;
            a.getXYZGradient(xyz);
            int n = index;
            gradient[n] = gradient[n] + xyz[0];
            int n2 = index + 1;
            gradient[n2] = gradient[n2] + xyz[1];
            int n3 = index + 2;
            gradient[n3] = gradient[n3] + xyz[2];
        }
    }

    public String toString() {
        int nAtoms = this.scatteringAtoms.length;
        int nActive = this.coordinateAtomList.size();
        int nUse = 0;
        for (Atom a : this.scatteringAtoms) {
            if (!a.getUse()) continue;
            ++nUse;
        }
        int nXYZ = this.getNumCoordParameters();
        int nBFactors = this.getNumBFactorParameters();
        int nOccupancies = this.getNumOccupancyParameters();
        int n = nXYZ + nBFactors + nOccupancies;
        StringBuilder sb = new StringBuilder("\n Refinement Model\n");
        sb.append(String.format("  Number of atoms:        %d\n", nAtoms));
        sb.append(String.format("  Atoms being used:       %d\n", nUse));
        sb.append(String.format("  Number of active atoms: %d\n", nActive));
        sb.append(String.format("  Number of variables:    %d (nXYZ %d, nB %d, nOcc %d)\n", n, nXYZ, nBFactors, nOccupancies));
        return sb.toString();
    }

    public RefinementMode getRefinementMode() {
        return this.refinementMode;
    }

    public List<RefinedParameter> getRefinedParameters() {
        return this.allParametersList;
    }

    public int getNumParameters() {
        return this.getNumCoordParameters() + this.getNumBFactorParameters() + this.getNumOccupancyParameters();
    }

    public int getNumCoordParameters() {
        if (this.refinementMode.includesCoordinates()) {
            return this.coordinateAtomList.size() * 3;
        }
        return 0;
    }

    public int getNumBFactorParameters() {
        int num = 0;
        if (this.refinementMode.includesBFactors()) {
            for (RefinedBFactor bFactor : this.refinedBFactors.values()) {
                num += bFactor.getNumberOfParameters();
            }
        }
        return num;
    }

    public int getNumOccupancyParameters() {
        if (this.refinementMode.includesOccupancies()) {
            return this.occupancyAtomList.size();
        }
        return 0;
    }

    public int getNumANISOU() {
        int numANISOU = 0;
        for (RefinedBFactor bFactor : this.refinedBFactors.values()) {
            if (!bFactor.isAnisou()) continue;
            ++numANISOU;
        }
        return numANISOU;
    }

    public void getParameters(double[] x) {
        for (RefinedParameter parameter : this.allParametersList) {
            parameter.getParameters(x);
        }
    }

    public void setParameters(double[] x) {
        for (RefinedParameter parameter : this.allParametersList) {
            parameter.setParameters(x);
        }
    }

    public void getVelocity(double[] x) {
        for (RefinedParameter parameter : this.allParametersList) {
            parameter.getVelocity(x);
        }
    }

    public void setVelocity(double[] x) {
        for (RefinedParameter parameter : this.allParametersList) {
            parameter.setVelocity(x);
        }
    }

    public void getAcceleration(double[] x) {
        for (RefinedParameter parameter : this.allParametersList) {
            parameter.getAcceleration(x);
        }
    }

    public void setAcceleration(double[] x) {
        for (RefinedParameter parameter : this.allParametersList) {
            parameter.setAcceleration(x);
        }
    }

    public void getPreviousAcceleration(double[] x) {
        for (RefinedParameter parameter : this.allParametersList) {
            parameter.getPreviousAcceleration(x);
        }
    }

    public void setPreviousAcceleration(double[] x) {
        for (RefinedParameter parameter : this.allParametersList) {
            parameter.setPreviousAcceleration(x);
        }
    }

    public void getMass(double[] mass) {
        for (RefinedParameter parameter : this.allParametersList) {
            if (parameter instanceof RefinedCoordinates) {
                parameter.getMass(mass, 5.0);
                continue;
            }
            if (parameter instanceof RefinedBFactor) {
                parameter.getMass(mass, this.bMass);
                continue;
            }
            if (!(parameter instanceof RefinedOccupancy)) continue;
            parameter.getMass(mass, this.occMass);
        }
    }

    public void loadOptimizationScaling(double[] optimizationScaling) {
        for (RefinedParameter parameter : this.allParametersList) {
            parameter.setOptimizationScaling(optimizationScaling);
        }
    }

    public void zeroGradient() {
        for (RefinedParameter parameter : this.allParametersList) {
            parameter.zeroGradient();
        }
    }

    public void getGradient(double[] gradient) {
        for (RefinedParameter parameter : this.allParametersList) {
            parameter.getGradient(gradient);
        }
    }

    private Map<Atom, RefinedCoordinates> createCoordinateModel(List<Atom> scatteringList) {
        Atom[] atomList;
        logger.fine("\n Creating Coordinate Refinement Model\n");
        MolecularAssembly rootAssembly = this.molecularAssemblies[0];
        IdentityHashMap<Atom, RefinedCoordinates> coordinateMap = new IdentityHashMap<Atom, RefinedCoordinates>();
        for (Atom a : atomList = rootAssembly.getAtomArray()) {
            scatteringList.add(a);
            if (!a.isActive()) continue;
            a.setXrayCoordIndex(this.coordinateAtomList.size());
            this.coordinateAtomList.add(a);
            coordinateMap.put(a, new RefinedCoordinates(a));
            logger.fine(" Active: " + String.valueOf(a));
        }
        for (int i = 1; i < this.molecularAssemblies.length; ++i) {
            MolecularAssembly molecularAssembly = this.molecularAssemblies[i];
            for (Atom a : atomList = molecularAssembly.getAtomArray()) {
                RefinedCoordinates refinedCoordinates;
                Character altLoc = a.getAltLoc();
                Atom rootAtom = rootAssembly.findAtom(a, false);
                Atom deuteriumMatch = rootAssembly.findAtom(a, true);
                if (rootAtom != null && rootAtom.getAltLoc().equals(altLoc)) {
                    if (rootAtom.isActive()) {
                        refinedCoordinates = (RefinedCoordinates)coordinateMap.get(rootAtom);
                        refinedCoordinates.addConstrainedAtom(a);
                        a.setXrayCoordIndex(rootAtom.getXrayCoordIndex());
                        continue;
                    }
                    a.setActive(false);
                    continue;
                }
                if (deuteriumMatch != null) {
                    scatteringList.add(a);
                    if (deuteriumMatch.isActive()) {
                        refinedCoordinates = (RefinedCoordinates)coordinateMap.get(deuteriumMatch);
                        refinedCoordinates.addConstrainedAtomThatScatters(a);
                        a.setXrayCoordIndex(deuteriumMatch.getXrayCoordIndex());
                        continue;
                    }
                    a.setActive(false);
                    continue;
                }
                scatteringList.add(a);
                if (!a.isActive()) continue;
                a.setXrayCoordIndex(this.coordinateAtomList.size());
                this.coordinateAtomList.add(a);
                coordinateMap.put(a, new RefinedCoordinates(a));
            }
        }
        return coordinateMap;
    }

    private Map<Atom, RefinedBFactor> createBFactorModel() {
        logger.fine("\n Creating B-Factor Refinement Model\n");
        MolecularAssembly rootAssembly = this.molecularAssemblies[0];
        IdentityHashMap<Atom, RefinedBFactor> bFactorMap = new IdentityHashMap<Atom, RefinedBFactor>();
        if (this.byResidue) {
            Object heavyAtom;
            Polymer[] polymers;
            for (Polymer polymer : polymers = rootAssembly.getChains()) {
                List residues = polymer.getResidues();
                heavyAtom = null;
                RefinedBFactor currentRefinedBFactor = null;
                for (int j = 0; j < residues.size(); ++j) {
                    Residue residue = (Residue)residues.get(j);
                    if (j % this.nResiduePerBFactor == 0 || heavyAtom == null) {
                        heavyAtom = residue.getFirstActiveHeavyAtom();
                        if (heavyAtom == null) continue;
                        this.bFactorAtomList.add((Atom)heavyAtom);
                        currentRefinedBFactor = new RefinedBFactor((Atom)heavyAtom);
                        bFactorMap.put((Atom)heavyAtom, currentRefinedBFactor);
                    }
                    for (Atom a : residue.getAtomList()) {
                        if (a == heavyAtom) continue;
                        currentRefinedBFactor.addConstrainedAtomThatScatters(a);
                    }
                }
            }
            List molecules = rootAssembly.getNodeList(true);
            for (MSNode m : molecules) {
                Atom heavyAtom2 = m.getFirstActiveHeavyAtom();
                if (heavyAtom2 == null) continue;
                this.bFactorAtomList.add(heavyAtom2);
                RefinedBFactor currentRefinedBFactor = new RefinedBFactor(heavyAtom2);
                bFactorMap.put(heavyAtom2, currentRefinedBFactor);
                for (Atom a : m.getAtomList()) {
                    if (a == heavyAtom2) continue;
                    currentRefinedBFactor.addConstrainedAtomThatScatters(a);
                }
            }
            for (int i = 1; i < this.molecularAssemblies.length; ++i) {
                MolecularAssembly molecularAssembly = this.molecularAssemblies[i];
                polymers = molecularAssembly.getChains();
                for (int j = 0; j < polymers.length; ++j) {
                    Polymer polymer = polymers[j];
                    List residues = polymer.getResidues();
                    for (int k = 0; k < residues.size(); ++k) {
                        Residue residue = (Residue)residues.get(k);
                        Residue rootResidue = rootAssembly.getResidue(j, k);
                        if (rootResidue == null) {
                            logger.severe(String.format(" Residue %s not found in the root conformation.", residue));
                            return null;
                        }
                        Atom rootAtom = rootResidue.getFirstActiveHeavyAtom();
                        if (rootAtom == null) continue;
                        RefinedBFactor refinedBFactor = (RefinedBFactor)bFactorMap.get(rootAtom);
                        for (Atom a : residue.getAtomList()) {
                            Character altLoc = a.getAltLoc();
                            if (!altLoc.equals(rootAtom.getAltLoc())) {
                                refinedBFactor.addConstrainedAtomThatScatters(a);
                                continue;
                            }
                            refinedBFactor.addConstrainedAtom(a);
                        }
                    }
                }
                molecules = this.molecularAssemblies[i].getNodeList(true);
                for (MSNode m : molecules) {
                    Character altLoc;
                    heavyAtom = m.getFirstActiveHeavyAtom();
                    if (heavyAtom == null || (altLoc = heavyAtom.getAltLoc()).equals(Character.valueOf(' ')) || altLoc.equals(Character.valueOf('A'))) continue;
                    this.bFactorAtomList.add((Atom)heavyAtom);
                    RefinedBFactor refinedBFactor = new RefinedBFactor((Atom)heavyAtom);
                    bFactorMap.put((Atom)heavyAtom, refinedBFactor);
                    for (Atom a : m.getAtomList()) {
                        if (a == heavyAtom) continue;
                        refinedBFactor.addConstrainedAtomThatScatters(a);
                    }
                }
            }
        } else if (this.ridingHydrogen) {
            for (Atom atom : this.coordinateAtomList) {
                if (atom.isHydrogen() && !atom.isDeuterium()) continue;
                this.bFactorAtomList.add(atom);
                RefinedBFactor refinedBFactor = new RefinedBFactor(atom);
                bFactorMap.put(atom, refinedBFactor);
                RefinedCoordinates refinedCoords = this.refinedCoordinates.get(atom);
                for (Atom a : refinedCoords.constrainedAtoms) {
                    refinedBFactor.addConstrainedAtom(a);
                }
                for (Atom a : refinedCoords.constrainedAtomsThatScatter) {
                    refinedBFactor.addConstrainedAtomThatScatters(a);
                }
            }
            for (Atom atom : this.coordinateAtomList) {
                Atom heavy;
                if (!atom.isHydrogen() || atom.isDeuterium() || !(heavy = ((Bond)atom.getBonds().getFirst()).get1_2(atom)).isActive()) continue;
                if (bFactorMap.containsKey(heavy)) {
                    RefinedBFactor refinedBFactor = (RefinedBFactor)bFactorMap.get(heavy);
                    refinedBFactor.addConstrainedAtomThatScatters(atom);
                    continue;
                }
                logger.info(" Could not locate a heavy atom B-factor for: " + String.valueOf(atom));
            }
        } else {
            for (Atom atom : this.coordinateAtomList) {
                this.bFactorAtomList.add(atom);
                RefinedBFactor refinedBFactor = new RefinedBFactor(atom);
                bFactorMap.put(atom, refinedBFactor);
                RefinedCoordinates refinedCoords = this.refinedCoordinates.get(atom);
                for (Atom a : refinedCoords.constrainedAtoms) {
                    refinedBFactor.addConstrainedAtom(a);
                }
            }
        }
        return bFactorMap;
    }

    private void collectBFactorRestraints() {
        this.bFactorRestraints.clear();
        MolecularAssembly rootAssembly = this.molecularAssemblies[0];
        List rootBonds = rootAssembly.getBondList();
        for (Bond bond : rootBonds) {
            Atom a1 = bond.getAtom(0);
            Atom a2 = bond.getAtom(1);
            if (!a1.isActive() && !a2.isActive()) continue;
            this.bFactorRestraints.add(new Atom[]{a1, a2});
        }
        for (int i = 1; i < this.molecularAssemblies.length; ++i) {
            MolecularAssembly molecularAssembly = this.molecularAssemblies[i];
            Character altLoc = molecularAssembly.getAlternateLocation();
            List bonds = molecularAssembly.getBondList();
            for (Bond bond : bonds) {
                Atom a1 = bond.getAtom(0);
                Atom a2 = bond.getAtom(1);
                if (!a1.isActive() && !a2.isActive()) continue;
                if (a1.getAltLoc().equals(altLoc) && a2.getAltLoc().equals(altLoc)) {
                    this.bFactorRestraints.add(new Atom[]{a1, a2});
                    continue;
                }
                if (a1.getAltLoc().equals(altLoc) && !a2.getAltLoc().equals(altLoc)) {
                    if ((a2 = rootAssembly.findAtom(a2)) == null) continue;
                    this.bFactorRestraints.add(new Atom[]{a1, a2});
                    continue;
                }
                if (a1.getAltLoc().equals(altLoc) || !a2.getAltLoc().equals(altLoc) || (a1 = rootAssembly.findAtom(a1)) == null) continue;
                this.bFactorRestraints.add(new Atom[]{a1, a2});
            }
        }
    }

    private Map<Atom, RefinedOccupancy> createOccupancyModel() {
        List molecules;
        logger.fine("\n Creating Occupancy Refinement Model\n");
        IdentityHashMap<Atom, RefinedOccupancy> refinedOccupancies = new IdentityHashMap<Atom, RefinedOccupancy>();
        boolean refineDeuterium = false;
        for (MolecularAssembly molecularAssembly : this.molecularAssemblies) {
            if (!molecularAssembly.hasDeuterium()) continue;
            refineDeuterium = true;
            break;
        }
        if (refineDeuterium) {
            MolecularAssembly rootAssembly = this.molecularAssemblies[0];
            Polymer[] polymers = rootAssembly.getChains();
            if (polymers != null) {
                for (Polymer polymer : polymers) {
                    List residues = polymer.getResidues();
                    for (Residue residue : residues) {
                        List atoms = residue.getAtomList();
                        for (Atom atom : atoms) {
                            RefinedOccupancy refinedOccupancy;
                            double occupancy;
                            if (!atom.isActive() || !atom.isHydrogen() || !((occupancy = atom.getOccupancy()) < 1.0)) continue;
                            Atom heavy = ((Bond)atom.getBonds().getFirst()).get1_2(atom);
                            if (refinedOccupancies.containsKey(heavy)) {
                                refinedOccupancy = (RefinedOccupancy)refinedOccupancies.get(heavy);
                                refinedOccupancy.addConstrainedAtomThatScatters(atom);
                                continue;
                            }
                            refinedOccupancy = new RefinedOccupancy(atom);
                            refinedOccupancies.put(heavy, refinedOccupancy);
                            this.occupancyAtomList.add(heavy);
                        }
                    }
                }
            }
            if (this.molecularAssemblies.length > 1) {
                MolecularAssembly molecularAssembly = this.molecularAssemblies[1];
                for (Atom heavy : this.occupancyAtomList) {
                    RefinedOccupancy refinedOccupancy = (RefinedOccupancy)refinedOccupancies.get(heavy);
                    ArrayList<Atom> atoms = new ArrayList<Atom>(refinedOccupancy.constrainedAtomsThatScatter);
                    atoms.add(refinedOccupancy.atom);
                    for (Atom atom : atoms) {
                        Atom match = molecularAssembly.findAtom(atom, true);
                        if (match == null) continue;
                        double o1 = atom.getOccupancy();
                        double o2 = match.getOccupancy();
                        if (this.resetHDOccupancy) {
                            logger.info(" Reset Occupancy for H/D Pair to 0.5/0.5:");
                            atom.setOccupancy(0.5);
                            match.setOccupancy(0.5);
                            logger.info(String.format(" %s: %6.3f", atom, atom.getOccupancy()));
                            logger.info(String.format(" %s: %6.3f", match, match.getOccupancy()));
                        } else if (o1 + o2 != 1.0) {
                            logger.info(" Occupancy Sum for H/D Pair is not 1.0:");
                            logger.info(String.format(" %s: %6.3f", atom, atom.getOccupancy()));
                            logger.info(String.format(" %s: %6.3f", match, match.getOccupancy()));
                            double delta = (1.0 - o1 - o2) / 2.0;
                            atom.setOccupancy(o1 + delta);
                            match.setOccupancy(o2 + delta);
                            logger.info(" Occupancy Sum for H/D Pair adjusted to 1.0:");
                            logger.info(String.format(" %s: %6.3f", atom, atom.getOccupancy()));
                            logger.info(String.format(" %s: %6.3f", match, match.getOccupancy()));
                        }
                        refinedOccupancy.addConstrainedAtomThatScattersComplement(match);
                    }
                }
            }
        } else {
            Polymer[] polymers = this.molecularAssemblies[0].getChains();
            if (polymers != null && polymers.length > 0) {
                for (int i = 0; i < polymers.length; ++i) {
                    List residues = polymers[i].getResidues();
                    for (int j = 0; j < residues.size(); ++j) {
                        List<Residue> list = this.getResidueConformers(i, j);
                        if (list == null || list.isEmpty()) continue;
                        this.altResidues.add(list);
                        for (Residue residue : list) {
                            double occupancy;
                            Character altLoc;
                            List atomList = residue.getAtomList();
                            RefinedOccupancy refinedOccupancy = null;
                            Atom refinedAtom = null;
                            for (Atom a : atomList) {
                                altLoc = a.getAltLoc();
                                occupancy = a.getOccupancy();
                                if (altLoc.equals(Character.valueOf(' ')) && !(occupancy < 1.0)) continue;
                                this.occupancyAtomList.add(a);
                                refinedOccupancy = new RefinedOccupancy(a);
                                refinedOccupancies.put(a, refinedOccupancy);
                                refinedAtom = a;
                                break;
                            }
                            if (refinedAtom == null) continue;
                            for (Atom a : atomList) {
                                if (a == refinedAtom) continue;
                                altLoc = a.getAltLoc();
                                occupancy = a.getOccupancy();
                                if (altLoc.equals(Character.valueOf(' ')) && !(occupancy < 1.0)) continue;
                                refinedOccupancy.addConstrainedAtomThatScatters(a);
                            }
                        }
                    }
                }
            }
        }
        if (this.refineMolOcc && (molecules = this.molecularAssemblies[0].getMolecules()) != null && !molecules.isEmpty()) {
            for (int i = 0; i < molecules.size(); ++i) {
                List<Molecule> list = this.getMoleculeConformers(i);
                if (list == null || list.isEmpty()) continue;
                this.altMolecules.add(list);
                for (Molecule molecule : list) {
                    double occupancy;
                    Character altLoc;
                    List atomList = molecule.getAtomList();
                    RefinedOccupancy refinedOccupancy = null;
                    Atom refinedAtom = null;
                    for (Atom a : atomList) {
                        altLoc = a.getAltLoc();
                        occupancy = a.getOccupancy();
                        if (altLoc.equals(Character.valueOf(' ')) && !(occupancy < 1.0)) continue;
                        this.occupancyAtomList.add(a);
                        refinedOccupancy = new RefinedOccupancy(a);
                        refinedOccupancies.put(a, refinedOccupancy);
                        logger.info(" Refined Occupancy: " + String.valueOf(a));
                        refinedAtom = a;
                        break;
                    }
                    if (refinedAtom == null) continue;
                    for (Atom a : atomList) {
                        if (a == refinedAtom) continue;
                        altLoc = a.getAltLoc();
                        occupancy = a.getOccupancy();
                        if (altLoc.equals(Character.valueOf(' ')) && !(occupancy < 1.0)) continue;
                        refinedOccupancy.addConstrainedAtomThatScatters(a);
                        logger.info("  Constrained Occupancy: " + String.valueOf(a));
                    }
                }
            }
        }
        return refinedOccupancies;
    }

    private List<Residue> getResidueConformers(int polymerID, int resID) {
        if (this.molecularAssemblies.length < 2) {
            return null;
        }
        double totalOccupancy = 0.0;
        ArrayList<Residue> residues = new ArrayList<Residue>();
        Residue residue = this.molecularAssemblies[0].getResidue(polymerID, resID);
        for (Atom a : residue.getAtomList()) {
            if (!a.getUse()) continue;
            Character altLoc = a.getAltLoc();
            double occupancy = a.getOccupancy();
            if (altLoc.equals(Character.valueOf(' ')) && !(occupancy < 1.0)) continue;
            logger.fine(String.format(" %s %c %5.3f", residue, altLoc, occupancy));
            totalOccupancy = occupancy;
            residues.add(residue);
            break;
        }
        if (residues.isEmpty()) {
            return null;
        }
        int numConformers = this.molecularAssemblies.length;
        block1: for (int i = 1; i < numConformers; ++i) {
            residue = this.molecularAssemblies[i].getResidue(polymerID, resID);
            for (Atom a : residue.getAtomList()) {
                Character altLoc;
                if (!a.getUse() || (altLoc = a.getAltLoc()).equals(Character.valueOf(' ')) || altLoc.equals(Character.valueOf('A'))) continue;
                double occupancy = a.getOccupancy();
                totalOccupancy += occupancy;
                residues.add(residue);
                logger.fine(String.format(" %s %c %5.3f", residue, altLoc, occupancy));
                continue block1;
            }
        }
        logger.fine("  Total occupancy: " + totalOccupancy);
        return residues;
    }

    private List<Molecule> getMoleculeConformers(int moleculeID) {
        ArrayList<Molecule> molecules = new ArrayList<Molecule>();
        double totalOccupancy = 0.0;
        List molList = this.molecularAssemblies[0].getMolecules();
        if (molList != null && !molList.isEmpty()) {
            Molecule molecule = (Molecule)molList.get(moleculeID);
            for (Atom a : molecule.getAtomList()) {
                if (!a.getUse()) continue;
                Character altLoc = a.getAltLoc();
                double occupancy = a.getOccupancy();
                if (altLoc.equals(Character.valueOf(' ')) && !(occupancy < 1.0)) continue;
                totalOccupancy = occupancy;
                molecules.add(molecule);
                logger.fine(String.format(" %s %c %5.3f", molecule, altLoc, occupancy));
                break;
            }
        }
        if (molecules.isEmpty()) {
            return null;
        }
        int numConformers = this.molecularAssemblies.length;
        block1: for (int i = 1; i < numConformers; ++i) {
            molList = this.molecularAssemblies[i].getMolecules();
            Molecule molecule = (Molecule)molList.get(moleculeID);
            for (Atom a : molecule.getAtomList()) {
                Character altLoc;
                if (!a.getUse() || (altLoc = a.getAltLoc()).equals(Character.valueOf(' ')) || altLoc.equals(Character.valueOf('A'))) continue;
                double occupancy = a.getOccupancy();
                totalOccupancy += occupancy;
                molecules.add(molecule);
                logger.fine(String.format(" %s %c %5.3f", molecule, altLoc, occupancy));
                continue block1;
            }
        }
        logger.fine("  Total occupancy: " + totalOccupancy);
        return molecules;
    }

    private void setParameterIndices() {
        int index = 0;
        for (RefinedParameter parameter : this.allParametersList) {
            parameter.setIndex(index);
            index += parameter.getNumberOfParameters();
        }
    }

    private void regularizeActiveAtoms() {
        for (MolecularAssembly molecularAssembly : this.molecularAssemblies) {
            if (this.ridingHydrogen) {
                for (Atom atom : molecularAssembly.getAtomList()) {
                    if (!atom.isHydrogen()) continue;
                    Atom other = ((Bond)atom.getBonds().getFirst()).get1_2(atom);
                    atom.setActive(other.isActive());
                }
            }
            if (!this.byResidue) continue;
            List nodeList = molecularAssembly.getNodeList();
            for (MSNode node : nodeList) {
                Atom activeAtom = node.getFirstActiveHeavyAtom();
                if (activeAtom != null) {
                    for (Atom atom : node.getAtomList()) {
                        atom.setActive(true);
                    }
                    continue;
                }
                for (Atom atom : node.getAtomList()) {
                    atom.setActive(false);
                }
            }
        }
    }

    private void addAnisotropicBFactors() {
        if (this.addAnisou) {
            for (MolecularAssembly molecularAssembly : this.molecularAssemblies) {
                int count = 0;
                List atomList = molecularAssembly.getAtomList();
                for (Atom a : atomList) {
                    if (!a.isHeavy() || !a.isActive() || a.getAnisou(null) != null) continue;
                    double u = ScalarMath.b2u((double)a.getTempFactor());
                    double[] anisou = new double[]{u, u, u, 0.0, 0.0, 0.0};
                    a.setAnisou(anisou);
                    ++count;
                }
                if (count <= 0) continue;
                Character c = molecularAssembly.getAlternateLocation();
                logger.info(String.format(" %d anisotropic B-factors were added to conformer %c.", count, c));
            }
        }
    }
}

