/*
 * Decompiled with CFR 0.152.
 */
package ffx.numerics.quickhull;

import ffx.numerics.quickhull.Face;
import ffx.numerics.quickhull.FaceList;
import ffx.numerics.quickhull.HalfEdge;
import ffx.numerics.quickhull.Point3d;
import ffx.numerics.quickhull.Vector3d;
import ffx.numerics.quickhull.Vertex;
import ffx.numerics.quickhull.VertexList;
import java.io.PrintStream;
import java.util.Arrays;
import java.util.Iterator;
import java.util.Vector;
import java.util.logging.Level;
import java.util.logging.Logger;

public class QuickHull3D {
    private static final Logger logger = Logger.getLogger(QuickHull3D.class.getName());
    public static final int CLOCKWISE = 1;
    public static final int INDEXED_FROM_ONE = 2;
    public static final int INDEXED_FROM_ZERO = 4;
    public static final int POINT_RELATIVE = 8;
    public static final double AUTOMATIC_TOLERANCE = -1.0;
    protected int findIndex = -1;
    protected double charLength;
    protected Vertex[] pointBuffer = new Vertex[0];
    protected int[] vertexPointIndices = new int[0];
    private Face[] discardedFaces = new Face[3];
    private Vertex[] maxVtxs = new Vertex[3];
    private Vertex[] minVtxs = new Vertex[3];
    protected Vector<Face> faces = new Vector(16);
    protected Vector<HalfEdge> horizon = new Vector(16);
    private FaceList newFaces = new FaceList();
    private VertexList unclaimed = new VertexList();
    private VertexList claimed = new VertexList();
    protected int numVertices;
    protected int numFaces;
    protected int numPoints;
    protected double explicitTolerance = -1.0;
    protected double tolerance;
    private static final double DOUBLE_PREC = 2.220446049250313E-16;
    private static final int NONCONVEX_WRT_LARGER_FACE = 1;
    private static final int NONCONVEX = 2;

    public double getDistanceTolerance() {
        return this.tolerance;
    }

    public void setExplicitDistanceTolerance(double tol) {
        this.explicitTolerance = tol;
    }

    public double getExplicitDistanceTolerance() {
        return this.explicitTolerance;
    }

    private void addPointToFace(Vertex vtx, Face face) {
        vtx.face = face;
        if (face.outside == null) {
            this.claimed.add(vtx);
        } else {
            this.claimed.insertBefore(vtx, face.outside);
        }
        face.outside = vtx;
    }

    private void removePointFromFace(Vertex vtx, Face face) {
        if (vtx == face.outside) {
            face.outside = vtx.next != null && vtx.next.face == face ? vtx.next : null;
        }
        this.claimed.delete(vtx);
    }

    private Vertex removeAllPointsFromFace(Face face) {
        if (face.outside != null) {
            Vertex end = face.outside;
            while (end.next != null && end.next.face == face) {
                end = end.next;
            }
            this.claimed.delete(face.outside, end);
            end.next = null;
            return face.outside;
        }
        return null;
    }

    public QuickHull3D() {
    }

    public QuickHull3D(double[] coords) throws IllegalArgumentException {
        this.build(coords, coords.length / 3);
    }

    public QuickHull3D(Point3d[] points) throws IllegalArgumentException {
        this.build(points, points.length);
    }

    private HalfEdge findHalfEdge(Vertex tail, Vertex head) {
        for (Face face : this.faces) {
            HalfEdge he = face.findEdge(tail, head);
            if (he == null) continue;
            return he;
        }
        return null;
    }

    protected void setHull(double[] coords, int nump, int[][] faceIndices, int numf) {
        this.initBuffers(nump);
        this.setPoints(coords, nump);
        this.computeMaxAndMin();
        for (int i = 0; i < numf; ++i) {
            Face face = Face.create(this.pointBuffer, faceIndices[i]);
            HalfEdge he = face.he0;
            do {
                HalfEdge heOpp;
                if ((heOpp = this.findHalfEdge(he.head(), he.tail())) == null) continue;
                he.setOpposite(heOpp);
            } while ((he = he.next) != face.he0);
            this.faces.add(face);
        }
    }

    public void printPoints(PrintStream ps) {
        for (int i = 0; i < this.numPoints; ++i) {
            Point3d pnt = this.pointBuffer[i].pnt;
            ps.println(pnt.x + ", " + pnt.y + ", " + pnt.z + ",");
        }
    }

    public void build(double[] coords) throws IllegalArgumentException {
        this.build(coords, coords.length / 3);
    }

    public void build(double[] coords, int nump) throws IllegalArgumentException {
        if (nump < 4) {
            throw new IllegalArgumentException("Less than four input points specified");
        }
        if (coords.length / 3 < nump) {
            throw new IllegalArgumentException("Coordinate array too small for specified number of points");
        }
        this.initBuffers(nump);
        this.setPoints(coords, nump);
        this.buildHull();
    }

    public void build(Point3d[] points) throws IllegalArgumentException {
        this.build(points, points.length);
    }

    public void build(Point3d[] points, int nump) throws IllegalArgumentException {
        if (nump < 4) {
            throw new IllegalArgumentException("Less than four input points specified");
        }
        if (points.length < nump) {
            throw new IllegalArgumentException("Point array too small for specified number of points");
        }
        this.initBuffers(nump);
        this.setPoints(points, nump);
        this.buildHull();
    }

    public void triangulate() {
        double minArea = 1000.0 * this.charLength * 2.220446049250313E-16;
        this.newFaces.clear();
        for (Face face : this.faces) {
            if (face.mark != 1) continue;
            face.triangulate(this.newFaces, minArea);
        }
        Face face = this.newFaces.first();
        while (face != null) {
            this.faces.add(face);
            face = face.next;
        }
    }

    protected void initBuffers(int nump) {
        if (this.pointBuffer.length < nump) {
            Vertex[] newBuffer = new Vertex[nump];
            this.vertexPointIndices = new int[nump];
            System.arraycopy(this.pointBuffer, 0, newBuffer, 0, this.pointBuffer.length);
            for (int i = this.pointBuffer.length; i < nump; ++i) {
                newBuffer[i] = new Vertex();
            }
            this.pointBuffer = newBuffer;
        }
        this.faces.clear();
        this.claimed.clear();
        this.numFaces = 0;
        this.numPoints = nump;
    }

    protected void setPoints(double[] coords, int nump) {
        int i = 0;
        while (i < nump) {
            Vertex vtx = this.pointBuffer[i];
            vtx.pnt.set(coords[i * 3 + 0], coords[i * 3 + 1], coords[i * 3 + 2]);
            vtx.index = i++;
        }
    }

    protected void setPoints(Point3d[] pnts, int nump) {
        int i = 0;
        while (i < nump) {
            Vertex vtx = this.pointBuffer[i];
            vtx.pnt.set(pnts[i]);
            vtx.index = i++;
        }
    }

    protected void computeMaxAndMin() {
        int i;
        Vector3d max = new Vector3d();
        Vector3d min = new Vector3d();
        for (i = 0; i < 3; ++i) {
            this.maxVtxs[i] = this.minVtxs[i] = this.pointBuffer[0];
        }
        max.set(this.pointBuffer[0].pnt);
        min.set(this.pointBuffer[0].pnt);
        for (i = 1; i < this.numPoints; ++i) {
            Point3d pnt = this.pointBuffer[i].pnt;
            if (pnt.x > max.x) {
                max.x = pnt.x;
                this.maxVtxs[0] = this.pointBuffer[i];
            } else if (pnt.x < min.x) {
                min.x = pnt.x;
                this.minVtxs[0] = this.pointBuffer[i];
            }
            if (pnt.y > max.y) {
                max.y = pnt.y;
                this.maxVtxs[1] = this.pointBuffer[i];
            } else if (pnt.y < min.y) {
                min.y = pnt.y;
                this.minVtxs[1] = this.pointBuffer[i];
            }
            if (pnt.z > max.z) {
                max.z = pnt.z;
                this.maxVtxs[2] = this.pointBuffer[i];
                continue;
            }
            if (!(pnt.z < min.z)) continue;
            min.z = pnt.z;
            this.minVtxs[2] = this.pointBuffer[i];
        }
        this.charLength = Math.max(max.x - min.x, max.y - min.y);
        this.charLength = Math.max(max.z - min.z, this.charLength);
        this.tolerance = this.explicitTolerance == -1.0 ? 6.661338147750939E-16 * (Math.max(Math.abs(max.x), Math.abs(min.x)) + Math.max(Math.abs(max.y), Math.abs(min.y)) + Math.max(Math.abs(max.z), Math.abs(min.z))) : this.explicitTolerance;
    }

    protected void createInitialSimplex() throws IllegalArgumentException {
        int i;
        double max = 0.0;
        int imax = 0;
        for (int i2 = 0; i2 < 3; ++i2) {
            double diff = this.maxVtxs[i2].pnt.get(i2) - this.minVtxs[i2].pnt.get(i2);
            if (!(diff > max)) continue;
            max = diff;
            imax = i2;
        }
        if (max <= this.tolerance) {
            throw new IllegalArgumentException("Input points appear to be coincident");
        }
        Vertex[] vtx = new Vertex[4];
        vtx[0] = this.maxVtxs[imax];
        vtx[1] = this.minVtxs[imax];
        Vector3d u01 = new Vector3d();
        Vector3d diff02 = new Vector3d();
        Vector3d nrml = new Vector3d();
        Vector3d xprod = new Vector3d();
        double maxSqr = 0.0;
        u01.sub(vtx[1].pnt, vtx[0].pnt);
        u01.normalize();
        for (int i3 = 0; i3 < this.numPoints; ++i3) {
            diff02.sub(this.pointBuffer[i3].pnt, vtx[0].pnt);
            xprod.cross(u01, diff02);
            double lenSqr = xprod.normSquared();
            if (!(lenSqr > maxSqr) || this.pointBuffer[i3] == vtx[0] || this.pointBuffer[i3] == vtx[1]) continue;
            maxSqr = lenSqr;
            vtx[2] = this.pointBuffer[i3];
            nrml.set(xprod);
        }
        if (Math.sqrt(maxSqr) <= 100.0 * this.tolerance) {
            throw new IllegalArgumentException("Input points appear to be colinear");
        }
        nrml.normalize();
        double maxDist = 0.0;
        double d0 = vtx[2].pnt.dot(nrml);
        for (int i4 = 0; i4 < this.numPoints; ++i4) {
            double dist = Math.abs(this.pointBuffer[i4].pnt.dot(nrml) - d0);
            if (!(dist > maxDist) || this.pointBuffer[i4] == vtx[0] || this.pointBuffer[i4] == vtx[1] || this.pointBuffer[i4] == vtx[2]) continue;
            maxDist = dist;
            vtx[3] = this.pointBuffer[i4];
        }
        if (Math.abs(maxDist) <= 100.0 * this.tolerance) {
            throw new IllegalArgumentException("Input points appear to be coplanar");
        }
        if (logger.isLoggable(Level.FINE)) {
            logger.fine("initial vertices:");
            logger.fine(vtx[0].index + ": " + String.valueOf(vtx[0].pnt));
            logger.fine(vtx[1].index + ": " + String.valueOf(vtx[1].pnt));
            logger.fine(vtx[2].index + ": " + String.valueOf(vtx[2].pnt));
            logger.fine(vtx[3].index + ": " + String.valueOf(vtx[3].pnt));
        }
        Face[] tris = new Face[4];
        if (vtx[3].pnt.dot(nrml) - d0 < 0.0) {
            tris[0] = Face.createTriangle(vtx[0], vtx[1], vtx[2]);
            tris[1] = Face.createTriangle(vtx[3], vtx[1], vtx[0]);
            tris[2] = Face.createTriangle(vtx[3], vtx[2], vtx[1]);
            tris[3] = Face.createTriangle(vtx[3], vtx[0], vtx[2]);
            for (i = 0; i < 3; ++i) {
                k = (i + 1) % 3;
                tris[i + 1].getEdge(1).setOpposite(tris[k + 1].getEdge(0));
                tris[i + 1].getEdge(2).setOpposite(tris[0].getEdge(k));
            }
        } else {
            tris[0] = Face.createTriangle(vtx[0], vtx[2], vtx[1]);
            tris[1] = Face.createTriangle(vtx[3], vtx[0], vtx[1]);
            tris[2] = Face.createTriangle(vtx[3], vtx[1], vtx[2]);
            tris[3] = Face.createTriangle(vtx[3], vtx[2], vtx[0]);
            for (i = 0; i < 3; ++i) {
                k = (i + 1) % 3;
                tris[i + 1].getEdge(0).setOpposite(tris[k + 1].getEdge(1));
                tris[i + 1].getEdge(2).setOpposite(tris[0].getEdge((3 - i) % 3));
            }
        }
        this.faces.addAll(Arrays.asList(tris).subList(0, 4));
        for (i = 0; i < this.numPoints; ++i) {
            Vertex v = this.pointBuffer[i];
            if (v == vtx[0] || v == vtx[1] || v == vtx[2] || v == vtx[3]) continue;
            maxDist = this.tolerance;
            Face maxFace = null;
            for (int k = 0; k < 4; ++k) {
                double dist = tris[k].distanceToPlane(v.pnt);
                if (!(dist > maxDist)) continue;
                maxFace = tris[k];
                maxDist = dist;
            }
            if (maxFace == null) continue;
            this.addPointToFace(v, maxFace);
        }
    }

    public int getNumVertices() {
        return this.numVertices;
    }

    public Point3d[] getVertices() {
        Point3d[] vtxs = new Point3d[this.numVertices];
        for (int i = 0; i < this.numVertices; ++i) {
            vtxs[i] = this.pointBuffer[this.vertexPointIndices[i]].pnt;
        }
        return vtxs;
    }

    public int getVertices(double[] coords) {
        for (int i = 0; i < this.numVertices; ++i) {
            Point3d pnt = this.pointBuffer[this.vertexPointIndices[i]].pnt;
            coords[i * 3 + 0] = pnt.x;
            coords[i * 3 + 1] = pnt.y;
            coords[i * 3 + 2] = pnt.z;
        }
        return this.numVertices;
    }

    public int[] getVertexPointIndices() {
        int[] indices = new int[this.numVertices];
        System.arraycopy(this.vertexPointIndices, 0, indices, 0, this.numVertices);
        return indices;
    }

    public int getNumFaces() {
        return this.faces.size();
    }

    public int[][] getFaces() {
        return this.getFaces(0);
    }

    public int[][] getFaces(int indexFlags) {
        int[][] allFaces = new int[this.faces.size()][];
        int k = 0;
        for (Face face : this.faces) {
            allFaces[k] = new int[face.numVertices()];
            this.getFaceIndices(allFaces[k], face, indexFlags);
            ++k;
        }
        return allFaces;
    }

    public void print(PrintStream ps) {
        this.print(ps, 0);
    }

    public void print(PrintStream ps, int indexFlags) {
        if ((indexFlags & 4) == 0) {
            indexFlags |= 2;
        }
        for (int i = 0; i < this.numVertices; ++i) {
            Point3d pnt = this.pointBuffer[this.vertexPointIndices[i]].pnt;
            ps.println("v " + pnt.x + " " + pnt.y + " " + pnt.z);
        }
        for (Face face : this.faces) {
            int[] indices = new int[face.numVertices()];
            this.getFaceIndices(indices, face, indexFlags);
            ps.print("f");
            for (int index : indices) {
                ps.print(" " + index);
            }
            ps.println();
        }
    }

    private void getFaceIndices(int[] indices, Face face, int flags) {
        boolean ccw = (flags & 1) == 0;
        boolean indexedFromOne = (flags & 2) != 0;
        boolean pointRelative = (flags & 8) != 0;
        HalfEdge hedge = face.he0;
        int k = 0;
        do {
            int idx = hedge.head().index;
            if (pointRelative) {
                idx = this.vertexPointIndices[idx];
            }
            if (indexedFromOne) {
                // empty if block
            }
            indices[k++] = ++idx;
        } while ((hedge = ccw ? hedge.next : hedge.prev) != face.he0);
    }

    protected void resolveUnclaimedPoints(FaceList newFaces) {
        Vertex vtxNext;
        Vertex vtx = vtxNext = this.unclaimed.first();
        while (vtx != null) {
            vtxNext = vtx.next;
            double maxDist = this.tolerance;
            Face maxFace = null;
            Face newFace = newFaces.first();
            while (newFace != null) {
                if (newFace.mark == 1) {
                    double dist = newFace.distanceToPlane(vtx.pnt);
                    if (dist > maxDist) {
                        maxDist = dist;
                        maxFace = newFace;
                    }
                    if (maxDist > 1000.0 * this.tolerance) break;
                }
                newFace = newFace.next;
            }
            if (maxFace != null) {
                this.addPointToFace(vtx, maxFace);
                if (logger.isLoggable(Level.FINE) && vtx.index == this.findIndex) {
                    logger.fine(this.findIndex + " CLAIMED BY " + maxFace.getVertexString());
                }
            } else if (logger.isLoggable(Level.FINE) && vtx.index == this.findIndex) {
                logger.fine(this.findIndex + " DISCARDED");
            }
            vtx = vtxNext;
        }
    }

    protected void deleteFacePoints(Face face, Face absorbingFace) {
        Vertex faceVtxs = this.removeAllPointsFromFace(face);
        if (faceVtxs != null) {
            if (absorbingFace == null) {
                this.unclaimed.addAll(faceVtxs);
            } else {
                Vertex vtxNext;
                Vertex vtx = vtxNext = faceVtxs;
                while (vtx != null) {
                    vtxNext = vtx.next;
                    double dist = absorbingFace.distanceToPlane(vtx.pnt);
                    if (dist > this.tolerance) {
                        this.addPointToFace(vtx, absorbingFace);
                    } else {
                        this.unclaimed.add(vtx);
                    }
                    vtx = vtxNext;
                }
            }
        }
    }

    protected double oppFaceDistance(HalfEdge he) {
        return he.face.distanceToPlane(he.opposite.face.getCentroid());
    }

    private boolean doAdjacentMerge(Face face, int mergeType) {
        HalfEdge hedge = face.he0;
        boolean convex = true;
        do {
            Face oppFace = hedge.oppositeFace();
            boolean merge = false;
            if (mergeType == 2) {
                if (this.oppFaceDistance(hedge) > -this.tolerance || this.oppFaceDistance(hedge.opposite) > -this.tolerance) {
                    merge = true;
                }
            } else if (face.area > oppFace.area) {
                if (this.oppFaceDistance(hedge) > -this.tolerance) {
                    merge = true;
                } else if (this.oppFaceDistance(hedge.opposite) > -this.tolerance) {
                    convex = false;
                }
            } else if (this.oppFaceDistance(hedge.opposite) > -this.tolerance) {
                merge = true;
            } else if (this.oppFaceDistance(hedge) > -this.tolerance) {
                convex = false;
            }
            if (!merge) continue;
            if (logger.isLoggable(Level.FINE)) {
                logger.fine("  merging " + face.getVertexString() + "  and  " + oppFace.getVertexString());
            }
            int numd = face.mergeAdjacentFace(hedge, this.discardedFaces);
            for (int i = 0; i < numd; ++i) {
                this.deleteFacePoints(this.discardedFaces[i], face);
            }
            if (logger.isLoggable(Level.FINE)) {
                logger.fine("  result: " + face.getVertexString());
            }
            return true;
        } while ((hedge = hedge.next) != face.he0);
        if (!convex) {
            face.mark = 2;
        }
        return false;
    }

    protected void calculateHorizon(Point3d eyePnt, HalfEdge edge0, Face face, Vector<HalfEdge> horizon) {
        this.deleteFacePoints(face, null);
        face.mark = 3;
        if (logger.isLoggable(Level.FINE)) {
            logger.fine("  visiting face " + face.getVertexString());
        }
        HalfEdge edge = edge0 == null ? (edge0 = face.getEdge(0)) : edge0.getNext();
        do {
            Face oppFace = edge.oppositeFace();
            if (oppFace.mark != 1) continue;
            if (oppFace.distanceToPlane(eyePnt) > this.tolerance) {
                this.calculateHorizon(eyePnt, edge.getOpposite(), oppFace, horizon);
                continue;
            }
            horizon.add(edge);
            if (!logger.isLoggable(Level.FINE)) continue;
            logger.fine("  adding horizon edge " + edge.getVertexString());
        } while ((edge = edge.getNext()) != edge0);
    }

    private HalfEdge addAdjoiningFace(Vertex eyeVtx, HalfEdge he) {
        Face face = Face.createTriangle(eyeVtx, he.tail(), he.head());
        this.faces.add(face);
        face.getEdge(-1).setOpposite(he.getOpposite());
        return face.getEdge(0);
    }

    protected void addNewFaces(FaceList newFaces, Vertex eyeVtx, Vector<HalfEdge> horizon) {
        newFaces.clear();
        HalfEdge hedgeSidePrev = null;
        HalfEdge hedgeSideBegin = null;
        for (HalfEdge horizonHe : horizon) {
            HalfEdge hedgeSide = this.addAdjoiningFace(eyeVtx, horizonHe);
            if (logger.isLoggable(Level.FINE)) {
                logger.fine("new face: " + hedgeSide.face.getVertexString());
            }
            if (hedgeSidePrev != null) {
                hedgeSide.next.setOpposite(hedgeSidePrev);
            } else {
                hedgeSideBegin = hedgeSide;
            }
            newFaces.add(hedgeSide.getFace());
            hedgeSidePrev = hedgeSide;
        }
        hedgeSideBegin.next.setOpposite(hedgeSidePrev);
    }

    protected Vertex nextPointToAdd() {
        if (!this.claimed.isEmpty()) {
            Face eyeFace = this.claimed.first().face;
            Vertex eyeVtx = null;
            double maxDist = 0.0;
            Vertex vtx = eyeFace.outside;
            while (vtx != null && vtx.face == eyeFace) {
                double dist = eyeFace.distanceToPlane(vtx.pnt);
                if (dist > maxDist) {
                    maxDist = dist;
                    eyeVtx = vtx;
                }
                vtx = vtx.next;
            }
            return eyeVtx;
        }
        return null;
    }

    protected void addPointToHull(Vertex eyeVtx) {
        this.horizon.clear();
        this.unclaimed.clear();
        if (logger.isLoggable(Level.FINE)) {
            logger.fine("Adding point: " + eyeVtx.index);
            logger.fine(" which is " + eyeVtx.face.distanceToPlane(eyeVtx.pnt) + " above face " + eyeVtx.face.getVertexString());
        }
        this.removePointFromFace(eyeVtx, eyeVtx.face);
        this.calculateHorizon(eyeVtx.pnt, null, eyeVtx.face, this.horizon);
        this.newFaces.clear();
        this.addNewFaces(this.newFaces, eyeVtx, this.horizon);
        Face face = this.newFaces.first();
        while (face != null) {
            if (face.mark == 1) {
                while (this.doAdjacentMerge(face, 1)) {
                }
            }
            face = face.next;
        }
        face = this.newFaces.first();
        while (face != null) {
            if (face.mark == 2) {
                face.mark = 1;
                while (this.doAdjacentMerge(face, 2)) {
                }
            }
            face = face.next;
        }
        this.resolveUnclaimedPoints(this.newFaces);
    }

    protected void buildHull() {
        Vertex eyeVtx;
        int cnt = 0;
        this.computeMaxAndMin();
        this.createInitialSimplex();
        while ((eyeVtx = this.nextPointToAdd()) != null) {
            this.addPointToHull(eyeVtx);
            logger.fine("iteration " + ++cnt + " done");
        }
        this.reindexFacesAndVertices();
        logger.fine("hull done");
    }

    private void markFaceVertices(Face face, int mark) {
        HalfEdge he0;
        HalfEdge he = he0 = face.getFirstEdge();
        do {
            he.head().index = mark;
        } while ((he = he.next) != he0);
    }

    protected void reindexFacesAndVertices() {
        for (int i = 0; i < this.numPoints; ++i) {
            this.pointBuffer[i].index = -1;
        }
        this.numFaces = 0;
        Iterator<Face> it = this.faces.iterator();
        while (it.hasNext()) {
            Face face = it.next();
            if (face.mark != 1) {
                it.remove();
                continue;
            }
            this.markFaceVertices(face, 0);
            ++this.numFaces;
        }
        this.numVertices = 0;
        for (int i = 0; i < this.numPoints; ++i) {
            Vertex vtx = this.pointBuffer[i];
            if (vtx.index != 0) continue;
            this.vertexPointIndices[this.numVertices] = i;
            ++this.numVertices;
            vtx.index = vtx.index;
        }
    }

    protected boolean checkFaceConvexity(Face face, double tol, PrintStream ps) {
        HalfEdge he = face.he0;
        do {
            face.checkConsistency();
            double dist = this.oppFaceDistance(he);
            if (dist > tol) {
                if (ps != null) {
                    ps.println("Edge " + he.getVertexString() + " non-convex by " + dist);
                }
                return false;
            }
            dist = this.oppFaceDistance(he.opposite);
            if (dist > tol) {
                if (ps != null) {
                    ps.println("Opposite edge " + he.opposite.getVertexString() + " non-convex by " + dist);
                }
                return false;
            }
            if (he.next.oppositeFace() != he.oppositeFace()) continue;
            if (ps != null) {
                ps.println("Redundant vertex " + he.head().index + " in face " + face.getVertexString());
            }
            return false;
        } while ((he = he.next) != face.he0);
        return true;
    }

    protected boolean checkFaces(double tol, PrintStream ps) {
        boolean convex = true;
        for (Face face : this.faces) {
            if (face.mark != 1 || this.checkFaceConvexity(face, tol, ps)) continue;
            convex = false;
        }
        return convex;
    }

    public boolean check(PrintStream ps) {
        return this.check(ps, this.getDistanceTolerance());
    }

    public boolean check(PrintStream ps, double tol) {
        double pointTol = 10.0 * tol;
        if (!this.checkFaces(this.tolerance, ps)) {
            return false;
        }
        for (int i = 0; i < this.numPoints; ++i) {
            Point3d pnt = this.pointBuffer[i].pnt;
            for (Face face : this.faces) {
                double dist;
                if (face.mark != 1 || !((dist = face.distanceToPlane(pnt)) > pointTol)) continue;
                if (ps != null) {
                    ps.println("Point " + i + " " + dist + " above face " + face.getVertexString());
                }
                return false;
            }
        }
        return true;
    }
}

