/*
 * Decompiled with CFR 0.152.
 */
package ffx.numerics.fft;

import ffx.numerics.fft.DataLayout1D;
import ffx.numerics.fft.MixedRadixFactor;
import ffx.numerics.fft.MixedRadixFactor2;
import ffx.numerics.fft.MixedRadixFactor3;
import ffx.numerics.fft.MixedRadixFactor4;
import ffx.numerics.fft.MixedRadixFactor5;
import ffx.numerics.fft.MixedRadixFactor6;
import ffx.numerics.fft.MixedRadixFactor7;
import ffx.numerics.fft.MixedRadixFactorPrime;
import ffx.numerics.fft.PassConstants;
import ffx.numerics.fft.PassData;
import java.util.Arrays;
import java.util.Random;
import java.util.Vector;
import java.util.logging.Level;
import java.util.logging.Logger;
import org.apache.commons.math3.util.FastMath;

public class Complex {
    private static final Logger logger = Logger.getLogger(Complex.class.getName());
    private static final int[] availableFactors = new int[]{7, 6, 5, 4, 3, 2};
    private static final int firstUnavailablePrime = 11;
    private final int n;
    private final int nFFTs;
    private final int externalIm;
    private final int im;
    private final int ii;
    private final int[] factors;
    private final double[] packedData;
    private final double[] scratch;
    private final MixedRadixFactor[] mixedRadixFactors;
    private final PassData[] passData;
    private boolean useSIMD;
    private int simdWidth;
    private int minSIMDLoopLength;
    private static int lastN = -1;
    private static int lastIm = -1;
    private static int lastNFFTs = -1;
    private static int[] factorsCache = null;
    private static MixedRadixFactor[] mixedRadixFactorsCache = null;

    public Complex(int n) {
        this(n, DataLayout1D.INTERLEAVED, 1);
    }

    public Complex(int n, DataLayout1D dataLayout, int imOffset) {
        this(n, dataLayout, imOffset, 1);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public Complex(int n, DataLayout1D dataLayout, int imOffset, int nFFTs) {
        assert (n > 1);
        this.n = n;
        this.nFFTs = nFFTs;
        this.externalIm = imOffset;
        if (dataLayout == DataLayout1D.INTERLEAVED) {
            this.im = 1;
            this.ii = 2;
        } else {
            this.im = n * nFFTs;
            this.ii = 1;
        }
        this.packedData = new double[2 * n * nFFTs];
        this.scratch = new double[2 * n * nFFTs];
        this.passData = new PassData[2];
        this.passData[0] = new PassData(1, this.packedData, 0, this.scratch, 0);
        this.passData[1] = new PassData(1, this.packedData, 0, this.scratch, 0);
        Class<Complex> clazz = Complex.class;
        synchronized (Complex.class) {
            if (this.n == lastN && this.im == lastIm && this.nFFTs == lastNFFTs) {
                this.factors = factorsCache;
                this.mixedRadixFactors = mixedRadixFactorsCache;
            } else {
                this.factors = Complex.factor(n);
                double[][][] twiddle = Complex.wavetable(n, this.factors);
                this.mixedRadixFactors = new MixedRadixFactor[this.factors.length];
                lastN = this.n;
                lastIm = this.im;
                lastNFFTs = this.nFFTs;
                factorsCache = this.factors;
                mixedRadixFactorsCache = this.mixedRadixFactors;
                int product = 1;
                block17: for (int i = 0; i < this.factors.length; ++i) {
                    int factor = this.factors[i];
                    PassConstants passConstants = new PassConstants(n, this.im, nFFTs, factor, product *= factor, twiddle[i]);
                    switch (factor) {
                        case 2: {
                            this.mixedRadixFactors[i] = new MixedRadixFactor2(passConstants);
                            continue block17;
                        }
                        case 3: {
                            this.mixedRadixFactors[i] = new MixedRadixFactor3(passConstants);
                            continue block17;
                        }
                        case 4: {
                            this.mixedRadixFactors[i] = new MixedRadixFactor4(passConstants);
                            continue block17;
                        }
                        case 5: {
                            this.mixedRadixFactors[i] = new MixedRadixFactor5(passConstants);
                            continue block17;
                        }
                        case 6: {
                            this.mixedRadixFactors[i] = new MixedRadixFactor6(passConstants);
                            continue block17;
                        }
                        case 7: {
                            this.mixedRadixFactors[i] = new MixedRadixFactor7(passConstants);
                            continue block17;
                        }
                        default: {
                            if (dataLayout == DataLayout1D.BLOCKED) {
                                throw new IllegalArgumentException(" Prime factors greater than 7 are only supported for interleaved data: " + factor);
                            }
                            this.mixedRadixFactors[i] = new MixedRadixFactorPrime(passConstants);
                        }
                    }
                }
            }
            this.useSIMD = true;
            String simd = System.getProperty("fft.simd", Boolean.toString(this.useSIMD));
            try {
                this.useSIMD = Boolean.parseBoolean(simd);
            }
            catch (Exception e) {
                logger.info(" Invalid value for fft.simd: " + simd);
                this.useSIMD = false;
            }
            this.simdWidth = MixedRadixFactor.LENGTH;
            String width = System.getProperty("fft.simd.width", Integer.toString(this.simdWidth));
            try {
                this.simdWidth = Integer.parseInt(width);
                if (this.simdWidth < 2 || this.simdWidth > MixedRadixFactor.LENGTH || this.simdWidth % 2 != 0) {
                    logger.info(" Invalid value for fft.simd.width: " + width);
                    this.simdWidth = MixedRadixFactor.LENGTH;
                }
            }
            catch (Exception e) {
                logger.info(" Invalid value for fft.simd.width: " + width);
                this.simdWidth = MixedRadixFactor.LENGTH;
            }
            for (MixedRadixFactor mixedRadixFactor : this.mixedRadixFactors) {
                mixedRadixFactor.setSIMDWidth(this.simdWidth);
            }
            this.minSIMDLoopLength = this.im == 1 ? MixedRadixFactor.LENGTH / 2 : MixedRadixFactor.LENGTH;
            String loop = System.getProperty("fft.minLoop", Integer.toString(this.minSIMDLoopLength));
            try {
                this.minSIMDLoopLength = Integer.max(this.minSIMDLoopLength, Integer.parseInt(loop));
            }
            catch (Exception e) {
                logger.info(" Invalid value for fft.minLoop: " + loop);
                this.minSIMDLoopLength = this.im == 1 ? MixedRadixFactor.LENGTH / 2 : MixedRadixFactor.LENGTH;
            }
            return;
        }
    }

    public String toString() {
        StringBuilder sb = new StringBuilder(" Complex FFT: n = " + this.n + ", nFFTs = " + this.nFFTs + ", im = " + this.externalIm);
        sb.append("\n  Factors: ").append(Arrays.toString(this.factors));
        return sb.toString();
    }

    public void setUseSIMD(boolean useSIMD) {
        this.useSIMD = useSIMD;
    }

    public void setMinSIMDLoopLength(int minSIMDLoopLength) {
        if (this.im == 1 && minSIMDLoopLength < 1) {
            throw new IllegalArgumentException(" Minimum SIMD loop length for interleaved data is 1 or greater.");
        }
        if (this.im > 2 && minSIMDLoopLength < 2) {
            throw new IllegalArgumentException(" Minimum SIMD loop length for blocked data is 2 or greater.");
        }
        this.minSIMDLoopLength = minSIMDLoopLength;
    }

    public static boolean preferredDimension(int dim) {
        if (dim < 2) {
            return false;
        }
        for (int factor : availableFactors) {
            while (dim % factor == 0) {
                dim /= factor;
            }
        }
        return dim <= 1;
    }

    public int[] getFactors() {
        return this.factors;
    }

    public void fft(double[] data, int offset, int stride) {
        this.transformInternal(data, offset, stride, -1, 2 * this.n);
    }

    public void fft(double[] data, int offset, int stride, int nextFFT) {
        this.transformInternal(data, offset, stride, -1, nextFFT);
    }

    public void ifft(double[] data, int offset, int stride) {
        this.transformInternal(data, offset, stride, 1, 2 * this.n);
    }

    public void ifft(double[] data, int offset, int stride, int nextFFT) {
        this.transformInternal(data, offset, stride, 1, nextFFT);
    }

    public void inverse(double[] data, int offset, int stride) {
        this.inverse(data, offset, stride, 2 * this.n);
    }

    public void inverse(double[] data, int offset, int stride, int nextFFT) {
        this.ifft(data, offset, stride, nextFFT);
        double norm = this.normalization();
        int index = 0;
        for (int f = 0; f < this.nFFTs; ++f) {
            for (int i = 0; i < 2 * this.n; ++i) {
                int n = index++;
                data[n] = data[n] * norm;
            }
        }
    }

    private void transformInternal(double[] data, int offset, int stride, int sign, int nextFFT) {
        this.passData[0].sign = sign;
        this.passData[0].in = data;
        this.passData[0].inOffset = offset;
        this.passData[0].out = this.scratch;
        this.passData[0].outOffset = 0;
        this.passData[1].sign = sign;
        this.passData[1].in = this.scratch;
        this.passData[1].inOffset = 0;
        this.passData[1].out = data;
        this.passData[1].outOffset = offset;
        boolean packed = false;
        if (stride > 2 || this.externalIm > this.n * this.nFFTs) {
            packed = true;
            this.pack(data, offset, stride, nextFFT);
            this.passData[0].in = this.packedData;
            this.passData[0].inOffset = 0;
            this.passData[1].out = this.packedData;
            this.passData[1].outOffset = 0;
        }
        int nfactors = this.factors.length;
        for (int i = 0; i < nfactors; ++i) {
            int pass = i % 2;
            MixedRadixFactor mixedRadixFactor = this.mixedRadixFactors[i];
            boolean applySIMD = false;
            if (this.useSIMD) {
                // empty if block
            }
            if (this.useSIMD && mixedRadixFactor.innerLoopLimit >= this.minSIMDLoopLength) {
                mixedRadixFactor.passSIMD(this.passData[pass]);
                continue;
            }
            mixedRadixFactor.passScalar(this.passData[pass]);
        }
        if (nfactors % 2 == 1) {
            if (stride <= 2 && this.im == this.externalIm && nextFFT == 2 * this.n) {
                System.arraycopy(this.scratch, 0, data, offset, 2 * this.n * this.nFFTs);
            } else {
                this.unpack(this.scratch, data, offset, stride, nextFFT);
            }
        } else if (packed) {
            this.unpack(this.packedData, data, offset, stride, nextFFT);
        }
    }

    private void pack(double[] data, int offset, int stride, int nextFFT) {
        int i = 0;
        for (int f = 0; f < this.nFFTs; ++f) {
            int inputOffset;
            int index = inputOffset = offset + f * nextFFT;
            int k = 0;
            while (k < this.n) {
                this.packedData[i] = data[index];
                this.packedData[i + this.im] = data[index + this.externalIm];
                ++k;
                i += this.ii;
                index += stride;
            }
        }
    }

    private void unpack(double[] source, double[] data, int offset, int stride, int nextFFT) {
        int i = 0;
        for (int f = 0; f < this.nFFTs; ++f) {
            int outputOffset;
            int index = outputOffset = offset + f * nextFFT;
            int k = 0;
            while (k < this.n) {
                data[index] = source[i];
                data[index + this.externalIm] = source[i + this.im];
                ++k;
                i += this.ii;
                index += stride;
            }
        }
    }

    private double normalization() {
        return 1.0 / (double)this.n;
    }

    protected static int[] factor(int n) {
        if (n < 2) {
            return null;
        }
        Vector<Integer> v = new Vector<Integer>();
        int nTest = n;
        for (int factor : availableFactors) {
            while (nTest % factor == 0) {
                nTest /= factor;
                v.add(factor);
            }
        }
        int factor = 11;
        while (nTest > 1) {
            while (nTest % factor != 0) {
                factor += 2;
            }
            nTest /= factor;
            v.add(factor);
        }
        int product = 1;
        int nf = v.size();
        int[] ret = new int[nf];
        for (int i = 0; i < nf; ++i) {
            ret[i] = (Integer)v.get(i);
            product *= ret[i];
        }
        if (product != n) {
            StringBuilder sb = new StringBuilder(" FFT factorization failed for " + n + "\n");
            for (int i = 0; i < nf; ++i) {
                sb.append(" ");
                sb.append(ret[i]);
            }
            sb.append("\n");
            sb.append(" Factor product = ");
            sb.append(product);
            sb.append("\n");
            logger.severe(sb.toString());
            System.exit(-1);
        } else if (logger.isLoggable(Level.FINEST)) {
            StringBuilder sb = new StringBuilder(" FFT factorization for " + n + " = ");
            for (int i = 0; i < nf - 1; ++i) {
                sb.append(ret[i]);
                sb.append(" * ");
            }
            sb.append(ret[nf - 1]);
            logger.finest(sb.toString());
        }
        return ret;
    }

    private static double[][][] wavetable(int n, int[] factors) {
        if (n < 2) {
            return null;
        }
        double TwoPI_N = Math.PI * -2 / (double)n;
        double[][][] ret = new double[factors.length][][];
        int product = 1;
        for (int i = 0; i < factors.length; ++i) {
            int factor = factors[i];
            int product_1 = product;
            int outLoopLimit = n / (product *= factor);
            if (factor >= 11) {
                ++outLoopLimit;
            }
            int nTwiddle = factor - 1;
            ret[i] = new double[outLoopLimit][2 * nTwiddle];
            double[][] twid = ret[i];
            for (int j = 0; j < factor - 1; ++j) {
                twid[0][2 * j] = 1.0;
                twid[0][2 * j + 1] = 0.0;
            }
            for (int k = 1; k < outLoopLimit; ++k) {
                int m = 0;
                for (int j = 0; j < nTwiddle; ++j) {
                    m += k * product_1;
                    double theta = TwoPI_N * (double)(m %= n);
                    twid[k][2 * j] = FastMath.cos((double)theta);
                    twid[k][2 * j + 1] = FastMath.sin((double)theta);
                }
            }
        }
        return ret;
    }

    public static void dft(double[] in, double[] out) {
        int n = in.length / 2;
        for (int k = 0; k < n; ++k) {
            double sumReal = 0.0;
            double simImag = 0.0;
            for (int t = 0; t < n; ++t) {
                double angle = Math.PI * 2 * (double)t * (double)k / (double)n;
                int re = 2 * t;
                int im = 2 * t + 1;
                sumReal = Math.fma(in[re], FastMath.cos((double)angle), sumReal);
                sumReal = Math.fma(in[im], FastMath.sin((double)angle), sumReal);
                simImag = Math.fma(-in[re], FastMath.sin((double)angle), simImag);
                simImag = Math.fma(in[im], FastMath.cos((double)angle), simImag);
            }
            int re = 2 * k;
            int im = 2 * k + 1;
            out[re] = sumReal;
            out[im] = simImag;
        }
    }

    public static void dftBlocked(double[] in, double[] out) {
        int n = in.length / 2;
        for (int k = 0; k < n; ++k) {
            double sumReal = 0.0;
            double simImag = 0.0;
            for (int t = 0; t < n; ++t) {
                double angle = Math.PI * 2 * (double)t * (double)k / (double)n;
                int re = t;
                int im = t + n;
                sumReal = Math.fma(in[re], FastMath.cos((double)angle), sumReal);
                sumReal = Math.fma(in[im], FastMath.sin((double)angle), sumReal);
                simImag = Math.fma(-in[re], FastMath.sin((double)angle), simImag);
                simImag = Math.fma(in[im], FastMath.cos((double)angle), simImag);
            }
            int re = k;
            int im = k + n;
            out[re] = sumReal;
            out[im] = simImag;
        }
    }

    public static void main(String[] args) throws Exception {
        int dimNotFinal = 128;
        int reps = 5;
        boolean blocked = false;
        try {
            dimNotFinal = Integer.parseInt(args[0]);
            if (dimNotFinal < 1) {
                dimNotFinal = 128;
            }
            if ((reps = Integer.parseInt(args[1])) < 1) {
                reps = 5;
            }
            blocked = Boolean.parseBoolean(args[2]);
        }
        catch (Exception exception) {
            // empty catch block
        }
        int dim = dimNotFinal;
        System.out.printf("Initializing a 1D array of length %d.\nThe best timing out of %d repetitions will be used.%n", dim, reps);
        DataLayout1D dataLayout1D = DataLayout1D.INTERLEAVED;
        int im = 1;
        if (blocked) {
            dataLayout1D = DataLayout1D.BLOCKED;
            im = dim;
        }
        Complex complex = new Complex(dim, dataLayout1D, im);
        double[] data = new double[dim * 2];
        Random random = new Random(1L);
        for (int i = 0; i < dim; ++i) {
            data[2 * i] = random.nextDouble();
        }
        double toSeconds = 1.0E-9;
        long seqTime = Long.MAX_VALUE;
        for (int i = 0; i < reps; ++i) {
            System.out.printf("Iteration %d%n", i + 1);
            long time = System.nanoTime();
            complex.fft(data, 0, 2);
            complex.ifft(data, 0, 2);
            time = System.nanoTime() - time;
            System.out.printf("Sequential: %12.9f%n", toSeconds * (double)time);
            if (time >= seqTime) continue;
            seqTime = time;
        }
        System.out.printf("Best Sequential Time:  %12.9f%n", toSeconds * (double)seqTime);
    }
}

