/*
 * Decompiled with CFR 0.152.
 */
package ffx.potential.bonded;

import ffx.numerics.math.DoubleMath;
import ffx.potential.MolecularAssembly;
import ffx.potential.bonded.AminoAcidUtils;
import ffx.potential.bonded.Atom;
import ffx.potential.bonded.Bond;
import ffx.potential.bonded.MSGroup;
import ffx.potential.bonded.MSNode;
import ffx.potential.bonded.Molecule;
import ffx.potential.bonded.NamingUtils;
import ffx.potential.bonded.Polymer;
import ffx.potential.bonded.Residue;
import ffx.potential.parameters.AtomType;
import ffx.potential.parameters.BioType;
import ffx.potential.parameters.BondType;
import ffx.potential.parameters.ForceField;
import ffx.utilities.StringUtils;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Comparator;
import java.util.List;
import java.util.Optional;
import java.util.logging.Level;
import java.util.logging.Logger;
import java.util.stream.Collectors;
import org.apache.commons.math3.util.FastMath;

public class BondedUtils {
    private static final Logger logger = Logger.getLogger(BondedUtils.class.getName());
    private static final double eps = 1.0E-7;

    public static boolean atomAttachedToAtom(Atom a1, Atom a2) {
        assert (a1 != a2);
        return a1.getBonds().stream().anyMatch(b -> b.get1_2(a1) == a2);
    }

    public static Bond buildBond(Atom a1, Atom a2, ForceField forceField, List<Bond> bondList) {
        Bond bond = new Bond(a1, a2);
        BondType bondType = forceField.getBondType(a1.getAtomType(), a2.getAtomType());
        if (bondType == null) {
            Bond.logNoBondType(a1, a2, forceField);
        } else {
            bond.setBondType(bondType);
        }
        if (bondList != null) {
            bondList.add(bond);
        }
        return bond;
    }

    public static Atom buildHeavy(MSGroup residue, String atomName, Atom bondedTo, int key, ForceField forceField, List<Bond> bondList) throws MissingHeavyAtomException, MissingAtomTypeException {
        Atom atom = (Atom)residue.getAtomNode(atomName);
        AtomType atomType = BondedUtils.findAtomType(key, forceField);
        if (atomType == null) {
            Residue res = (Residue)residue;
            throw new MissingAtomTypeException(res, atom);
        }
        if (atom == null) {
            throw new MissingHeavyAtomException(atomName, atomType, bondedTo);
        }
        atom.setAtomType(atomType);
        if (bondedTo != null) {
            BondedUtils.buildBond(atom, bondedTo, forceField, bondList);
        }
        return atom;
    }

    public static Atom buildHeavy(MSGroup residue, String atomName, Atom ia, double bond, Atom ib, double angle1, Atom ic, double angle2, int chiral, int lookUp, ForceField forceField) {
        AtomType atomType = BondedUtils.findAtomType(lookUp, forceField);
        return BondedUtils.buildHeavyAtom(residue, atomName, ia, bond, ib, angle1, ic, angle2, chiral, atomType);
    }

    public static Atom buildHeavy(MSGroup residue, String atomName, Atom ia, double bond, Atom ib, double angle1, Atom ic, double angle2, int chiral, int lookUp, ForceField forceField, List<Bond> bondList) {
        AtomType atomType = BondedUtils.findAtomType(lookUp, forceField);
        return BondedUtils.buildHeavyAtom(residue, atomName, ia, bond, ib, angle1, ic, angle2, chiral, atomType, forceField, bondList);
    }

    public static Atom buildHeavy(MSGroup residue, AminoAcidUtils.SideChainType atomName, Atom ia, double bond, Atom ib, double angle1, Atom ic, double angle2, int chiral, ForceField forceField, List<Bond> bondList) {
        AtomType atomType = BondedUtils.findAtomType(atomName.getType(), forceField);
        return BondedUtils.buildHeavyAtom(residue, atomName.name(), ia, bond, ib, angle1, ic, angle2, chiral, atomType, forceField, bondList);
    }

    public static Atom buildH(MSGroup residue, String atomName, Atom ia, double bond, Atom ib, double angle1, Atom ic, double angle2, int chiral, int lookUp, ForceField forceField, List<Bond> bondList) {
        AtomType atomType = BondedUtils.findAtomType(lookUp, forceField);
        return BondedUtils.buildHydrogenAtom(residue, atomName, ia, bond, ib, angle1, ic, angle2, chiral, atomType, forceField, bondList);
    }

    public static Atom buildH(MSGroup residue, AminoAcidUtils.SideChainType atomName, Atom ia, double bond, Atom ib, double angle1, Atom ic, double angle2, int chiral, ForceField forceField, List<Bond> bondList) {
        AtomType atomType = BondedUtils.findAtomType(atomName.getType(), forceField);
        return BondedUtils.buildHydrogenAtom(residue, atomName.name(), ia, bond, ib, angle1, ic, angle2, chiral, atomType, forceField, bondList);
    }

    public static Atom buildHydrogenAtom(MSGroup residue, String atomName, Atom ia, double bond, Atom ib, double angle1, Atom ic, double angle2, int chiral, AtomType atomType, ForceField forceField, List<Bond> bondList) {
        Molecule molecule;
        if (atomType == null) {
            return null;
        }
        Atom atom = (Atom)residue.getAtomNode(atomName);
        if (atom == null) {
            String dAtomName = atomName.replaceFirst("H", "D");
            atom = (Atom)residue.getAtomNode(dAtomName);
        }
        if (atom == null && residue instanceof Molecule && StringUtils.looksLikeWater((String)(molecule = (Molecule)residue).getName())) {
            atom = (Atom)molecule.getAtomNode("H");
            if (atom == null) {
                atom = (Atom)molecule.getAtomNode("D");
            }
            if (atom != null) {
                atom.setName(atomName);
            }
        }
        if (atom == null) {
            boolean buildDeuterium = forceField.getBoolean("build-deuterium", false);
            if (buildDeuterium && atomName.startsWith("H")) {
                atomName = atomName.replaceFirst("H", "D");
            }
            String resName = ia.getResidueName();
            int resSeq = ia.getResidueNumber();
            Character chainID = ia.getChainID();
            Character altLoc = ia.getAltLoc();
            String segID = ia.getSegID();
            double occupancy = ia.getOccupancy();
            double tempFactor = ia.getTempFactor();
            atom = new Atom(0, atomName, altLoc, new double[3], resName, resSeq, chainID, occupancy, tempFactor, segID, true);
            residue.addMSNode(atom);
            BondedUtils.intxyz(atom, ia, bond, ib, angle1, ic, angle2, chiral);
        }
        atom.setAtomType(atomType);
        BondedUtils.buildBond(ia, atom, forceField, bondList);
        return atom;
    }

    public static AtomType findAtomType(int key, ForceField forceField) {
        BioType bioType = forceField.getBioType(Integer.toString(key));
        if (bioType != null) {
            AtomType atomType = forceField.getAtomType(Integer.toString(bioType.atomType));
            if (atomType != null) {
                return atomType;
            }
            logger.severe(String.format("The atom type %s was not found for biotype %s.", bioType.atomType, bioType));
        }
        return null;
    }

    public static List<Atom> findAtomsOfElement(Residue residue, int element) {
        return residue.getAtomList().stream().filter(a -> a.getAtomType().atomicNumber == element).collect(Collectors.toList());
    }

    public static List<Atom> findBondedAtoms(Atom atom, int element) {
        return BondedUtils.findBondedAtoms(atom, null, element);
    }

    public static List<Atom> findBondedAtoms(Atom atom, Atom toExclude, int element) {
        return atom.getBonds().stream().map(b -> b.get1_2(atom)).filter(a -> a != toExclude).filter(a -> a.getAtomType().atomicNumber == element).collect(Collectors.toList());
    }

    public static Atom findNitrogenAtom(Residue residue) {
        assert (residue.getResidueType() == Residue.ResidueType.AA);
        List<Object> nitrogenCandidates = new ArrayList(2);
        switch (residue.getAminoAcid3()) {
            case LYS: 
            case LYD: {
                List<Atom> nitrogenList = BondedUtils.findAtomsOfElement(residue, 7);
                for (Atom atom : nitrogenList) {
                    List<Atom> carbons = BondedUtils.findBondedAtoms(atom, 6);
                    if (carbons.size() == 2) {
                        nitrogenCandidates.add(atom);
                        continue;
                    }
                    if (BondedUtils.findBondedAtoms(carbons.get(0), 1).size() >= 2) continue;
                    nitrogenCandidates.add(atom);
                }
                if (!nitrogenCandidates.isEmpty()) break;
                throw new IllegalArgumentException(String.format(" Could not identify N atom of residue %s!", residue));
            }
            case ARG: 
            case HIS: 
            case HIE: 
            case HID: {
                List<Atom> nitrogenList = BondedUtils.findAtomsOfElement(residue, 7);
                Atom commonC = BondedUtils.findAtomsOfElement(residue, 6).stream().filter(carbon -> BondedUtils.findBondedAtoms(carbon, 7).size() >= 2).findAny().get();
                nitrogenCandidates = nitrogenList.stream().filter(nitr -> !BondedUtils.atomAttachedToAtom(nitr, commonC)).collect(Collectors.toList());
                break;
            }
            case ASN: 
            case GLN: {
                List<Atom> bondedCarbs;
                List<Atom> nitrogenList = BondedUtils.findAtomsOfElement(residue, 7);
                for (Atom atom : nitrogenList) {
                    bondedCarbs = BondedUtils.findBondedAtoms(atom, 6);
                    for (Atom carbon2 : bondedCarbs) {
                        if (BondedUtils.hasAttachedAtom(carbon2, 8)) continue;
                        nitrogenCandidates.add(atom);
                    }
                }
                if (!nitrogenCandidates.isEmpty()) break;
                throw new IllegalArgumentException(String.format(" Could not identify N atom of residue %s!", residue));
            }
            case TRP: {
                List<Atom> bondedCarbs;
                List<Atom> nitrogenList = BondedUtils.findAtomsOfElement(residue, 7);
                for (Atom atom : nitrogenList) {
                    bondedCarbs = BondedUtils.findBondedAtoms(atom, 6);
                    if (bondedCarbs.size() == 1) {
                        nitrogenCandidates.add(atom);
                    }
                    for (Atom carbon2 : bondedCarbs) {
                        if (!BondedUtils.hasAttachedAtom(carbon2, 8)) continue;
                        nitrogenCandidates.add(atom);
                    }
                }
                if (!nitrogenCandidates.isEmpty()) break;
                throw new IllegalArgumentException(String.format(" Could not identify N atom of residue %s!", residue));
            }
            case ACE: {
                return null;
            }
            default: {
                nitrogenCandidates = BondedUtils.findAtomsOfElement(residue, 7);
            }
        }
        switch (nitrogenCandidates.size()) {
            case 0: {
                logger.warning(" Did not find any atoms that might be the amide nitrogen for residue " + String.valueOf(residue));
                return null;
            }
            case 1: {
                return (Atom)nitrogenCandidates.get(0);
            }
            case 2: {
                logger.fine(String.format(" Probable NME C-terminal cap attached to residue %s, some atom names may be duplicated!", residue));
                Atom N = null;
                for (Atom atom : nitrogenCandidates) {
                    atom.setName("N");
                    Optional<Atom> capMethyl = BondedUtils.findBondedAtoms(atom, 6).stream().filter(carb -> BondedUtils.findBondedAtoms(carb, 1).size() == 3).findAny();
                    if (capMethyl.isPresent()) {
                        BondedUtils.findBondedAtoms(atom, 1).get(0).setName("H");
                        Atom theCap = capMethyl.get();
                        theCap.setName("CH3");
                        List<Atom> capHydrogenList = BondedUtils.findBondedAtoms(theCap, 1);
                        for (int i = 0; i < 3; ++i) {
                            capHydrogenList.get(i).setName(String.format("H%d", i + 1));
                        }
                        continue;
                    }
                    N = atom;
                }
                return N;
            }
        }
        if (logger.isLoggable(Level.FINE)) {
            logger.warning(String.format(" Nitrogen could not be mapped for amide nitrogen of residue %s ", residue));
        }
        return null;
    }

    public static Optional<Atom> findNucleotideO4s(Residue residue) {
        assert (residue.getResidueType() == Residue.ResidueType.NA);
        return BondedUtils.findAtomsOfElement(residue, 8).stream().filter(o -> BondedUtils.findBondedAtoms(o, 6).size() == 2).findAny();
    }

    public static Atom getAlphaCarbon(Residue residue, Atom N) {
        List<Atom> resAtoms = residue.getAtomList();
        List<Atom> caCandidates = BondedUtils.findBondedAtoms(N, 6).stream().filter(resAtoms::contains).toList();
        if (residue.getAminoAcid3() == AminoAcidUtils.AminoAcid3.PRO) {
            Atom CA = null;
            Atom CD = null;
            Atom aceC = null;
            for (Atom caCand : caCandidates) {
                if (BondedUtils.hasAttachedAtom(caCand, 8)) {
                    aceC = caCand;
                    continue;
                }
                List<Atom> attachedH = BondedUtils.findBondedAtoms(caCand, 1);
                if (attachedH.size() == 1) {
                    CA = caCand;
                    continue;
                }
                if (attachedH.size() == 2) {
                    CD = caCand;
                    continue;
                }
                throw new IllegalArgumentException(String.format(" Error in parsing proline %s", residue));
            }
            assert (CA != null && CD != null);
            if (aceC != null) {
                NamingUtils.nameAcetylCap(residue, aceC);
            }
            return CA;
        }
        if (caCandidates.size() == 1) {
            return caCandidates.get(0);
        }
        Atom CA = null;
        Atom aceC = null;
        for (Atom caCand : caCandidates) {
            if (BondedUtils.hasAttachedAtom(caCand, 8)) {
                aceC = caCand;
                continue;
            }
            CA = caCand;
        }
        NamingUtils.nameAcetylCap(residue, aceC);
        return CA;
    }

    public static boolean hasAttachedAtom(Atom atom, int element) {
        return atom.getBonds().stream().map(b -> b.get1_2(atom)).anyMatch(a -> a.getAtomType().atomicNumber == element);
    }

    public static void intxyz(Atom atom, Atom ia, double bond, Atom ib, double angle1, Atom ic, double angle2, int chiral) {
        double[] xa = new double[3];
        xa = ia == null ? null : ia.getXYZ(xa);
        double[] xb = new double[3];
        xb = ib == null ? null : ib.getXYZ(xb);
        double[] xc = new double[3];
        xc = ic == null ? null : ic.getXYZ(xc);
        atom.moveTo(BondedUtils.determineIntxyz(xa, bond, xb, angle1, xc, angle2, chiral));
    }

    public static void numberAtoms(MolecularAssembly molecularAssembly) {
        Polymer[] polymers;
        int index = 1;
        for (Atom a : molecularAssembly.getAtomArray()) {
            a.setXyzIndex(index++);
        }
        --index;
        if (logger.isLoggable(Level.INFO)) {
            logger.info(String.format(" Total number of atoms: %d\n", index));
        }
        if ((polymers = molecularAssembly.getChains()) != null) {
            for (Polymer p : polymers) {
                List<Residue> residues = p.getResidues();
                for (Residue r : residues) {
                    r.reOrderAtoms();
                }
            }
        }
        List<MSNode> molecules = molecularAssembly.getMolecules();
        for (MSNode n : molecules) {
            MSGroup m = (MSGroup)n;
            m.reOrderAtoms();
        }
        List<MSNode> water = molecularAssembly.getWater();
        for (MSNode n : water) {
            MSGroup m = (MSGroup)n;
            m.reOrderAtoms();
        }
        List<MSNode> ions = molecularAssembly.getIons();
        for (MSNode n : ions) {
            MSGroup m = (MSGroup)n;
            m.reOrderAtoms();
        }
    }

    public static Atom[] sortAtomsByDistance(Atom reference, List<Atom> toCompare) {
        Atom[] theAtoms = toCompare.toArray(new Atom[0]);
        BondedUtils.sortAtomsByDistance(reference, theAtoms);
        return theAtoms;
    }

    public static void sortAtomsByDistance(Atom reference, Atom[] toCompare) {
        double[] refXYZ = reference.getXYZ(new double[3]);
        Arrays.sort(toCompare, Comparator.comparingDouble(a -> {
            double[] atXYZ = a.getXYZ(new double[3]);
            return DoubleMath.dist2((double[])refXYZ, (double[])atXYZ);
        }));
    }

    private static Atom buildHeavyAtom(MSGroup residue, String atomName, Atom ia, double bond, Atom ib, double angle1, Atom ic, double angle2, int chiral, AtomType atomType) {
        Atom atom = (Atom)residue.getAtomNode(atomName);
        if (atomType == null) {
            return null;
        }
        if (atom == null) {
            String resName = ia.getResidueName();
            int resSeq = ia.getResidueNumber();
            Character chainID = ia.getChainID();
            Character altLoc = ia.getAltLoc();
            String segID = ia.getSegID();
            double occupancy = ia.getOccupancy();
            double tempFactor = ia.getTempFactor();
            atom = new Atom(0, atomName, altLoc, new double[3], resName, resSeq, chainID, occupancy, tempFactor, segID, true);
            residue.addMSNode(atom);
            BondedUtils.intxyz(atom, ia, bond, ib, angle1, ic, angle2, chiral);
        }
        atom.setAtomType(atomType);
        return atom;
    }

    private static Atom buildHeavyAtom(MSGroup residue, String atomName, Atom ia, double bond, Atom ib, double angle1, Atom ic, double angle2, int chiral, AtomType atomType, ForceField forceField, List<Bond> bondList) {
        Atom atom = BondedUtils.buildHeavyAtom(residue, atomName, ia, bond, ib, angle1, ic, angle2, chiral, atomType);
        BondedUtils.buildBond(ia, atom, forceField, bondList);
        return atom;
    }

    public static double[] determineIntxyz(double[] ia, double bond, double[] ib, double angle1, double[] ic, double angle2, int chiral) {
        if (ia != null && !Double.isFinite(bond)) {
            throw new IllegalArgumentException(String.format(" Passed bond length is non-finite %f", bond));
        }
        if (ib != null && !Double.isFinite(angle1)) {
            throw new IllegalArgumentException(String.format(" Passed angle is non-finite %f", angle1));
        }
        if (ic != null && !Double.isFinite(angle2)) {
            throw new IllegalArgumentException(String.format(" Passed dihedral/improper is non-finite %f", angle2));
        }
        if (chiral == 3) {
            double[] negChiral = BondedUtils.determineIntxyz(ia, bond, ib, angle1, ic, angle2, -1);
            double[] posChiral = BondedUtils.determineIntxyz(ia, bond, ib, angle1, ic, angle2, 1);
            double[] displacement = new double[3];
            double dispMag = 0.0;
            for (int i = 0; i < 3; ++i) {
                displacement[i] = 0.5 * (posChiral[i] + negChiral[i]);
                int n = i;
                displacement[n] = displacement[n] - ia[i];
                dispMag += displacement[i] * displacement[i];
            }
            dispMag = FastMath.sqrt((double)dispMag);
            double extend = bond / dispMag;
            assert (extend > 0.999);
            double[] outXYZ = new double[3];
            for (int i = 0; i < 3; ++i) {
                int n = i;
                displacement[n] = displacement[n] * extend;
                outXYZ[i] = displacement[i] + ia[i];
            }
            return outXYZ;
        }
        angle1 = FastMath.toRadians((double)angle1);
        angle2 = FastMath.toRadians((double)angle2);
        double zcos0 = FastMath.cos((double)angle1);
        double zcos1 = FastMath.cos((double)angle2);
        double zsin0 = FastMath.sin((double)angle1);
        double zsin1 = FastMath.sin((double)angle2);
        double[] ret = new double[3];
        double[] x = new double[3];
        if (ia == null) {
            x[2] = 0.0;
            x[1] = 0.0;
            x[0] = 0.0;
        } else if (ib == null) {
            double[] xa = new double[3];
            System.arraycopy(ia, 0, xa, 0, ia.length);
            x[0] = xa[0];
            x[1] = xa[1];
            x[2] = xa[2] + bond;
        } else if (ic == null) {
            double sing;
            double cosg;
            double[] xa = new double[3];
            double[] xb = new double[3];
            double[] xab = new double[3];
            for (int i = 0; i < ia.length; ++i) {
                xa[i] = ia[i];
                xb[i] = ib[i];
            }
            DoubleMath.sub((double[])xa, (double[])xb, (double[])xab);
            double rab = DoubleMath.length((double[])xab);
            DoubleMath.normalize((double[])xab, (double[])xab);
            double cosb = xab[2];
            double sinb = FastMath.sqrt((double)(xab[0] * xab[0] + xab[1] * xab[1]));
            if (sinb == 0.0) {
                cosg = 1.0;
                sing = 0.0;
            } else {
                cosg = xab[1] / sinb;
                sing = xab[0] / sinb;
            }
            double xtmp = bond * zsin0;
            double ztmp = rab - bond * zcos0;
            x[0] = xb[0] + xtmp * cosg + ztmp * sing * sinb;
            x[1] = xb[1] - xtmp * sing + ztmp * cosg * sinb;
            x[2] = xb[2] + ztmp * cosb;
        } else if (chiral == 0) {
            double[] xa = new double[3];
            double[] xb = new double[3];
            double[] xc = new double[3];
            double[] xab = new double[3];
            double[] xbc = new double[3];
            double[] xt = new double[3];
            double[] xu = new double[3];
            for (int i = 0; i < ia.length; ++i) {
                xa[i] = ia[i];
                xb[i] = ib[i];
                xc[i] = ic[i];
            }
            DoubleMath.sub((double[])xa, (double[])xb, (double[])xab);
            DoubleMath.normalize((double[])xab, (double[])xab);
            DoubleMath.sub((double[])xb, (double[])xc, (double[])xbc);
            DoubleMath.normalize((double[])xbc, (double[])xbc);
            xt[0] = xab[2] * xbc[1] - xab[1] * xbc[2];
            xt[1] = xab[0] * xbc[2] - xab[2] * xbc[0];
            xt[2] = xab[1] * xbc[0] - xab[0] * xbc[1];
            double cosine = xab[0] * xbc[0] + xab[1] * xbc[1] + xab[2] * xbc[2];
            double sine = FastMath.sqrt((double)FastMath.max((double)(1.0 - cosine * cosine), (double)1.0E-7));
            if (FastMath.abs((double)cosine) >= 1.0) {
                logger.warning("Undefined Dihedral");
            }
            DoubleMath.scale((double[])xt, (double)(1.0 / sine), (double[])xt);
            xu[0] = xt[1] * xab[2] - xt[2] * xab[1];
            xu[1] = xt[2] * xab[0] - xt[0] * xab[2];
            xu[2] = xt[0] * xab[1] - xt[1] * xab[0];
            x[0] = xa[0] + bond * (xu[0] * zsin0 * zcos1 + xt[0] * zsin0 * zsin1 - xab[0] * zcos0);
            x[1] = xa[1] + bond * (xu[1] * zsin0 * zcos1 + xt[1] * zsin0 * zsin1 - xab[1] * zcos0);
            x[2] = xa[2] + bond * (xu[2] * zsin0 * zcos1 + xt[2] * zsin0 * zsin1 - xab[2] * zcos0);
        } else if (FastMath.abs((int)chiral) == 1) {
            double b;
            double a;
            double c;
            double[] xa = new double[3];
            double[] xb = new double[3];
            double[] xc = new double[3];
            double[] xba = new double[3];
            double[] xac = new double[3];
            double[] xt = new double[3];
            for (int i = 0; i < ia.length; ++i) {
                xa[i] = ia[i];
                xb[i] = ib[i];
                xc[i] = ic[i];
            }
            DoubleMath.sub((double[])xb, (double[])xa, (double[])xba);
            DoubleMath.normalize((double[])xba, (double[])xba);
            DoubleMath.sub((double[])xa, (double[])xc, (double[])xac);
            DoubleMath.normalize((double[])xac, (double[])xac);
            xt[0] = xba[2] * xac[1] - xba[1] * xac[2];
            xt[1] = xba[0] * xac[2] - xba[2] * xac[0];
            xt[2] = xba[1] * xac[0] - xba[0] * xac[1];
            double cosine = xba[0] * xac[0] + xba[1] * xac[1] + xba[2] * xac[2];
            double sine2 = FastMath.max((double)(1.0 - cosine * cosine), (double)1.0E-7);
            if (FastMath.abs((double)cosine) >= 1.0) {
                logger.warning("Defining Atom Colinear");
            }
            if ((c = (1.0 + (a = (-zcos1 - cosine * zcos0) / sine2) * zcos1 - (b = (zcos0 + cosine * zcos1) / sine2) * zcos0) / sine2) > 1.0E-7) {
                c = (double)chiral * FastMath.sqrt((double)c);
            } else if (c < -1.0E-7) {
                c = FastMath.sqrt((double)((a * xac[0] + b * xba[0]) * (a * xac[0] + b * xba[0]) + (a * xac[1] + b * xba[1]) * (a * xac[1] + b * xba[1]) + (a * xac[2] + b * xba[2]) * (a * xac[2] + b * xba[2])));
                a /= c;
                b /= c;
                c = 0.0;
            } else {
                c = 0.0;
            }
            x[0] = xa[0] + bond * (a * xac[0] + b * xba[0] + c * xt[0]);
            x[1] = xa[1] + bond * (a * xac[1] + b * xba[1] + c * xt[1]);
            x[2] = xa[2] + bond * (a * xac[2] + b * xba[2] + c * xt[2]);
        }
        System.arraycopy(x, 0, ret, 0, ret.length);
        return ret;
    }

    public static class MissingAtomTypeException
    extends Exception {
        private static final long serialVersionUID = 1L;
        public final Residue residue;
        public final Atom atom;

        public MissingAtomTypeException(Residue residue, Atom atom) {
            this.residue = residue;
            this.atom = atom;
        }

        @Override
        public String toString() {
            StringBuilder sb = new StringBuilder();
            sb.append(String.format(" Atom %s", this.atom));
            if (this.residue != null) {
                sb.append(String.format("\n of residue %c-%s", this.residue.getChainID(), this.residue));
            }
            sb.append("\n could not be assigned an atom type.\n");
            return sb.toString();
        }
    }

    public static class MissingHeavyAtomException
    extends Exception {
        private static final long serialVersionUID = 1L;
        public final String atomName;
        public final AtomType atomType;
        final Atom bondedTo;

        public MissingHeavyAtomException(String atomName, AtomType atomType, Atom bondedTo) {
            this.atomName = atomName;
            this.atomType = atomType;
            this.bondedTo = bondedTo;
        }

        @Override
        public String toString() {
            StringBuilder sb = new StringBuilder();
            if (this.atomType != null) {
                sb.append(String.format("\n An atom of type\n %s\n", this.atomType));
            } else {
                sb.append(String.format("\n Atom %s", this.atomName));
            }
            sb.append(" was not found");
            if (this.bondedTo != null) {
                sb.append(String.format(" bonded to atom %s ", this.bondedTo));
            }
            sb.append(".\n");
            return sb.toString();
        }
    }
}

