/*
 * Decompiled with CFR 0.152.
 */
package ffx.algorithms.dynamics.thermostats;

import ffx.algorithms.dynamics.thermostats.ThermostatEnum;
import ffx.numerics.Constraint;
import ffx.numerics.Potential;
import ffx.potential.SystemState;
import java.util.ArrayList;
import java.util.List;
import java.util.Random;
import java.util.logging.Level;
import java.util.logging.Logger;
import org.apache.commons.math3.util.FastMath;

public abstract class Thermostat {
    private static final Logger logger = Logger.getLogger(Thermostat.class.getName());
    private final double[] centerOfMass = new double[3];
    private final double[] linearMomentum = new double[3];
    private final double[] angularMomentum = new double[3];
    private final int constrainedDoF;
    protected ThermostatEnum name;
    protected double kT;
    protected int degreesOfFreedom;
    protected final SystemState state;
    protected Potential.VARIABLE_TYPE[] type;
    protected Random random;
    protected List<Constraint> constraints;
    double targetTemperature;
    private boolean removeCenterOfMassMotion;
    private boolean quiet = false;

    public Thermostat(SystemState state, Potential.VARIABLE_TYPE[] type, double targetTemperature) {
        this(state, type, targetTemperature, new ArrayList<Constraint>());
    }

    public Thermostat(SystemState state, Potential.VARIABLE_TYPE[] type, double targetTemperature, List<Constraint> constraints) {
        this.state = state;
        this.type = type;
        int n = state.getNumberOfVariables();
        assert (n > 3);
        assert (type.length == n);
        this.random = new Random();
        this.setTargetTemperature(targetTemperature);
        this.constraints = new ArrayList<Constraint>(constraints);
        double nConstrained = constraints.stream().mapToInt(Constraint::getNumDegreesFrozen).sum();
        double[] mass = state.getMass();
        int massConstrained = 0;
        for (int i = 0; i < n; ++i) {
            if (!(mass[i] <= 0.0)) continue;
            ++massConstrained;
        }
        if (massConstrained > 0 && nConstrained > 0.0) {
            logger.severe("Mass-constraints with other constraints are not supported.");
        }
        this.constrainedDoF = (int)FastMath.max((double)nConstrained, (double)massConstrained);
        this.removeCenterOfMassMotion = true;
        this.degreesOfFreedom = n - 3 - this.constrainedDoF;
        this.computeKineticEnergy();
    }

    public static ThermostatEnum parseThermostat(String str) {
        try {
            return ThermostatEnum.valueOf(str.toUpperCase());
        }
        catch (Exception e) {
            logger.info(String.format(" Could not parse %s as a thermostat; defaulting to Berendsen.", str));
            return ThermostatEnum.BERENDSEN;
        }
    }

    public void centerOfMassMotion(boolean remove, boolean print) {
        double totalMass = 0.0;
        for (int i = 0; i < 3; ++i) {
            this.centerOfMass[i] = 0.0;
            this.linearMomentum[i] = 0.0;
            this.angularMomentum[i] = 0.0;
        }
        int nVariables = this.state.getNumberOfVariables();
        double[] x = this.state.x();
        double[] v = this.state.v();
        double[] mass = this.state.getMass();
        int index = 0;
        while (index < nVariables) {
            double m = mass[index];
            if (m <= 0.0 || this.type[index] == Potential.VARIABLE_TYPE.OTHER) {
                ++index;
                continue;
            }
            assert (this.type[index] == Potential.VARIABLE_TYPE.X);
            double xx = x[index];
            double vx = v[index++];
            assert (this.type[index] == Potential.VARIABLE_TYPE.Y);
            double yy = x[index];
            double vy = v[index++];
            assert (this.type[index] == Potential.VARIABLE_TYPE.Z);
            double zz = x[index];
            double vz = v[index++];
            totalMass += m;
            this.centerOfMass[0] = this.centerOfMass[0] + xx * m;
            this.centerOfMass[1] = this.centerOfMass[1] + yy * m;
            this.centerOfMass[2] = this.centerOfMass[2] + zz * m;
            this.linearMomentum[0] = this.linearMomentum[0] + vx * m;
            this.linearMomentum[1] = this.linearMomentum[1] + vy * m;
            this.linearMomentum[2] = this.linearMomentum[2] + vz * m;
            this.angularMomentum[0] = this.angularMomentum[0] + (yy * vz - zz * vy) * m;
            this.angularMomentum[1] = this.angularMomentum[1] + (zz * vx - xx * vz) * m;
            this.angularMomentum[2] = this.angularMomentum[2] + (xx * vy - yy * vx) * m;
        }
        this.angularMomentum[0] = this.angularMomentum[0] - (this.centerOfMass[1] * this.linearMomentum[2] - this.centerOfMass[2] * this.linearMomentum[1]) / totalMass;
        this.angularMomentum[1] = this.angularMomentum[1] - (this.centerOfMass[2] * this.linearMomentum[0] - this.centerOfMass[0] * this.linearMomentum[2]) / totalMass;
        this.angularMomentum[2] = this.angularMomentum[2] - (this.centerOfMass[0] * this.linearMomentum[1] - this.centerOfMass[1] * this.linearMomentum[0]) / totalMass;
        this.centerOfMass[0] = this.centerOfMass[0] / totalMass;
        this.centerOfMass[1] = this.centerOfMass[1] / totalMass;
        this.centerOfMass[2] = this.centerOfMass[2] / totalMass;
        this.linearMomentum[0] = this.linearMomentum[0] / totalMass;
        this.linearMomentum[1] = this.linearMomentum[1] / totalMass;
        this.linearMomentum[2] = this.linearMomentum[2] / totalMass;
        if (print) {
            String sb = String.format("  Center of Mass   (%12.3f,%12.3f,%12.3f)\n  Linear Momentum  (%12.3f,%12.3f,%12.3f)\n  Angular Momentum (%12.3f,%12.3f,%12.3f)", this.centerOfMass[0], this.centerOfMass[1], this.centerOfMass[2], this.linearMomentum[0], this.linearMomentum[1], this.linearMomentum[2], this.angularMomentum[0], this.angularMomentum[1], this.angularMomentum[2]);
            logger.info(sb);
        }
        if (remove) {
            this.removeCenterOfMassMotion(print);
            this.centerOfMassMotion(false, print);
        }
    }

    public final void computeKineticEnergy() {
        double e = 0.0;
        double[] v = this.state.v();
        double[] mass = this.state.getMass();
        for (int i = 0; i < this.state.getNumberOfVariables(); ++i) {
            double m = mass[i];
            if (!(m > 0.0)) continue;
            double velocity = v[i];
            double v2 = velocity * velocity;
            e += m * v2;
        }
        this.state.setTemperature(e / (0.831446261815324 * (double)this.degreesOfFreedom));
        this.state.setKineticEnergy(e *= 0.0011950286806883365);
    }

    public abstract void halfStep(double var1);

    public abstract void fullStep(double var1);

    public double getCurrentTemperature() {
        return this.state.getTemperature();
    }

    public int getDegreesOfFreedom() {
        return this.degreesOfFreedom;
    }

    public double getKineticEnergy() {
        return this.state.getKineticEnergy();
    }

    public boolean getRemoveCenterOfMassMotion() {
        return this.removeCenterOfMassMotion;
    }

    public void setRemoveCenterOfMassMotion(boolean remove) {
        this.removeCenterOfMassMotion = remove;
        int nVariables = this.state.getNumberOfVariables();
        this.degreesOfFreedom = this.removeCenterOfMassMotion ? nVariables - 3 - this.constrainedDoF : nVariables - this.constrainedDoF;
    }

    public double getTargetTemperature() {
        return this.targetTemperature;
    }

    public void setTargetTemperature(double t) {
        assert (t > 0.0);
        this.targetTemperature = t;
        this.kT = t * 0.831446261815324;
    }

    public void maxwell(double targetTemperature) {
        if (logger.isLoggable(Level.FINE)) {
            logger.fine("\n Initializing velocities to target temperature");
        }
        this.setTargetTemperature(targetTemperature);
        double[] v = this.state.v();
        double[] mass = this.state.getMass();
        for (int i = 0; i < this.state.getNumberOfVariables(); ++i) {
            double m = mass[i];
            if (!(m > 0.0)) continue;
            v[i] = this.random.nextGaussian() * FastMath.sqrt((double)(0.831446261815324 * targetTemperature / m));
        }
        if (this.removeCenterOfMassMotion) {
            this.centerOfMassMotion(true, !this.quiet);
        }
        this.computeKineticEnergy();
        double scale = FastMath.sqrt((double)(targetTemperature / this.state.getTemperature()));
        for (int i = 0; i < this.state.getNumberOfVariables(); ++i) {
            double m = mass[i];
            if (!(m > 0.0)) continue;
            int n = i;
            v[n] = v[n] * scale;
        }
        this.computeKineticEnergy();
        this.log(Level.INFO);
    }

    public void maxwell() {
        this.maxwell(this.targetTemperature);
    }

    public double[] maxwellIndividual(double mass) {
        double[] vv = new double[3];
        if (mass > 0.0) {
            for (int i = 0; i < 3; ++i) {
                vv[i] = this.random.nextGaussian() * FastMath.sqrt((double)(0.831446261815324 * this.targetTemperature / mass));
            }
        }
        return vv;
    }

    public void setQuiet(boolean quiet) {
        this.quiet = quiet;
    }

    public void setRandomSeed(long seed) {
        this.random.setSeed(seed);
    }

    public String toString() {
        return this.logTemp();
    }

    public String logTemp() {
        StringBuilder sb = new StringBuilder(String.format("  Target temperature:           %7.2f Kelvin\n", this.targetTemperature));
        sb.append(String.format("  Current temperature:          %7.2f Kelvin\n", this.state.getTemperature()));
        sb.append(String.format("  Number of variables:          %7d\n", this.state.getNumberOfVariables()));
        sb.append(String.format("  Number of degrees of freedom: %7d\n", this.degreesOfFreedom));
        sb.append(String.format("  Kinetic Energy:               %7.2f\n", this.state.getKineticEnergy()));
        sb.append(String.format("  kT per degree of freedom:     %7.2f", 418.4 * this.state.getKineticEnergy() / ((double)this.degreesOfFreedom * this.kT)));
        return sb.toString();
    }

    protected void log(Level level) {
        if (logger.isLoggable(level) && !this.quiet) {
            logger.log(level, this.logTemp());
        }
    }

    private void removeCenterOfMassMotion(boolean print) {
        double xx = 0.0;
        double yy = 0.0;
        double zz = 0.0;
        double xy = 0.0;
        double xz = 0.0;
        double yz = 0.0;
        int index = 0;
        double[] x = this.state.x();
        double[] mass = this.state.getMass();
        while (index < this.state.getNumberOfVariables()) {
            if (this.type[index] == Potential.VARIABLE_TYPE.OTHER) {
                ++index;
                continue;
            }
            double m = mass[index];
            assert (this.type[index] == Potential.VARIABLE_TYPE.X);
            double xi = x[index++] - this.centerOfMass[0];
            assert (this.type[index] == Potential.VARIABLE_TYPE.Y);
            double yi = x[index++] - this.centerOfMass[1];
            assert (this.type[index] == Potential.VARIABLE_TYPE.Z);
            double zi = x[index++] - this.centerOfMass[2];
            xx += xi * xi * m;
            yy += yi * yi * m;
            zz += zi * zi * m;
            xy += xi * yi * m;
            xz += xi * zi * m;
            yz += yi * zi * m;
        }
        index = 0;
        double[] v = this.state.v();
        while (index < this.state.getNumberOfVariables()) {
            if (this.type[index] == Potential.VARIABLE_TYPE.OTHER) {
                ++index;
                continue;
            }
            double m = mass[index];
            if (m > 0.0) {
                int n = index++;
                v[n] = v[n] - this.linearMomentum[0] / m;
                int n2 = index++;
                v[n2] = v[n2] - this.linearMomentum[1] / m;
                int n3 = index++;
                v[n3] = v[n3] - this.linearMomentum[2] / m;
                continue;
            }
            index += 3;
        }
        if (print) {
            logger.info("  Center of mass motion removed.");
        }
    }
}

