/*
 * Decompiled with CFR 0.152.
 */
package ffx.algorithms.dynamics;

import edu.rit.pj.Comm;
import ffx.algorithms.AlgorithmListener;
import ffx.algorithms.Terminatable;
import ffx.algorithms.dynamics.Barostat;
import ffx.algorithms.dynamics.MDEngine;
import ffx.algorithms.dynamics.MDVerbosity;
import ffx.algorithms.dynamics.MDWriteAction;
import ffx.algorithms.dynamics.MolecularDynamicsOpenMM;
import ffx.algorithms.dynamics.NonEquilbriumDynamics;
import ffx.algorithms.dynamics.integrators.BetterBeeman;
import ffx.algorithms.dynamics.integrators.Integrator;
import ffx.algorithms.dynamics.integrators.IntegratorEnum;
import ffx.algorithms.dynamics.integrators.Respa;
import ffx.algorithms.dynamics.integrators.Stochastic;
import ffx.algorithms.dynamics.integrators.VelocityVerlet;
import ffx.algorithms.dynamics.thermostats.Adiabatic;
import ffx.algorithms.dynamics.thermostats.Berendsen;
import ffx.algorithms.dynamics.thermostats.Bussi;
import ffx.algorithms.dynamics.thermostats.Thermostat;
import ffx.algorithms.dynamics.thermostats.ThermostatEnum;
import ffx.algorithms.thermodynamics.OrthogonalSpaceTempering;
import ffx.crystal.Crystal;
import ffx.numerics.Potential;
import ffx.numerics.math.RunningStatistics;
import ffx.potential.MolecularAssembly;
import ffx.potential.SystemState;
import ffx.potential.UnmodifiableState;
import ffx.potential.bonded.Atom;
import ffx.potential.bonded.LambdaInterface;
import ffx.potential.extended.ExtendedSystem;
import ffx.potential.openmm.OpenMMDualTopologyEnergy;
import ffx.potential.openmm.OpenMMEnergy;
import ffx.potential.parameters.ForceField;
import ffx.potential.parsers.DYNFilter;
import ffx.potential.parsers.PDBFilter;
import ffx.potential.parsers.XPHFilter;
import ffx.potential.parsers.XYZFilter;
import ffx.utilities.FileUtils;
import ffx.utilities.TinkerUtils;
import java.io.File;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.EnumSet;
import java.util.List;
import java.util.logging.Level;
import java.util.logging.Logger;
import org.apache.commons.configuration2.CompositeConfiguration;
import org.apache.commons.io.FilenameUtils;

public class MolecularDynamics
implements Runnable,
Terminatable {
    private static final Logger logger = Logger.getLogger(MolecularDynamics.class.getName());
    private static final double DEFAULT_LOG_INTERVAL = 0.25;
    private static final double DEFAULT_RESTART_INTERVAL = 1.0;
    private static final double DEFAULT_TRAJECTORY_INTERVAL = 10.0;
    private static final int DEFAULT_DYNAMICS_SLEEP_TIME = 25000;
    protected MolecularAssembly[] molecularAssembly;
    private final Potential potential;
    protected final AlgorithmListener algorithmListener;
    private final Integrator integrator;
    private Thermostat thermostat;
    boolean constantPressure = false;
    Barostat barostat = null;
    protected final SystemState state;
    protected UnmodifiableState initialState;
    private UnmodifiableState storedState;
    private final RunningStatistics temperatureStats = new RunningStatistics();
    private final RunningStatistics potentialEnergyStats = new RunningStatistics();
    private final RunningStatistics kineticEnergyStats = new RunningStatistics();
    private final RunningStatistics totalEnergyStats = new RunningStatistics();
    protected double dt = 0.001;
    private long nSteps = 1000L;
    double targetTemperature = 298.15;
    protected boolean initVelocities = true;
    protected boolean automaticWriteouts = true;
    protected double totalSimTime = 0.0;
    protected double restartInterval = 1.0;
    protected int restartFrequency;
    private double trajectoryInterval = 10.0;
    protected int trajectoryFrequency;
    private double logInterval = 0.25;
    protected int logFrequency;
    protected String fileType = "XYZ";
    boolean saveSnapshotAsPDB = true;
    PDBFilter[] pdbFilter = null;
    File restartFile = null;
    boolean loadRestart = false;
    DYNFilter dynFilter = null;
    private File fallbackDynFile;
    Level basicLogging = Level.INFO;
    private MDVerbosity verbosityLevel = MDVerbosity.VERBOSE;
    boolean initialized = false;
    private boolean terminate = false;
    protected boolean done;
    private ExtendedSystem esvSystem;
    private Stochastic esvIntegrator;
    private Adiabatic esvThermostat;
    private int printEsvFrequency = -1;
    private boolean nonEquilibriumLambda;
    private NonEquilbriumDynamics nonEquilibriumDynamics;

    public MolecularDynamics(MolecularAssembly assembly, Potential potentialEnergy, AlgorithmListener listener, ThermostatEnum requestedThermostat, IntegratorEnum requestedIntegrator) {
        this.molecularAssembly = new MolecularAssembly[1];
        this.molecularAssembly[0] = assembly;
        this.algorithmListener = listener;
        this.potential = potentialEnergy;
        if (potentialEnergy instanceof Barostat) {
            this.constantPressure = true;
            this.barostat = (Barostat)potentialEnergy;
        }
        int numberOfVariables = potentialEnergy.getNumberOfVariables();
        this.state = new SystemState(numberOfVariables);
        this.state.setMass(this.potential.getMass());
        CompositeConfiguration properties = this.molecularAssembly[0].getProperties();
        if (requestedIntegrator == null) {
            String integrate = properties.getString("integrate", "VERLET").trim();
            try {
                integrate = integrate.toUpperCase().replace("-", "_");
                requestedIntegrator = IntegratorEnum.valueOf(integrate.toUpperCase());
            }
            catch (Exception e) {
                requestedIntegrator = IntegratorEnum.VERLET;
            }
        }
        boolean oMMLogging = this.potential instanceof OpenMMEnergy;
        List constraints = potentialEnergy.getConstraints();
        this.integrator = switch (requestedIntegrator) {
            case IntegratorEnum.RESPA, IntegratorEnum.MTS -> {
                Respa respa = new Respa(this.state);
                int in = this.molecularAssembly[0].getProperties().getInt("respa-dt", 4);
                if (in < 2) {
                    in = 2;
                }
                if (!oMMLogging) {
                    respa.setInnerTimeSteps(in);
                }
                logger.log(Level.FINE, String.format(" Created a RESPA integrator with %d inner time steps.", in));
                yield respa;
            }
            case IntegratorEnum.STOCHASTIC, IntegratorEnum.LANGEVIN -> {
                double friction = properties.getDouble("friction", 91.0);
                logger.log(Level.FINE, String.format(" Friction set at %.3f collisions/picosecond", friction));
                Stochastic stochastic = new Stochastic(friction, this.state);
                if (properties.containsKey("randomseed")) {
                    stochastic.setRandomSeed(properties.getInt("randomseed", 0));
                }
                requestedThermostat = ThermostatEnum.ADIABATIC;
                yield stochastic;
            }
            case IntegratorEnum.BEEMAN -> new BetterBeeman(this.state);
            default -> new VelocityVerlet(this.state);
        };
        this.integrator.addConstraints(constraints);
        if (requestedThermostat == null) {
            String thermo = properties.getString("thermostat", "Berendsen").trim();
            try {
                thermo = thermo.toUpperCase().replace("-", "_");
                requestedThermostat = ThermostatEnum.valueOf(thermo.toUpperCase());
            }
            catch (Exception e) {
                requestedThermostat = ThermostatEnum.BERENDSEN;
            }
        }
        switch (requestedThermostat) {
            case BERENDSEN: {
                double tau = properties.getDouble("tau-temperature", 0.2);
                Thermostat thermostat = new Berendsen(this.state, potentialEnergy.getVariableTypes(), this.targetTemperature, tau, constraints);
                break;
            }
            case BUSSI: {
                double tau = properties.getDouble("tau-temperature", 0.2);
                Bussi bussi = new Bussi(this.state, potentialEnergy.getVariableTypes(), this.targetTemperature, tau, constraints);
                if (properties.containsKey("randomseed")) {
                    bussi.setRandomSeed(properties.getInt("randomseed", 0));
                }
                Thermostat thermostat = bussi;
                break;
            }
            default: {
                Thermostat thermostat = this.thermostat = new Adiabatic(this.state, potentialEnergy.getVariableTypes(), constraints);
            }
        }
        if (properties.containsKey("randomseed")) {
            this.thermostat.setRandomSeed(properties.getInt("randomseed", 0));
        }
        if (this.integrator instanceof Stochastic) {
            boolean removecom = assembly.getForceField().getBoolean("removecom", false);
            this.thermostat.setRemoveCenterOfMassMotion(removecom);
            if (removecom) {
                logger.info(" Removing center of mass motion from stochastic simulation.");
            }
        }
        this.done = true;
        this.fallbackDynFile = MolecularDynamics.defaultFallbackDyn(assembly);
    }

    public static MolecularDynamics dynamicsFactory(MolecularAssembly assembly, Potential potentialEnergy, AlgorithmListener listener, ThermostatEnum requestedThermostat, IntegratorEnum requestedIntegrator) {
        return MolecularDynamics.dynamicsFactory(assembly, potentialEnergy, listener, requestedThermostat, requestedIntegrator, MolecularDynamics.defaultEngine(assembly, potentialEnergy));
    }

    public static MolecularDynamics dynamicsFactory(MolecularAssembly assembly, Potential potentialEnergy, AlgorithmListener listener, ThermostatEnum requestedThermostat, IntegratorEnum requestedIntegrator, MDEngine engine) {
        switch (engine) {
            case OPENMM: 
            case OMM: {
                Barostat barostat = null;
                Potential openmmPotential = potentialEnergy;
                if (potentialEnergy instanceof Barostat) {
                    barostat = (Barostat)potentialEnergy;
                    logger.info(" dynamicsFactory: Molecular dynamics has barostat:" + String.valueOf(barostat));
                    openmmPotential = barostat.getCrystalPotential();
                }
                if (openmmPotential instanceof OpenMMEnergy || openmmPotential instanceof OpenMMDualTopologyEnergy) {
                    MolecularDynamicsOpenMM molecularDynamicsOpenMM = new MolecularDynamicsOpenMM(assembly, openmmPotential, listener, requestedThermostat, requestedIntegrator);
                    if (barostat != null) {
                        molecularDynamicsOpenMM.setBarostat(barostat);
                    }
                    return molecularDynamicsOpenMM;
                }
                logger.info(String.format(" Requested OpenMM molecular dynamics engine %s, but the potential does not use OpenMM: %s", new Object[]{engine, potentialEnergy}));
                return new MolecularDynamics(assembly, potentialEnergy, listener, requestedThermostat, requestedIntegrator);
            }
        }
        return new MolecularDynamics(assembly, potentialEnergy, listener, requestedThermostat, requestedIntegrator);
    }

    private static MDEngine defaultEngine(MolecularAssembly molecularAssembly, Potential potentialEnergy) {
        CompositeConfiguration properties = molecularAssembly.getProperties();
        String mdEngine = properties.getString("MD-engine");
        if (mdEngine != null) {
            if (mdEngine.equalsIgnoreCase("OMM")) {
                logger.info(" Creating OpenMM Dynamics Object");
                return MDEngine.OPENMM;
            }
            logger.info(" Creating FFX Dynamics Object");
            return MDEngine.FFX;
        }
        boolean ommLeaves = potentialEnergy.getUnderlyingPotentials().stream().anyMatch(p -> p instanceof OpenMMEnergy);
        boolean bl = ommLeaves = ommLeaves || potentialEnergy instanceof OpenMMEnergy;
        if (ommLeaves) {
            return MDEngine.OPENMM;
        }
        return MDEngine.FFX;
    }

    private static File defaultFallbackDyn(MolecularAssembly assembly) {
        String firstFileName = FilenameUtils.removeExtension((String)assembly.getFile().getAbsolutePath());
        return new File(firstFileName + ".dyn");
    }

    public void addAssembly(MolecularAssembly assembly) {
        this.molecularAssembly = Arrays.copyOf(this.molecularAssembly, this.molecularAssembly.length + 1);
        this.molecularAssembly[this.molecularAssembly.length - 1] = assembly;
    }

    public void attachExtendedSystem(ExtendedSystem system, double reportFreq) {
        if (this.esvSystem != null) {
            logger.warning("An ExtendedSystem is already attached to this MD!");
        }
        this.esvSystem = system;
        SystemState esvState = this.esvSystem.getState();
        this.esvIntegrator = new Stochastic(this.esvSystem.getThetaFriction(), esvState);
        if (!this.esvSystem.getConstraints().isEmpty()) {
            this.esvIntegrator.addConstraints(this.esvSystem.getConstraints());
        }
        this.esvThermostat = new Adiabatic(esvState, this.potential.getVariableTypes());
        this.printEsvFrequency = this.intervalToFreq(reportFreq, "Reporting (logging) interval");
        logger.info(String.format("  Attached extended system (%s) to molecular dynamics.", this.esvSystem.toString()));
        logger.info(String.format("  Extended System Theta Friction: %f", this.esvSystem.getThetaFriction()));
        logger.info(String.format("  Extended System Theta Mass: %f", this.esvSystem.getThetaMass()));
        logger.info(String.format("  Extended System Lambda Print Frequency: %d (fsec)", this.printEsvFrequency));
    }

    public void setNonEquilibriumLambda(boolean nonEquilibrium, int nEQSteps, boolean reverseNEQ) {
        this.nonEquilibriumLambda = nonEquilibrium;
        this.nonEquilibriumDynamics = this.nonEquilibriumLambda ? new NonEquilbriumDynamics(nEQSteps, reverseNEQ) : null;
    }

    public void dynamic(long nSteps, double timeStep, double loggingInterval, double trajectoryInterval, double temperature, boolean initVelocities, String fileType, double restartInterval, File dyn) {
        this.fileType = fileType;
        this.setRestartFrequency(restartInterval);
        this.dynamic(nSteps, timeStep, loggingInterval, trajectoryInterval, temperature, initVelocities, dyn);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void dynamic(long nSteps, double timeStep, double loggingInterval, double trajectoryInterval, double temperature, boolean initVelocities, File dyn) {
        if (!this.done) {
            logger.warning(" Programming error - a thread invoked dynamic when it was already running.");
            return;
        }
        this.init(nSteps, timeStep, loggingInterval, trajectoryInterval, this.fileType, this.restartInterval, temperature, initVelocities, dyn);
        Thread dynamicThread = new Thread(this);
        dynamicThread.start();
        MolecularDynamics molecularDynamics = this;
        synchronized (molecularDynamics) {
            try {
                while (dynamicThread.isAlive()) {
                    this.wait(0L, 25000);
                }
            }
            catch (InterruptedException e) {
                String message = " Molecular dynamics interrupted.";
                logger.log(Level.WARNING, message, e);
            }
        }
        if (!this.verbosityLevel.isQuiet()) {
            logger.info(" Done with an MD round.");
        }
    }

    public MolecularAssembly[] getAssemblyArray() {
        return (MolecularAssembly[])this.molecularAssembly.clone();
    }

    public File getDynFile() {
        return this.restartFile;
    }

    public double getInitialKineticEnergy() {
        return this.initialState.kineticEnergy();
    }

    public double getInitialPotentialEnergy() {
        return this.initialState.potentialEnergy();
    }

    public double getInitialTemperature() {
        return this.initialState.temperature();
    }

    public double getInitialTotalEnergy() {
        return this.initialState.getTotalEnergy();
    }

    public int getIntervalSteps() {
        return 1;
    }

    public void setIntervalSteps(int intervalSteps) {
    }

    public double getKineticEnergy() {
        return this.state.getKineticEnergy();
    }

    public double getPotentialEnergy() {
        return this.state.getPotentialEnergy();
    }

    public double getTemperature() {
        return this.state.getTemperature();
    }

    public Thermostat getThermostat() {
        return this.thermostat;
    }

    public void setThermostat(Thermostat thermostat) {
        this.thermostat = thermostat;
    }

    public double getTimeStep() {
        return this.dt;
    }

    private void setTimeStep(double step) {
        this.dt = step * 0.001;
        this.setRestartFrequency(this.restartInterval);
        this.setLoggingFrequency(this.logInterval);
        this.setTrajectoryFrequency(this.trajectoryInterval);
    }

    public double getTotalEnergy() {
        return this.state.getTotalEnergy();
    }

    public MDVerbosity getVerbosityLevel() {
        return this.verbosityLevel;
    }

    public void setVerbosityLevel(MDVerbosity level) {
        this.verbosityLevel = level;
        this.basicLogging = level == MDVerbosity.SILENT ? Level.FINE : Level.INFO;
    }

    public void init(long nSteps, double timeStep, double loggingInterval, double trajectoryInterval, String fileType, double restartInterval, double temperature, boolean initVelocities, File dyn) {
        if (!this.done) {
            logger.warning(" Programming error - attempt to modify parameters of a running MolecularDynamics instance.");
            return;
        }
        if (this.integrator instanceof Stochastic) {
            if (this.constantPressure) {
                logger.log(this.basicLogging, "\n Stochastic dynamics in the NPT ensemble");
            } else {
                logger.log(this.basicLogging, "\n Stochastic dynamics in the NVT ensemble");
            }
        } else if (!(this.thermostat instanceof Adiabatic)) {
            if (this.constantPressure) {
                logger.log(this.basicLogging, "\n Molecular dynamics in the NPT ensemble");
            } else {
                logger.log(this.basicLogging, "\n Molecular dynamics in the NVT ensemble");
            }
        } else if (this.constantPressure) {
            logger.severe("\n NPT Molecular dynamics requires a thermostat");
        } else {
            logger.log(this.basicLogging, "\n Molecular dynamics in the NVE ensemble");
        }
        this.nSteps = nSteps;
        this.totalSimTime = 0.0;
        this.setTimeStep(timeStep);
        this.setLoggingFrequency(loggingInterval);
        this.setTrajectoryFrequency(trajectoryInterval);
        this.setRestartFrequency(restartInterval);
        this.checkFrequency("Reporting           ", this.logFrequency);
        if (this.automaticWriteouts) {
            this.checkFrequency("Trajectory Write-Out", this.trajectoryFrequency);
            this.checkFrequency("Restart             ", this.restartFrequency);
        }
        this.saveSnapshotAsPDB = true;
        if (fileType.equalsIgnoreCase("XYZ") || fileType.equalsIgnoreCase("ARC")) {
            this.saveSnapshotAsPDB = false;
        } else if (!fileType.equalsIgnoreCase("PDB")) {
            logger.warning("Snapshot file type unrecognized; saving snapshots as PDB.\n");
        }
        this.setArchiveFile();
        this.restartFile = dyn == null ? this.fallbackDynFile : dyn;
        boolean bl = this.loadRestart = this.restartFile.exists() && !this.initialized;
        if (this.dynFilter == null) {
            this.dynFilter = new DYNFilter(this.molecularAssembly[0].getName());
        }
        this.targetTemperature = temperature;
        this.initVelocities = initVelocities;
        this.done = false;
        if (this.loadRestart) {
            logger.info("  Continuing from " + this.restartFile.getAbsolutePath());
        }
        if (!this.verbosityLevel.isQuiet()) {
            logger.info(String.format("  Integrator:          %15s", this.integrator.toString()));
            logger.info(String.format("  Thermostat:          %15s", this.thermostat.toString()));
            logger.info(String.format("  Number of steps:     %8d", nSteps));
            logger.info(String.format("  Time step:           %8.3f (fsec)", timeStep));
            logger.info(String.format("  Print interval:      %8.3f (psec)", loggingInterval));
            logger.info(String.format("  Save interval:       %8.3f (psec)", trajectoryInterval));
            if (this.molecularAssembly.length > 1) {
                for (int i = 0; i < this.molecularAssembly.length; ++i) {
                    File archiveFile = this.molecularAssembly[i].getArchiveFile();
                    logger.info(String.format("  Archive file %3d: %s", i + 1, archiveFile.getAbsolutePath()));
                }
            } else {
                logger.info(String.format("  Archive file:     %s", this.molecularAssembly[0].getArchiveFile().getAbsolutePath()));
            }
            logger.info(String.format("  Restart file:     %s", this.restartFile.getAbsolutePath()));
        }
    }

    public void init(long nSteps, double timeStep, double loggingInterval, double trajectoryInterval, double temperature, boolean initVelocities, File dyn) {
        this.init(nSteps, timeStep, loggingInterval, trajectoryInterval, "XYZ", this.restartInterval, temperature, initVelocities, dyn);
    }

    @Override
    public void run() {
        try {
            this.preRunOps();
        }
        catch (IllegalStateException ise) {
            return;
        }
        this.initializeEnergies();
        this.postInitEnergies();
        this.mainLoop();
        this.postRun();
    }

    public void setAutomaticWriteouts(boolean autoWriteout) {
        this.automaticWriteouts = autoWriteout;
    }

    public void setFallbackDynFile(File fallback) {
        this.fallbackDynFile = fallback;
    }

    public void setFileType(String fileType) {
        this.fileType = fileType;
    }

    public void setObtainVelAcc(boolean obtainVA) {
    }

    public void setRestartFrequency(double restartInterval) throws IllegalArgumentException {
        this.restartFrequency = this.intervalToFreq(restartInterval, "Restart interval");
        this.restartInterval = restartInterval;
    }

    public void setArchiveFiles(File[] archiveFiles) {
        logger.info(" Setting archive files:\n " + Arrays.toString(archiveFiles));
        int n = this.molecularAssembly.length;
        assert (archiveFiles.length == n);
        for (int i = 0; i < n; ++i) {
            this.molecularAssembly[i].setArchiveFile(archiveFiles[i]);
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public void terminate() {
        this.terminate = true;
        while (!this.done) {
            MolecularDynamics molecularDynamics = this;
            synchronized (molecularDynamics) {
                try {
                    this.wait(1L);
                }
                catch (Exception e) {
                    logger.log(Level.WARNING, " Exception terminating dynamics.\n", e);
                }
            }
        }
    }

    public EnumSet<MDWriteAction> writeFilesForStep(long step, boolean trySnapshot, boolean tryRestart) {
        return this.writeFilesForStep(step, trySnapshot, tryRestart, null);
    }

    public EnumSet<MDWriteAction> writeFilesForStep(long step, boolean trySnapshot, boolean tryRestart, String[] extraLines) {
        Comm world;
        ArrayList<String> linesList;
        ArrayList<String> arrayList = linesList = extraLines == null ? new ArrayList<String>() : new ArrayList<String>(Arrays.asList(extraLines));
        if (this.potential instanceof LambdaInterface) {
            String lamString = String.format("Lambda: %.8f", ((LambdaInterface)this.potential).getLambda());
            linesList.add(lamString);
        }
        String tempString = String.format("Temp: %.2f", this.thermostat.getTargetTemperature());
        linesList.add(tempString);
        String timeString = String.format(" Time: %7.3e", this.totalSimTime);
        linesList.add(timeString);
        if (this.esvSystem != null) {
            String pHString = String.format("pH: %.2f", this.esvSystem.getConstantPh());
            linesList.add(pHString);
        }
        if ((world = Comm.world()) != null && world.size() > 1) {
            String rankString = String.format("Rank: %d", world.rank());
            linesList.add(rankString);
        }
        String[] allLines = new String[linesList.size()];
        allLines = linesList.toArray(allLines);
        EnumSet<MDWriteAction> written = EnumSet.noneOf(MDWriteAction.class);
        if (step != 0L) {
            if (trySnapshot && this.trajectoryFrequency > 0 && step % (long)this.trajectoryFrequency == 0L) {
                this.potential.setCoordinates(this.state.x());
                if (this.totalEnergyStats.getCount() > 0L) {
                    logger.info(String.format("\n Average Values for the Last %d Out of %d Dynamics Steps\n", this.trajectoryFrequency, step));
                    logger.info(String.format("  Simulation Time  %16.4f Picosecond", (double)step * this.dt));
                    logger.info(String.format("  Total Energy     %16.4f Kcal/mole   (+/-%9.4f)", this.totalEnergyStats.getMean(), this.totalEnergyStats.getStandardDeviation()));
                    logger.info(String.format("  Potential Energy %16.4f Kcal/mole   (+/-%9.4f)", this.potentialEnergyStats.getMean(), this.potentialEnergyStats.getStandardDeviation()));
                    logger.info(String.format("  Kinetic Energy   %16.4f Kcal/mole   (+/-%9.4f)", this.kineticEnergyStats.getMean(), this.kineticEnergyStats.getStandardDeviation()));
                    logger.info(String.format("  Temperature      %16.4f Kelvin      (+/-%9.4f)\n", this.temperatureStats.getMean(), this.temperatureStats.getStandardDeviation()));
                    this.totalEnergyStats.reset();
                    this.potentialEnergyStats.reset();
                    this.kineticEnergyStats.reset();
                    this.temperatureStats.reset();
                }
                if (this.esvSystem != null) {
                    for (Atom atom : this.molecularAssembly[0].getAtomList()) {
                        int atomIndex = atom.getIndex() - 1;
                        atom.setOccupancy(this.esvSystem.getTitrationLambda(atomIndex));
                        atom.setTempFactor(this.esvSystem.getTautomerLambda(atomIndex));
                    }
                }
                this.appendSnapshot(allLines);
                written.add(MDWriteAction.SNAPSHOT);
            }
            if (tryRestart && this.restartFrequency > 0 && step % (long)this.restartFrequency == 0L) {
                this.writeRestart();
                written.add(MDWriteAction.RESTART);
            }
        }
        return written;
    }

    public void writeRestart() {
        String dynName = FileUtils.relativePathTo((File)this.restartFile).toString();
        double[] x = this.state.x();
        double[] v = this.state.v();
        double[] a = this.state.a();
        double[] aPrevious = this.state.aPrevious();
        if (this.dynFilter.writeDYN(this.restartFile, this.molecularAssembly[0].getCrystal(), x, v, a, aPrevious)) {
            logger.log(this.basicLogging, String.format(" Wrote dynamics restart to:  %s.", dynName));
        } else {
            logger.log(this.basicLogging, String.format(" Writing dynamics restart failed:  %s.", dynName));
        }
        if (this.esvSystem != null) {
            this.esvSystem.writeRestart();
            this.esvSystem.writeLambdaHistogram(false);
        }
        this.potential.writeAdditionalRestartInfo(true);
    }

    private void setArchiveFile() {
        for (MolecularAssembly assembly : this.molecularAssembly) {
            File file = assembly.getFile();
            String filename = FilenameUtils.removeExtension((String)file.getAbsolutePath());
            File archiveFile = assembly.getArchiveFile();
            if (archiveFile != null) continue;
            archiveFile = new File(filename + ".arc");
            assembly.setArchiveFile(XYZFilter.version((File)archiveFile));
        }
    }

    private int intervalToFreq(double interval, String describe) throws IllegalArgumentException {
        if (!Double.isFinite(interval) || interval <= 0.0) {
            throw new IllegalArgumentException(String.format(" %s must be positive finite value in ps, was %10.4g", describe, interval));
        }
        if (interval >= this.dt) {
            return (int)(interval / this.dt);
        }
        logger.warning(String.format(" Specified %s of %.6f ps < time step %.6f ps; interval is set to once per time step!", describe, interval, this.dt));
        return 1;
    }

    private void setLoggingFrequency(double logInterval) {
        this.logFrequency = this.intervalToFreq(logInterval, "Reporting (logging) interval");
        this.logInterval = logInterval;
    }

    private void setTrajectoryFrequency(double snapshotInterval) {
        this.trajectoryFrequency = this.intervalToFreq(snapshotInterval, "Trajectory writeout interval");
        this.trajectoryInterval = snapshotInterval;
    }

    void preRunOps() throws IllegalStateException {
        this.done = false;
        this.terminate = false;
        this.thermostat.setTargetTemperature(this.targetTemperature);
        boolean quiet = this.verbosityLevel.isQuiet();
        this.thermostat.setQuiet(quiet);
        Integrator integrator = this.integrator;
        if (integrator instanceof Stochastic) {
            Stochastic stochastic = (Stochastic)integrator;
            stochastic.setTemperature(this.targetTemperature);
        }
        this.integrator.setTimeStep(this.dt);
        if (!this.initialized) {
            if (this.loadRestart) {
                double[] aPrevious;
                double[] a;
                double[] v;
                double[] x;
                Crystal crystal = this.molecularAssembly[0].getCrystal();
                if (!this.dynFilter.readDYN(this.restartFile, crystal, x = this.state.x(), v = this.state.v(), a = this.state.a(), aPrevious = this.state.aPrevious())) {
                    String message = " Could not load the restart file - dynamics terminated.";
                    logger.log(Level.WARNING, message);
                    this.done = true;
                    throw new IllegalStateException(message);
                }
                this.molecularAssembly[0].setCrystal(crystal);
                this.potential.setCoordinates(x);
                this.potential.setVelocity(v);
                this.potential.setAcceleration(a);
                this.potential.setPreviousAcceleration(aPrevious);
            } else {
                this.potential.getCoordinates(this.state.x());
                if (this.initVelocities) {
                    this.thermostat.maxwell(this.targetTemperature);
                } else {
                    Arrays.fill(this.state.v(), 0.0);
                }
            }
        } else if (this.initVelocities) {
            this.thermostat.maxwell(this.targetTemperature);
        }
    }

    private void initializeEnergies() {
        OrthogonalSpaceTempering orthogonalSpaceTempering;
        double[] x = this.state.x();
        double[] gradient = this.state.gradient();
        boolean propagateLambda = true;
        Potential potential = this.potential;
        if (potential instanceof OrthogonalSpaceTempering) {
            orthogonalSpaceTempering = (OrthogonalSpaceTempering)potential;
            propagateLambda = orthogonalSpaceTempering.getPropagateLambda();
            orthogonalSpaceTempering.setPropagateLambda(false);
        }
        if (this.esvSystem != null && this.potential instanceof OpenMMEnergy) {
            this.state.setPotentialEnergy(((OpenMMEnergy)this.potential).energyAndGradientFFX(x, gradient));
        } else {
            this.state.setPotentialEnergy(this.potential.energyAndGradient(x, gradient));
        }
        potential = this.potential;
        if (potential instanceof OrthogonalSpaceTempering) {
            orthogonalSpaceTempering = (OrthogonalSpaceTempering)potential;
            orthogonalSpaceTempering.setPropagateLambda(propagateLambda);
        }
        if (!this.loadRestart || this.initialized || this.integrator instanceof Respa) {
            if (this.integrator instanceof Respa) {
                this.potential.setEnergyTermState(Potential.STATE.SLOW);
                this.potential.energyAndGradient(x, gradient);
            }
            int numberOfVariables = this.state.getNumberOfVariables();
            double[] a = this.state.a();
            double[] mass = this.state.getMass();
            for (int i = 0; i < numberOfVariables; ++i) {
                a[i] = -418.4 * gradient[i] / mass[i];
            }
            this.state.copyAccelerationsToPrevious();
        }
        if (this.esvSystem != null) {
            SystemState esvState = this.esvSystem.getState();
            double[] esvA = esvState.a();
            double[] esvMass = esvState.getMass();
            int nESVs = esvState.getNumberOfVariables();
            double[] gradESV = this.esvSystem.postForce();
            for (int i = 0; i < nESVs; ++i) {
                esvA[i] = -418.4 * gradESV[i] / esvMass[i];
            }
        }
        this.thermostat.computeKineticEnergy();
        if (this.esvSystem != null) {
            this.esvThermostat.computeKineticEnergy();
            double kineticEnergy = this.thermostat.getKineticEnergy();
            double esvKineticEnergy = this.esvThermostat.getKineticEnergy();
            this.state.setKineticEnergy(kineticEnergy + esvKineticEnergy);
        }
        this.initialState = new UnmodifiableState(this.state);
        this.temperatureStats.reset();
        this.potentialEnergyStats.reset();
        this.kineticEnergyStats.reset();
        this.totalEnergyStats.reset();
    }

    void postInitEnergies() {
        this.initialized = true;
        logger.log(this.basicLogging, String.format("\n  %8s %12s %12s %12s %8s %8s", "Time", "Kinetic", "Potential", "Total", "Temp", "CPU"));
        logger.log(this.basicLogging, String.format("  %8s %12s %12s %12s %8s %8s", "psec", "kcal/mol", "kcal/mol", "kcal/mol", "K", "sec"));
        logger.log(this.basicLogging, String.format("  %8s %12.4f %12.4f %12.4f %8.2f", "", this.state.getKineticEnergy(), this.state.getPotentialEnergy(), this.state.getTotalEnergy(), this.state.getTemperature()));
        this.storeState();
    }

    void postRun() {
        if (this.integrator instanceof Respa) {
            this.potential.setEnergyTermState(Potential.STATE.BOTH);
        }
        if (!this.terminate) {
            logger.log(this.basicLogging, String.format(" Completed %8d time steps\n", this.nSteps));
        }
        this.done = true;
        this.terminate = false;
    }

    protected void appendSnapshot(String[] extraLines) {
        int numAssemblies = this.molecularAssembly.length;
        int currentAssembly = 0;
        for (MolecularAssembly assembly : this.molecularAssembly) {
            File archiveFile = assembly.getArchiveFile();
            ForceField forceField = assembly.getForceField();
            CompositeConfiguration properties = assembly.getProperties();
            String name = assembly.getName();
            String[] tokens = name.split(" +");
            StringBuilder stringBuilder = new StringBuilder();
            int numTokens = tokens.length;
            for (int i = 0; i < numTokens; ++i) {
                if (tokens[i].equalsIgnoreCase("Energy:") || tokens[i].equalsIgnoreCase("Density:")) {
                    ++i;
                    continue;
                }
                stringBuilder.append(" ").append(tokens[i]);
            }
            assembly.setName(stringBuilder.toString());
            if (archiveFile != null && !this.saveSnapshotAsPDB) {
                String aiName = FileUtils.relativePathTo((File)archiveFile).toString();
                if (this.esvSystem == null) {
                    XYZFilter xyzFilter = new XYZFilter(archiveFile, assembly, forceField, properties);
                    if (xyzFilter.writeFile(archiveFile, true, extraLines)) {
                        logger.log(this.basicLogging, String.format(" Appended snapshot to:       %s", aiName));
                    } else {
                        logger.warning(String.format(" Appending snapshot failed:  %s", aiName));
                    }
                } else {
                    XPHFilter xphFilter = new XPHFilter(archiveFile, assembly, forceField, properties, this.esvSystem);
                    if (xphFilter.writeFile(archiveFile, true, extraLines)) {
                        logger.log(this.basicLogging, String.format(" Appended to XPH archive %s", aiName));
                    } else {
                        logger.warning(String.format(" Appending to XPH archive %s failed.", aiName));
                    }
                }
            } else if (this.saveSnapshotAsPDB) {
                if (this.pdbFilter == null) {
                    this.pdbFilter = new PDBFilter[numAssemblies];
                }
                if (this.pdbFilter[currentAssembly] == null) {
                    File pdbFile;
                    File file = assembly.getFile();
                    String extName = FilenameUtils.getExtension((String)file.getName());
                    if (extName.toLowerCase().startsWith("pdb")) {
                        pdbFile = TinkerUtils.version((File)file);
                    } else {
                        String filename = FilenameUtils.removeExtension((String)file.getAbsolutePath());
                        pdbFile = new File(filename + ".pdb");
                    }
                    this.pdbFilter[currentAssembly] = new PDBFilter(pdbFile, assembly, forceField, properties);
                    this.pdbFilter[currentAssembly].setModelNumbering(0);
                }
                File pdbFile = this.pdbFilter[currentAssembly].getFile();
                String aiName = FileUtils.relativePathTo((File)pdbFile).toString();
                if (this.pdbFilter[currentAssembly].writeFile(pdbFile, true, extraLines)) {
                    logger.log(this.basicLogging, String.format(" Appended to PDB file %s", aiName));
                } else {
                    logger.warning(String.format(" Appending to PDB file to %s failed.", aiName));
                }
            }
            ++currentAssembly;
        }
    }

    protected long logThermoForTime(long step, long time) {
        if (step % (long)this.logFrequency == 0L) {
            return this.logThermodynamics(time);
        }
        return time;
    }

    private long logThermodynamics(long time) {
        time = System.nanoTime() - time;
        logger.log(this.basicLogging, String.format(" %7.3e %12.4f %12.4f %12.4f %8.2f %8.3f", this.totalSimTime, this.state.getKineticEnergy(), this.state.getPotentialEnergy(), this.state.getTotalEnergy(), this.state.getTemperature(), (double)time * 1.0E-9));
        return System.nanoTime();
    }

    private void mainLoop() {
        long time = System.nanoTime();
        int removeCOMMotionFrequency = this.molecularAssembly[0].getForceField().getInteger("REMOVE-COM-FREQUENCY", Integer.valueOf(100));
        if (this.thermostat.getRemoveCenterOfMassMotion()) {
            logger.info(String.format(" COM will be removed every %3d step(s).", removeCOMMotionFrequency));
        }
        if (this.nonEquilibriumLambda) {
            this.nSteps = this.nonEquilibriumDynamics.setMDSteps(this.nSteps);
            LambdaInterface lambdaInterface = (LambdaInterface)this.potential;
            double lambda = this.nonEquilibriumDynamics.getInitialLambda();
            lambdaInterface.setLambda(lambda);
        }
        for (long step = 1L; step <= this.nSteps; ++step) {
            List constraints;
            long constraintFails2;
            if (this.nonEquilibriumLambda && this.nonEquilibriumDynamics.isUpdateStep(step)) {
                Potential.STATE respaState = this.potential.getEnergyTermState();
                if (this.integrator instanceof Respa && respaState != Potential.STATE.BOTH) {
                    this.potential.setEnergyTermState(Potential.STATE.BOTH);
                }
                LambdaInterface lambdaInterface = (LambdaInterface)this.potential;
                double currentLambda = lambdaInterface.getLambda();
                double currentEnergy = this.state.getPotentialEnergy();
                double newLambda = this.nonEquilibriumDynamics.getNextLambda(step, currentLambda);
                lambdaInterface.setLambda(newLambda);
                double newEnergy = this.potential.energy(this.state.x());
                double dW = newEnergy - currentEnergy;
                this.nonEquilibriumDynamics.addWork(dW);
                logger.info(String.format(" Non-equilibrium L=%5.3f Work=%12.6f", newLambda, this.nonEquilibriumDynamics.getWork()));
                if (this.integrator instanceof Respa) {
                    this.potential.setEnergyTermState(respaState);
                }
            }
            if (step > 1L && (constraintFails2 = (constraints = this.potential.getConstraints()).stream().filter(c -> !c.constraintSatisfied(this.state.x(), this.state.v(), 1.0E-7, 1.0E-7)).count()) > 0L) {
                logger.info(String.format(" %d constraint failures in step %d", constraintFails2, step));
            }
            this.thermostat.halfStep(this.dt);
            this.integrator.preForce(this.potential);
            if (this.esvSystem != null) {
                this.esvIntegrator.preForce(this.potential);
                this.esvSystem.preForce();
            }
            if (this.esvSystem != null && this.potential instanceof OpenMMEnergy) {
                this.state.setPotentialEnergy(((OpenMMEnergy)this.potential).energyAndGradientFFX(this.state.x(), this.state.gradient()));
            } else {
                this.state.setPotentialEnergy(this.potential.energyAndGradient(this.state.x(), this.state.gradient()));
            }
            Integrator constraintFails2 = this.integrator;
            if (constraintFails2 instanceof Respa) {
                Respa r = (Respa)constraintFails2;
                double potentialEnergy = this.state.getPotentialEnergy();
                this.state.setPotentialEnergy(potentialEnergy + r.getHalfStepEnergy());
            }
            this.integrator.postForce(this.state.gradient());
            if (this.esvSystem != null) {
                double[] dEdL = this.esvSystem.postForce();
                this.esvIntegrator.postForce(dEdL);
            }
            this.thermostat.computeKineticEnergy();
            this.thermostat.fullStep(this.dt);
            this.thermostat.computeKineticEnergy();
            if (this.esvSystem != null) {
                this.esvThermostat.computeKineticEnergy();
            }
            if (this.thermostat.getRemoveCenterOfMassMotion() && step % (long)removeCOMMotionFrequency == 0L) {
                this.thermostat.centerOfMassMotion(true, false);
            }
            if (this.esvSystem != null) {
                double kineticEnergy = this.thermostat.getKineticEnergy();
                double esvKineticEnergy = this.esvThermostat.getKineticEnergy();
                this.state.setKineticEnergy(kineticEnergy + esvKineticEnergy);
            }
            this.temperatureStats.addValue(this.state.getTemperature());
            this.potentialEnergyStats.addValue(this.state.getPotentialEnergy());
            this.kineticEnergyStats.addValue(this.state.getKineticEnergy());
            this.totalEnergyStats.addValue(this.state.getTotalEnergy());
            this.potential.setVelocity(this.state.v());
            this.potential.setAcceleration(this.state.a());
            this.potential.setPreviousAcceleration(this.state.aPrevious());
            this.totalSimTime += this.dt;
            time = this.logThermoForTime(step, time);
            if (step % (long)this.printEsvFrequency == 0L && this.esvSystem != null) {
                logger.log(this.basicLogging, String.format(" %s", this.esvSystem.getLambdaList()));
            }
            if (this.automaticWriteouts) {
                this.writeFilesForStep(step, true, true);
            }
            if (this.algorithmListener != null && step % (long)this.logFrequency == 0L) {
                for (MolecularAssembly assembly : this.molecularAssembly) {
                    this.algorithmListener.algorithmUpdate(assembly);
                }
            }
            if (!this.terminate) continue;
            logger.info(String.format("\n Terminating after %8d time steps\n", step));
            break;
        }
    }

    private void checkFrequency(String describe, int frequency) {
        if ((long)frequency > this.nSteps) {
            logger.fine(String.format(" Specified %s frequency of %d is greater than the number of steps %d", describe, frequency, this.nSteps));
        }
    }

    public void setCoordinates(double[] coords) {
        this.state.setCoordinates(coords);
    }

    public double[] getCoordinates() {
        return this.state.getCoordinatesCopy();
    }

    public void storeState() {
        this.storedState = this.state.getUnmodifiableState();
    }

    public void revertState() throws Exception {
        if (this.storedState == null) {
            throw new Exception();
        }
        this.revertState(this.storedState);
    }

    private void revertState(UnmodifiableState state) {
        this.state.revertState(state);
        this.potential.setVelocity(state.v());
        this.potential.setAcceleration(state.a());
        this.potential.setPreviousAcceleration(state.aPrevious());
        int numberOfVariables = this.state.getNumberOfVariables();
        Atom[] atoms = this.molecularAssembly[0].getActiveAtomArray();
        if (atoms.length * 3 == numberOfVariables) {
            int index = 0;
            double[] x = state.x();
            double[] gradient = state.gradient();
            for (Atom atom : atoms) {
                atom.moveTo(x[index], x[index + 1], x[index + 2]);
                atom.setXYZGradient(gradient[index], gradient[index + 1], gradient[index + 2]);
                index += 3;
            }
        }
    }
}

