/*
 * Decompiled with CFR 0.152.
 */
package ffx.algorithms.dynamics;

import ffx.crystal.Crystal;
import ffx.crystal.CrystalPotential;
import ffx.crystal.LatticeSystem;
import ffx.crystal.SpaceGroup;
import ffx.numerics.Potential;
import ffx.numerics.math.RunningStatistics;
import ffx.numerics.math.ScalarMath;
import ffx.potential.MolecularAssembly;
import ffx.potential.bonded.Atom;
import ffx.potential.parsers.XYZFilter;
import java.io.File;
import java.util.ArrayList;
import java.util.List;
import java.util.logging.Level;
import java.util.logging.Logger;
import org.apache.commons.io.FilenameUtils;
import org.apache.commons.math3.util.FastMath;

public class Barostat
implements CrystalPotential {
    private static final Logger logger = Logger.getLogger(Barostat.class.getName());
    private final MolecularAssembly molecularAssembly;
    private Crystal crystal;
    private Crystal unitCell;
    private final double mass;
    private final CrystalPotential potential;
    private final double[] x;
    private final int nSymm;
    private final SpaceGroup spaceGroup;
    private final int nMolecules;
    private double kT;
    private double pressure = 1.0;
    private boolean isotropic = false;
    private boolean active = true;
    private double maxAngleMove = 0.5;
    private double maxVolumeMove = 1.0;
    private double minDensity = 0.75;
    private double maxDensity = 1.6;
    private int meanBarostatInterval = 1;
    private int barostatCount = 0;
    private final RunningStatistics unitCellMoves = new RunningStatistics();
    private final RunningStatistics sideMoves = new RunningStatistics();
    private final RunningStatistics angleMoves = new RunningStatistics();
    private Potential.STATE state = Potential.STATE.BOTH;
    private MoveType moveType = MoveType.SIDE;
    private boolean moveAccepted = false;
    private double currentDensity = 0.0;
    private double a = 0.0;
    private double b = 0.0;
    private double c = 0.0;
    private double alpha = 0.0;
    private double beta = 0.0;
    private double gamma = 0.0;
    private final RunningStatistics aStats = new RunningStatistics();
    private final RunningStatistics bStats = new RunningStatistics();
    private final RunningStatistics cStats = new RunningStatistics();
    private final RunningStatistics alphaStats = new RunningStatistics();
    private final RunningStatistics betaStats = new RunningStatistics();
    private final RunningStatistics gammaStats = new RunningStatistics();
    private final RunningStatistics densityStats = new RunningStatistics();
    private int printFrequency = 1000;

    public Barostat(MolecularAssembly molecularAssembly, CrystalPotential potential) {
        this(molecularAssembly, potential, 298.15);
    }

    public Barostat(MolecularAssembly molecularAssembly, CrystalPotential potential, double temperature) {
        this.molecularAssembly = molecularAssembly;
        this.potential = potential;
        this.kT = temperature * 0.0019872042586408316;
        this.crystal = potential.getCrystal();
        this.unitCell = this.crystal.getUnitCell();
        this.spaceGroup = this.unitCell.spaceGroup;
        Atom[] atoms = molecularAssembly.getAtomArray();
        int nAtoms = atoms.length;
        this.nSymm = this.spaceGroup.getNumberOfSymOps();
        this.mass = molecularAssembly.getMass();
        this.x = new double[3 * nAtoms];
        this.nMolecules = molecularAssembly.fractionalCount();
    }

    public double density() {
        return this.unitCell.getDensity(this.mass);
    }

    public boolean destroy() {
        return this.potential.destroy();
    }

    public double energy(double[] x) {
        return this.potential.energy(x);
    }

    public double energyAndGradient(double[] x, double[] g) {
        double energy = this.potential.energyAndGradient(x, g);
        if (this.active && this.state != Potential.STATE.FAST && FastMath.random() < 1.0 / (double)this.meanBarostatInterval) {
            this.moveAccepted = false;
            this.applyBarostat(energy);
            this.collectStats();
            if (this.moveAccepted) {
                energy = this.potential.energyAndGradient(x, g);
            }
        }
        return energy;
    }

    public boolean isIsotropic() {
        return this.isotropic;
    }

    public void setIsotropic(boolean isotropic) {
        this.isotropic = isotropic;
    }

    public double[] getAcceleration(double[] acceleration) {
        return this.potential.getAcceleration(acceleration);
    }

    public double[] getCoordinates(double[] parameters) {
        return this.potential.getCoordinates(parameters);
    }

    public void setCoordinates(double[] parameters) {
        this.potential.setCoordinates(parameters);
    }

    public Crystal getCrystal() {
        return this.potential.getCrystal();
    }

    public void setCrystal(Crystal crystal) {
        this.potential.setCrystal(crystal);
    }

    public Potential.STATE getEnergyTermState() {
        return this.state;
    }

    public void setEnergyTermState(Potential.STATE state) {
        this.state = state;
        this.potential.setEnergyTermState(state);
    }

    public double[] getMass() {
        return this.potential.getMass();
    }

    public int getMeanBarostatInterval() {
        return this.meanBarostatInterval;
    }

    public void setMeanBarostatInterval(int meanBarostatInterval) {
        this.meanBarostatInterval = meanBarostatInterval;
    }

    public int getNumberOfVariables() {
        return this.potential.getNumberOfVariables();
    }

    public double getPressure() {
        return this.pressure;
    }

    public void setPressure(double pressure) {
        this.pressure = pressure;
    }

    public double[] getPreviousAcceleration(double[] previousAcceleration) {
        return this.potential.getPreviousAcceleration(previousAcceleration);
    }

    public double[] getScaling() {
        return this.potential.getScaling();
    }

    public void setScaling(double[] scaling) {
        this.potential.setScaling(scaling);
    }

    public double getTotalEnergy() {
        return this.potential.getTotalEnergy();
    }

    public List<Potential> getUnderlyingPotentials() {
        ArrayList<Potential> underlying = new ArrayList<Potential>();
        underlying.add((Potential)this.potential);
        underlying.addAll(this.potential.getUnderlyingPotentials());
        return underlying;
    }

    public CrystalPotential getCrystalPotential() {
        return this.potential;
    }

    public Potential.VARIABLE_TYPE[] getVariableTypes() {
        return this.potential.getVariableTypes();
    }

    public double[] getVelocity(double[] velocity) {
        return this.potential.getVelocity(velocity);
    }

    public boolean isActive() {
        return this.active;
    }

    public void setActive(boolean active) {
        this.active = active;
    }

    public void setAcceleration(double[] acceleration) {
        this.potential.setAcceleration(acceleration);
    }

    public void setDensity(double density) {
        this.molecularAssembly.computeFractionalCoordinates();
        this.crystal.setDensity(density, this.mass);
        this.potential.setCrystal(this.crystal);
        this.molecularAssembly.moveToFractionalCoordinates();
    }

    public void setMaxAngleMove(double maxAngleMove) {
        this.maxAngleMove = maxAngleMove;
    }

    public void setMaxVolumeMove(double maxVolumeMove) {
        this.maxVolumeMove = maxVolumeMove;
    }

    public void setMaxDensity(double maxDensity) {
        this.maxDensity = maxDensity;
    }

    public void setMinDensity(double minDensity) {
        this.minDensity = minDensity;
    }

    public void setBarostatPrintFrequency(int frequency) {
        this.printFrequency = frequency;
    }

    public void setPreviousAcceleration(double[] previousAcceleration) {
        this.potential.setPreviousAcceleration(previousAcceleration);
    }

    public void setTemperature(double temperature) {
        this.kT = temperature * 0.0019872042586408316;
    }

    public void setVelocity(double[] velocity) {
        this.potential.setVelocity(velocity);
    }

    private double mcStep(double currentE, double currentV) {
        double den = this.density();
        if (Double.isNaN(den) || Double.isNaN(this.a) || Double.isNaN(this.b) || Double.isNaN(this.c) || Double.isNaN(this.alpha) || Double.isNaN(this.beta) || Double.isNaN(this.gamma)) {
            logger.warning(String.format(" Found value of NaN: density: %7.4f a: %7.4f b: %7.4f c: %7.4f alpha: %7.4f beta: %7.4f gamma: %7.4f", den, this.a, this.b, this.c, this.alpha, this.beta, this.gamma));
            File errorFile = new File(FilenameUtils.removeExtension((String)this.molecularAssembly.getFile().getName()) + "_err.xyz");
            XYZFilter.version((File)errorFile);
            XYZFilter writeFilter = new XYZFilter(errorFile, this.molecularAssembly, this.molecularAssembly.getForceField(), this.molecularAssembly.getProperties());
            writeFilter.writeFile(errorFile, true, null);
        }
        if (den < this.minDensity || den > this.maxDensity) {
            if (logger.isLoggable(Level.FINE)) {
                logger.fine(String.format(" MC Density %10.6f is outside the range %10.6f - %10.6f.", den, this.minDensity, this.maxDensity));
            }
            this.crystal.changeUnitCellParameters(this.a, this.b, this.c, this.alpha, this.beta, this.gamma);
            return currentE;
        }
        double minInterfacialRadius = 1.2;
        double currentMinInterfacialRadius = FastMath.min((double)FastMath.min((double)this.unitCell.interfacialRadiusA, (double)this.unitCell.interfacialRadiusB), (double)this.unitCell.interfacialRadiusC);
        if (currentMinInterfacialRadius < minInterfacialRadius) {
            if (logger.isLoggable(Level.FINE)) {
                logger.fine(String.format(" MC An interfacial radius (%10.6f,%10.6f,%10.6f) is below the minimum %10.6f", this.unitCell.interfacialRadiusA, this.unitCell.interfacialRadiusB, this.unitCell.interfacialRadiusC, minInterfacialRadius));
            }
            this.crystal.changeUnitCellParameters(this.a, this.b, this.c, this.alpha, this.beta, this.gamma);
            return currentE;
        }
        this.potential.setCrystal(this.crystal);
        this.molecularAssembly.moveToFractionalCoordinates();
        double newV = this.unitCell.volume / (double)this.nSymm;
        this.potential.getCoordinates(this.x);
        double newE = this.potential.energy(this.x);
        double dPE = newE - currentE;
        double dEV = this.pressure * (newV - currentV) / 68568.4112;
        double dES = (double)(-this.nMolecules) * this.kT * FastMath.log((double)(newV / currentV));
        double dE = dPE + dEV + dES;
        if (dE < 0.0) {
            if (logger.isLoggable(Level.FINE)) {
                logger.fine(String.format(" MC Accept: %12.6f (dPE) + %12.6f (dEV) + %12.6f (dES) = %12.6f with (V=%12.6f, E=%12.6f)", dPE, dEV, dES, dE, newV, newE));
            }
            this.moveAccepted = true;
            switch (this.moveType.ordinal()) {
                case 0: {
                    this.sideMoves.addValue(1.0);
                    break;
                }
                case 1: {
                    this.angleMoves.addValue(1.0);
                    break;
                }
                case 2: {
                    this.unitCellMoves.addValue(1.0);
                }
            }
            return newE;
        }
        double acceptanceProbability = FastMath.exp((double)(-dE / this.kT));
        double metropolis = FastMath.random();
        if (metropolis > acceptanceProbability) {
            this.rejectMove();
            if (logger.isLoggable(Level.FINE)) {
                logger.fine(String.format(" MC Reject: %12.6f (dPE) + %12.6f (dEV) + %12.6f (dES) = %12.6f with (V=%12.6f, E=%12.6f)", dPE, dEV, dES, dE, currentV, currentE));
            }
            switch (this.moveType.ordinal()) {
                case 0: {
                    this.sideMoves.addValue(0.0);
                    break;
                }
                case 1: {
                    this.angleMoves.addValue(0.0);
                    break;
                }
                case 2: {
                    this.unitCellMoves.addValue(0.0);
                }
            }
            return currentE;
        }
        if (logger.isLoggable(Level.FINE)) {
            logger.fine(String.format(" MC Accept: %12.6f (dPE) + %12.6f (dEV) + %12.6f (dES) = %12.6f with (V=%12.6f, E=%12.6f)", dPE, dEV, dES, dE, newV, newE));
        }
        this.moveAccepted = true;
        switch (this.moveType.ordinal()) {
            case 0: {
                this.sideMoves.addValue(1.0);
                break;
            }
            case 1: {
                this.angleMoves.addValue(1.0);
                break;
            }
            case 2: {
                this.unitCellMoves.addValue(1.0);
            }
        }
        return newE;
    }

    private void rejectMove() {
        this.crystal.changeUnitCellParameters(this.a, this.b, this.c, this.alpha, this.beta, this.gamma);
        this.potential.setCrystal(this.crystal);
        this.molecularAssembly.moveToFractionalCoordinates();
    }

    private double mcA(double currentE) {
        double dVdA;
        this.moveType = MoveType.SIDE;
        double currentAUV = this.unitCell.volume / (double)this.nSymm;
        double dAUVolume = this.maxVolumeMove * (2.0 * FastMath.random() - 1.0);
        double dA = dAUVolume / (dVdA = this.unitCell.dVdA / (double)this.nSymm);
        boolean succeed = this.crystal.changeUnitCellParameters(this.a + dA, this.b, this.c, this.alpha, this.beta, this.gamma);
        if (succeed) {
            if (logger.isLoggable(Level.FINE)) {
                logger.fine(String.format(" Proposing MC change of the a-axis to (%6.3f) with volume change %6.3f (A^3)", this.a, dAUVolume));
            }
            return this.mcStep(currentE, currentAUV);
        }
        return currentE;
    }

    private double mcB(double currentE) {
        double dVdB;
        this.moveType = MoveType.SIDE;
        double currentAUV = this.unitCell.volume / (double)this.nSymm;
        double dAUVolume = this.maxVolumeMove * (2.0 * FastMath.random() - 1.0);
        double dB = dAUVolume / (dVdB = this.unitCell.dVdB / (double)this.nSymm);
        boolean succeed = this.crystal.changeUnitCellParameters(this.a, this.b + dB, this.c, this.alpha, this.beta, this.gamma);
        if (succeed) {
            if (logger.isLoggable(Level.FINE)) {
                logger.fine(String.format(" Proposing MC change of the b-axis to (%6.3f) with volume change %6.3f (A^3)", this.b, dAUVolume));
            }
            return this.mcStep(currentE, currentAUV);
        }
        return currentE;
    }

    private double mcC(double currentE) {
        double dVdC;
        this.moveType = MoveType.SIDE;
        double currentAUV = this.unitCell.volume / (double)this.nSymm;
        double dAUVolume = this.maxVolumeMove * (2.0 * FastMath.random() - 1.0);
        double dC = dAUVolume / (dVdC = this.unitCell.dVdC / (double)this.nSymm);
        boolean succeed = this.crystal.changeUnitCellParameters(this.a, this.b, this.c + dC, this.alpha, this.beta, this.gamma);
        if (succeed) {
            if (logger.isLoggable(Level.FINE)) {
                logger.fine(String.format(" Proposing MC change to the c-axis to (%6.3f) with volume change %6.3f (A^3)", this.c, dAUVolume));
            }
            return this.mcStep(currentE, currentAUV);
        }
        return currentE;
    }

    private double mcAB(double currentE) {
        double dVdAB;
        this.moveType = MoveType.SIDE;
        double currentAUV = this.unitCell.volume / (double)this.nSymm;
        double dAUVolume = this.maxVolumeMove * (2.0 * FastMath.random() - 1.0);
        double dAB = dAUVolume / (dVdAB = (this.unitCell.dVdA + this.unitCell.dVdB) / (double)this.nSymm);
        boolean succeed = this.crystal.changeUnitCellParametersAndVolume(this.a + dAB, this.b + dAB, this.c, this.alpha, this.beta, this.gamma, currentAUV + dAUVolume);
        if (succeed) {
            if (logger.isLoggable(Level.FINE)) {
                logger.fine(String.format(" Proposing MC change to the a,b-axes to (%6.3f) with volume change %6.3f (A^3)", this.a, dAUVolume));
            }
            return this.mcStep(currentE, currentAUV);
        }
        return currentE;
    }

    private double mcABC(double currentE) {
        double dVdABC;
        this.moveType = MoveType.SIDE;
        double currentAUV = this.unitCell.volume / (double)this.nSymm;
        double dAUVolume = this.maxVolumeMove * (2.0 * FastMath.random() - 1.0);
        double dABC = dAUVolume / (dVdABC = (this.unitCell.dVdA + this.unitCell.dVdB + this.unitCell.dVdC) / (double)this.nSymm);
        boolean succeed = this.crystal.changeUnitCellParametersAndVolume(this.a + dABC, this.b + dABC, this.c + dABC, this.alpha, this.beta, this.gamma, currentAUV + dAUVolume);
        if (succeed) {
            if (logger.isLoggable(Level.FINE)) {
                logger.fine(String.format(" Proposing MC change to the a,b,c-axes to (%6.3f) with volume change %6.3f (A^3)", this.a, dAUVolume));
            }
            return this.mcStep(currentE, currentAUV);
        }
        return currentE;
    }

    private double mcAlpha(double currentE) {
        this.moveType = MoveType.ANGLE;
        double move = this.maxAngleMove * (2.0 * FastMath.random() - 1.0);
        double currentAUV = this.unitCell.volume / (double)this.nSymm;
        double newAlpha = ScalarMath.mirrorDegrees((double)(this.alpha + move));
        boolean succeed = this.crystal.changeUnitCellParametersAndVolume(this.a, this.b, this.c, newAlpha, this.beta, this.gamma, currentAUV);
        if (succeed) {
            if (logger.isLoggable(Level.FINE)) {
                logger.fine(String.format(" Proposing MC change of alpha to %6.3f.", newAlpha));
            }
            return this.mcStep(currentE, currentAUV);
        }
        return currentE;
    }

    private double mcBeta(double currentE) {
        this.moveType = MoveType.ANGLE;
        double move = this.maxAngleMove * (2.0 * FastMath.random() - 1.0);
        double currentAUV = this.unitCell.volume / (double)this.nSymm;
        double newBeta = ScalarMath.mirrorDegrees((double)(this.beta + move));
        boolean succeed = this.crystal.changeUnitCellParametersAndVolume(this.a, this.b, this.c, this.alpha, newBeta, this.gamma, currentAUV);
        if (succeed) {
            if (logger.isLoggable(Level.FINE)) {
                logger.fine(String.format(" Proposing MC change of beta to %6.3f.", newBeta));
            }
            return this.mcStep(currentE, currentAUV);
        }
        return currentE;
    }

    private double mcGamma(double currentE) {
        this.moveType = MoveType.ANGLE;
        double move = this.maxAngleMove * (2.0 * FastMath.random() - 1.0);
        double currentAUV = this.unitCell.volume / (double)this.nSymm;
        double newGamma = ScalarMath.mirrorDegrees((double)(this.gamma + move));
        boolean succeed = this.crystal.changeUnitCellParametersAndVolume(this.a, this.b, this.c, this.alpha, this.beta, newGamma, currentAUV);
        if (succeed) {
            if (logger.isLoggable(Level.FINE)) {
                logger.fine(String.format(" Proposing MC change of gamma to %6.3f.", newGamma));
            }
            return this.mcStep(currentE, currentAUV);
        }
        return currentE;
    }

    private double mcABG(double currentE) {
        double newGamma;
        double newBeta;
        this.moveType = MoveType.ANGLE;
        double move = this.maxAngleMove * (2.0 * FastMath.random() - 1.0);
        double currentAUV = this.unitCell.volume / (double)this.nSymm;
        double newAlpha = ScalarMath.mirrorDegrees((double)(this.alpha + move));
        boolean succeed = this.crystal.changeUnitCellParametersAndVolume(this.a, this.b, this.c, newAlpha, newBeta = ScalarMath.mirrorDegrees((double)(this.beta + move)), newGamma = ScalarMath.mirrorDegrees((double)(this.gamma + move)), currentAUV);
        if (succeed) {
            if (logger.isLoggable(Level.FINE)) {
                logger.fine(String.format(" Proposing MC change to all angles to %6.3f.", newAlpha));
            }
            return this.mcStep(currentE, currentAUV);
        }
        return currentE;
    }

    private double applyBarostat(double currentE) {
        this.molecularAssembly.computeFractionalCoordinates();
        this.crystal = this.potential.getCrystal();
        this.unitCell = this.crystal.getUnitCell();
        this.a = this.unitCell.a;
        this.b = this.unitCell.b;
        this.c = this.unitCell.c;
        this.alpha = this.unitCell.alpha;
        this.beta = this.unitCell.beta;
        this.gamma = this.unitCell.gamma;
        if (this.isotropic) {
            currentE = this.mcABC(currentE);
        } else {
            block0 : switch (this.spaceGroup.latticeSystem) {
                case MONOCLINIC_LATTICE: {
                    int move = (int)FastMath.floor((double)(FastMath.random() * 4.0));
                    switch (move) {
                        case 0: {
                            currentE = this.mcA(currentE);
                            break block0;
                        }
                        case 1: {
                            currentE = this.mcB(currentE);
                            break block0;
                        }
                        case 2: {
                            currentE = this.mcC(currentE);
                            break block0;
                        }
                        case 3: {
                            currentE = this.mcBeta(currentE);
                            break block0;
                        }
                    }
                    logger.severe(" Barostat programming error.");
                    break;
                }
                case ORTHORHOMBIC_LATTICE: {
                    int move = (int)FastMath.floor((double)(FastMath.random() * 3.0));
                    switch (move) {
                        case 0: {
                            currentE = this.mcA(currentE);
                            break block0;
                        }
                        case 1: {
                            currentE = this.mcB(currentE);
                            break block0;
                        }
                        case 2: {
                            currentE = this.mcC(currentE);
                            break block0;
                        }
                    }
                    logger.severe(" Barostat programming error.");
                    break;
                }
                case TETRAGONAL_LATTICE: {
                    int move = (int)FastMath.floor((double)(FastMath.random() * 2.0));
                    switch (move) {
                        case 0: {
                            currentE = this.mcAB(currentE);
                            break block0;
                        }
                        case 1: {
                            currentE = this.mcC(currentE);
                            break block0;
                        }
                    }
                    logger.severe(" Barostat programming error.");
                    break;
                }
                case RHOMBOHEDRAL_LATTICE: {
                    int move = (int)FastMath.floor((double)(FastMath.random() * 2.0));
                    switch (move) {
                        case 0: {
                            currentE = this.mcABC(currentE);
                            break block0;
                        }
                        case 1: {
                            currentE = this.mcABG(currentE);
                            break block0;
                        }
                    }
                    logger.severe(" Barostat programming error.");
                    break;
                }
                case HEXAGONAL_LATTICE: {
                    int move = (int)FastMath.floor((double)(FastMath.random() * 2.0));
                    switch (move) {
                        case 0: {
                            currentE = this.mcAB(currentE);
                            break block0;
                        }
                        case 1: {
                            currentE = this.mcC(currentE);
                            break block0;
                        }
                    }
                    logger.severe(" Barostat programming error.");
                    break;
                }
                case CUBIC_LATTICE: {
                    currentE = this.mcABC(currentE);
                    break;
                }
                case TRICLINIC_LATTICE: {
                    if (LatticeSystem.check((double)this.a, (double)this.b) && LatticeSystem.check((double)this.b, (double)this.c) && LatticeSystem.check((double)this.alpha, (double)90.0) && LatticeSystem.check((double)this.beta, (double)90.0) && LatticeSystem.check((double)this.gamma, (double)90.0)) {
                        currentE = this.mcABC(currentE);
                        break;
                    }
                    int move = (int)FastMath.floor((double)(FastMath.random() * 6.0));
                    switch (move) {
                        case 0: {
                            currentE = this.mcA(currentE);
                            break block0;
                        }
                        case 1: {
                            currentE = this.mcB(currentE);
                            break block0;
                        }
                        case 2: {
                            currentE = this.mcC(currentE);
                            break block0;
                        }
                        case 3: {
                            currentE = this.mcAlpha(currentE);
                            break block0;
                        }
                        case 4: {
                            currentE = this.mcBeta(currentE);
                            break block0;
                        }
                        case 5: {
                            currentE = this.mcGamma(currentE);
                            break block0;
                        }
                    }
                    logger.severe(" Barostat programming error.");
                }
            }
        }
        this.currentDensity = this.density();
        if (this.moveAccepted) {
            if (logger.isLoggable(Level.FINE)) {
                StringBuilder sb = new StringBuilder(" MC Barostat Acceptance:");
                if (this.sideMoves.getCount() > 0L) {
                    sb.append(String.format(" Side %5.1f%%", this.sideMoves.getMean() * 100.0));
                }
                if (this.angleMoves.getCount() > 0L) {
                    sb.append(String.format(" Angle %5.1f%%", this.angleMoves.getMean() * 100.0));
                }
                if (this.unitCellMoves.getCount() > 0L) {
                    sb.append(String.format(" UC %5.1f%%", this.unitCellMoves.getMean() * 100.0));
                }
                sb.append(String.format("\n Density: %5.3f  UC: %s", this.currentDensity, this.unitCell.toShortString()));
                logger.fine(sb.toString());
            }
        } else if (this.unitCell.a != this.a || this.unitCell.b != this.b || this.unitCell.c != this.c || this.unitCell.alpha != this.alpha || this.unitCell.beta != this.beta || this.unitCell.gamma != this.gamma) {
            logger.severe(" Reversion of unit cell parameters did not succeed after failed Barostat MC move.");
        }
        return currentE;
    }

    private void collectStats() {
        ++this.barostatCount;
        if (Double.isNaN(this.currentDensity) || Double.isNaN(this.a) || Double.isNaN(this.b) || Double.isNaN(this.c) || Double.isNaN(this.alpha) || Double.isNaN(this.beta) || Double.isNaN(this.gamma)) {
            logger.warning(String.format(" Statistic Value was NaN: Density: %5.3f A: %5.3f B: %5.3f C: %5.3f Alpha: %5.3f Beta: %5.3f Gamma: %5.3f", this.currentDensity, this.a, this.b, this.c, this.alpha, this.beta, this.gamma));
        }
        this.densityStats.addValue(this.currentDensity);
        this.aStats.addValue(this.a);
        this.bStats.addValue(this.b);
        this.cStats.addValue(this.c);
        this.alphaStats.addValue(this.alpha);
        this.betaStats.addValue(this.beta);
        this.gammaStats.addValue(this.gamma);
        if (this.barostatCount % this.printFrequency == 0) {
            logger.info(String.format(" Barostat statistics for the last %d moves:", this.printFrequency));
            logger.info(String.format("  Density: %5.3f\u00b1%5.3f with range %5.3f .. %5.3f (g/cc)", this.densityStats.getMean(), this.densityStats.getStandardDeviation(), this.densityStats.getMin(), this.densityStats.getMax()));
            this.densityStats.reset();
            logger.info(String.format("  Lattice a-axis: %5.2f\u00b1%3.2f b-axis: %5.2f\u00b1%3.2f c-axis: %5.2f\u00b1%3.2f", this.aStats.getMean(), this.aStats.getStandardDeviation(), this.bStats.getMean(), this.bStats.getStandardDeviation(), this.cStats.getMean(), this.cStats.getStandardDeviation()));
            logger.info(String.format("          alpha:  %5.2f\u00b1%3.2f beta:   %5.2f\u00b1%3.2f gamma:  %5.2f\u00b1%3.2f", this.alphaStats.getMean(), this.alphaStats.getStandardDeviation(), this.betaStats.getMean(), this.betaStats.getStandardDeviation(), this.gammaStats.getMean(), this.gammaStats.getStandardDeviation()));
            this.aStats.reset();
            this.bStats.reset();
            this.cStats.reset();
            this.alphaStats.reset();
            this.betaStats.reset();
            this.gammaStats.reset();
            if (this.sideMoves.getCount() > 0L) {
                logger.info(String.format("  Axis length moves: %4d (%5.2f%%)", this.sideMoves.getCount(), this.sideMoves.getMean() * 100.0));
                this.sideMoves.reset();
            }
            if (this.angleMoves.getCount() > 0L) {
                logger.info(String.format("  Angle moves:       %4d (%5.2f%%)", this.angleMoves.getCount(), this.angleMoves.getMean() * 100.0));
                this.angleMoves.reset();
            }
            if (this.unitCellMoves.getCount() > 0L) {
                logger.info(String.format("  Unit cell moves:   %4d (%5.2f%%)", this.unitCellMoves.getCount(), this.unitCellMoves.getMean() * 100.0));
                this.unitCellMoves.reset();
            }
        }
    }

    private static enum MoveType {
        SIDE,
        ANGLE,
        UNIT;

    }
}

